/************************************************************************
 * SGA - A C++ library to help develop Simple Genetic Algorithms        *
 * Copyright (C) 2005 Dorival M. Pedroso                                *
 *                                                                      *
 * This program is free software: you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation, either version 3 of the License, or    *
 * any later version.                                                   *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the         *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program. If not, see <http://www.gnu.org/licenses/>  *
 ************************************************************************/


/* Each individual corresponds to a set with all circles.
 * The DNA of an individual contains the circle information,
 * organized as follows (row-major):
 *
 *  NBASES = 1:
 *    x0 y0 r0   x1 y1 r1   x2 y2 r2  ==
 *   _          _
 *  |  x0 y0 r0  |
 *  |  x1 y1 r1  |
 *  |    ....
 *  |_ xN yN rN _|   N => N circles
 *
 *  NBASES = 2:
 *    x00 x01 y00 y01 r00 r01   x10 x11 y10 y11 r10 r11   x20 x21 y20 y21 r20 r21  ==
 *   _                           _
 *  |  x00 x01  y00 y01  r00 r01  |
 *  |  x10 y11  y10 y11  r10 r11  |
 *  |            ....
 *  |_ xN0 xN1  yN0 yN1  rN0 rN1 _|   N => N circles
 *
 * Check the Representation for further details.
 */


// STL
#include <iostream>
#include <fstream>
#include <sstream>
#include <cmath>    // for pow(x,a) == x^a

// FLTK
#include <FL/Fl.H>
#include <FL/Fl_Button.H>
#include <FL/Fl_Input.H>
#include <FL/fl_draw.H>
#include <FL/Fl_Double_Window.H>

// OpenMPI
#include <mpi.h>

// SGA
#include "random.h"
#include "numstreams.h"
#include "triangleio.h"
#include "truss2d.h"

// Constants
const uint32_t NBASES          = 10;    // Number of bases in the DNA
const double   PI              = 3.141592653589793238462;
const bool     IS_MINIMIZATION = true;
const double   XMIN            = 0.0;
const double   XMAX            = 1.0;
const double   YMIN            = 0.0;
const double   YMAX            = 1.0;
const uint32_t TAG_FINISHED    = 1111; // Tag for Send/recv finished variable
const uint32_t TAG_PRMS1       = 2222; // Tag for Send/recv prms1
const uint32_t TAG_PRMS2       = 3333; // Tag for Send/recv prms2
const uint32_t TAG_BEST_OV     = 4444; // Tag for Send/recv best individual
const uint32_t TAG_BEST_IND    = 5555; // Tag for Send/recv best individual

// Global Variables (set by root which will send to all other processes)
double   gTotVol;    // Total volume
double   gVoidRatio; // Void ratio
double   gMinRadius; // Minimum radius of circles
double   gMaxRadius; // Maximum radius of circles
uint32_t gNCircles;  // Number of circles (NC)     NOTE: The minimum of circles must be 3
uint32_t gNIndiv;    // Number of individuals
uint32_t gEliteSize; // Number of individuals in the elite
uint32_t gMaxNCross; // Maximum number of sections during crossover
double   gProbCross; // Probability of crossover
double   gProbMut;   // Probability of mutation
uint32_t gNGenerat;  // Number of generations
uint32_t gRefInterv; // Interval for refinements
uint32_t gMigInterv; // Migration interval
uint32_t gNIndivMig; // Number of individuals that migrate
uint32_t gRptInterv; // Report interval
uint32_t gNGenes;    // == 3*NBASES*gNCircles, ex.: x0 y0 r0  x1 y1 r1  x2 y2 r2, for NBASES=1

// Memory for refinements
bool     * gEp;     // size = gNCircles*2
double   * gNbc;    // size = gNCircles*2
double   * gRadii;  // size = gNCircles
uint32_t * gIndIdx; // size = gNIndiv

/** Representation of the DNA.
 * G[] corresponds to ONE individual.
 *
 * Example:
 *  NBASES = 4
 *
 *  DNA of ONE individual:
 *   0 : x00 x01 x02 x03   y00 y01 y02 y03   r00 r01 r02 r03 : circle# 0
 *  12 : x10 x11 x12 x13   y10 y11 y12 y13   r10 r11 r12 r13 : circle# 1
 *  24 : x20 x21 x22 x23   y20 y21 y22 y23   r20 r21 r22 r23 : circle# 2
 *   ^     0   1   2   3     4   5   6   7     8   9  10  11
 *   |
 *  iCircle*3*NBASES, where 3 is due to x, y and r values
 *
 *  circle# 0:             circle# 1:             circle# 2:
 *   x0 = x00+x01+x02+x03   x1 = x10+x11+x12+x13   x2 = x20+x21+x22+x23
 *   y0 = y00+y01+y02+y03   y1 = y10+y11+y12+y13   y2 = y20+y21+y22+y23
 *   r0 = r00+r01+r02+r03   r1 = r10+r11+r12+r13   r2 = r20+r21+r22+r23
 */
inline void Representation(double const G[], uint32_t iCircle, double & xc, double & yc, double & r)
{ // {{{

	xc=0.0; for (uint32_t j=       0; j<  NBASES; ++j) xc += G[iCircle*3*NBASES+j]; // 3 => x y r
	yc=0.0; for (uint32_t j=  NBASES; j<2*NBASES; ++j) yc += G[iCircle*3*NBASES+j]; // 3 => x y r
	r =0.0; for (uint32_t j=2*NBASES; j<3*NBASES; ++j) r  += G[iCircle*3*NBASES+j]; // 3 => x y r

} // }}}

/** Fill arrays Xc, Yc, and R with the values of an individual.
 * size(Xc) = size(Yc) = size(R) == gNCircles
 * NOTE: these arrays must be allocated before.
 */
inline void Indiv2Arrays(double const G[], double * Xc, double * Yc, double * R)
{ // {{{

	double xc, yc, r;
	for (uint32_t i=0; i<gNCircles; ++i)
	{
		Representation (G,i,xc,yc,r);
		Xc[i] = xc;
		Yc[i] = yc;
		R [i] = r;
	}

} // }}}

//////////////////////////////////////////////////////////////////////////////////////////////// Callbacks /////

// The following two functions must be declared before: #include "island.h"
inline void MutationFun (double & OldGene)
{ // {{{

	if (Rnd::Flip(0.5)) OldGene /= 2.0;
	else                OldGene *= 2.0;

} // }}}

inline double ObjectiveFun (double const G[], uint32_t nGenes) // G[] corresponds to ONE individual
{ // {{{

	// Variables
	double xc, yc, r;      // center and radius
	double xck, yck, rk;   // center and radius of another circle
	double ld, rd, bd, ud; // left, right, bottom and upper drifts
	double dij, eij;       // distance and difference between the sum of radius and the distance of two circles

	// Loop over all circles
	double sum_r  = 0.0; // Sum of all radii
	double Vs     = 0.0; // Volume of solids == sum area of circles
	double err_xy = 0.0; // Error on position == sum of drift of all circles
	double err_di = 0.0; // Error on distances between circles
	double err_ra = 0.0; // Error on radius
	double err_co = 0.0; // Error on lack of contact (only for circles closer to the diameter of one cirlce)
	for (uint32_t i=0; i<gNCircles; ++i)
	{
		// Geometry 
		Representation (G,i,xc,yc,r);
		sum_r += r;       // Sum of all radii
		Vs    += r*r;     // Sum volume of solids
		// Error on position
		ld = XMIN-(xc-r); // left drift
		rd = (xc+r)-XMAX; // right drift
		bd = YMIN-(yc-r); // bottom drift
		ud = (yc+r)-YMAX; // upper drift
		if (ld>0) err_xy += ld;
		if (rd>0) err_xy += rd;
		if (bd>0) err_xy += bd;
		if (ud>0) err_xy += ud;
		// Error on distances
		for (uint32_t k=i+1; k<gNCircles; ++k)
		{
			Representation (G,k,xck,yck,rk);
			dij = sqrt(pow(xc-xck,2.0)+pow(yc-yck,2.0));
			eij = r+rk-dij;
			if ((dij<=2*r) && (eij<0)) err_co += fabs(eij);
			if (eij>0) err_di += eij;
		}
		// Error on radius
		if (r<gMinRadius) err_ra += gMinRadius-r;
		if (r>gMaxRadius) err_ra += r-gMaxRadius;
	}
	Vs *= PI;                                             // Volume of solids
	double vr     = gTotVol/Vs - 1.0;                     // Current void ratio
	double err_vr = fabs(vr - gVoidRatio);                // Error on void ratio
	return err_vr + (err_xy + 1.5*err_di + err_co + 0.8*err_ra);// /sum_r; // Minimization problem

} // }}}

#include "island.h"
#include "evolver.h"

/** Generate random population
 * size(max) = size(min) == gNGenes == 3*NBASES*gNCircles
 * NOTE: these arrays must be allocated before.
 */
inline void GenerateRandom(SGA::Population<double> & Pop)
{ // {{{

	double     ra  = (gMinRadius+gMaxRadius)/2.0; // average radius
	uint32_t   nx  = static_cast<uint32_t>(sqrt(gNCircles)+1);
	uint32_t   ny  = static_cast<uint32_t>(sqrt(gNCircles)+1);
	double     dx  = (XMAX-XMIN)/nx;
	double     dy  = (YMAX-YMIN)/ny;
	double   * Xc  = new double   [nx*ny];
	double   * Yc  = new double   [ny*ny];
	uint32_t * idx = new uint32_t [ny*ny];

	uint32_t n = 0;
	for (uint32_t j=0; j<ny; ++j)
	{
		for (uint32_t i=0; i<nx; ++i)
		{
			double  alp = (Rnd::Flip(0.5)?0.0:1.0);
			double  bet = Rnd::T<double>::Gen(0.8,1.2);
			Xc [j*nx+i] = XMIN+ra*alp+i*dx*bet;
			Yc [j*nx+i] = YMIN+ra*alp+j*dy*bet;
			idx[j*nx+i] = n;
			n++;
		}
	}

	for (uint32_t i=0; i<gNIndiv; ++i)
	{
		Rnd::Shuffle(nx*ny, idx);
		uint32_t m = 0;
		for (uint32_t j=0; j<gNCircles; ++j)
		{
			for (uint32_t k=       0; k<  NBASES; ++k) Pop(i,j*3*NBASES+k) = Xc[idx[m]]/NBASES;
			for (uint32_t k=  NBASES; k<2*NBASES; ++k) Pop(i,j*3*NBASES+k) = Yc[idx[m]]/NBASES;
			for (uint32_t k=2*NBASES; k<3*NBASES; ++k) Pop(i,j*3*NBASES+k) = (Rnd::Flip(0.5)?gMinRadius:gMaxRadius)/NBASES;
			m++;
		}
	}

	delete [] Xc;
	delete [] Yc;
	delete [] idx;

} // }}}

inline double RefineStep (uint32_t type, double * Points, Truss2D & Truss)
{ // {{{

	// Clear previous U, F and Fint values
	Truss.Initialize ();

	// Calculate "internal" forces due to Circles under/overlapping
	//Truss.CirclesForce (1.0, gRadii, gNbc, /*RepulsionOnly*/ +1);
	//Truss.CirclesForce (1.0, gRadii, gNbc, /*RepulsionOnly*/ -1);
	Truss.CirclesForce (1.0, gRadii, gNbc, /*Both*/type);

	// Solve for these "internal" forces
	Truss.Solve ();

	// Update mesh
	for (uint32_t j=0; j<2*gNCircles; ++j) Points[j] += Truss.U()[j];

	// Calculate error
	//double error = Truss.CirclesError (gRadii, /*RepulsionOnly*/ +1);
	//std::cout << "Error = " << error << std::endl;

	return 0;

} // }}}

/** Refine one individual. */
inline void RefineInd (uint32_t type, uint32_t nIt, double G[])
{ // {{{

	// Triangulate
	TriangleIO in(gNCircles);
	TriangleIO out;
	double xc, yc, r; // center and radius
	for (uint32_t j=0; j<gNCircles; ++j)
	{
		Representation (G,j,xc,yc,r);
		in.R().pointlist[j*2  ] = xc;
		in.R().pointlist[j*2+1] = yc;
		gRadii          [j]     = r;
	}
	triangulate("Qze",in.P(),out.P(),NULL);

	// Solve FEM
	Truss2D truss (gNCircles, out.R().numberofedges, in.R().pointlist, out.R().edgelist, gEp, gNbc);
	for (uint32_t j=0; j<nIt; ++j) RefineStep (type, in.R().pointlist, truss);
	
	// Set new coordinates
	for (uint32_t j=0; j<gNCircles; ++j)
	{
		for (uint32_t k=     0; k<  NBASES; ++k) G[j*3*NBASES+k] = in.R().pointlist[j*2  ]/NBASES;
		for (uint32_t k=NBASES; k<2*NBASES; ++k) G[j*3*NBASES+k] = in.R().pointlist[j*2+1]/NBASES;
	}

} // }}}

/** Refine a population. */
inline void RefinePop(uint32_t type, uint32_t nIt, uint32_t nIndivToRefine, SGA::Population<double> & Pop)
{ // {{{

	Rnd::Shuffle (gNIndiv, gIndIdx);
	for (uint32_t i=0; i<nIndivToRefine; ++i)
		RefineInd (type, nIt, Pop(gIndIdx[i]));

} // }}}

/////////////////////////////////////////////////////////////////////////////////////////////// MainWindow /////

// MainWindow
class MainWindow : public Fl_Double_Window
{
public:
	// DrawArea structure
	class DrawArea : public Fl_Widget
	{ // {{{
	public:
		// Constructor
		DrawArea (int    xmin, int    ymin, int    width, int    height, // Screen coordinates
		          double Xmin, double Ymin, double Width, double Height) // Real coordinates
			: Fl_Widget (xmin,ymin,width,height,0),
			  _Xmin     (Xmin),
			  _Ymin     (Ymin),
			  _Width    (Width),
			  _Height   (Height),
			  _ind      (NULL)
		{}

		// Methods
		void SetBestIndividual (uint32_t iGen, double const * Ind) { _igen=iGen; _ind=Ind; }

		// Internal methods
		void draw()
		{
			if (_ind!=NULL)
			{
				// Scale factor
				_sf = (w()<h() ? static_cast<double>(w()-2/_Width) : static_cast<double>(h()-2/_Height));

				// Clear the background
				fl_color (FL_BACKGROUND_COLOR);
				fl_rectf (x(),y(),w(),h());

				// Draw circles
				double Xc,Yc,R;
				for (uint32_t i=0; i<gNCircles; ++i)
				{
					Representation (_ind,i,Xc,Yc,R);
					int xc = _x(Xc);
					int yc = _y(Yc);
					int r  = _l(R );
					fl_color  (FL_RED);
					fl_pie    (xc-r, yc-r, 2*r+1, 2*r+1, 0,360);
					fl_color  (FL_BLACK);
					fl_circle (xc, yc, r);
				}
			}
		}

	private:
		// Data
		double         _Xmin;   // Real coordinates
		double         _Ymin;   // Real coordinates
		double         _Width;  // Real coordinates: Xmax - Xmin
		double         _Height; // Real coordinates: Ymax - Ymin
		double         _sf;     // Scale factor: x=xmin+(X-Xmin)*sf and y=ymin+ymax-(Y-Ymin)*sf, where x and y are screen coordinates
		uint32_t       _cseed;  // Color seed
		uint32_t       _igen;   // Index of a generation
		double const * _ind;    // Each individual corresponds to a set of circles (size==gNGenes==3*gNCircles)

		// Private methods
		inline int _x (double   X) const { return static_cast<int>((x()+1)        +_sf*(X-_Xmin)); } // X in screen coordinates
		inline int _y (double   Y) const { return static_cast<int>((y()+1)+(h()-2)-_sf*(Y-_Ymin)); } // Y in screen coordinates
		inline int _l (double   L) const { return static_cast<int>(_sf*L)                        ; } // Length in screen coordinates

	}; // class DrawArea }}}

	// Constructor
	MainWindow(uint32_t width=1024, uint32_t height=824) : Fl_Double_Window (width,height,"Circles packing using Genetic Algorithms")
	{ // {{{

		// Initialize global variables
		gTotVol    = 1;
		gVoidRatio = 0.5;
		gMinRadius = 0.01;
		gMaxRadius = 0.1;
		gNCircles  = 100;
		gNGenes    = 3*NBASES*gNCircles;
		gNIndiv    = 100;
		gEliteSize = 20;
		gProbCross = 1.0;
		gProbMut   = 0.5;
		gNGenerat  = 1000;
		gRefInterv = 50;
		gMigInterv = 30;
		gNIndivMig = 5;
		gRptInterv = 5;

		// Add widgets to window
		begin();
			// Inputs
			_wTotVol    = new Fl_Input (100,   0, 100, 30, "TotVol    = ");
			_wVoidRatio = new Fl_Input (100,  30, 100, 30, "VoidRatio = ");
			_wMinRadius = new Fl_Input (100,  60, 100, 30, "MinRadius = ");
			_wMaxRadius = new Fl_Input (100,  90, 100, 30, "MaxRadius = ");
			_wNCircles  = new Fl_Input (100, 120, 100, 30, "NCircles  = ");
			_wNIndiv    = new Fl_Input (100, 150, 100, 30, "NIndiv    = ");
			_wEliteSize = new Fl_Input (100, 180, 100, 30, "EliteSize = ");
			_wProbCross = new Fl_Input (100, 210, 100, 30, "ProbCross = ");
			_wProbMut   = new Fl_Input (100, 240, 100, 30, "ProbMut   = ");
			_wNGenerat  = new Fl_Input (100, 270, 100, 30, "NGenerat  = ");
			_wRefInterv = new Fl_Input (100, 300, 100, 30, "RefInterv = ");
			_wMigInterv = new Fl_Input (100, 330, 100, 30, "MigInterv = ");
			_wNIndivMig = new Fl_Input (100, 360, 100, 30, "NIndivMig = ");
			_wRptInterv = new Fl_Input (100, 390, 100, 30, "RptInterv = ");
			char buf[256];
			snprintf(buf, 256, "%f", gTotVol   ); _wTotVol   ->value(buf);
			snprintf(buf, 256, "%f", gVoidRatio); _wVoidRatio->value(buf);
			snprintf(buf, 256, "%f", gMinRadius); _wMinRadius->value(buf);
			snprintf(buf, 256, "%f", gMaxRadius); _wMaxRadius->value(buf);
			snprintf(buf, 256, "%d", gNCircles ); _wNCircles ->value(buf);
			snprintf(buf, 256, "%d", gNIndiv   ); _wNIndiv   ->value(buf);
			snprintf(buf, 256, "%d", gEliteSize); _wEliteSize->value(buf);
			snprintf(buf, 256, "%f", gProbCross); _wProbCross->value(buf);
			snprintf(buf, 256, "%f", gProbMut  ); _wProbMut  ->value(buf);
			snprintf(buf, 256, "%d", gNGenerat ); _wNGenerat ->value(buf);
			snprintf(buf, 256, "%d", gRefInterv); _wRefInterv->value(buf);
			snprintf(buf, 256, "%d", gMigInterv); _wMigInterv->value(buf);
			snprintf(buf, 256, "%d", gNIndivMig); _wNIndivMig->value(buf);
			snprintf(buf, 256, "%d", gRptInterv); _wRptInterv->value(buf);
			// Buttons
			_wquit = new Fl_Button(  0, 420, 100, 30, "&Quit");
			_wrun  = new Fl_Button(100, 420, 100, 30, "&Run" );
			_wquit->callback(_quit_cb, this);
			_wrun ->callback(_run_cb , this);
			// Draw area
			double el = pow(gTotVol,1.0/2.0); // Edge length (should be 1/3, but we are using the area)
			_wda = new DrawArea(/*xmin*/200, /*ymin*/0  , /*width*/w()-200, /*height*/h(),     // 200 = 100 from label + 100 from input box
			                    /*Xmin*/0.0, /*Ymin*/0.0, /*Width*/el-0.0 , /*Height*/el-0.0); // -0.0 => -Xmin and -Xmax
		end();

		// Set focus
		_wrun->take_focus();

		// Set resizable and show window
		//resizable(this);
		resizable(_wda);
		show();

	} // }}}

	// Destructor
	~MainWindow()
	{ // {{{
		delete _wTotVol;   
		delete _wVoidRatio;
		delete _wNCircles; 
		delete _wNIndiv;   
		delete _wNGenerat; 
		delete _wMigInterv;
		delete _wNIndivMig;
		delete _wRptInterv;
		delete _wquit;     
		delete _wrun;      
		delete _wda;      
	} // }}}

private:
	// Widgets // {{{
	Fl_Input  * _wTotVol;    // Input: Total volume
	Fl_Input  * _wVoidRatio; // Input: Void ratio
	Fl_Input  * _wMinRadius; // Input: Minimum radius
	Fl_Input  * _wMaxRadius; // Input: Maximum radius
	Fl_Input  * _wNCircles;  // Input: Number of circles (NC)
	Fl_Input  * _wNIndiv;    // Input: Number of individuals
	Fl_Input  * _wEliteSize; // Input: Number of individuals in the elite
	Fl_Input  * _wProbCross; // Input: Probability of crossover
	Fl_Input  * _wProbMut;   // Input: Probability of mutation
	Fl_Input  * _wNGenerat;  // Input: Number of generations
	Fl_Input  * _wRefInterv; // Input: Interval for refinements
	Fl_Input  * _wMigInterv; // Input: Migration interval
	Fl_Input  * _wNIndivMig; // Input: Number of individuals that migrate
	Fl_Input  * _wRptInterv; // Input: Report interval
	Fl_Button * _wquit;      // Quit button
	Fl_Button * _wrun;       // Run Button
	DrawArea  * _wda;        // Draw area // }}}

	// Callbacks (must be static)
	static void _run_cb  (Fl_Widget * o, void * v) { ((MainWindow*)v)->_run (); }
	static void _quit_cb (Fl_Widget * o, void * v) { ((MainWindow*)v)->_quit(); }

	// Run callback
	inline void _run ()
	{ // {{{

		// Read input values
		gTotVol    = atof(_wTotVol   ->value());
		gVoidRatio = atof(_wVoidRatio->value());
		gMinRadius = atof(_wMinRadius->value());
		gMaxRadius = atof(_wMaxRadius->value());
		gNCircles  = atoi(_wNCircles ->value());
		gNIndiv    = atoi(_wNIndiv   ->value());
		gEliteSize = atoi(_wEliteSize->value());
		gProbCross = atof(_wProbCross->value());
		gProbMut   = atof(_wProbMut  ->value());
		gNGenerat  = atoi(_wNGenerat ->value());
		gRefInterv = atoi(_wRefInterv->value());
		gMigInterv = atoi(_wMigInterv->value());
		gNIndivMig = atoi(_wNIndivMig->value());
		gRptInterv = atoi(_wRptInterv->value());
		gNGenes    = 3*NBASES*gNCircles;
		gMaxNCross = gNGenes-1;

		// Allocate memory
		gEp     = new bool     [gNCircles*2];
		gNbc    = new double   [gNCircles*2];
		gRadii  = new double   [gNCircles];
		gIndIdx = new uint32_t [gNIndiv];
		for (uint32_t i=0; i<gNCircles; ++i)
		{
			gEp [i*2]   = false;
			gEp [i*2+1] = false;
		}
		uint32_t nfix = static_cast<uint32_t>(0.1*static_cast<double>(gNCircles)+2);
		for (uint32_t i=0; i<nfix; ++i) gEp[i]=true; // fix some circles
		for (uint32_t i=0; i<gNIndiv; ++i) gIndIdx[i]=i;

		// Number of processors (== number of islands)
		uint32_t nprocs = MPI::COMM_WORLD.Get_size();

		// Sent input values to all other processes
		bool     finished  = false;
		double   prms1[ 6] = { gTotVol, gVoidRatio, gMinRadius, gMaxRadius, gProbCross, gProbMut };
		uint32_t prms2[10] = { gNCircles, gNIndiv, gEliteSize, gNGenerat, gRefInterv, gMigInterv, gNIndivMig, gRptInterv, gNGenes, gMaxNCross };
		for (uint32_t i=1; i<nprocs; ++i)
		{
			MPI::COMM_WORLD.Send(&finished, 1, MPI::BOOL    , i, TAG_FINISHED);
			MPI::COMM_WORLD.Send(prms1    , 6, MPI::DOUBLE  , i, TAG_PRMS1   );
			MPI::COMM_WORLD.Send(prms2    ,10, MPI::UNSIGNED, i, TAG_PRMS2   );
		}

		// My Island
		SGA::Island<double> isl(gNIndiv,gNGenes,IS_MINIMIZATION,gEliteSize,gMaxNCross,gProbCross,gProbMut);

		// Generate a random island
		GenerateRandom (isl.Pop());

		// Draw this initial population
		_wda->SetBestIndividual (-1, isl.Pop()(0));
		_wda->redraw();
		Fl::wait(0);
		_wda->SetBestIndividual (-1, NULL);

		// Evolve
		uint32_t nindref = static_cast<uint32_t>(0.4*static_cast<double>(gNIndiv)+2);
		for (uint32_t i=0; i<gNGenerat; ++i)
		{
			isl.Reproduction();
			if ((i>0) && (i%gMigInterv==0)) SGA::Migrate (gNIndivMig, isl);
			if (          i%gRptInterv==0 ) _report      (i, isl);
			if ((i>0) && (i%gRefInterv==0)) RefinePop    (/*type*/0,/*nIt*/5,/*nIndivToRefine*/nindref,isl.Pop());
		}

		// Get the best individual from all other processes (islands)
		double best_ov;                       // Best objective value of all individuals of all islands
		double aisl_ov;                       // Best objective value from another island
	   	double * best = new double [gNGenes]; // Best individual of all individuals of all islands
		double * aisl = new double [gNGenes]; // Best Individual from another island
		double const * my_best = isl.Best (best_ov, /*ReCalculateOVs*/true);
		for (uint32_t i=0; i<gNGenes; ++i) best[i] = my_best[i];
		for (uint32_t i=1; i<nprocs; ++i)
		{
			MPI::COMM_WORLD.Recv(&aisl_ov,       1, MPI::DOUBLE, i, TAG_BEST_OV );
			MPI::COMM_WORLD.Recv( aisl   , gNGenes, MPI::DOUBLE, i, TAG_BEST_IND);
			if (aisl_ov<best_ov)
			{
				best_ov = aisl_ov;
				for (uint32_t j=0; j<gNGenes; ++j) best[j] = aisl[j];
			}
		}

		// Draw best circles
		_wda->SetBestIndividual (gNGenerat, best);
		_wda->redraw();
		Fl::wait(0);

		// Fill arrays with best results
		double * Xc = new double [gNCircles];
		double * Yc = new double [gNCircles];
		double * R  = new double [gNCircles];
		Indiv2Arrays (best, Xc,Yc,R);

		// Write file and calculate volume of solids
		std::ofstream of("circles.out",std::ios::out);
		double Vs = 0.0; // Volume of solids == sum area of circles
		for (uint32_t i=0; i<gNCircles; ++i)
		{
			of << Xc[i] << " " << Yc[i] << " " << R[i] << std::endl;
		   	Vs += R[i]*R[i];
		}
		of.close();
		Vs *= PI;                     // Volume of solids
		double vr = gTotVol/Vs - 1.0; // Current void ratio

		// Output
		std::cout << "BestOV = " << best_ov << ",  VoidRatio = " << vr << std::endl;

		// Clean up
		delete [] best;
		delete [] aisl;
		delete [] Xc;
		delete [] Yc;
		delete [] R;

		// Deallocate memory
		delete [] gEp;
		delete [] gNbc;
		delete [] gRadii;
		delete [] gIndIdx;

	} // }}}

	// Quit callback
	inline void _quit()
	{ // {{{

		uint32_t nprocs   = MPI::COMM_WORLD.Get_size();
		bool     finished = true;
		for (uint32_t i=1; i<nprocs; ++i)
			MPI::COMM_WORLD.Send(&finished, 1, MPI::BOOL, i, TAG_FINISHED);
		hide();

	} // }}}

	// Methods
	inline void _report (uint32_t iGeneration, SGA::Island<double> const & Isl)
	{ // {{{
		uint32_t my_id = MPI::COMM_WORLD.Get_rank(); // my processor ID
		if (my_id==0) // Root (anyway, only Root should allocate a MainWindow...)
		{
			// Output results to console
			double best_ov;  double const * pbest = Isl.Best (best_ov );
			double worst_ov;                        Isl.Worst(worst_ov);
			if (iGeneration==0) std::cout << SGA::_6 << "Gen"       << SGA::_6_3 << "BestOV" << SGA::_6_3 << "WorstOV" << std::endl;
			else                std::cout << SGA::_6 << iGeneration << SGA::_6_3 <<  best_ov << SGA::_6_3 <<  worst_ov << std::endl;
			// Draw circles
			_wda->SetBestIndividual (iGeneration, pbest);
			_wda->redraw();
			Fl::wait(0);
		}
	} // }}}

}; // class MainWindow

void SlaveRun()
{ // {{{

	// Wait information from root
	double   prms1[6]; // = { gTotVol, gVoidRatio, gMinRadius, gMaxRadius, gProbCross, gProbMut };
	uint32_t prms2[9]; // = { gNCircles, gNIndiv, gEliteSize, gNGenerat, gRefInterv, gMigInterv, gNIndivMig, gRptInterv, gNGenes, gMaxNCross };
	MPI::COMM_WORLD.Recv(prms1, 6, MPI::DOUBLE  , /*Root*/0, TAG_PRMS1);
	MPI::COMM_WORLD.Recv(prms2,10, MPI::UNSIGNED, /*Root*/0, TAG_PRMS2);

	// Set parameters 1
	gTotVol    = prms1[0];
	gVoidRatio = prms1[1];
	gMinRadius = prms1[2];
	gMaxRadius = prms1[3];
	gProbCross = prms1[4];
	gProbMut   = prms1[5];

	// Set parameters 2
	gNCircles  = prms2[0];
	gNIndiv    = prms2[1];
	gEliteSize = prms2[2];
	gNGenerat  = prms2[3];
	gRefInterv = prms2[4];
	gMigInterv = prms2[5];
	gNIndivMig = prms2[6];
	gRptInterv = prms2[7];
	gNGenes    = prms2[8];
	gMaxNCross = prms2[9];

	// Allocate memory
	gEp     = new bool     [gNCircles*2];
	gNbc    = new double   [gNCircles*2];
	gRadii  = new double   [gNCircles];
	gIndIdx = new uint32_t [gNIndiv];
	for (uint32_t i=0; i<gNCircles; ++i)
	{
		gEp [i*2]   = false;
		gEp [i*2+1] = false;
	}
	for (uint32_t i=0; i<6; ++i) gEp[i]=true; // fix some circles
	for (uint32_t i=0; i<gNIndiv; ++i) gIndIdx[i]=i;

	// My Island
	SGA::Island<double> isl(gNIndiv,gNGenes,IS_MINIMIZATION,gEliteSize,gMaxNCross,gProbCross,gProbMut);

	// Generate a random island
	GenerateRandom (isl.Pop());

	// Evolve
	for (uint32_t i=0; i<gNGenerat; ++i)
	{
		isl.Reproduction();
		if ((i>0) && (i%gMigInterv==0)) SGA::Migrate (gNIndivMig, isl);
		if ((i>0) && (i%gRefInterv==0)) RefinePop    (/*type*/0,/*nIt*/1,/*nIndivToRefine*/1,isl.Pop());
	}

	// Send the best individual to Root
	double best_ov;
	double const * best = isl.Best (best_ov, /*ReCalculateOVs*/true);
	MPI::COMM_WORLD.Send(&best_ov,       1, MPI::DOUBLE, /*Root*/0, TAG_BEST_OV );
	MPI::COMM_WORLD.Send( best   , gNGenes, MPI::DOUBLE, /*Root*/0, TAG_BEST_IND);

	// Deallocate memory
	delete [] gEp;
	delete [] gNbc;
	delete [] gRadii;
	delete [] gIndIdx;
} // }}}

///////////////////////////////////////////////////////////////////////////////////////////////////// main /////

int main(int argc, char **argv) try
{ // {{{

	// Initialize MPI
	MPI::Init(argc, argv);
	uint32_t my_id = MPI::COMM_WORLD.Get_rank(); // my processor ID

	// Initialize random numbers generator
	if (argc>1) Rnd::Init(my_id*atoi(argv[1]));
	else        Rnd::Init(1234);

	// Run
	if (my_id==0) // Root
	{
		// Initialize window
		MainWindow win;
		Fl::run();
	}
	else
	{
		bool finished = false;
		while (!finished)
		{
			// Wait message from Root to run
			MPI::COMM_WORLD.Recv(&finished, 1, MPI::BOOL, /*Root*/0, TAG_FINISHED);
			if (!finished) SlaveRun();
		}
	}

	std::cout << "Before Finilize # " << my_id << std::endl;

	// End
	MPI::Finalize();
	return 0;

} // }}}
catch (char const * m) // {{{
{
	std::cout << "Fatal: " << m << std::endl;
	MPI::Finalize();
	exit (1);
}
catch (...)
{
	std::cout << "Some exception (...) ocurred\n";
	MPI::Finalize();
	exit (1);
} //}}} 

// vim:fdm=marker
