/************************************************************************
 * SGA - A C++ library to help develop Simple Genetic Algorithms        *
 * Copyright (C) 2005 Dorival M. Pedroso                                *
 *                                                                      *
 * This program is free software: you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation, either version 3 of the License, or    *
 * any later version.                                                   *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the         *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program. If not, see <http://www.gnu.org/licenses/>  *
 ************************************************************************/

#ifndef SGA_TRUSS_2D_H
#define SGA_TRUSS_2D_H

// STL
#include <sstream>
#include <cmath>

// LAPACK
extern "C"
{
	void dgesv_(int *Np, int *NRHSp, double *A, int *LDAp, int *IPIVp, double *B, int *LDBp, int *INFOp);
}

/** 2D Truss for Finite Element Analysis. */
class Truss2D
{
public:
	/** Constructor.
	 * Example:
	 * \verbatim
	                               1.0 ^
	                                   |
	                                   |3
	                                   o----> 2.0
	                                 ,'|
	                               ,'  |
	                             ,'    |
	                   EA/L=20 ,'      |
	                     (3) ,'        | EA/L=5
	                       ,'          | (2)
	                     ,'            |
	                   ,'              |
	    y            ,'                |
	    |         1,'        (1)       |2
	    |         o--------------------o
	    |____x   /_\      EA/L=10     /_\
	            ////                  ___
	
	  const int nn = 3; // num nodes
	  const int nr = 3; // num rods
	  double nodes    [nn*2] = {0,0, 10,0, 10,10};
	  int    con      [nr*2] = {0,1, 1,2, 2,0};
	  double props    [nr  ] = {10,5,20};
	  bool   ep       [nn*2] = {true,true, false,true, false,false};
	  double ebc      [nn*2] = {0,-0.5, 0,0.4, 0,0};
	  double nbc      [nn*2] = {0,0, 0,0, 2,1};
	  double ucorrect [nn*2] = {0,-0.5, 0,0.4, -0.5,0.2};
	  double fcorrect [nn*2] = {-2,-2, 0,1, 2,1};

	  Truss2D truss (nn,nr,nodes,con,ep,nbc,ebc,props);
	  truss.Solve ();
	
	 Notation:
	   X,Y : coordinates
	   R   : radii
	   C   : connectivities (l:left node, r: right node)
	   P   : properties
	   ep  : is essential prescribed ?
	   bc  : boundary conditions
	   ebc : essential boundary conditions == displacements
	   nbc : natural boundary conditions == forces
	
	 \endverbatim */
	Truss2D(int            NNodes,           ///< Num of nodes
	        int            NRods,            ///< Number of rods
	        double const * Nodes,            ///< Coordinates x0 y0  x1 y1  ...  xnn ynn (size=2*nn)
	        int    const * Connects,         ///< Connectivities 0 1  0 2  1 2  (size=2*nr)
	        bool   const * EssenPresc,       ///< Essential (displ.) prescribed? (size=2*nn)
	        double const * NaturalBC,        ///< Natural (force) boundary conditions (size=2*nn)
	        double const * EssenBC   =NULL,  ///< Essential (displ.) boundary conditions (size=2*nn)
	        double const * Props     =NULL,  ///< Properties (size=nr)
	        bool           DividePbyL=false) ///< Divide P (properties) by L (length) ?
		: _nn    (NNodes    ), // {{{
		  _nr    (NRods     ),
		  _nodes (Nodes     ),
		  _con   (Connects  ),
		  _props (Props     ),
		  _ep    (EssenPresc),
		  _ebc   (EssenBC   ),
		  _nbc   (NaturalBC ),
		  _PbyL  (DividePbyL)
	{
		_ndof  = 2*_nn;
		_U     = new double [_ndof];
		_F     = new double [_ndof];
		_Fint  = new double [_ndof];
		_Res   = new double [_ndof];
		_dU    = new double [_ndof];
		_dF    = new double [_ndof];
		_dFint = new double [_ndof];
		_K     = new double [_ndof*_ndof];
		_Kcpy  = new double [_ndof*_ndof];
		_ipiv  = new int    [_ndof];
		Initialize();
	} // }}}

	/** Clear U, F, Fint and Res (displacements, forces, internal forces, and residuals). */
	void Initialize () { for (int i=0; i<_ndof; ++i) { _U[i]=0.0; _F[i]=0.0; _Fint[i]=0.0; _Res[i]=0.0; } }

	/** (re)set number of rods (elements) (NOTE: this will only work if Props==NULL). */
	void SetNRods (int NRods) { _nr = NRods; }

	/** (re)set connectivities (NOTE: this will only work if Props==NULL). */
	void SetCon (int const * Con) { _con = Con; }

	/** Destructor. */
	~Truss2D()
	{ // {{{
		delete [] _U;
		delete [] _F;
		delete [] _Fint;
		delete [] _Res;
		delete [] _dU;
		delete [] _dF;
		delete [] _dFint;
		delete [] _K;
		delete [] _Kcpy;
		delete [] _ipiv;
	} // }}}

	double X (int i)        const { return _nodes[i*2];   }                               ///< Returns the abcissa of node i
	double Y (int i)        const { return _nodes[i*2+1]; }                               ///< Returns the ordinate of node i
	double L (int i, int j) const { return sqrt(pow(X(i)-X(j),2.0)+pow(Y(i)-Y(j),2.0)); } ///< Returns the distance between node i and j

	/** Calculate the element stiffness (Ke).
	 * \param e index of an element
	 */
	void CalcKe (int e)
	{ // {{{
		int    i = _con[e*2];
		int    j = _con[e*2+1];
		double d = L(i,j);
		double c = (X(j)-X(i))/d;
		double s = (Y(j)-Y(i))/d;
		double p = (_props==NULL ? 1.0 : (_PbyL ? _props[e]/d : _props[e]));
		_Ke[0]= p*c*c;   _Ke[4]= p*c*s;  _Ke[ 8]=-p*c*c;   _Ke[12]=-p*c*s;
		_Ke[1]=_Ke[ 4];  _Ke[5]= p*s*s;  _Ke[ 9]=_Ke[12];  _Ke[13]=-p*s*s;
		_Ke[2]=_Ke[ 8];  _Ke[6]=_Ke[ 9]; _Ke[10]=_Ke[ 0];  _Ke[14]=_Ke[4];
		_Ke[3]=_Ke[12];  _Ke[7]=_Ke[13]; _Ke[11]=_Ke[14];  _Ke[15]=_Ke[5];
	} // }}}

	/** Calculate the global stiffness (K). */
	void CalcK ()
	{ // {{{
		for (int i=0; i<_ndof*_ndof; ++i) _K[i] = 0.0;
		for (int e=0; e<_nr; ++e)
		{
			CalcKe (e);
			int l    = _con[e*2];                // left node
			int r    = _con[e*2+1];              // right node
			int m[4] = {l*2, l*2+1, r*2, r*2+1}; // map (local=>global)
			for (int i=0; i<4; ++i)
			for (int j=0; j<4; ++j)
				_K[m[j]*_ndof+m[i]] += _Ke[j*4+i];
		}
	} // }}}

	/** Modify K for prescribed displacements.
	 * \param h step-size
	 */
	void ModifyK (double h=1)
	{ // {{{
		// Set dF and dU (workspace) vectors
		for (int i=0; i<_ndof; ++i) { _dF[i]=0.0; _dU[i]=0.0; _dFint[i]=0.0; }
		for (int i=0; i<_ndof; ++i)
		{
			if (_ep[i]) _dU[i] = (_ebc==NULL ? 0.0 : _ebc[i]*h);
			else
			{
				_dF[i] = _dU[i] = _nbc[i]*h;
				for (int j=0; j<_ndof; ++j)
					if (_ep[j]) _dU[i] -= (_ebc==NULL ? 0.0 : _K[j*_ndof+i]*_ebc[j]*h);
			}
		}
		// Clear lines and columns of K for prescribed displacements => modified stiffness
		for (int i=0; i<_ndof; ++i)
		{
			if (_ep[i])
			{
				for (int j=0; j<_ndof; ++j)
				{
					_K[j*_ndof+i] = 0.0;
					_K[i*_ndof+j] = 0.0;
				}
				_K[i*_ndof+i] = 1.0;
			}
		}
	} // }}}

	/** Calculate internal forces. */
	void CalcDFint ()
	{ // {{{
		for (int e=0; e<_nr; ++e)
		{
			int    i     = _con[e*2];
			int    j     = _con[e*2+1];
			double d     = L(i,j);
			double c     = (X(j)-X(i))/d;
			double s     = (Y(j)-Y(i))/d;
			double p     = (_props==NULL ? 1.0 : (_PbyL ? _props[e]/d : _props[e]));
			double elong = c*_dU[j*2]+s*_dU[j*2+1] - (c*_dU[i*2]+s*_dU[i*2+1]);                 // elongation
			double dfn   = elong*p;                                                             // normal force increment
			_dFint[i*2]+=-c*dfn; _dFint[i*2+1]+=-s*dfn; _dFint[j*2]+=c*dfn; _dFint[j*2+1]+=s*dfn; // global coordinates
		}
	} // }}}

	/** Circles: Calculate internal axial force.
	 * \param R Radius
	 * \param i Index of left node on a rod
	 * \param j Index of right node on a rod
	 * \param Type the type of force, according to:
	 * \verbatim Type:
	 *  -1: only attraction
	 *   0: attraction and repulsion
	 *  +1: only repulsion
	 * \endverbatim */
	double CircleAxialForce (double const * R, int i, int j, int Type=0)
	{ // {{{
		double dij = L(i,j);
		double eij = (R[i]+R[j])-dij;
		if (Type<0) eij = (eij<0 ? eij : 0.0);
		if (Type>0) eij = (eij>0 ? eij : 0.0);
		return eij*R[i]*R[j]/(dij*dij);
	} // }}}

	/** Circles: Calculate error.
	 * \param R Radius (size==nnodes)
	 * \param Type the type of force, according to:
	 * \verbatim Type:
	 *  -1: only attraction
	 *   0: attraction and repulsion
	 *  +1: only repulsion
	 * \endverbatim */
	double CirclesError (double const * R, int Type=0)
	{ // {{{
		double error = 0.0;
		for (int e=0; e<_nr; ++e)
		{
			int i = _con[e*2];
			int j = _con[e*2+1];
			error += fabs(CircleAxialForce(R,i,j,Type));
		}
		return error;
	} // }}}

	/** Circles: Calculate forces on nodes.
	 * \param sf Scale factor
	 * \param R Radius (size==nnodes)
	 * \param dF Internal force (size==2*nnodes)
	 * \param Type the type of force, according to:
	 * \verbatim Type:
	 *  -1: only attraction
	 *   0: attraction and repulsion
	 *  +1: only repulsion
	 * \endverbatim */
	void CirclesForce (double sf, double const * R, double * dF, int Type=0)
	{ // {{{
		for (int i=0; i<_ndof; ++i) dF[i]=0.0;
		for (int e=0; e<_nr; ++e)
		{
			int    i   = _con[e*2];
			int    j   = _con[e*2+1];
			double dij = L(i,j);
			double c   = (X(j)-X(i))/dij;
			double s   = (Y(j)-Y(i))/dij;
			double dfn = -CircleAxialForce(R,i,j,Type)*sf;                        // normal force increment
			dF[i*2]-=-c*dfn; dF[i*2+1]-=-s*dfn; dF[j*2]-=c*dfn; dF[j*2+1]-=s*dfn; // global coordinates
			// considering that dFext==0, dResidual=dFext-dFint => dResidual=-dFint
		}
		//for (int i=0; i<_ndof; ++i) std::cout<<dF[i]<<" "; std::cout<<std::endl;
	} // }}}

	/** Solve equilibrium via FEM.
	 * \param nInc Number of increments
	 */
	void Solve (int nInc=1)
	{ // {{{
		double h = 1.0/nInc;
		for (int i=0; i<nInc; ++i)
		{
			// Assembly
			CalcK ();
			// Save a copy of K for later recovering of dF (could save only K21 and K22)
			for (int j=0; j<_ndof*_ndof; ++j) _Kcpy[j] = _K[j];
			// Modify K for prescribed displacements
			ModifyK (h);
			// Solve dU = inv(K)*dF
			int info = 0;
			int nrhs = 1;
			dgesv_ (&_ndof, &nrhs, _K, &_ndof, _ipiv, _dU, &_ndof, &info);
			// Solve external forces increments for prescribed essential (displ.) dofs
			for (int j=0; j<_ndof; ++j)
				if (_ep[j]) for (int m=0; m<_ndof; ++m) _dF[j] += _Kcpy[m*_ndof+j]*_dU[m];
			// Calc internal forces
			CalcDFint ();
			// Update
			for (int j=0; j<_ndof; ++j)
			{
				_U    [j] += _dU    [j];
				_F    [j] += _dF    [j];
				_Fint [j] += _dFint [j];
				_Res  [j]  = _F[j] - _Fint[j];
			}
		}
	} // }}}

	double const * U    () const { return _U;    } ///< Displacements
	double const * F    () const { return _F;    } ///< Forces
	double const * Fint () const { return _Fint; } ///< Internal forces
	double const * Res  () const { return _Res;  } ///< Residual

private:
	int            _nn;     ///< Num of nodes
	int            _nr;     ///< Number of rods
	double const * _nodes;  ///< Coordinates x0 y0  x1 y1  ...  xnn ynn (size=2*nn)
	int    const * _con;    ///< Connectivities 0 1  0 2  1 2  (size=2*nr)
	double const * _props;  ///< Properties (size=nr)
	bool   const * _ep;     ///< Essential (displ.) prescribed?
	double const * _ebc;    ///< Essential (displ.) boundary conditions
	double const * _nbc;    ///< Natural (force) boundary conditions
	bool           _PbyL;   ///< Divide P (properties) by L (length) ?
	double         _Ke[16]; ///< Space to hold a stiffness matrix of an element. Set by Ke() method. (must be in col-major format for LAPACK)
	int            _ndof;   ///< Number of degrees of freedom == 2*nn
	double       * _U;      ///< Global displacements (size=_ndof)
	double       * _F;      ///< Global forces (size=_ndof)
	double       * _Fint;   ///< Internal forces of an element (size=_ndof)
	double       * _Res;    ///< Residual = F - Fint (size=_ndof)
	double       * _dU;     ///< Global displacements increments (size=_ndof)
	double       * _dF;     ///< Global forces increments (size=_ndof)
	double       * _dFint;  ///< Internal forces increments of an element (size=_ndof)
	double       * _K;      ///< Global stiffness matrix. (must be in col-major forces for LAPACK). (size=_ndof*_ndof)
	double       * _Kcpy;   ///< Global stiffness matrix. (must be in col-major forces for LAPACK). (size=_ndof*_ndof)
	int          * _ipiv;   ///< Index pivot for LAPACK (size=_ndof)
}; // class Truss2D

#endif // SGA_TRUSS_2D_H

// vim:fdm=marker
