/************************************************************************
 * SGA - A C++ library to help develop Simple Genetic Algorithms        *
 * Copyright (C) 2005 Dorival M. Pedroso                                *
 *                                                                      *
 * This program is free software: you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation, either version 3 of the License, or    *
 * any later version.                                                   *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the         *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program. If not, see <http://www.gnu.org/licenses/>  *
 ************************************************************************/

#ifndef SGA_RANDOM_H
#define SGA_RANDOM_H

// STL
#include <cstdlib>
#include <ctime>
#include <inttypes.h>

extern "C"
{
	// From SFMT (SIMD-oriented Fast Mersenne Twister)
	void     init_gen_rand (uint32_t Seed);
	uint32_t gen_rand32    ();
	uint64_t gen_rand64    ();

  #ifdef HAVE_DSFMT
	// From dSFMT (double SIMD-oriented Fast Mersenne Twister)
	void   dinit_gen_rand       (uint32_t Seed);
	double genrand_close1_open2 ();
  #endif
}

/** \namespace Rnd Random number generators. */
namespace Rnd
{

/** Generates a random number on [0,1) with 53-bit resolution. */
inline static double genrand_res53 () { return static_cast<double>(gen_rand64()*(1.0/18446744073709551616.0L)); }

/** Initialize pseudo random generators. */
inline static void Init (uint32_t Seed)
{
	init_gen_rand(Seed);
  #ifdef HAVE_DSFMT
	dinit_gen_rand(Seed);
  #endif
}

/** Generate a Bernoulli variable (Throw a coin with probability "p") */
#ifdef HAVE_DSFMT
  inline bool Flip (double p) { return (p==1.0 ? true : (p==0.0 ? false : ((genrand_close1_open2()-1.0)<=p ? true : false))); }
#else
  inline bool Flip (double p) { return (p==1.0 ? true : (p==0.0 ? false : ((genrand_res53())<=p ? true : false))); }
#endif

/** Shuffle a list of Types. */
template<typename Type>
inline void Shuffle (uint32_t size, Type list[])
{ // {{{
	uint32_t j;
	Type     tmp;
	for (uint32_t i=size-1; i>0; --i)
	{
		j       = gen_rand32() % i;
		tmp     = list[j];
		list[j] = list[i];
		list[i] = tmp;
	}
} // }}}

/** Generate different type of random numbers.
 * Usage:
 *        Rnd::T<Type>::Gen()
 *   or   Rnd::T<Type>::Gen(Lo,Hi)
 */
template<typename Out_T> class T // Traits
{
public:
	/** Generate a random number (from 0 to 2^32-i for uint32_t and inside [0,1) for double). */
	static inline Out_T Gen();

	/** Generate a random number inside [Lo,Hi] for integers and [Lo,Hi) for reals. */
	static inline Out_T Gen(Out_T Lo, Out_T Hi);
};

/** Generate a uint32_t from 0 to 2^32-1 (efficient).
 * Usage:
 *        Rnd::T<uint32_t>::Gen()
 *   or   Rnd::T<uint32_t>::Gen(Lo,Hi)
 */
template <> class T<uint32_t>
{
public:
	/** Generate a uint32_t from 0 to 2^32-1 (efficient) */
	static inline uint32_t Gen() { return gen_rand32(); }

	/** Generate a uint32_t inside [Lo,Hi] */
	static inline uint32_t Gen(uint32_t Lo, uint32_t Hi) { return (gen_rand32() % (Hi-Lo+1) + Lo); }
};

/** Generate a double (efficient).
 * Usage:
 *        Rnd::T<double>::Gen()
 *   or   Rnd::T<double>::Gen(Lo,Hi)
 */
template <> class T<double>
{
public:
  #ifdef HAVE_DSFMT
	/** Generate a double inside [0,1) (efficient) */
	static inline double Gen() { return genrand_close1_open2()-1.0; }

	/** Generate a double inside [Lo,Hi] */
	static inline double Gen(double Lo, double Hi) { return Lo+(Hi-Lo)*(genrand_close1_open2()-1.0); }
  #else
	/** Generate a double inside [0,1) (efficient) */
	static inline double Gen() { return genrand_res53(); }

	/** Generate a double inside [Lo,Hi] */
	static inline double Gen(double Lo, double Hi) { return Lo+(Hi-Lo)*(genrand_res53()); }
  #endif
};

/** Generate a long number (by means of double numbers generator).
 * Usage (only for the case of numbers inside [Lo,Hi]):
 *
 *        Rnd::T<long>::Gen(Lo,Hi)
 */
template <> class T<long>
{
public:
	/** Generate a long number inside [Lo,Hi] */
  #ifdef HAVE_DSFMT
	static inline long Gen(long Lo, long Hi) { return static_cast<long>(Lo+(Hi-Lo+1)*(genrand_close1_open2()-1.0)); }
  #else
	static inline long Gen(long Lo, long Hi) { return static_cast<long>(Lo+(Hi-Lo+1)*(genrand_res53())); }
  #endif
};

}; // namespace Rnd

#endif // SGA_RANDOM_H

// vim:fdm=marker
