/************************************************************************
 * SGA - A C++ library to help develop Simple Genetic Algorithms        *
 * Copyright (C) 2005 Dorival M. Pedroso                                *
 *                                                                      *
 * This program is free software: you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation, either version 3 of the License, or    *
 * any later version.                                                   *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the         *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program. If not, see <http://www.gnu.org/licenses/>  *
 ************************************************************************/

#ifndef SGA_EVOLVER_H
#define SGA_EVOLVER_H

// OpenMPI
#include "mpi.h"

// SGA
#include "random.h"
#include "island.h"
#include "mpitraits.h"

namespace SGA
{

const uint32_t TAG_SET_DEST_ISL = 1000; ///< MPI tag for setting the destination island
const uint32_t TAG_SENT_GENE    = 2000; ///< MPI tag for sending/receiving an individual (a bunch of genes)

/** Migrate this island to another island in parallel.
 * \param nIndivMig Number of individuals that migrate
 * \param Isl The island
 */
template<typename Gene_T>
inline void Migrate (uint32_t nIndivMig, SGA::Island<Gene_T> & Isl)
{ // {{{
	
	// Variables
	uint32_t my_id = MPI::COMM_WORLD.Get_rank(); // my processor ID

	// Synchronize
	MPI::COMM_WORLD.Barrier();

	// Set destination island for migration
	uint32_t idest; // Index of a destination island (other than my_id)
	if (my_id==0) // Root
	{
		// Number of processors (== number of islands)
		uint32_t nprocs = MPI::COMM_WORLD.Get_size();

		// Allocate destinations array
		uint32_t * destinations = new uint32_t [nprocs];

		// The next line assures that a situation where (idest==my_id) will never happen
		// because the Shuffle will (should) always move all indexes inside destinations
		for (uint32_t i=0; i<nprocs; ++i) destinations[i] = i;

		// Find destinations
		Rnd::Shuffle (nprocs, destinations); // Indexes of destination islands

		// Send destinations to all other processors
		for (uint32_t i=0; i<nprocs; ++i)
		{
			if (i==0) idest = destinations[i]; // Root
			else MPI::COMM_WORLD.Send(&destinations[i], 1, MPI::INT, i, TAG_SET_DEST_ISL); // Send destinations to all other processes
		}

		// Clean up
		delete [] destinations;
	}
	else
		MPI::COMM_WORLD.Recv(&idest, 1, MPI::INT, /*Root*/0, TAG_SET_DEST_ISL); // Receive idest from root

	//std::cout << "Process/Island #" << my_id << " has island #" << idest << " as destination for migration\n";

	// Do migration
	if (idest!=my_id) // Avoid sending to myself
	{
		uint32_t migi; // Index of an individual that will migrate
		for (uint32_t i=0; i<nIndivMig; ++i)
		{
			// Randomly select an individual
			migi = Rnd::T<uint32_t>::Gen (0,Isl.nIndiv()-1);
			
			// Post requests to send genes
			MPI::Request req_send = MPI::COMM_WORLD.Isend(Isl.Pop()(migi), Isl.nGenes(), MPITraits<Gene_T>::Type(), idest, TAG_SENT_GENE);
			
			// Post requests to receive genes
			MPI::Request req_recv = MPI::COMM_WORLD.Irecv(Isl.Pop()(migi), Isl.nGenes(), MPITraits<Gene_T>::Type(), MPI::ANY_SOURCE, TAG_SENT_GENE);

			// Wait for all requests to finish
			req_send.Wait();
			MPI::Status st;
			req_recv.Wait(st);

			//std::cout << "Process/Island #" << my_id << " received an individual from Island #" << st.Get_source() << std::endl;
		}
	}
} // }}}

}; // namespace SGA

#endif // SGA_EVOLVER_H

// vim:fdm=marker
