/*
SDX: Documentary System in XML.
Copyright (C) 2000, 2001, 2002  Ministere de la culture et de la communication (France), AJLSM

Ministere de la culture et de la communication,
Mission de la recherche et de la technologie
3 rue de Valois, 75042 Paris Cedex 01 (France)
mrt@culture.fr, michel.bottin@culture.fr

AJLSM, 17, rue Vital Carles, 33000 Bordeaux (France)
sevigny@ajlsm.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
or connect to:
http://www.fsf.org/copyleft/gpl.html
*/
package fr.gouv.culture.sdx.repository;

import fr.gouv.culture.sdx.document.Document;
import fr.gouv.culture.sdx.document.ParsableDocument;
import fr.gouv.culture.sdx.exception.SDXException;
import fr.gouv.culture.sdx.exception.SDXExceptionCode;
import fr.gouv.culture.sdx.framework.FrameworkImpl;
import fr.gouv.culture.sdx.utils.Utilities;
import org.apache.avalon.excalibur.datasource.DataSourceComponent;
import org.apache.avalon.excalibur.io.IOUtil;
import org.apache.avalon.excalibur.xml.Parser;
import org.apache.avalon.framework.component.ComponentException;
import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.framework.component.ComponentSelector;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.cocoon.xml.XMLConsumer;
import org.xml.sax.ContentHandler;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.sql.*;
import java.util.StringTokenizer;

/** An implementation of a repository using a JDBC datasource.
 * <p>
 * This repository can be used by itself, or be subclassed for specific RDBMS.
 * A great care has been taken to isolate what could be vendor specific.
 */
public class JDBCRepository extends AbstractDataSourceComponentBackedRepository {

    public static final int NO_SPECIFIC_TYPE = -1;


    /**Constant for a field name in a table*/
    protected final static String FIELD_DATA = "data";


    protected final int PARAM_INDEX_FIELD_DATA_ORIGINAL = 2;


    /**
     * Creates a repository.
     *
     * <p>
     * A logger must be set and then this repository must be configured and initialized.
     *
     * @see #enableLogging
     * @see #configure
     * @see #init
     */
    public JDBCRepository() {
    }


    /** Gets a connection for manipulating repository's content.
     * @return The connection.
     * @throws SDXException
     */
    public RepositoryConnection getConnection() throws SDXException {
        try {
            JDBCRepositoryConnection conn = new JDBCRepositoryConnection(getDataSourceComponent());
            conn.enableLogging(this.logger);
            return conn;
        } catch (SQLException e) {
            String[] args = new String[2];
            args[0] = this.getId();
            args[1] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_GET_CONNECTION, args, e);
        } catch (ComponentException e) {
            String[] args = new String[1];
            args[0] = e.getMessage();
            //null logger passed to prevent double logging
            SDXException sdxE = new SDXException(null, SDXExceptionCode.ERROR_ACQUIRE_DATASOURCE, args, e);

            String[] args2 = new String[2];
            args2[0] = this.getId();
            args2[1] = sdxE.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_GET_CONNECTION, args2, sdxE);
        }
    }

    /** Releases a previously opened connection.
     *
     * The releases both the SQL connection to the database
     * and the datasource component connection from cocoon's
     * database selector component.
     *
     * @param c A connection.
     * @throws SDXException
     */
    public void releaseConnection(RepositoryConnection c) throws SDXException {
        if (c != null) {
            //closing the sql connection if there is one open
            if (((JDBCRepositoryConnection) c).getConnection() != null)
                ((JDBCRepositoryConnection) c).releaseSqlConnection();
            //releasing the datasource component from cocoon's dbSelector
            this.releaseDataSourceComponent(((JDBCRepositoryConnection) c).getDataSource());
        }
    }

    /** Compose the object so that we get the <code>Component</code>s we need from the
     * <code>ComponentManager</code>.
     * @param manager   The component manager from Cocoon that allow us to acquire a database selector component.
     * @throws ComponentException
     */
    public void compose(ComponentManager manager) throws ComponentException {
        //setting the manager
        super.compose(manager);

        if (this.manager != null)
            this.dbSelector = (ComponentSelector) this.manager.lookup(DataSourceComponent.ROLE + "Selector");
    }

    /** Configures this repository.
     *
     * <p>
     *  In addition to the parameters needed in the base configuration handled by the parent class,
     *  the following parameter is required: data source identifier (dsi), the value of the "name"
     *  attribute of the "jdbc" subElement for the  "datasources" element in cocoon.xconf
     *  (user must create this information in cocoon.xconf)
     *
     * @param   configuration   The configuration for this repository (based on a xml file).
     *
     *<p> Sample configuration entry:
     *<p> &lt;sdx:repository sdx:type = "JDBC" sdx:id = "myRepoId" dsi = "datasource identifier from cocoon.xconf"/>
     *
     */
    public void configure(Configuration configuration) throws ConfigurationException {
        Utilities.checkConfiguration(configuration);
        //configuring using overridden DataSourceComponentBacked configure method
        super.configure(configuration);
        // Let the superclass handle basic configurations
        loadBaseConfiguration(configuration);

        // building a name for the table, it is the appPathName(app directory name) and the repositoryId separated by an underscore, appPath_repoId,
        /*TODO: maybe in the future if the need arises, but this method of building the table name isn't portable
        *if the user decides to change the app name when moving the app, as the table name would change too
        */
        this.tableName = (String) props.get(FrameworkImpl.APP_PATH_NAME) + "_" + this.id;
    }

    /** Initializes the repository.
     *
     * If there are no tables in the database,
     * we create the necessary table
     *
     * @throws SDXException  */
    public void init() throws SDXException {

        /** First try to access the database and see if the tables exist. */
        Connection conn = null;
        ResultSet rs = null;

        JDBCRepositoryConnection repoConn = (JDBCRepositoryConnection) this.getConnection();
        try {
            conn = repoConn.getConnection();
            DatabaseMetaData dbmd = conn.getMetaData();
            String tableName = getTableName();
            StringTokenizer tnTokens = new StringTokenizer(tableName, "_", true);
            String escapedTableName = "";
            while (tnTokens.hasMoreTokens()) {
                String nextToken = tnTokens.nextToken();
                if (nextToken.equals("_") || nextToken.equals("%"))
                    nextToken = dbmd.getSearchStringEscape() + nextToken;
                escapedTableName += nextToken;
            }
            if (Utilities.checkString(escapedTableName))
                tableName = escapedTableName;
            rs = dbmd.getTables(null, null, tableName, null);
            if (!rs.next()) {
                //we don't have a table name with mixed case
                //we will try with lower case
                rs = dbmd.getTables(null, null, tableName.toLowerCase(), null);                // The table doesn't exist, so we should create it.
                if (!rs.next()) {
                    //we don't have a table name with lower case
                    //we will try with upper case
                    rs = dbmd.getTables(null, null, tableName.toUpperCase(), null);                // The table doesn't exist, so we should create it.
                    if (!rs.next())//no table we will create it
                        createTable(conn);
                }
            }
        } catch (SQLException e) {
            String[] args = new String[2];
            args[0] = this.getId();
            args[1] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_INIT_REPO, args, e);
        } finally {
            if (rs != null) {
                try {
                    rs.close();
                } catch (SQLException e) {
                    String[] args = new String[2];
                    args[0] = this.getId();
                    args[1] = e.getMessage();
                    throw new SDXException(logger, SDXExceptionCode.ERROR_CLOSE_RESULT_SET, args, e);
                }
            }
            repoConn.commit();
            //this closes the connection
            this.releaseConnection(repoConn);

        }
    }

    /** Returns the number of documents within the repository (all kind of documents).
     *
     * NOT IMPLEMENTED YET!
     *
     * @return The number.
     * @throws SDXException
     */
    public long size() throws SDXException {
        return size(NO_SPECIFIC_TYPE);
    }

    /** Returns the number of documents of specified type within the store.
     *
     *	@param	type	The type of document, must be defined in fr.gouv.culture.sdx.documents.Document constants.
     */
    private long size(int type) throws SDXException {

        JDBCRepositoryConnection repoConn = (JDBCRepositoryConnection) this.getConnection();
        PreparedStatement ps = null;
        Connection conn = null;
        try {
            conn = repoConn.getConnection();
            String queryString = "SELECT count(*) FROM ?";

            //Well, as far as I've seen, no "t" field :-( -pb

            ps = conn.prepareStatement(queryString);
            ps.setString(1, getTableName());
            if (type != NO_SPECIFIC_TYPE) {
                queryString += " WHERE " + FIELD_DATA + "=?";//at this point we have "SELECT count(*) FROM ?  WHERE data=type"
                ps.setInt(2, type);
            }
            ResultSet rs = ps.executeQuery();
            int ret = 0;
            if (rs.next()) ret = rs.getInt(PARAM_INDEX_FIELD_ID);
            rs.close();
            return ret;
        } catch (SQLException e) {
            String[] args = new String[2];
            args[0] = this.getId();
            args[1] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_SIZE_REPO, args, e);

        } finally {
            if (ps != null) {
                try {
                    ps.close();
                } catch (SQLException e) {
                    String[] args = new String[2];
                    args[0] = this.getId();
                    args[1] = e.getMessage();
                    throw new SDXException(logger, SDXExceptionCode.ERROR_CLOSE_SQL_PREPARED_STATEMENT, args, e);
                }
            }
            //this closes the connection
            this.releaseConnection(repoConn);
        }

    }

    /** Lists the repository content as SAX events.
     * <p>
     * The exact structure is still to be defined, but it should be very simple,
     * with only one element per document and a few properties as attributes.
     *
     * @param hdl A SAX content handler to feed with events.
     * @throws SDXException
     */
    public void lists(ContentHandler hdl) throws SDXException {
        //TODOImplement
    }

    /** Adds a document to the repository.
     *
     * @param doc   A document.
     * @param c     A connection to the repository.
     * @throws SDXException
     */
    public synchronized void add(Document doc, RepositoryConnection c) throws SDXException {
        //ensuring we have valid objects
        super.add(doc, c);
        Connection conn = ((JDBCRepositoryConnection) c).getConnection();
        String queryString = getDocumentAddQuery();
        PreparedStatement ps = null;
        try {
            ps = conn.prepareStatement(queryString);
            ps.setString(PARAM_INDEX_FIELD_ID, doc.getId());
            //this is will give us the original file
            ps.setBinaryStream(PARAM_INDEX_FIELD_DATA_ORIGINAL, doc.openStream(), doc.getLength());
            //TODO?: remove the transformed data field as we dont use it anymore
            ps.executeUpdate();
        } catch (SQLException e) {
            String[] args = new String[3];
            args[0] = doc.getId();
            args[1] = this.getId();
            args[2] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_ADD_DOC, args, e);
        } finally {
            if (ps != null) {
                try {
                    ps.close();
                } catch (SQLException e) {
                    String[] args = new String[2];
                    args[0] = this.getId();
                    args[1] = e.getMessage();
                    throw new SDXException(logger, SDXExceptionCode.ERROR_CLOSE_SQL_PREPARED_STATEMENT, args, e);
                }
            }
        }

    }

    /** Deletes all documents from the repository.
     * @throws SDXException
     */
    public synchronized void empty() throws SDXException {
        Connection conn = null;
        PreparedStatement ps = null;
        JDBCRepositoryConnection repoConn = (JDBCRepositoryConnection) this.getConnection();

        try {
            conn = repoConn.getConnection();
            String queryString = getDeleteAllQuery();
            ps = conn.prepareStatement(queryString);
            ps.executeUpdate();
        } catch (SQLException e) {
            String[] args = new String[2];
            args[0] = this.getId();
            args[1] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_EMPTY, args, e);
        } finally {
            if (ps != null) {
                try {
                    ps.close();
                } catch (SQLException e) {
                    String[] args = new String[2];
                    args[0] = this.getId();
                    args[1] = e.getMessage();
                    throw new SDXException(logger, SDXExceptionCode.ERROR_CLOSE_SQL_PREPARED_STATEMENT, args, e);
                }
            }
            repoConn.commit();
            //this closes the connection
            this.releaseConnection(repoConn);
        }
    }

    /** Deletes a document from the repository.
     *
     * @param doc A document.
     * @param c A connection to the repository.
     * @throws SDXException
     */
    public synchronized void delete(Document doc, RepositoryConnection c) throws SDXException {
        super.delete(doc, c);
        Connection conn = ((JDBCRepositoryConnection) c).getConnection();
        String queryString = getDocumentDeleteQuery();
        PreparedStatement ps = null;
        try {
            ps = conn.prepareStatement(queryString);
            ps.setString(PARAM_INDEX_FIELD_ID, doc.getId());
            ps.executeUpdate();
        } catch (SQLException e) {
            String[] args = new String[3];
            args[0] = doc.getId();
            args[1] = this.getId();
            args[2] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_DELETE_DOC, args, e);
        } finally {
            if (ps != null) {
                try {
                    ps.close();
                } catch (SQLException e) {
                    String[] args = new String[2];
                    args[0] = this.getId();
                    args[1] = e.getMessage();
                    throw new SDXException(logger, SDXExceptionCode.ERROR_CLOSE_SQL_PREPARED_STATEMENT, args, e);
                }
            }
        }

    }

    /** Retrieves an SDX document as SAX events.
     *
     * @param doc       A ParsableDocument, ie XMLDocument or HTMLDocument.
     * @param consumer  A SAX content handler to feed with events.
     * <p>The wrapped contentHandler for including events within an XSP page contentHandler should be created using
     * IncludeXMLConsumer stripper = new IncludeXMLConsumer(xspContentHandler);</p>
     * @param c A connection to the repository.
     * @throws SDXException
     */
    public void toSAX(ParsableDocument doc, XMLConsumer consumer, RepositoryConnection c) throws SDXException {
        /*TODORefactor?:can we refactor these methods into abstract repository given the changes i made there with the
        *necessity to make specific calls to the openStream methods of the subclasses?-rbp
        */
        //ensuring we have valid objects
        super.toSAX(doc, consumer, c);
        Parser parser = null;
        try {
            doc.setContent(openStream(doc, null, c));
            parser = (Parser) this.manager.lookup(Parser.ROLE);
            doc.parse(parser, consumer);
        } catch (ComponentException e) {
            String[] args = new String[1];
            args[0] = e.getMessage();
            //null logger passed to prevent double logging
            SDXException sdxE = new SDXException(null, SDXExceptionCode.ERROR_ACQUIRE_PARSER, args, e);

            String[] args2 = new String[2];
            args2[0] = this.getId();
            args2[1] = sdxE.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_GET_DOC, args2, sdxE);
        } /*catch (SDXException e) {
            //is this catch necessary, i think the finally will be exectued either way, but for safety we'll catch it now?-rbp
            throw e;
        } */ finally {
            if (parser != null) this.manager.release(parser);
        }
    }

    /** Opens an input stream to read the content of a document.
     *
     * @return          The stream.
     * @param encoding  An encoding (may be null).
     * <p>If <code> null</code> or invalid we use a default.
     * <p>TODOImplement use of encoding currently not implemented , will do soon.
     * @param doc       A document to read.
     * @param c         A connection to the repository.
     * @throws SDXException
     */
    public InputStream openStream(Document doc, String encoding, RepositoryConnection c) throws SDXException {
        //ensuring we have valid objects
        super.openStream(doc, encoding, c);
        Connection conn = ((JDBCRepositoryConnection) c).getConnection();
        String queryString = getDocumentGetQuery();
        PreparedStatement ps = null;
        ResultSet rs = null;
        try {
            ps = conn.prepareStatement(queryString);
            ps.setString(1, doc.getId());
            rs = ps.executeQuery();
            InputStream is;
            if (rs.next()) {
                //was FIELD_DATA_TRANSFORMED, but since we dont store that yet, it is no good
                is = rs.getBinaryStream(FIELD_DATA);
                return is;
            } else {
                String[] args = new String[2];
                args[0] = doc.getId();
                args[1] = this.getId();
                throw new SDXException(logger, SDXExceptionCode.ERROR_GET_DOC, args, null);
            }
        } catch (SQLException e) {
            String[] args = new String[3];
            args[0] = doc.getId();
            args[1] = this.getId();
            args[2] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_GET_DOC, args, e);
        } finally {
            if (ps != null) {
                try {
                    if (ps != null) ps.close();
                } catch (SQLException e) {
                    String[] args = new String[2];
                    args[0] = this.getId();
                    args[1] = e.getMessage();
                    throw new SDXException(logger, SDXExceptionCode.ERROR_CLOSE_SQL_PREPARED_STATEMENT, args, e);
                }
            }

            if (rs != null) {
                try {
                    if (rs != null) rs.close();
                } catch (SQLException e) {
                    String[] args = new String[2];
                    args[0] = this.getId();
                    args[1] = e.getMessage();
                    throw new SDXException(logger, SDXExceptionCode.ERROR_CLOSE_RESULT_SET, args, e);
                }
            }

        }

    }

    /** Writes the content of a document to an output stream.
     *
     * @param doc A document.
     * @param os An output stream.
     * @param c A connection to the repository.
     * @throws SDXException
     */
    public void get(Document doc, OutputStream os, RepositoryConnection c) throws SDXException {
        //ensuring we have valid objects
        super.get(doc, os, c);
        Connection conn = ((JDBCRepositoryConnection) c).getConnection();
        String queryString = getDocumentGetQuery();
        PreparedStatement ps = null;
        ResultSet rs = null;
        try {
            ps = conn.prepareStatement(queryString);
            ps.setString(PARAM_INDEX_FIELD_ID, doc.getId());
            rs = ps.executeQuery();
            if (rs.next()) IOUtil.copy(rs.getBinaryStream(FIELD_DATA), os);
        } catch (SQLException e) {
            String[] args = new String[3];
            args[0] = doc.getId();
            args[1] = this.getId();
            args[2] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_GET_DOC, args, e);
        } catch (IOException e) {
            String[] args = new String[3];
            args[0] = doc.getId();
            args[1] = this.getId();
            args[2] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_GET_DOC, args, e);
        } finally {
            if (ps != null) {
                try {
                    ps.close();
                } catch (SQLException e) {
                    String[] args = new String[2];
                    args[0] = this.getId();
                    args[1] = e.getMessage();
                    throw new SDXException(logger, SDXExceptionCode.ERROR_CLOSE_SQL_PREPARED_STATEMENT, args, e);
                }
            }

            if (rs != null) {
                try {
                    rs.close();
                } catch (SQLException e) {
                    String[] args = new String[2];
                    args[0] = this.getId();
                    args[1] = e.getMessage();
                    throw new SDXException(logger, SDXExceptionCode.ERROR_CLOSE_RESULT_SET, args, e);
                }

            }
        }

    }

    /** Returns an SQL query that could create the table.
     * <p>
     * This query should not have any parameter.
     * @return The query.
     */
    protected String getTableCreationQuery() {
        //changed dataType of FIELD_DATA_TRANSFORMED from "BINARY NOT NULL" TO only "BINARY"
        return "CREATE TABLE " + getTableName() + " ( " + FIELD_ID + " VARCHAR(255) NOT NULL, " + FIELD_DATA + " BINARY, PRIMARY KEY (" + FIELD_ID + "))";
    }

    /** Returns an SQL query that could add a document.
     * <p>
     * This query should have three parameters, the first for the id, the second
     * for the doctype and the third for the data.
     * @return The query.
     */
    protected String getDocumentAddQuery() {
        //TODOImplement: at some time, implement the proper handling of the original document (column data_t).
        return "INSERT INTO " + getTableName() + " (" + FIELD_ID + "," + FIELD_DATA + ") VALUES (?, ?)";
    }

    /** Returns an SQL query that could delete a document.
     * <p>
     * This query should have one parameter for the document's id.
     * @return The query.
     */
    protected String getDocumentDeleteQuery() {
        return "DELETE FROM " + getTableName() + " WHERE " + FIELD_ID + " = ?";
    }

    /** Returns an SQL query that could delete all documents.
     * <p>
     * This query should not have any parameter.
     * @return The query.
     */
    protected String getDeleteAllQuery() {
        return "DELETE FROM " + getTableName();
    }


}
