;; tests.scm: test procedures for SDOM core
;; Copyright (C) 2010 Julian Graham

;; SDOM free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; SDOM is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SDOM.  If not, see <http://www.gnu.org/licenses/>.

#!r6rs

(import (rnrs))
(import (sxml ssax))
(import (sdom core))
(import (sdom events))
; (import (sdom ls))

(define (current-milliseconds)
  (let ((tod (gettimeofday)))
    (+ (* (car tod) 1000000)
       (cdr tod))))

(define xmlns "http://www.w3.org/XML/1998/namespace")
(define xhtmlns "http://www.w3.org/1999/xhtml")

(define domns "http://www.w3.org/DOM")

(define hc-staff 
  (sdom:xml->sdom (open-input-file "test-xml/hc_staff.xml") '()))
(define hc-nodtd-staff 
  (sdom:xml->sdom (open-input-file "test-xml/hc_nodtdstaff.xml") '()))
(define barfoo (sdom:xml->sdom (open-input-file "test-xml/barfoo.xml") '()))
(define barfoo-base 
  (sdom:xml->sdom (open-input-file "test-xml/barfoo_base.xml") '()))
(define barfoo-nodefaultns
  (sdom:xml->sdom (open-input-file "test-xml/barfoo_nodefaultns.xml") '()))

(define test-error-handler 
  (lambda err (raise (make-message-condition "test-error-handler invoked!"))))

(define attrisid-1 
  (lambda () 
    (let* ((hc-staff-dom (sdom:clone-node hc-staff #t))
	   (elem-list (sdom:get-elements-by-tag-name hc-staff-dom "acronym"))
	   (acronym-elem (list-ref elem-list 0))
	   (attr (sdom:get-attribute-node acronym-elem "title")))
      (if (sdom:is-id attr)
	  (raise (make-message-condition "attr is not id!")) #t))))

(define attrisid-2
  (lambda ()
    (let* ((hc-staff-dom (sdom:clone-node hc-staff #t))
	   (elem-list (sdom:get-elements-by-tag-name hc-staff-dom "acronym"))
	   (acronym-elem (list-ref elem-list 0)))
      (sdom:set-attribute! acronym-elem "xml:lang" "FR-fr" xmlns)
      (sdom:set-id-attribute! acronym-elem "lang" #t xmlns)
      (let ((attr (sdom:get-attribute-node acronym-elem "lang" xmlns)))
	(if (not (sdom:is-id attr))
	    (raise (make-message-condition "attr is id!")) #t)))))

(define attrisid-3
  (lambda ()
    (let* ((hc-staff-dom (sdom:clone-node hc-staff #t))
	   (elem-list (sdom:get-elements-by-tag-name hc-staff-dom "acronym"))
	   (acronym-elem (list-ref elem-list 0)))
      (sdom:set-attribute! acronym-elem "xml:lang" "FR-fr" xmlns)
      (sdom:set-id-attribute! acronym-elem "lang" #f xmlns)
      (let ((attr (sdom:get-attribute-node acronym-elem "lang" xmlns)))
	(if (sdom:is-id attr)
	    (raise (make-message-condition "attr is not id!")) #t)))))

(define attrisid-6
  (lambda ()
    (let* ((hc-staff-dom (sdom:clone-node hc-staff #t))
	   (attr (sdom:create-attribute hc-staff-dom "xml:lang" xmlns)))
      (if (sdom:is-id attr) 
	  (raise (make-message-condition "attr is not id!")) #t))))

(define attrisid-7
  (lambda ()
    (let* ((hc-staff-dom (sdom:clone-node hc-staff #t))
	   (elt-lst (sdom:get-elements-by-tag-name hc-staff-dom "acronym" "*"))
	   (acronym-elem (list-ref elt-lst 0)))
      (sdom:set-attribute! acronym-elem "dom3:newAttr" "null" domns)
      (sdom:set-id-attribute! acronym-elem "newAttr" #t domns)
      (let ((attr (sdom:get-attribute-node acronym-elem "newAttr" domns)))
	(if (not (sdom:is-id attr))
	    (raise (make-message-condition "attr is id!"))
	    (let ((imported-attr (sdom:import-node hc-staff-dom attr #f)))
	      (if (sdom:is-id imported-attr)
		  (raise (make-message-condition "attr is not id!"))
		  #t)))))))

(define canonicalform-1
  (lambda ()
    (let* ((barfoo-dom (sdom:clone-node barfoo #t))
	   (p-list (sdom:get-elements-by-tag-name barfoo-dom "p"))
	   (p-elem (list-ref p-list 0))
	   (entref (sdom:create-entity-reference barfoo-dom "ent1")))
      (sdom:set-dom-config-parameter! barfoo-dom "canonical-form" #t) 
      (sdom:append-child! p-elem entref) 
      (sdom:normalize-document! barfoo-dom)
      (let* ((p-list (sdom:get-elements-by-tag-name barfoo-dom "p"))
	     (p-elem (list-ref p-list 0))
	     (child (sdom:last-child p-elem)))
	(if (not child)
	    (raise (make-message-condition
		    "last child should not be #f after normalization")))
	(if (not (equal? (sdom:node-name child) "#text"))
	    (raise (make-message-condition
		    "first child name should be #text after normalization")))
	(if (not (equal? (sdom:node-value child) "barfoo"))
	    (raise (make-message-condition
		    "first child value should barfoo after normalization"))
	    #t)))))

(define canonicalform-2
  (lambda ()
    (let* ((barfoo-dom (sdom:clone-node barfoo #t))
	   (p-list (sdom:get-elements-by-tag-name barfoo-dom "p"))
	   (p-elem (list-ref p-list 0))
	   (text (sdom:create-text-node barfoo-dom "suc on")))
      (sdom:set-dom-config-parameter! barfoo-dom "canonical-form" #t) 
      (sdom:append-child! p-elem text) 
      (sdom:normalize-document! barfoo-dom)
      (let* ((p-list (sdom:get-elements-by-tag-name barfoo-dom "p"))
	     (p-elem (list-ref p-list 0))
	     (child (sdom:last-child p-elem)))
	(if (not (equal? (sdom:node-value child) "barsuc on"))
	    (raise (make-message-condition
		    "no character normalization under canonical form"))
	    #t)))))

(define canonicalform-3
  (lambda ()
    (let* ((hc-staff-dom (sdom:clone-node hc-staff #t)))
      (sdom:set-dom-config-parameter! hc-staff-dom "canonical-form" #t) 
      (sdom:normalize-document! hc-staff-dom)
      (let* ((elem-list (sdom:get-elements-by-tag-name hc-staff-dom "strong"))
	     (elem-name (list-ref elem-list 1))
	     (text (sdom:last-child elem-name)))
	(if (not (equal? (sdom:node-name text) "#text"))
	    (raise (make-message-condition 
		    "cdata sections should be removed during canonicalization"))
	    #t)))))

(define canonicalform-4
  (lambda ()
    (let* ((barfoo-dom (sdom:clone-node barfoo #t)))
      (sdom:set-dom-config-parameter! barfoo-dom "canonical-form" #t) 
      (sdom:normalize-document! barfoo-dom)
      (let* ((doc-elem (sdom:document-element barfoo-dom))
	     (xmlns-attr (sdom:get-attribute-node doc-elem "xmlns")))
	(if (null? xmlns-attr)
	    (raise (make-message-condition
		    "xmlns should be set on document element"))
	    #t)))))

(define cdatasections-1
  (lambda ()
    (let* ((barfoo-dom (sdom:clone-node barfoo #t))
	   (elem-list (sdom:get-elements-by-tag-name barfoo-dom "p"))
	   (new-cdata (sdom:create-cdata-section barfoo-dom "CDATA")))
      (sdom:append-child! (list-ref elem-list 0) new-cdata)
      (sdom:set-dom-config-parameter! barfoo-dom "cdata-sections" #t)
      (sdom:set-dom-config-parameter! 
       barfoo-dom "error-handler" test-error-handler)
      (sdom:normalize! barfoo-dom)
      (let* ((elem-list (sdom:get-elements-by-tag-name barfoo-dom "p"))
	     (cdata (sdom:last-child (list-ref elem-list 0))))
	(if (not (equal? (sdom:node-name cdata) "#cdata-section"))
	    (raise (make-message-condition
		    "cdata section should be named \"#cdata-section\"") )
	    #t)))))

(define comments-1
  (lambda ()
    (let* ((barfoo-dom (sdom:clone-node barfoo #t))
	   (elem-list (sdom:get-elements-by-tag-name barfoo-dom "p"))
	   (comment (sdom:create-comment barfoo-dom "COMMENT_NODE")))
      (sdom:append-child! (list-ref elem-list 0) comment)
      (sdom:set-dom-config-parameter! barfoo-dom "cdata-sections" #f)
      (sdom:set-dom-config-parameter! 
       barfoo-dom "error-handler" test-error-handler)
      (sdom:normalize! barfoo-dom)
      (let* ((elem-list (sdom:get-elements-by-tag-name barfoo-dom "p"))
	     (cdata (sdom:last-child (list-ref elem-list 0))))
	(if (not (equal? (sdom:node-name cdata) "#comment"))
	    (raise (make-message-condition
		    "comment should be named \"#comment\"")))
	#t))))

(define documentadoptnode-3
  (lambda ()
    (let* ((hc-staff-dom (sdom:clone-node hc-staff #t))
	   (attr (sdom:create-attribute hc-staff-dom "xml:lang" xmlns)))
      (sdom:adopt-node! hc-staff-dom attr)
      (cond ((not (equal? (sdom:node-name attr) "xml:lang"))
	     (raise (make-message-condition 
		     "attr name should be \"xml:lang\"")))
	    ((not (equal? (sdom:namespace-uri attr) xmlns))
	     (raise (make-message-condition 
		     "attr namespace-uri should be xmlns")))
	    ((not (equal? (sdom:prefix attr) "xml"))
	     (raise (make-message-condition "attr prefix should be \"xml\"")))
	    ((sdom:owner-element attr)
	     (raise (make-message-condition
		     "attr owner-element should be null")))
	    ((not (sdom:specified attr))
	     (raise (make-message-condition "attr should be specified"))))
      #t)))

(define documentadoptnode-4
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doc-elem (sdom:document-element doc))
	   (root-name (sdom:tag-name doc-elem))
	   (root-ns (sdom:namespace-uri doc-elem))
	   (new-doc (sdom:create-document root-name #f root-ns))
	   (attr (sdom:create-attribute doc "xml:lang" xmlns)))
      (sdom:adopt-node! new-doc attr)
      (cond ((not (equal? (sdom:node-name attr) "xml:lang"))
	     (raise (make-message-condition 
		     "attr name should be \"xml:lang\"")))
	    ((not (equal? (sdom:namespace-uri attr) xmlns))
	     (raise (make-message-condition
		     "attr namespace-uri should be xmlns")))
	    ((not (equal? (sdom:prefix attr) "xml"))
	     (raise (make-message-condition "attr prefix should be \"xml\"")))
	    ((sdom:owner-element attr)
	     (raise (make-message-condition
		     "attr owner-element should be null")))
	    ((not (sdom:specified attr))
	     (raise (make-message-condition "attr should be specified"))))
      #t)))

(define documentadoptnode-7
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (success #f))
      (guard (ex ((sdom:exception? ex) 
		  (if (eqv? (sdom:exception-code ex)
			    sdom:exception-code-not-supported-err)
		      (set! success #t))))
	     (sdom:adopt-node! doc doc))
      (if (not success)
	  (raise (make-message-condition
		  "adopting self as node should not be supported!"))
	  #t))))

(define documentadoptnode-8
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doc-elem (sdom:document-element doc))
	   (root-name (sdom:tag-name doc-elem))
	   (root-ns (sdom:namespace-uri doc-elem))
	   (new-doc (sdom:create-document root-name #f root-ns))
	   (success #f))
      (guard (ex ((sdom:exception? ex)
		  (if (eqv? (sdom:exception-code ex) 
			    sdom:exception-code-not-supported-err)
		      (set! success #t))))
	     (sdom:adopt-node! doc new-doc))
      (if (not success)
	  (raise (make-message-condition
		  "document adoption should not be supported!"))
	  #t))))

(define documentadoptnode-9
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doc-elem (sdom:document-element doc))
	   (root-name (sdom:tag-name doc-elem))
	   (root-ns (sdom:namespace-uri doc-elem))
	   (new-doc (sdom:create-document root-name #f root-ns))
	   (success #f))
      (guard (ex ((sdom:exception? ex)
		  (if (eqv? (sdom:exception-code ex) 
			    sdom:exception-code-not-supported-err)
		      (set! success #t))))
	     (sdom:adopt-node! new-doc doc))
      (if (not success)
	  (raise (make-message-condition
		  "document adoption should not be supported!"))
	  #t))))

(define documentadoptnode-11
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doc-elem (sdom:document-element doc))
	   (root-name (sdom:tag-name doc-elem))
	   (doctype (sdom:create-document-type root-name '() '()))
	   (success #f))
      (guard (ex ((sdom:exception? ex)
		  (if (eqv? (sdom:exception-code ex) 
			    sdom:exception-code-not-supported-err)
		      (set! success #t))))
	     (sdom:adopt-node! doc doctype))
      (if (not success)
	  (raise (make-message-condition
		  "document type adoption should not be supported!"))
	  #t))))

(define documentadoptnode-12
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doc-elem (sdom:document-element doc))
	   (root-name (sdom:tag-name doc-elem))
	   (root-ns (sdom:namespace-uri doc-elem))
	   (doctype (sdom:create-document-type root-name '() '()))
	   (new-doc (sdom:create-document root-name doctype root-ns))
	   (success #f))
      (guard (ex ((sdom:exception? ex)
		  (if (eqv? (sdom:exception-code ex) 
			    sdom:exception-code-not-supported-err)
		      (set! success #t))))
	     (sdom:adopt-node! new-doc doctype))
      (if (not success)
	  (raise (make-message-condition
		  "document type adoption should not be supported!"))
	  #t))))

(define documentadoptnode-13
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doc-frag (sdom:create-document-fragment doc))
	   (child-list (sdom:get-elements-by-tag-name doc "acronym"))
	   (acronym-node (list-ref child-list 0)))
      (sdom:append-child! doc-frag acronym-node)
      (sdom:adopt-node! doc doc-frag)
      (if (not (sdom:has-child-nodes? doc-frag))
	  (raise (make-message-condition
		  "document fragment children should be adopted recursively"))
	  #t))))

(define documentadoptnode-14
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doc-elem (sdom:document-element doc))
	   (root-name (sdom:tag-name doc-elem))
	   (root-ns (sdom:namespace-uri doc-elem))
	   (new-doc (sdom:create-document root-name #f root-ns))
	   (doc-frag (sdom:create-document-fragment new-doc))
	   (imported (sdom:import-node new-doc doc-elem #t))
	   (doc-elem (sdom:document-element new-doc)))
      (sdom:append-child! doc-elem imported)
      (sdom:append-child! 
       doc-frag (list-ref (sdom:get-elements-by-tag-name new-doc "acronym") 0))
      (sdom:adopt-node! new-doc doc-frag)
      (if (not (sdom:has-child-nodes? doc-frag))
	  (raise (make-message-condition
		  "document fragment children should be adopted recursively"))
	  #t))))

(define documentadoptnode-15
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doc-frag (sdom:create-document-fragment doc)))
      (sdom:adopt-node! doc doc-frag)
      (if (sdom:has-child-nodes? doc-frag)
	  (raise (make-message-condition
		  "newly-created document fragment should have no children"))
	  #t))))

(define documentadoptnode-21
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (child-list (sdom:get-elements-by-tag-name doc "acronym"))
	   (attr (sdom:get-attribute-node (list-ref child-list 0) "title")))
      (sdom:adopt-node! doc attr)
      (if (not (equal? (sdom:node-name attr) "title"))
	  (raise (make-message-condition
		  "node adoption should preserve node-name")))
      (if (not (sdom:attr? attr))
	  (raise (make-message-condition
		  "node adoption should preserve node-type")))
      (if (not (equal? (sdom:node-value attr) "Yes"))
	  (raise (make-message-condition
		  "node adoption should preserve node-value")))
      (if (sdom:owner-element attr)
	  (raise (make-message-condition
		  "owner-element should be null for newly-adopted nodes")))
      #t)))

(define documentadoptnode-22
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doc-elem (sdom:document-element doc))
	   (adopted-node (sdom:adopt-node! doc doc-elem)))
      (if (not (sdom:has-child-nodes? adopted-node))
	  (raise (make-message-condition
		  "children of adopted node should be recursively adopted")))
      (if (not (equal? (sdom:node-name doc-elem) (sdom:node-name adopted-node)))
	  (raise (make-message-condition
		  "node adoption should preserve node-name")))
      #t)))

(define documentadoptnode-23
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (child-list (sdom:get-elements-by-tag-name doc "acronym"))
	   (acronym-elem (list-ref child-list 0))
	   (adopted-node (sdom:adopt-node! doc acronym-elem)))
      (if (not (eqv? (length (sdom:child-nodes acronym-elem))
		     (length (sdom:child-nodes adopted-node))))
	  (raise (make-message-condition
		  "node adoption should preserve number of child nodes"))
	  #t))))

(define documentadoptnode-24
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (new-doc (sdom:create-document 
		     "dom:test" #f "http://www.w3.org/DOM/Test"))
	   (child-list (sdom:get-elements-by-tag-name doc "code" "*"))
	   (code-elem (list-ref child-list 0))
	   (adopted-node (sdom:adopt-node! new-doc code-elem)))
      (if (not (eqv? (length (sdom:child-nodes code-elem))
		     (length (sdom:child-nodes adopted-node))))
	  (raise (make-message-condition
		  "node adoption should preserve number of nodes"))
	  #t))))

(define documentadoptnode-25
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doc-elem (sdom:document-element doc))
	   (root-ns (sdom:namespace-uri doc-elem))
	   (root-name (sdom:tag-name doc-elem))
	   (new-elem (sdom:create-element 
		      doc "th" "http://www.w3.org/1999/xhtml"))
	   (new-doc (sdom:create-document root-name #f root-ns))
	   (adopted-node (sdom:adopt-node! new-doc new-elem)))
      (if (not (equal? "th" (sdom:node-name adopted-node)))
	  (raise (make-message-condition
		  "node adoption should preserve node-name")))
      (if (not (equal? "http://www.w3.org/1999/xhtml"
		       (sdom:namespace-uri adopted-node)))
	  (raise (make-message-condition
		  "node adoption should preserve namespace-uri"))
	  #t))))

(define documentadoptnode-26
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doc-elem (sdom:document-element doc))
	   (root-ns (sdom:namespace-uri doc-elem))
	   (root-tagname (sdom:tag-name doc-elem))
	   (new-doc (sdom:create-document root-tagname #f root-ns))
	   (new-elem (sdom:create-element 
		      new-doc "head" "http://www.w3.org/1999/xhtml")))
      (sdom:set-attribute! doc-elem "xml:lang" "en-US" xmlns)
      (let* ((doc-elem (sdom:document-element new-doc))
	     (appended-child (sdom:append-child! doc-elem new-elem))
	     (adopted-node (sdom:adopt-node! doc new-elem)))
	(if (not (equal? (sdom:node-name adopted-node) "head"))
	    (raise (make-message-condition
		    "node adoption should preserve node-name")))
	(if (not (equal? (sdom:namespace-uri adopted-node)
			 xhtmlns))
	    (raise (make-message-condition
		    "node adoption should preserve namespace-uri"))
	    #t)))))

(define documentadoptnode-27
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doc-elem (sdom:document-element doc))
	   (root-ns (sdom:namespace-uri doc-elem))
	   (root-tagname (sdom:tag-name doc-elem))
	   (new-doc (sdom:create-document root-tagname #f root-ns))
	   (new-elem (sdom:create-element new-doc "xhtml:head" xhtmlns)))
      (sdom:set-attribute! new-elem "xml:lang" "en-US" xmlns)
      (let* ((doc-elem (sdom:document-element new-doc))
	     (appended-child (sdom:append-child! doc-elem new-elem))
	     (new-imp-elem (sdom:import-node doc new-elem #t))
	     (adopted-node (sdom:adopt-node! doc new-imp-elem)))
	(if (not (equal? (sdom:node-name adopted-node) "xhtml:head"))
	    (raise (make-message-condition
		    "node adoption should preserve node-name")))
	(if (not (equal? (sdom:namespace-uri adopted-node) xhtmlns))
	    (raise (make-message-condition
		    "node adoption should preserve namespace-uri"))
	    #t)))))

(define documentadoptnode-30
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (new-text (sdom:create-text-node 
		      doc "sdom:adopt-node! test for a TEXT_NODE"))
	   (adopted-text (sdom:adopt-node! doc new-text)))
      (or (equal? (sdom:node-value adopted-text)
		  "sdom:adopt-node! test for a TEXT_NODE")
	  (raise (make-message-condition
		  "node adoption should preserve text node value"))))))

(define documentadoptnode-31
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doc-elem (sdom:document-element doc))
	   (root-name (sdom:tag-name doc-elem))
	   (root-ns (sdom:namespace-uri doc-elem))
	   (new-doc (sdom:create-document root-name #f root-ns))
	   (new-text (sdom:create-text-node 
		      new-doc "new sdom:adopt-node! test for a TEXT_NODE"))
	   (adopted-text (sdom:adopt-node! doc new-text)))
      (if (not (equal? (sdom:node-value	adopted-text)
		       "new sdom:adopt-node! test for a TEXT_NODE"))
	  (raise (make-message-condition
		  "node adoption should preserve text node value")) 
	  #t))))

(define documentadoptnode-32
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doc-adopter (sdom:clone-node hc-staff #t))
	   (new-cdata (sdom:create-cdata-section
		       doc "sdom:adopt-node! test for a CDATASECTION_NODE"))
	   (adopted-cdata (sdom:adopt-node! doc-adopter new-cdata)))
      (if (not (equal? (sdom:node-value	adopted-cdata)
		       "sdom:adopt-node! test for a CDATASECTION_NODE"))
	  (raise (make-message-condition
		  "node adoption should preserve cdata section value"))
	  #t))))

(define documentadoptnode-33
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doc-elem (sdom:document-element doc))
	   (root-ns (sdom:namespace-uri doc-elem))
	   (root-name (sdom:tag-name doc-elem))
	   (new-doc (sdom:create-document root-name #f root-ns))
	   (new-cdata (sdom:create-cdata-section
		       new-doc
		       "sdom:adopt-node! test for a CDATASECTION_NODE"))
	   (adopted-cdata (sdom:adopt-node! doc new-cdata)))
      (if (not (equal? (sdom:node-value	adopted-cdata)
		       "sdom:adopt-node! test for a CDATASECTION_NODE"))
	  (raise (make-message-condition
		  "node adoption should preserve cdata section value"))
	  #t))))

(define documentadoptnode-34
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doc-elem (sdom:document-element doc))
	   (root-ns (sdom:namespace-uri doc-elem))
	   (root-name (sdom:tag-name doc-elem))
	   (new-doc (sdom:create-document root-name #f root-ns))
	   (new-comment (sdom:create-comment
			 new-doc
			 "sdom:adopt-node! test for a COMMENT_NODE"))
	   (adopted-comment (sdom:adopt-node! new-doc new-comment)))
      (if (not (equal? (sdom:node-value adopted-comment)
		       "sdom:adopt-node! test for a COMMENT_NODE"))
	  (raise (make-message-condition
		  "node adoption should preserve comment value"))
	  #t))))

(define documentadoptnode-35
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doc-elem (sdom:document-element doc))
	   (root-ns (sdom:namespace-uri doc-elem))
	   (root-name (sdom:tag-name doc-elem))
	   (new-doc (sdom:create-document root-name #f root-ns))
	   (new-pi (sdom:create-processing-instruction
		    new-doc "PITarget" "PIData"))
	   (adopted-pi (sdom:adopt-node! doc new-pi)))
      (if (not (equal? (sdom:target adopted-pi) "PITarget"))
	  (raise (make-message-condition
		  "node adoption should preserve PI target")))
      (if (not (equal? (sdom:data adopted-pi) "PIData"))
	  (raise (make-message-condition
		  "node adoption should preserve PI data"))
	  #t))))

(define documentadoptnode-36
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doc-elem (sdom:document-element doc))
	   (root-ns (sdom:namespace-uri doc-elem))
	   (root-name (sdom:node-name doc-elem))
	   (new-doc (sdom:create-document root-name #f root-ns))
	   (new-pi-1 (sdom:create-processing-instruction
		      new-doc "PITarget" "PIData"))
	   (new-pi-2 (sdom:create-processing-instruction
		      doc "PITarget" "PIData"))
	   (adopted-pi-1 (sdom:adopt-node! new-doc new-pi-1))
	   (adopted-pi-2 (sdom:adopt-node! new-doc new-pi-2)))
      (if (or (not (equal? (sdom:target adopted-pi-1) "PITarget"))
	      (not (equal? (sdom:target adopted-pi-2) "PITarget")))
	  (raise (make-message-condition
		  "node adoption should preserve PI target")))
      (if (or (not (equal? (sdom:data adopted-pi-1) "PIData"))
	      (not (equal? (sdom:data adopted-pi-2) "PIData")))
	  (raise (make-message-condition
		  "node adoption should preserve PI data"))
	  #t))))

(define documentgetstricterrorchecking-1
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (strict (sdom:get-dom-config-parameter 
		    doc "strict-error-checking")))
      (if (not strict) 
	  (raise (make-message-condition
		  "error checking should be strict by default"))
	  #t))))

(define documentgetstricterrorchecking-2
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doc-elem (sdom:document-element doc))
	   (root-ns (sdom:namespace-uri doc-elem))
	   (root-name (sdom:tag-name doc-elem))
	   (new-doc (sdom:create-document root-name #f root-ns))
	   (strict (sdom:get-dom-config-parameter 
		    new-doc "strict-error-checking")))
      (if (not strict)
	  (raise (make-message-condition
		  "error checking should be strict for new documents"))
	  #t))))

(define documentnormalizedocument-1
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doc-elem (sdom:document-element doc))
	   (orig-node-name (sdom:node-name doc-elem))
	   (found-error #f))
      (sdom:set-dom-config-parameter! doc "error-handler"
				      (lambda err (set! found-error #t)))
      (sdom:normalize-document! doc)
      (if found-error (raise (make-message-condition 
			      "error during document normalization")))
      (let* ((doc-elem (sdom:document-element doc))
	     (node-name (sdom:node-name doc-elem)))
	(if (not (equal? orig-node-name node-name))
	    (raise (make-message-condition
		    "document normalization should not affect node name"))
	    #t)))))

(define documentnormalizedocument-3
  (lambda () 
    (let* ((doc (sdom:clone-node barfoo #t))
	   (p-list (sdom:get-elements-by-tag-name doc "p"))
	   (elem (list-ref p-list 0))
	   (new-cdata (sdom:create-cdata-section doc "CDATA"))
	   (appended-child (sdom:append-child! elem new-cdata))
	   (found-error #f))
      (sdom:set-dom-config-parameter! doc "cdata-sections" #t)
      (sdom:set-dom-config-parameter! doc "error-handler"
				      (lambda err (set! found-error #t)))
      (sdom:normalize-document! doc)
      (if found-error (raise (make-message-condition
			      "error during document normalization")))
      (let* ((p-list (sdom:get-elements-by-tag-name doc "p"))
	     (elem (list-ref p-list 0))
	     (cdata (sdom:last-child elem))
	     (node-name (sdom:node-name cdata)))
	(if (not (equal? node-name "#cdata-section"))
	    (raise (make-message-condition
		    "document normalization should preserve cdata sections")))
	(sdom:set-dom-config-parameter! doc "cdata-sections" #f)
	(sdom:normalize-document! doc)
	(if found-error 
	    (raise (make-message-condition 
		    "error during document normalization")))
	(let* ((p-list (sdom:get-elements-by-tag-name doc "p"))
	       (elem (list-ref p-list 0))
	       (text (sdom:last-child elem))
	       (node-name (sdom:node-name text)))
	  (if (not (equal? node-name "#text"))
	      (raise (make-message-condition 
		      "document normalization should convert cdata sections")))
	  (let* ((node-value (sdom:node-value text)))
	    (if (not (equal? node-value "barCDATA"))
		(raise (make-message-condition
			"cdata section content should be converted"))
		#t)))))))

(define documentnormalizedocument-4
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo #t))
	   (p-list (sdom:get-elements-by-tag-name doc "p"))
	   (elem (list-ref p-list 0))
	   (new-comment (sdom:create-comment doc "COMMENT_NODE"))
	   (appended-child (sdom:append-child! elem new-comment))
	   (found-error #f))	 
      (sdom:set-dom-config-parameter! doc "comments" #t)
      (sdom:set-dom-config-parameter! doc "error-handler" 
				      (lambda err (set! found-error #t)))
      (sdom:normalize-document! doc)
      (if found-error (raise (make-message-condition
			      "error during document normalization")))
      (let* ((p-list (sdom:get-elements-by-tag-name doc "p"))
	     (elem (list-ref p-list 0))
	     (last-child (sdom:last-child elem))
	     (node-name (sdom:node-name last-child)))
	(if (not (equal? node-name "#comment"))
	    (raise (make-message-condition
		    "document normalization should preserve comments")))
	(sdom:set-dom-config-parameter! doc "comments" #f)
	(sdom:normalize-document! doc)
	(if found-error (raise (make-message-condition
				"error during document normalization")))
	(let* ((p-list (sdom:get-elements-by-tag-name doc "p"))
	       (elem (list-ref p-list 0))
	       (last-child (sdom:last-child elem))
	       (node-name (sdom:node-name last-child)))
	  (if (not (equal? node-name "#text"))
	      (raise (make-message-condition
		      "document normalization should remove comments"))
	      #t))))))

(define documentnormalizedocument-8
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo #t))
	   (elem-list (sdom:get-elements-by-tag-name doc "p"))
	   (elem (list-ref elem-list 0))
	   (new-child (sdom:create-cdata-section doc "this is not ]]> good"))
	   (old-child (sdom:first-child elem))
	   (ret-val (sdom:replace-child! elem new-child old-child))
	   (new-child (sdom:create-cdata-section doc "this is not ]]> good"))
	   (ret-val (sdom:append-child! elem new-child))
	   (split-warnings 0))
      (sdom:set-dom-config-parameter! doc "split-cdata-sections" #t)
      (sdom:set-dom-config-parameter! 
       doc "error-handler" 
       (lambda (s m t e d l)
	 (if (eqv? s sdom:error-severity-warning)
	     (if (equal? t "cdata-sections-splitted")
		 (set! split-warnings (+ split-warnings 1)))
	     (raise (make-message-condition
		     "error or severe error during normalization")))))
      (sdom:normalize-document! doc)
      (if (not (eqv? split-warnings 2))
	  (raise (make-message-condition
		  "normalization should have split 2 cdata sections")))
      (let* ((elem-list (sdom:get-elements-by-tag-name doc "p"))
	     (elem (list-ref elem-list 0))
	     (child-nodes (sdom:child-nodes elem))
	     (len (length child-nodes)))
	(if (not (> len 3))
	    (raise (make-message-condition
		    "document normalize should split cdata sections"))
	    #t)))))

(define documentnormalizedocument-11
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t)))
      (sdom:set-dom-config-parameter! doc "namespace-declarations" #t)
      (sdom:normalize-document! doc)
      (let ((elem-list (sdom:get-elements-by-tag-name doc "acronym" "*")))
	(if (null? elem-list)
	    (raise (make-message-condition
		    "normalization should preserve node name")))
	(if (sdom:can-set-dom-config-parameter? doc "namespace-declarations" 
						#f)
	    (sdom:set-dom-config-parameter! doc "namespace-declarations" #f))
	(sdom:normalize-document! doc)
	(let* ((elem-list (sdom:get-elements-by-tag-name doc "acronym" "*"))
	       (elem-name (list-ref elem-list 0))
	       (node-name (sdom:node-name elem-name)))
	  (if (not (equal? node-name "acronym"))
	      (raise (make-message-condition
		      "normalization should preserve node name"))
	      #t))))))

(define documentrenamenode-1
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (child-list (sdom:get-elements-by-tag-name doc "acronym"))
	   (element (list-ref child-list 0))
	   (attr (sdom:get-attribute-node element "title"))
	   (renamedclass (sdom:rename-node! 
			  attr "renamedNode" "http://www.w3.org/DOM/Test"))
	   (node-name (sdom:node-name renamedclass))
	   (namespace-uri (sdom:namespace-uri renamedclass))
	   (node-type (sdom:node-type renamedclass)))
      (if (not (equal? node-name "renamedNode"))
	  (raise (make-message-condition
		  "renaming node should change node name")))
      (if (not (eqv? node-type 2))
	  (raise (make-message-condition
		  "renaming node should preserve node type")))
      (if (not (equal? namespace-uri "http://www.w3.org/DOM/Test"))
	  (raise (make-message-condition
		  "renaming node should change namespace URI")))
      #t)))

(define documentrenamenode-2
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (child-list (sdom:get-elements-by-tag-name doc "acronym"))
	   (element (list-ref child-list 0))
	   (attr (sdom:get-attribute-node element "title"))
	   (renamedclass (sdom:rename-node! attr "prefi0x:renamedNode"
					    "http://www.w3.org/DOM/Test"))
	   (node-name (sdom:node-name renamedclass))
	   (namespace-uri (sdom:namespace-uri renamedclass))
	   (node-type (sdom:node-type renamedclass)))
      (if (not (equal? node-name "prefi0x:renamedNode"))
	  (raise (make-message-condition "renaming node should change prefix")))
      (if (not (equal? namespace-uri "http://www.w3.org/DOM/Test"))
	  (raise (make-message-condition
		  "renaming node should change namespace URI")))
      #t)))

(define documentrenamenode-3
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (attr (sdom:create-attribute doc "test"))
	   (renamed-node (sdom:rename-node! attr "pre0:fix1" 
					    "http://www.w3.org/DOM/Test"))
	   (node-name (sdom:node-name renamed-node))
	   (namespace-uri (sdom:namespace-uri renamed-node)))
      (if (not (equal? node-name "pre0:fix1"))
	  (raise (make-message-condition
		  "renaming node should change node name")))
      (if (not (equal? namespace-uri "http://www.w3.org/DOM/Test"))
	  (raise (make-message-condition
		  "renaming node should change namespace URI")))
      #t)))

(define documentrenamenode-4
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (attr (sdom:create-attribute doc "xml:lang" xmlns))
	   (renamed-node (sdom:rename-node! attr "title" ""))
	   (node-name (sdom:node-name renamed-node))
	   (namespace-uri (sdom:namespace-uri renamed-node)))
      (if (not (equal? node-name "title"))
	  (raise (make-message-condition 
		  "renaming node should change node name")))
      (if namespace-uri (make-message-condition 
			 "renaming node should change namespace URI"))
      #t)))

(define documentrenamenode-5
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (attr (sdom:create-attribute doc "xml:lang" xmlns))
	   (renamed-node (sdom:rename-node! attr "title" '()))
	   (node-name (sdom:node-name renamed-node))
	   (namespace-uri (sdom:namespace-uri renamed-node)))
      (if (not (equal? node-name "title"))
	  (raise (make-message-condition 
		  "renaming node should change node name")))
      (if namespace-uri (make-message-condition 
			 "renaming node should change namespace URI"))
      #t)))

(define documentrenamenode-7
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doc-elem (sdom:document-element doc))
	   (root-ns (sdom:namespace-uri doc-elem))
	   (root-name (sdom:tag-name doc-elem))
	   (new-doc (sdom:create-document root-name #f root-ns))
	   (attr (sdom:create-attribute new-doc "xml:lang" xmlns))
	   (renamed-node (sdom:rename-node! attr "xml:dom" xmlns))
	   (node-name (sdom:node-name renamed-node))
	   (namespace-uri (sdom:namespace-uri renamed-node)))
      (if (not (equal? node-name "xml:dom"))
	  (raise (make-message-condition
		  "renaming node should change node name")))
      (if (not (equal? namespace-uri xmlns))
	  (raise (make-message-condition
		  "renaming node should change namespace URI")))
      #t)))

(define documentrenamenode-10
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (text-node (sdom:create-text-node doc "hello"))
	   (qualified-names (list "_:" ":0" ":" "a0:0" "_:0;" "a:::::c"))
	   (success #f))
      (for-each 
       (lambda (x) 
	 (guard (ex ((sdom:exception? ex) 
		     (if (eqv? (sdom:exception-code ex)
			       sdom:exception-code-not-supported-err)
			 (set! success #t))))
		(sdom:rename-node! text-node x xmlns)))
       qualified-names)
      (if (not success)
	  (raise (make-message-condition
		  "renaming node should raise error on invalid node type")))
      #t)))

(define documentrenamenode-11
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (text-node (sdom:create-text-node doc "hello"))
	   (success #f))
      (guard (ex ((sdom:exception? ex)
		  (if (eqv? (sdom:exception-code ex)
			    sdom:exception-code-not-supported-err)
		      (set! success #t))))
	     (sdom:rename-node! text-node "pre:fix" '()))
      (if (not success)
	  (raise (make-message-condition
		  "renaming node should raise error on invalid node type")))
      #t)))

(define documentrenamenode-12
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (text-node (sdom:create-text-node doc "hello"))
	   (success #f))
      (guard (ex ((sdom:exception? ex)
		  (if (eqv? (sdom:exception-code ex)
			    sdom:exception-code-not-supported-err)
		      (set! success #t))))
	     (sdom:rename-node! text-node "xml:prefix" xmlns))
      (if (not success)
	  (raise (make-message-condition
		  "renaming node should raise error on invalid node type")))
      #t)))

(define documentrenamenode-13
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (text-node (sdom:create-text-node doc "hello"))
	   (success #f))
      (guard (ex ((sdom:exception? ex)
		  (if (eqv? (sdom:exception-code ex)
			    sdom:exception-code-not-supported-err)
		      (set! success #t))))
	     (sdom:rename-node! text-node "xmlns:prefix" xmlns))
      (if (not success)
	  (raise (make-message-condition
		  "renaming node should raise error on invalid node type")))
      #t)))

(define documentrenamenode-14
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (text-node (sdom:create-text-node doc "hello"))
	   (success #f))
      (guard (ex ((sdom:exception? ex)
		  (if (eqv? (sdom:exception-code ex)
			    sdom:exception-code-not-supported-err)
		      (set! success #t))))
	     (sdom:rename-node! 
	      text-node "xmlns" "http://www.w3.org/2000/xmlns"))
      (if (not success)
	  (raise (make-message-condition
		  "renaming node should raise error on invalid node type")))
      #t)))

(define documentrenamenode-15
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (child-list (sdom:get-elements-by-tag-name doc "acronym"))
	   (element (list-ref child-list 0))
	   (renamedclass (sdom:rename-node! element "qnam:renamedNode"
					    "http://www.w3.org/DOM/Test"))
	   (node-name (sdom:node-name renamedclass))
	   (namespace-uri (sdom:namespace-uri renamedclass))
	   (node-type (sdom:node-type renamedclass)))
      (if (not (equal? node-name "qnam:renamedNode"))
	  (raise (make-message-condition
		  "renaming node should change node name")))
      (if (not (eqv? node-type 1))
	  (raise (make-message-condition
		  "renaming node should preserve node type")))
      (if (not (equal? namespace-uri "http://www.w3.org/DOM/Test"))
	  (raise (make-message-condition
		  "renaming node should change namespace URI")))
      #t)))

(define documentrenamenode-16
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (child-list (sdom:get-elements-by-tag-name doc "acronym"))
	   (element (list-ref child-list 0))
	   (renamedclass (sdom:rename-node! element "renamedNode" #f))
	   (node-name (sdom:node-name renamedclass))
	   (namespace-uri (sdom:namespace-uri renamedclass)))
      (if (not (equal? node-name "renamedNode"))
	  (raise (make-message-condition 
		  "renaming node should change node name")))
      (if (not (sdom:element? element))
	  (raise (make-message-condition
		  "renaming node should preserve node type")))
      (if namespace-uri (raise (make-message-condition
				"renaming node should change namespace URI")))
      #t)))

(define documentrenamenode-17
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo #t))
	   (doc-elem (sdom:document-element doc))
	   (root-ns (sdom:namespace-uri doc-elem))
	   (root-tagname (sdom:tag-name doc-elem))
	   (new-doc (sdom:create-document root-tagname #f root-ns))
	   (element (sdom:create-element new-doc "body" xhtmlns))
	   (renamed-node (sdom:rename-node! element "xhtml:head" xhtmlns))
	   (node-name (sdom:node-name renamed-node))
	   (namespace-uri (sdom:namespace-uri renamed-node))
	   (node-type (sdom:node-type renamed-node)))
      (if (not (equal? node-name "xhtml:head"))
	  (raise (make-message-condition
		  "renaming node should change node name")))
      (if (not (eqv? node-type 1))
	  (raise (make-message-condition
		  "renaming node should preserve node type")))
      (if (not (equal? namespace-uri xhtmlns))
	  (raise (make-message-condition
		  "renaming node should change namespace URI")))
      #t)))

(define documentrenamenode-19
  (lambda ()
    (let* ((qualified-names (list "a_:" "_:" ":" "::0;" "a:-:c"))
	   (doc (sdom:clone-node hc-staff #t))
	   (new-doc (sdom:create-document 
		     "newD" #f "http://www.w3.org/DOM/Test"))
	   (element (sdom:create-element doc "test"
					 "http://www.w3.org/DOM/Test"))
	   (success #f))
      (for-each 
       (lambda (x) 
	 (guard (ex ((sdom:exception? ex)
		     (if (or (eqv? (sdom:exception-code ex)
				   sdom:exception-code-namespace-err)
			     (eqv? (sdom:exception-code ex)
				   sdom:exception-code-invalid-character-err))
			 (set! success #t))))
		(sdom:rename-node! 
		 element x "http://www.w3.org/2000/XMLNS"))
	 (if (not success)
	     (raise (make-message-condition
		     "renaming node should fail with bad qname"))))
       
       qualified-names)
      #t)))

(define documentrenamenode-20
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo #t))
	   (doc-elem (sdom:document-element doc))
	   (root-ns (sdom:namespace-uri doc-elem))
	   (root-tagname (sdom:tag-name doc-elem))
	   (element (sdom:create-element doc root-tagname root-ns))
	   (success #f))
      (guard (ex ((sdom:exception? ex)
		  (if (eqv? (sdom:exception-code ex)
			    sdom:exception-code-namespace-err)
		      (set! success #t))))
	     (sdom:rename-node! 
	      element "xml:html" "http://www.example.com/xml"))
      (if (not success)
	  (raise (make-message-condition
		  "renaming node should fail with illegal namespace"))
	  #t))))

(define documentrenamenode-21
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo #t))
	   (doc-elem (sdom:document-element doc))
	   (root-ns (sdom:namespace-uri doc-elem))
	   (root-tagname (sdom:tag-name doc-elem))
	   (attr (sdom:create-attribute doc "xml:lang" xmlns))
	   (success #f))
      (guard (ex ((sdom:exception? ex)
		  (if (eqv? (sdom:exception-code ex)
			    sdom:exception-code-namespace-err)
		      (set! success #t))))
	     (sdom:rename-node! 
	      attr "xmlns:xml" "http://www.w3.org/2000/XMLNS"))
      (if (not success)
	  (raise (make-message-condition
		  "renaming node should fail with illegal namespace"))
	  #t))))

(define documentrenamenode-22
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (attr (sdom:create-attribute doc "xml:lang" xmlns))
	   (success #f))
      (guard (ex ((sdom:exception? ex)
		  (if (eqv? (sdom:exception-code ex)
			    sdom:exception-code-namespace-err)
		      (set! success #t))))
	     (sdom:rename-node! attr "xmlns" "http://www.w3.org/1999/xmlns/"))
      (if (not success)
	  (raise (make-message-condition
		  "renaming node should fail with illegal namespace"))
	  #t))))

(define documentrenamenode-23
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (success #f))
      (guard (ex ((sdom:exception? ex)
		  (if (eqv? (sdom:exception-code ex)
			    sdom:exception-code-not-supported-err)
		      (set! success #t))))
	     (sdom:rename-node! doc "root" "http://www.w3.org/DOM/Test"))
      (if (not success)
	  (raise (make-message-condition
		  "renaming node should fail on illegal node type"))
	  #t))))

(define documentrenamenode-24
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (success #f))
      (guard (ex ((sdom:exception? ex)
		  (if (eqv? (sdom:exception-code ex)
			    sdom:exception-code-not-supported-err)
		      (set! success #t))))
	     (sdom:rename-node! doc "doc:root" #f))
      (if (not success)
	  (raise (make-message-condition
		  "renaming node should fail on illegal node type"))
	  #t))))

(define documentrenamenode-26
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doc-frag (sdom:create-document-fragment doc))
	   (success #f))
      (guard (ex ((sdom:exception? ex)
		  (if (eqv? (sdom:exception-code ex)
			    sdom:exception-code-not-supported-err)
		      (set! success #t))))
	     (sdom:rename-node! doc-frag "root" "http://www.w3.org/DOM/Test"))
      (if (not success)
	  (raise (make-message-condition
		  "renaming node should fail on illegal node type"))
	  #t))))

(define documentrenamenode-27
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doc-elem (sdom:document-element doc))
	   (root-ns (sdom:namespace-uri doc-elem))
	   (root-name (sdom:tag-name doc-elem))
	   (new-doc (sdom:create-document root-name #f root-ns))
	   (text (sdom:create-text-node new-doc "text"))
	   (comment (sdom:create-comment new-doc "comment"))
	   (cdata (sdom:create-cdata-section new-doc "cdata"))
	   (pi (sdom:create-processing-instruction new-doc "pit" "pid"))
	   (entref (sdom:create-entity-reference new-doc "alpha")))
      (let ((success #f))
	(guard (ex ((sdom:exception? ex)
		    (if (eqv? (sdom:exception-code ex)
			      sdom:exception-code-not-supported-err)
			(set! success #t))))
	       (sdom:rename-node! text "text" "http://www.w3.org/DOM/Test"))
	(if (not success) 
	    (raise (make-message-condition
		    "renaming node should fail for text node"))))
      (let ((success #f))
	(guard (ex ((sdom:exception? ex)
		    (if (eqv? (sdom:exception-code ex)
			      sdom:exception-code-not-supported-err)
			(set! success #t))))
	       (sdom:rename-node! 
		comment "comment" "http://www.w3.org/DOM/Test"))
	(if (not success)
	    (raise (make-message-condition
		    "renaming node should fail for comment node"))))
      (let ((success #f))
	(guard (ex ((sdom:exception? ex)
		    (if (eqv? (sdom:exception-code ex)
			      sdom:exception-code-not-supported-err)
			(set! success #t))))
	       (sdom:rename-node! cdata "cdata" "http://www.w3.org/DOM/Test"))
	(if (not success)
	    (raise (make-message-condition
		    "renaming node should fail for cdata section"))))
      (let ((success #f))
	(guard (ex ((sdom:exception? ex)
		    (if (eqv? (sdom:exception-code ex)
			      sdom:exception-code-not-supported-err)
			(set! success #t))))
	       (sdom:rename-node! pi "pi" "http://www.w3.org/DOM/Test"))
	(if (not success)
	    (raise 
	     (make-message-condition 
	      "renaming node should fail on processing instruction nodes"))))
      (let ((success #f))
	(guard (ex ((sdom:exception? ex)
		    (if (eqv? (sdom:exception-code ex)
			      sdom:exception-code-not-supported-err)
			(set! success #t))))
	       (sdom:rename-node! entref "entref" "http://www.w3.org/DOM/Test"))
	(if (not success)
	    (raise (make-message-condition
		    "renaming node should fail on entity reference"))))
      #t)))

(define documentrenamenode-29
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doc-elem (sdom:document-element doc))
	   (success #f))
      (guard (ex ((sdom:exception? ex)
		  (if (eqv? (sdom:exception-code ex)
			    sdom:exception-code-invalid-character-err)
		      (set! success #t))))
	     (sdom:rename-node! doc-elem "@" "http://www.w3.org/DOM/Test"))
      (if (not success)
	  (raise (make-message-condition
		  "renaming node should fail for invalid characters"))
	  #t))))

(define domconfigcdatasections-1
  (lambda ()
    (let* ((parameter "cDaTa-sections")
	   (doc (sdom:create-document "html" #f xhtmlns))
	   (state (sdom:get-dom-config-parameter doc parameter)))
      (if (not state) (raise (make-message-condition
			      "cdata-sections should be true by default")))
      (if (not (sdom:can-set-dom-config-parameter? doc parameter #f))
	  (raise (make-message-condition
		  "should be able to set cdata-sections to #f")))
      (if (not (sdom:can-set-dom-config-parameter? doc parameter #t))
	  (raise (make-message-condition
		  "should be able to set cdata-sections to #t")))
      (sdom:set-dom-config-parameter! doc parameter #f)
      (let ((state (sdom:get-dom-config-parameter doc parameter)))
	(if state (raise (make-message-condition
			  "cdata-sections should be #f after set!"))
	    #t)))))

(define domconfigcomments-1
  (lambda ()
    (let* ((parameter "cOmments")
	   (doc (sdom:create-document "html" #f xhtmlns))
	   (state (sdom:get-dom-config-parameter doc parameter)))
      (if (not state) (raise (make-message-condition
			      "comments should be true by default")))
      (if (not (sdom:can-set-dom-config-parameter? doc parameter #f))
	  (raise (make-message-condition
		  "should be able to set comments to #f")))
      (if (not (sdom:can-set-dom-config-parameter? doc parameter #t))
	  (raise (make-message-condition
		  "should be able to set comments to #t")))
      (sdom:set-dom-config-parameter! doc parameter #f)
      (let ((state (sdom:get-dom-config-parameter doc parameter)))
	(if state (raise (make-message-condition
			  "comments should be #f after set!"))
	    #t)))))

(define domconfigentities-1
  (lambda ()
    (let* ((parameter "eNtIties")
	   (doc (sdom:create-document "html" #f xhtmlns))
	   (state (sdom:get-dom-config-parameter doc parameter)))
      (if (not state) (raise (make-message-condition
			      "entities should be true by default")))
      (if (not (sdom:can-set-dom-config-parameter? doc parameter #f))
	  (raise (make-message-condition
		  "should be able to set entities to #f")))
      (if (not (sdom:can-set-dom-config-parameter? doc parameter #t))
	  (raise (make-message-condition
		  "should be able to set entities to #t")))
      (sdom:set-dom-config-parameter! doc parameter #f)
      (let ((state (sdom:get-dom-config-parameter doc parameter)))
	(if state (raise (make-message-condition
			  "entities should be #f after set!"))
	    #t)))))

(define domconfigerrorhandler-1
  (lambda ()
    (let* ((parameter "eRrOr-handler")
	   (doc (sdom:create-document "html" #f xhtmlns))
	   (error-handler (lambda () #t))
	   (orig-handler (sdom:get-dom-config-parameter doc parameter)))
      (if (not (sdom:can-set-dom-config-parameter? 
		doc parameter error-handler))
	  (raise (make-message-condition
		  "should be able to set new error-handler")))
      (if (not (sdom:can-set-dom-config-parameter? doc parameter orig-handler))
	  (raise (make-message-condition
		  "should be able to reset original error-handler")))
      (sdom:set-dom-config-parameter! doc parameter error-handler)
      (let ((state (sdom:get-dom-config-parameter doc parameter)))
	(if (not (eq? state error-handler))
	    (raise (make-message-condition
		    "error-handler should be new after set!")))
	(sdom:set-dom-config-parameter! doc parameter orig-handler)
	(let ((state (sdom:get-dom-config-parameter doc parameter)))
	  (if (not (eq? state orig-handler))
	      (raise (make-message-condition
		      "error-handler should be new after set!"))
	      #t))))))

(define domconfigerrorhandler-2
  (lambda ()
    (let* ((parameter "error-handler")
	   (doc (sdom:create-document "html" #f xhtmlns)))
      (if (not (sdom:can-set-dom-config-parameter? doc parameter '()))
	  (raise (make-message-condition
		  "should be able to set null error handler")))
      (sdom:set-dom-config-parameter! doc parameter error-handler)
      (let ((state (sdom:get-dom-config-parameter doc parameter)))
	(if state (raise (make-message-condition
			  "error-handler should be null after set!"))
	    #t)))))

(define domconfiginfoset-1
  (lambda ()
    (let* ((parameter "iNfOset")
	   (doc (sdom:create-document "html" #f xhtmlns)))
      (if (sdom:get-dom-config-parameter doc parameter)
	  (raise (make-message-condition
		  "infoset should be false before setting")))
      (if (not (sdom:can-set-dom-config-parameter? doc parameter #f))
	  (raise (make-message-condition
		  "should be able to set infoset to false")))
      (if (not (sdom:can-set-dom-config-parameter? doc parameter #t))
	  (raise (make-message-condition
		  "should be able to set infoset to true")))
      (sdom:set-dom-config-parameter! doc parameter #t)
      (if (not (sdom:get-dom-config-parameter doc parameter))
	  (raise (make-message-condition
		  "infoset should be true after set")))
      (if (sdom:get-dom-config-parameter doc "entities")
	  (raise (make-message-condition
		  "entities should be false after setting infoset true")))
      (if (sdom:get-dom-config-parameter doc "cdata-sections")
	  (raise (make-message-condition
		  "cdata-sections should be false after setting infoset")))
      (sdom:set-dom-config-parameter! doc parameter #f)
      (if (not (sdom:get-dom-config-parameter doc parameter))
	  (raise (make-message-condition
		  "setting infoset false should be a no-op")))
      (sdom:set-dom-config-parameter! doc "entities" #t)
      (if (not (sdom:get-dom-config-parameter doc parameter))
	  (raise (make-message-condition
		  "setting entities true should invalidate infoset"))
	  #t))))

(define domconfignamespacedeclarations-1
  (lambda ()
    (let* ((parameter "nAmEspace-declarations")
	   (doc (sdom:create-document "html" #f xhtmlns)))
      (if (not (sdom:get-dom-config-parameter doc parameter))
	  (raise (make-message-condition
		  "namespace-declarations should be true before setting")))
      (if (not (sdom:can-set-dom-config-parameter? doc parameter #f))
	  (raise (make-message-condition
		  "should be able to set namespace-declarations to false")))
      (if (not (sdom:can-set-dom-config-parameter? doc parameter #t))
	  (raise (make-message-condition
		  "should be able to set namespace-declarations to true")))
      (sdom:set-dom-config-parameter! doc parameter #t)
      (if (not (sdom:get-dom-config-parameter doc parameter))
	  (raise (make-message-condition
		  "namespace-declaration should be true after set")))
      (sdom:set-dom-config-parameter! doc parameter #f)
      (if (sdom:get-dom-config-parameter doc parameter)
	  (raise (make-message-condition
		  "namespace-declaration should be false after set"))
	  #t))))

(define domconfigparameternames-1
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo #t))
	   (match-count 0))
      (for-each (lambda (x) 
		  (let ((def (sdom:get-dom-config-parameter doc x)))
		    (if (not (sdom:can-set-dom-config-parameter? doc x def))
			(raise (make-message-condition
				"should be able to set config to default")))
		    (sdom:set-dom-config-parameter! doc x def)
		    (if (member x `("canonical-form" 
				    "cdata-sections" 
				    "check-character-normalization"
				    "comments" 
				    "datatype-normalization" 
				    "element-content-whitespace" 
				    "entities"
				    "error-handler" 
				    "infoset" 
				    "namespaces" 
				    "namespace-declarations"
				    "normalize-characters" 
				    "split-cdata-sections" 
				    "validate"
				    "validate-if-schema" 
				    "well-formed"))
			(set! match-count (+ match-count 1)))))
		sdom:config-parameter-names)
      (if (not (eqv? match-count 16))
	  (raise (make-message-condition
		  "one or more config parameters not found"))
	  #t))))

(define domconfigsplitcdatasections-1
  (lambda () 
    (let* ((parameter "sPlIt-cdata-sections")
	   (doc (sdom:create-document "html" #f xhtmlns)))
      (if (not (sdom:get-dom-config-parameter doc parameter))
	  (raise (make-message-condition
		  "split-cdata-sections should be true by default")))
      (if (not (sdom:can-set-dom-config-parameter? doc parameter #f))
	  (raise (make-message-condition
		  "should be able to set split-cdata-sections to false")))
      (if (not (sdom:can-set-dom-config-parameter? doc parameter #t))
	  (raise (make-message-condition
		  "should be able to set split-cdata-sections to true")))
      (sdom:set-dom-config-parameter! doc parameter #t)
      (if (not (sdom:get-dom-config-parameter doc parameter))
	  (raise (make-message-condition
		  "split-cdata-sections should be true after set")))
      (sdom:set-dom-config-parameter! doc parameter #f)
      (if (sdom:get-dom-config-parameter doc parameter)
	  (raise (make-message-condition
		  "split-cdata-sections should be false after set")))
      #t)))

(define domconfigurationcansetparameter-1
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (new-comment (sdom:create-comment doc "This is a new Comment node"))
	   (doc-elem (sdom:document-element doc)))
      (sdom:append-child! doc-elem new-comment)
      (if (not (sdom:can-set-dom-config-parameter? doc "comments" #f))
	  (raise (make-message-condition
		  "should be able to set comments to false")))
      (sdom:normalize-document! doc)
      (let ((last-child (sdom:last-child doc-elem)))
	(if (not (equal? (sdom:node-value last-child)
			 "This is a new Comment node"))
	    (raise (make-message-condition
		    "can-set? should not affect configuration setting")))
	#t))))

(define domconfigurationgetparameter-1
  (lambda ()
    (let ((doc (sdom:clone-node hc-staff #t)))
      (if (not (sdom:get-dom-config-parameter doc "comments"))
	  (raise (make-message-condition "comments should be true by default")))
      (if (not (sdom:get-dom-config-parameter doc "cdata-sections"))
	  (raise (make-message-condition
		  "cdata-sections should be true by default")))
      (if (not (sdom:get-dom-config-parameter doc "entities"))
	  (raise (make-message-condition "entities should be true by default")))
      (if (not (sdom:get-dom-config-parameter doc "namespace-declarations"))
	  (raise (make-message-condition
		  "namespace-declarations should be true by default")))
      (if (sdom:get-dom-config-parameter doc "infoset")
	  (raise (make-message-condition "infoset should be false by default")))
      #t)))

(define domconfigurationgetparameter-2
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (failure #t))
      (guard (ex ((sdom:exception? ex) (set! failure #f)))
	     (sdom:get-dom-config-parameter doc "not-found-param"))
      (if failure 
	  (raise (make-message-condition
		  "get should fail for invalid parameter"))
	  #t))))

(define elementsetidattribute-3
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elem-list (sdom:get-elements-by-tag-name doc "acronym"))
	   (acronym-node (list-ref elem-list 0)))
      (sdom:set-attribute! acronym-node "class" "Maybe")
      (sdom:set-id-attribute! acronym-node "class" #t)
      (let ((attr-node (sdom:get-attribute-node acronym-node "class")))
	(if (not (sdom:is-id attr-node))
	    (raise (make-message-condition
		    "attr should be id after set-id-attribute #t"))))
      (let ((elem (sdom:get-element-by-id doc "Maybe")))
	(if (not (equal? (sdom:tag-name elem) "acronym"))
	    (raise (make-message-condition
		    "id should turn up under get-element-by-id"))))
      (sdom:set-id-attribute! acronym-node "class" #f)
      (let ((attr-node (sdom:get-attribute-node acronym-node "class")))
	(if (sdom:is-id attr-node)
	    (raise (make-message-condition
		    "attr should not be id after set-id-attribute #f"))
	    #t)))))

(define elementsetidattribute-5
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elem-list (sdom:get-elements-by-tag-name doc "strong"))
	   (strong-node (list-ref elem-list 0))
	   (success #f))
      (guard (ex ((sdom:exception? ex)
		  (if (eqv? (sdom:exception-code ex)
			    sdom:exception-code-not-found-err)
		      (set! success #t))))
	     (sdom:set-id-attribute! strong-node "hasMiddleName" #t))
      (if (not success)
	  (raise (make-message-condition
		  "should throw exception on set-id-attribute with bad name"))
	  #t))))

(define elementsetidattribute-9
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elem-list1 (sdom:get-elements-by-tag-name doc "strong"))
	   (elem-list2 (sdom:get-elements-by-tag-name doc "sup"))
	   (name-node (list-ref elem-list1 0))
	   (salary-node (list-ref elem-list2 0)))
      (sdom:set-attribute! name-node "hasMiddleName" "Antoine")
      (sdom:set-attribute! salary-node "annual" "2002")
      (sdom:set-id-attribute! name-node "hasMiddleName" #t)
      (sdom:set-id-attribute! salary-node "annual" #t)
      (let ((attr-node (sdom:get-attribute-node name-node "hasMiddleName")))
	(if (not (sdom:is-id attr-node))
	    (raise (make-message-condition
		    "attr should be id after set-id-attribute"))))
      (let ((attr-node (sdom:get-attribute-node salary-node "annual")))
	(if (not (sdom:is-id attr-node))
	    (raise (make-message-condition
		    "attr should be id after set-id-attribute"))))
      (let ((elem (sdom:get-element-by-id doc "Antoine")))
	(if (not (equal? (sdom:tag-name elem) "strong"))
	    (raise (make-message-condition
		    "id should turn up under get-element-by-id"))))
      (let ((elem (sdom:get-element-by-id doc "2002")))
	(if (not (equal? (sdom:tag-name elem) "sup"))
	    (raise (make-message-condition
		    "id should turn up under get-element-by-id"))
	    #t)))))

(define elementsetidattribute-10	   
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elem-list (sdom:get-elements-by-tag-name doc "acronym"))
	   (acronym-node (list-ref elem-list 0)))
      (sdom:set-id-attribute! acronym-node "title" #t)
      (let ((attr-node (sdom:get-attribute-node acronym-node "title")))
	(if (not (sdom:is-id attr-node))
	    (raise (make-message-condition
		    "attr should be id after set-id-attribute")))
	(sdom:set-id-attribute! acronym-node "title" #t)
	(if (not (sdom:is-id attr-node))
	    (raise (make-message-condition
		    "attr should be id after set-id-attribute")))
	(sdom:set-id-attribute! acronym-node "title" #f)
	(if (sdom:is-id attr-node)
	    (raise (make-message-condition
		    "attr should be id after set-id-attribute"))
	    #t)))))

(define elementsetidattributenode-2
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elem-list (sdom:get-elements-by-tag-name doc "acronym"))
	   (acronym-node (list-ref elem-list 0))
	   (attr-node (sdom:get-attribute-node acronym-node "title")))
      (sdom:set-id-attribute-node! attr-node #t)
      (if (not (sdom:is-id attr-node))
	  (raise (make-message-condition
		  "attr should be id after set-id-attribute-node #t")))
      (let ((elem (sdom:get-element-by-id doc "Yes")))
	(if (not (equal? (sdom:tag-name elem) "acronym"))
	    (raise (make-message-condition
		    "id should turn up under get-element-by-id"))))
      (sdom:set-id-attribute-node! attr-node #f)
      (if (sdom:is-id attr-node)
	  (raise (make-message-condition
		  "attr should not be id after set-id-attribute-node #f"))
	  #t))))

(define elementsetidattributenode-3
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elem-list (sdom:get-elements-by-tag-name doc "strong"))
	   (strong-node (list-ref elem-list 0)))
      (sdom:set-attribute! strong-node "foo" "Karen")
      (let ((attr-node (sdom:get-attribute-node strong-node "foo")))
	(sdom:set-id-attribute-node! attr-node #t)
	(if (not (sdom:is-id attr-node))
	    (raise (make-message-condition
		    "attr should be id after set-id-attribute-node #t")))
	(let ((elem (sdom:get-element-by-id doc "Karen")))
	  (if (not (equal? (sdom:tag-name elem) "strong"))
	      (raise (make-message-condition
		      "id should turn up under get-element-by-id"))))
	(sdom:set-id-attribute-node! attr-node #f)
	(if (sdom:is-id attr-node)
	    (raise (make-message-condition
		   "attr should not be id after set-id-attribute-node #f"))
	    #t)))))

(define entities-1
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo #t))
	   (p-list (sdom:get-elements-by-tag-name doc "p"))
	   (p-elem (list-ref p-list 0))
	   (ent-ref (sdom:create-entity-reference doc "ent1"))) 
      (sdom:set-dom-config-parameter! doc "entities" #t)
      (sdom:append-child! p-elem ent-ref)
      (sdom:normalize-document! doc)
      (let* ((p-list (sdom:get-elements-by-tag-name doc "p"))
	     (p-elem (list-ref p-list 0))
	     (child (sdom:last-child p-elem)))
	(if (null? child)
	    (raise (make-message-condition
		    "entity references should be preserved")))
	(if (not (equal? (sdom:node-name child) "ent1"))
	    (raise (make-message-condition
		    "entity references should be preserved")))
	(if (not (find (lambda (x) (equal? (sdom:node-name x) "ent2"))
		       (sdom:entities (sdom:doctype doc))))
	    (raise (make-message-condition
		    "entity definitions should be preserved"))
	    #t)))))

(define entities-2
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo #t))
	   (p-list (sdom:get-elements-by-tag-name doc "p"))
	   (p-elem (list-ref p-list 0))
	   (ent-ref (sdom:create-entity-reference doc "ent1"))) 
      (sdom:set-dom-config-parameter! doc "entities" #f)
      (sdom:append-child! p-elem ent-ref)
      (sdom:normalize-document! doc)
      (let* ((p-list (sdom:get-elements-by-tag-name doc "p"))
	     (p-elem (list-ref p-list 0))
	     (child (sdom:last-child p-elem)))
	(if (null? child)
	    (raise (make-message-condition 
		    "entity references should be resolved")))
	(if (not (equal? (sdom:node-name child) "#text"))
	    (raise (make-message-condition
		    "entity references should be resolved")))
	(if (not (equal? (sdom:node-value child) "barfoo"))
	    (raise (make-message-condition
		    "entity references should be resolved")))
	(if (not (find (lambda (x) (equal? (sdom:node-name x) "ent2"))
		       (sdom:entities (sdom:doctype doc))))
	    (raise (make-message-condition
		    "entity definitions should be preserved"))
	    #t)))))

(define entities-3
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo #t))
	   (p-list (sdom:get-elements-by-tag-name doc "p"))
	   (p-elem (list-ref p-list 0))
	   (ent-ref (sdom:create-entity-reference doc "ent3"))) 
      (sdom:set-dom-config-parameter! doc "entities" #f)
      (sdom:append-child! p-elem ent-ref)
      (sdom:normalize-document! doc)
      (let* ((p-list (sdom:get-elements-by-tag-name doc "p"))
	     (p-elem (list-ref p-list 0))
	     (child (sdom:last-child p-elem)))
	(if (null? child)
	    (raise (make-message-condition
		    "entity references should be left unresolved")))
	(if (not (eqv? (sdom:node-type child) 5))
	    (raise (make-message-condition
		    "entity references should be left unresolved")))
	(if (not (equal? (sdom:node-name child) "ent3")) 
	    (raise (make-message-condition
		    "entity references should be left unresolved")))
	#t))))

(define entities-4
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo #t))
	   (p-list (sdom:get-elements-by-tag-name doc "p"))
	   (p-elem (list-ref p-list 0))
	   (ent-ref (sdom:create-entity-reference doc "ent1"))) 
      (sdom:set-dom-config-parameter! doc "entities" #f)
      (sdom:append-child! p-elem ent-ref)
      (sdom:normalize! doc)
      (let* ((p-list (sdom:get-elements-by-tag-name doc "p"))
	     (p-elem (list-ref p-list 0))
	     (child (sdom:last-child p-elem)))
	(if (null? child)
	    (raise (make-message-condition
		    "entity references should be left unresolved")))
	(if (not (equal? (sdom:node-name child) "ent1"))
	    (raise (make-message-condition
		    "entity references should be left unresolved")))
	(if (not (find (lambda (x) (equal? (sdom:node-name x) "ent2"))
		       (sdom:entities (sdom:doctype doc))))
	    (raise (make-message-condition
		    "entity definitions should be preserved"))
	    #t)))))

(define handleerror-1
  (lambda ()
    (let* ((handler (lambda (severity msg type excep data loc) #f))
	   (doc (sdom:clone-node barfoo #t))
	   (elem-list (sdom:get-elements-by-tag-name doc "p"))
	   (elem (list-ref elem-list 0))
	   (old-child (sdom:first-child elem))
	   (new-child (sdom:create-cdata-section doc "this is not ]]> good")))
      (sdom:replace-child! elem new-child old-child)
      (let ((new-child (sdom:create-cdata-section doc "this is not ]]> bad")))
	(sdom:append-child! elem new-child))
      (sdom:set-dom-config-parameter! doc "split-cdata-sections" #t)
      (sdom:set-dom-config-parameter! doc "error-handler" handler)
      (sdom:normalize-document! doc)
      (let* ((elem-list (sdom:get-elements-by-tag-name doc "p"))
	     (elem (list-ref elem-list 0))
	     (child (sdom:last-child elem))
	     (value (sdom:node-value child)))
	(if (equal? value "this is not ]]> bad")
	    (if (not (eqv? (sdom:node-type child) 4))
		(raise (make-message-condition
			"last child should be CDATA after normalization"))
		(let* ((child (sdom:first-child elem))
		       (value (sdom:node-value child)))
		  (if (equal? value "this is not ]]> good")
		      (raise (make-message-condition
			      "first child should not be intact"))
		      #t)))
	    (let* ((child (sdom:first-child elem))
		   (value (sdom:node-value child)))
	      (if (not (equal? value "this is not ]]> good"))
		  (raise (make-message-condition
			  "first child should be intact"))
		  #t)))))))

(define infoset-2
  (lambda ()
    (let* ((failure #f)
	   (handler (lambda (severity msg type excep data loc) 
		      (if (>= severity 2) (set! failure #t)) #t))
	   (doc (sdom:clone-node barfoo #t)))
      (sdom:set-dom-config-parameter! doc "infoset" #t)
      (sdom:set-dom-config-parameter! doc "error-handler" handler)
      (let* ((plist (sdom:get-elements-by-tag-name doc "p"))
	     (pelem (list-ref plist 0))
	     (entref (sdom:create-entity-reference doc "ent3"))) 
	(sdom:append-child! pelem entref)
	(sdom:normalize-document! doc)
	(if failure (raise (make-message-condition
			    "should not have been severe errors")))
	(let* ((plist (sdom:get-elements-by-tag-name doc "p"))
	       (pelem (list-ref plist 0))
	       (child (sdom:last-child pelem)))
	  (if (not child) (raise (make-message-condition
				  "last child should not be null")))
	  (if (not (sdom:entity-reference? child))
	      (raise (make-message-condition
		      "last child should have type entity-reference")))
	  (if (not (equal? (sdom:node-name child) "ent3"))
	      (raise (make-message-condition
		      "last child should have name ent3"))
	      #t))))))

(define infoset-4
  (lambda ()
    (let* ((failure #f)
	   (handler (lambda (severity msg type excep data loc)
		      (if (>= severity 2) (set! failure #t)) #t))
	   (doc (sdom:clone-node barfoo #t))
	   (plist (sdom:get-elements-by-tag-name doc "p"))
	   (pelem (list-ref plist 0))
	   (newcdata (sdom:create-cdata-section doc "CDATA"))) 
      (sdom:append-child! pelem newcdata)
      (sdom:set-dom-config-parameter! doc "infoset" #t)
      (sdom:set-dom-config-parameter! doc "error-handler" handler)
      (sdom:normalize-document! doc)
      (if failure (raise (make-message-condition
			  "should not have been severe errors")))
      (let* ((plist (sdom:get-elements-by-tag-name doc "p"))
	     (pelem (list-ref plist 0))
	     (text (sdom:last-child pelem)))
	(if (not (equal? (sdom:node-name text) "#text"))
	    (raise (make-message-condition
		    "last element should have been converted to text node")))
	(if (not (equal? (sdom:node-value text) "barCDATA"))
	    (raise (make-message-condition
		    "last element should have been coalesced with CDATA"))
	    #t)))))

(define infoset-8
  (lambda ()
    (let* ((failure #f)
	   (handler (lambda (severity msg type excep data loc)
		      (if (>= severity 2) (set! failure #t)) #t))
	   (doc (sdom:clone-node barfoo #t)))
      (sdom:set-dom-config-parameter! doc "infoset" #t)
      (sdom:set-dom-config-parameter! doc "error-handler" handler)
      (let* ((bodylist (sdom:get-elements-by-tag-name doc "body"))
	     (body (list-ref bodylist 0))
	     (child (sdom:first-child body))
	     (text (sdom:create-text-node doc "	  ")))
	(sdom:insert-before! body text child))
      (sdom:normalize-document! doc)
      (if failure (raise (make-message-condition
			  "should not have been severe errors")))
      (let* ((bodylist (sdom:get-elements-by-tag-name doc "body"))
	     (body (list-ref bodylist 0))
	     (child (sdom:first-child body)))
	(if (not child) (raise (make-message-condition
				"first child should not be null")))
	(if (not (equal? (sdom:node-name child) "#text"))
	    (raise (make-message-condition "first child should be text node")))
	(let ((child (sdom:next-sibling child)))
	  (if (not child) (raise (make-message-condition
				  "second child should not be null")))
	  (if (not (equal? (sdom:node-name child) "p"))
	      (raise (make-message-condition 
		      "second child should be p element"))
	      #t))))))

(define infoset-9
  (lambda ()
    (let* ((failure #f)
	   (handler (lambda (severity msg type excep data loc)
		      (if (>= severity 2) (set! failure #t)) #t))
	   (doc (sdom:clone-node barfoo #t))
	   (plist (sdom:get-elements-by-tag-name doc "p"))
	   (elem (list-ref plist 0))
	   (newcomment (sdom:create-comment doc "COMMENT_NODE")))
      (sdom:append-child! elem newcomment)
      (sdom:set-dom-config-parameter! doc "comments" #f)
      (sdom:set-dom-config-parameter! doc "infoset" #t)
      (sdom:set-dom-config-parameter! doc "error-handler" handler)
      (sdom:normalize-document! doc)
      (if failure (raise (make-message-condition
			  "should not have been severe errors")))
      (let* ((plist (sdom:get-elements-by-tag-name doc "p"))
	     (elem (list-ref plist 0))
	     (lastchild (sdom:last-child elem)))
	(if (not (equal? (sdom:node-name lastchild) "#comment"))
	    (raise (make-message-condition
		    "comments should have been preserved"))
	    #t)))))

(define nodeappendchild-2
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo #t))
	   (docelem (sdom:document-element doc))
	   (tagname (sdom:tag-name docelem))
	   (rootns (sdom:namespace-uri docelem))
	   (newelem (sdom:create-element doc tagname rootns))) 
      (guard (ex ((sdom:exception? ex)
		  (if (not (or (eqv? (sdom:exception-code ex) 3)
			       (eqv? (sdom:exception-code ex) 9)))
		      (raise (make-message-condition 
			      "wrong exception type was thrown")))))
	     (begin (sdom:append-child! doc newelem)
		    (raise (make-message-condition
			    "exception should have been thrown"))))
      #t)))

(define nodecomparedocumentposition-3
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doccomp (sdom:clone-node hc-staff #t))
	   (pos1 (sdom:compare-document-position doc doccomp))
	   (pos2 (sdom:compare-document-position doccomp doc)))
      (if (not (eqv? (logand 33 57) (logand pos1 57)))
	  (raise (make-message-condition
		  "documents should be disconnected")))
      (if (eqv? (logand pos1 2) (logand pos2 2))
	  (raise (make-message-condition
		  "documents should not both be preceding")))
      (if (eqv? (logand pos1 4) (logand pos2 4))
	  (raise (make-message-condition
		  "documents should not both be following")))
      (let ((pos3 (sdom:compare-document-position doc doccomp)))
	(if (not (eqv? pos1 pos3))
	    (raise (make-message-condition
		    "document position should be consistent across calls"))
	    #t)))))

(define nodecomparedocumentposition-4
  (lambda ()
    (let ((doc (sdom:clone-node hc-staff #t)))
      (if (not (eqv? (sdom:compare-document-position doc doc) 0))
	  (raise (make-message-condition
		  "no flags should be set when comparing doc with itself"))
	  #t))))

(define nodecomparedocumentposition-5
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc))
	   (rootname (sdom:tag-name docelem))
	   (rootns (sdom:namespace-uri docelem))
	   (newdoc (sdom:create-document rootname #f rootns))
	   (pos1 (sdom:compare-document-position doc newdoc))
	   (pos2 (sdom:compare-document-position newdoc doc)))
      (if (not (eqv? (logand 33 57) (logand pos1 57)))
	  (raise (make-message-condition
		  "documents should be disconnected")))
      (if (eqv? (logand pos1 2) (logand pos2 2))
	  (raise (make-message-condition
		  "documents should not both be preceding")))
      (if (eqv? (logand pos1 4) (logand pos2 4))
	  (raise (make-message-condition
		  "documents should not both be following")))
      (let ((pos3 (sdom:compare-document-position doc newdoc)))
	(if (not (eqv? pos1 pos3))
	    (raise (make-message-condition
		    "document position should be consistent across calls"))
	    #t)))))

(define nodecomparedocumentposition-6
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc))
	   (pos1 (sdom:compare-document-position doc docelem))
	   (pos2 (sdom:compare-document-position docelem doc)))
      (if (not (eqv? 20 pos1))
	  (raise (make-message-condition
		  "document element should be contained and following")))
      (if (not (eqv? 10 pos2))
	  (raise (make-message-condition
		  "document should be preceding and containing"))
	  #t))))

(define nodecomparedocumentposition-7
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc))
	   (newelem (sdom:create-element doc "br" xhtmlns)))
      (sdom:append-child! docelem newelem)
      (let ((pos1 (sdom:compare-document-position doc newelem))
	    (pos2 (sdom:compare-document-position newelem doc)))
	(if (not (eqv? 20 pos1))
	    (raise (make-message-condition
		    "new element should be contained and following")))
	(if (not (eqv? 10 pos2))
	    (raise (make-message-condition
		    "document should be preceding and containing"))
	    #t)))))

(define nodecomparedocumentposition-8
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elemlist (sdom:get-elements-by-tag-name doc "p"))
	   (elem (list-ref elemlist 0))
	   (pos1 (sdom:compare-document-position doc elem))
	   (pos2 (sdom:compare-document-position elem doc)))
      (if (not (eqv? 20 pos1))
	  (raise (make-message-condition
		  "element should be contained and following")))
      (if (not (eqv? 10 pos2))
	  (raise (make-message-condition
		  "document should be preceding and containing"))
	  #t))))

(define nodecomparedocumentposition-9
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elemlist (sdom:get-elements-by-tag-name doc "p"))
	   (elem (list-ref elemlist 0))
	   (newelem (sdom:create-element doc "br" xhtmlns)))
      (sdom:append-child! elem newelem)
      (let ((pos1 (sdom:compare-document-position doc newelem))
	    (pos2 (sdom:compare-document-position newelem doc)))
	(if (not (eqv? 20 pos1))
	    (raise (make-message-condition
		    "new element should be contained and following")))
	(if (not (eqv? 10 pos2))
	    (raise (make-message-condition
		    "document should be preceding and containing"))
	    #t)))))

(define nodecomparedocumentposition-11
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elemlist (sdom:get-elements-by-tag-name doc "p"))
	   (elem (list-ref elemlist 0))
	   (newattr (sdom:create-attribute doc "xml:lang" xmlns)))
      (sdom:set-attribute-node! elem newattr)
      (if (not (eqv? (sdom:compare-document-position newattr doc) 10))
	  (raise (make-message-condition
		  "document should precede and contain attribute"))
	  #t))))

(define nodecomparedocumentposition-12
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (pi (sdom:create-processing-instruction doc "PITarget" "PIDATA")))
      (sdom:append-child! doc pi)
      (if (not (eqv? (sdom:compare-document-position doc pi) 20))
	  (raise (make-message-condition
		  "PI should be contained and following")))
      (if (not (eqv? (sdom:compare-document-position pi doc) 10))
	  (raise (make-message-condition
		  "document should precede and contain PI"))
	  #t))))

(define nodecomparedocumentposition-13
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (comment (sdom:create-comment doc "Another Comment"))
	   (elemlist (sdom:get-elements-by-tag-name doc "p"))
	   (elem (list-ref elemlist 0)))	   
      (sdom:append-child! elem comment)
      (if (not (eqv? (sdom:compare-document-position doc comment) 20))
	  (raise (make-message-condition
		  "comment should be contained and following")))
      (if (not (eqv? (sdom:compare-document-position comment doc) 10))
	  (raise (make-message-condition
		  "document should precede and contain comment"))
	  #t))))

(define nodecomparedocumentposition-14
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc))
	   (docfrag (sdom:create-document-fragment doc)))
      (sdom:append-child! docfrag docelem)
      (let* ((docfragchild (sdom:first-child docfrag))
	     (dfpos1 (sdom:compare-document-position docfrag docfragchild))
	     (dfpos2 (sdom:compare-document-position docfragchild docfrag)))
	(if (not (eqv? dfpos1 20))
	    (raise (make-message-condition
		    "element should be contained and following")))
	(if (not (eqv? dfpos2 10))
	    (raise (make-message-condition
		    "document fragment should precede and contain element"))
	    #t)))))

(define nodecomparedocumentposition-15
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc))
	   (docfrag (sdom:create-document-fragment doc))
	   (attr (sdom:create-attribute doc "xml:lang" xmlns)))
      (sdom:set-attribute-node! docelem attr)
      (sdom:append-child! docfrag docelem)
      (let ((docfragchild (sdom:first-child docfrag)))
	(if (not (eqv? (sdom:compare-document-position docfragchild attr) 20))
	    (raise (make-message-condition
		    "attr should be contained and following")))
	(if (not (eqv? (sdom:compare-document-position attr docfragchild) 10))
	    (raise (make-message-condition
		    "element should precede and contain attribute"))
	    #t)))))

(define nodecomparedocumentposition-16
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docfrag (sdom:create-document-fragment doc))
	   (attr (sdom:create-attribute doc "xml:lang" xmlns))
	   (attrcloned (sdom:clone-node attr #t))
	   (pos1 (sdom:compare-document-position docfrag attrcloned))
	   (pos2 (sdom:compare-document-position attrcloned docfrag)))
      (if (not (eqv? (logand 33 57) (logand pos1 57)))
	  (raise (make-message-condition
		  "fragment and attribute should be disconnected")))
      (if (eqv? (logand pos1 2) (logand pos2 2))
	  (raise (make-message-condition
		  "nodes should not both be preceding")))
      (if (eqv? (logand pos1 4) (logand pos2 4))
	  (raise (make-message-condition
		  "nodes should not both be following")))
      (if (not (eqv? pos1 (sdom:compare-document-position docfrag attrcloned)))
	  (raise (make-message-condition
		  "node position should be consistent"))
	  #t))))

(define nodecomparedocumentposition-17
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (pi1 (sdom:create-processing-instruction doc "PI1" ""))
	   (pi2 (sdom:create-processing-instruction doc "PI2" "")))
      (sdom:append-child! doc pi1)
      (sdom:append-child! doc pi2)
      (if (not (eqv? (sdom:compare-document-position pi1 pi2) 4))
	  (raise (make-message-condition
		  "second PI should be following first PI")))
      (if (not (eqv? (sdom:compare-document-position pi2 pi1) 2))
	  (raise (make-message-condition
		  "first PI should be preceding second PI"))
	  #t))))

(define nodecomparedocumentposition-18
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc))
	   (txt1 (sdom:create-text-node doc "T1"))
	   (txt2 (sdom:create-text-node doc "T2")))
      (sdom:append-child! docelem txt1)
      (sdom:append-child! docelem txt2)
      (if (not (eqv? (sdom:compare-document-position txt1 txt2) 4))
	  (raise (make-message-condition
		  "second text node should be following first text node")))
      (if (not (eqv? (sdom:compare-document-position txt2 txt1) 2))
	  (raise (make-message-condition
		  "first text node should be preceding second text node"))
	  #t))))

(define nodecomparedocumentposition-19
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elemlist (sdom:get-elements-by-tag-name doc "strong" "*"))
	   (elemstrong (list-ref elemlist 0))
	   (textnode (sdom:first-child elemstrong))
	   (cdata1 (sdom:create-cdata-section doc "FOO"))
	   (cdata2 (sdom:create-cdata-section doc "BAR")))
      (sdom:insert-before! elemstrong cdata1 textnode)
      (sdom:insert-after! elemstrong cdata2 textnode)
      (if (not (eqv? (sdom:compare-document-position cdata1 cdata2) 4))
	  (raise (make-message-condition
		  "second cdata section should be following first")))
      (if (not (eqv? (sdom:compare-document-position cdata2 cdata1) 2))
	  (raise (make-message-condition
		  "first cdata section should be preceding second"))
	  #t))))

(define nodecomparedocumentposition-20
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elemlist (sdom:get-elements-by-tag-name doc "strong" "*"))
	   (elemstrong (list-ref elemlist 0))
	   (textnode (sdom:first-child elemstrong))
	   (cdata1 (sdom:create-cdata-section doc "FOO")))
      (sdom:insert-after! elemstrong cdata1 textnode)
      (if (not (eqv? (sdom:compare-document-position textnode cdata1) 4))
	  (raise (make-message-condition
		  "cdata section should follow text node")))
      (if (not (eqv? (sdom:compare-document-position cdata1 textnode) 2))
	  (raise (make-message-condition
		  "text node should precede cdata section"))
	  #t))))

(define nodecomparedocumentposition-21
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (plist (sdom:get-elements-by-tag-name doc "p"))
	   (blist (sdom:get-elements-by-tag-name doc "body"))
	   (pelem (list-ref plist 0))
	   (belem (list-ref blist 0))
	   (pclone (sdom:clone-node pelem #t)))
      (sdom:append-child! belem pclone)
      (let* ((elemlist (sdom:get-elements-by-tag-name doc "strong"))
	     (name1 (list-ref elemlist 0))
	     (name2 (list-ref elemlist 1))
	     (txt1 (sdom:first-child name1))
	     (txt2 (sdom:first-child name2)))
	(if (not (eqv? (sdom:compare-document-position txt1 txt2) 4))
	    (raise (make-message-condition
		    "second text node should follow first text node")))
	(if (not (eqv? (sdom:compare-document-position txt2 txt1) 2))
	    (raise (make-message-condition
		    "first text node should precede second text node"))
	    #t)))))

(define nodecomparedocumentposition-25
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elemlist (sdom:get-elements-by-tag-name doc "var"))
	   (elem (list-ref elemlist 0))
	   (entref (sdom:create-entity-reference doc "F")))
      (sdom:insert-before! elem entref (sdom:first-child elem))
      (if (not (eqv? (sdom:compare-document-position elem entref) 20))
	  (raise (make-message-condition
		  "entity reference should be contained and following")))
      (if (not (eqv? (sdom:compare-document-position entref elem) 10))
	  (raise (make-message-condition
		  "element should be containing and preceding"))
	  #t))))

(define nodecomparedocumentposition-30
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elemlist (sdom:get-elements-by-tag-name doc "strong"))
	   (strong (list-ref elemlist 0))
	   (positionlist (sdom:get-elements-by-tag-name doc "code"))
	   (code (list-ref positionlist 0)))
      (if (not (eqv? (sdom:compare-document-position code strong) 2))
	  (raise (make-message-condition
		  "strong element should precede code element"))
	  #t))))

(define nodecomparedocumentposition-31
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (p (sdom:clone-node (list-ref (sdom:get-elements-by-tag-name 
					  doc "p") 0) 
			       #t))
	   (body (list-ref (sdom:get-elements-by-tag-name doc "body") 0)))
      (sdom:append-child! body (sdom:first-child p))
      (let* ((namelist (sdom:get-elements-by-tag-name doc "strong"))
	     (strong (list-ref namelist 0))
	     (positionlist (sdom:get-elements-by-tag-name doc "em"))
	     (code (list-ref positionlist 1))
	     (newelem (sdom:create-element doc "br" xhtmlns)))
	(sdom:append-child! code newelem)
	(if (not (eqv? (sdom:compare-document-position strong newelem) 4))
	    (raise (make-message-condition
		    "strong element should precede new element")))
	(if (not (eqv? (sdom:compare-document-position newelem strong) 2))
	    (raise (make-message-condition
		    "new element should follow strong element"))
	    #t)))))

(define nodecomparedocumentposition-33
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elem (sdom:create-element doc "br" xhtmlns))
	   (attr (sdom:create-attribute doc "xml:lang" xmlns)))
      (sdom:set-attribute-node! elem attr)
      (if (not (eqv? (sdom:compare-document-position elem attr) 20))
	  (raise (make-message-condition
		  "attr should be contained by and follow element")))
      (if (not (eqv? (sdom:compare-document-position attr elem) 10))
	  (raise (make-message-condition
		  "element should contain and precede attr")))
      (let ((pos1 (sdom:compare-document-position doc elem))
	    (pos2 (sdom:compare-document-position elem doc)))
	(if (not (eqv? (logand pos1 57) (logand 33 57)))
	    (raise (make-message-condition
		    "document and new element should be disconnected")))
	(if (eqv? (logand pos1 2) (logand pos2 2))
	    (raise (make-message-condition
		    "document and element should not both be following")))
	(if (eqv? (logand pos1 4) (logand pos2 4))
	    (raise (make-message-condition
		    "document and element should not both be preceding")))
	(if (not (eqv? pos1 (sdom:compare-document-position doc elem)))
	    (raise (make-message-condition
		    "document position should be consistent"))
	    #t)))))

(define nodecomparedocumentposition-34
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elemmain (sdom:create-element doc "p" xhtmlns))
	   (elem (sdom:create-element doc "br" xhtmlns))
	   (txt (sdom:create-text-node doc "TEXT"))
	   (pi (sdom:create-processing-instruction doc "PIT" "PID")))
      (sdom:append-child! elemmain txt)
      (sdom:append-child! elemmain elem)
      (sdom:append-child! elemmain pi)
      (if (not (eqv? (sdom:compare-document-position txt elem) 4))
	  (raise (make-message-condition
		  "element should be following text node")))
      (if (not (eqv? (sdom:compare-document-position pi txt) 2))
	  (raise (make-message-condition
		  "processing instruction should precede text node"))
	  #t))))

(define nodecomparedocumentposition-37
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elemlist (sdom:get-elements-by-tag-name doc "acronym"))
	   (elem (list-ref elemlist 0))
	   (attr (sdom:get-attribute-node elem "title"))
	   (txt (sdom:first-child elem)))
      (if (not (eqv? (sdom:compare-document-position attr txt) 4))
	  (raise (make-message-condition
		  "text node should follow attribute node")))
      (if (not (eqv? (sdom:compare-document-position txt attr) 2))
	  (raise (make-message-condition
		  "attribute node should precede text node"))
	  #t))))

(define nodecomparedocumentposition-38
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elemlist (sdom:get-elements-by-tag-name doc "acronym"))
	   (elem (list-ref elemlist 0))
	   (attr (sdom:get-attribute-node elem "title"))
	   (txt (sdom:first-child attr)))
      (if (not (eqv? (sdom:compare-document-position attr txt) 20))
	  (raise (make-message-condition
		  "text node should be contained by and follow attr")))
      (if (not (eqv? (sdom:compare-document-position txt attr) 10))
	  (raise (make-message-condition
		  "attr should contain and precede txt"))
	  #t))))

(define nodecomparedocumentposition-39
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elemlist (sdom:get-elements-by-tag-name doc "acronym"))
	   (elem (list-ref elemlist 3))
	   (attr1 (sdom:get-attribute-node elem "class"))
	   (attr2 (sdom:get-attribute-node 
		   elem "xsi:noNamespaceSchemaLocation"))
	   (attrpos (sdom:compare-document-position attr1 attr2)))
      (if (not (eqv? (logand attrpos 32) (logand 32 32)))
	  (raise (make-message-condition
		  "relative attribute position should be impl. specific")))
      (if (not (eqv? (logand attrpos 25) (logand 0 25)))
	  (raise (make-message-condition "other bits should be zero")))
      (if (eqv? (logand 0 6) (logand attrpos 6))
	  (raise (make-message-condition 
		  "should be either preceding or following")))
      (let ((swappedpos (sdom:compare-document-position attr2 attr1)))
	(if (eqv? (logand swappedpos 2) (logand attrpos 2))
	    (raise (make-message-condition 
		    "only one attr should be preceding")))
	(if (eqv? (logand swappedpos 4) (logand attrpos 4))
	    (raise (make-message-condition "only one attr should be following"))
	    #t)))))

(define nodecomparedocumentposition-40
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elemlist (sdom:get-elements-by-tag-name doc "acronym"))
	   (elem (list-ref elemlist 0))
	   (attr1 (sdom:get-attribute-node elem "title")))
      (sdom:set-attribute! elem "xml:lang" "FR-fr" xmlns)
      (let* ((attr2 (sdom:get-attribute-node elem "xml:lang"))
	     (attrpos (sdom:compare-document-position attr1 attr2)))
	(if (not (eqv? (logand 32 32) (logand attrpos 32)))
	    (raise (make-message-condition
		    "relative attribute position should be impl. specific")))
	(if (not (eqv? (logand 0 25) (logand attrpos 25)))
	    (raise (make-message-condition
		    "other bits should be zero")))
	(if (eqv? (logand 0 6) (logand attrpos 6))
	    (raise (make-message-condition
		    "should be either preceding or following")))
	(let ((swappedpos (sdom:compare-document-position attr2 attr1)))
	  (if (eqv? (logand swappedpos 2) (logand attrpos 2))
	      (raise (make-message-condition
		      "only one attr should be preceding")))
	  (if (eqv? (logand swappedpos 4) (logand attrpos 4))
	      (raise (make-message-condition
		      "only one attr should be following"))
	      #t))))))

(define nodegetbaseuri-2
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo #t))
	   (docelem (sdom:document-element doc))
	   (rootns (sdom:namespace-uri docelem))
	   (rootname (sdom:tag-name docelem))
	   (newdoc (sdom:create-document rootname #f rootns))
	   (baseuri (sdom:base-uri newdoc)))
      (if baseuri 
	  (raise (make-message-condition
		  "base URI should be null for newly created document")))
      (sdom:set-document-uri! newdoc "http://www.example.com/sample.xml")
      (if (not (equal? (sdom:base-uri newdoc)
		       "http://www.example.com/sample.xml"))
	  (raise (make-message-condition
		  "base URI should be equal to document URI"))
	  #t))))

(define nodegetbaseuri-4
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo #t))
	   (docelem (sdom:document-element doc))
	   (baseuri (sdom:base-uri docelem))
	   (docuri (sdom:document-uri doc)))
      (if (not (equal? baseuri docuri))
	  (raise (make-message-condition
		  "base URI should be equal to document URI"))
	  #t))))

(define nodegetbaseuri-5
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo-base #t))
	   (docelem (sdom:document-element doc))
	   (baseuri (sdom:base-uri docelem)))
      (if (not (equal? baseuri "http://www.w3.org/DOM/L3Test"))
	  (raise (make-message-condition
		  "base URI should be equal to base attribute"))
	  #t))))

(define nodegetbaseuri-7
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo-base #t))
	   (bodylist (sdom:get-elements-by-tag-name doc "body"))
	   (bodyelem (list-ref bodylist 0))
	   (newelem (sdom:create-element doc "meta" xhtmlns)))
      (sdom:set-attribute! newelem "content" "text/xml")
      (sdom:append-child! bodyelem newelem)
      (if (not (equal? (sdom:base-uri newelem)
		       "http://www.w3.org/DOM/EmployeeID"))
	  (raise (make-message-condition
		  "base URI should be equal to inherited base attribute"))
	  #t))))

(define nodegetbaseuri-9
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo-base #t))
	   (bodylist (sdom:get-elements-by-tag-name doc "body"))
	   (bodyelem (list-ref bodylist 0))
	   (baseuri (sdom:base-uri bodyelem)))
      (if (not (equal? baseuri "http://www.w3.org/DOM/EmployeeID"))
	  (raise (make-message-condition
		  "explicit base attribute should produce base URI"))
	  #t))))

(define nodegetbaseuri-10
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo-base #t))
	   (docelem (sdom:document-element doc))
	   (newpi (sdom:create-processing-instruction doc "TARGET" "DATA")))
      (sdom:append-child! docelem newpi)
      (if (not (equal? (sdom:base-uri newpi) "http://www.w3.org/DOM/L3Test"))
	  (raise (make-message-condition
		  "processing instruction should inherit URI from element"))
	  #t))))

(define nodegetbaseuri-15
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo-base #t))
	   (bodylist (sdom:get-elements-by-tag-name doc "body"))
	   (bodyelem (list-ref bodylist 0))
	   (attrnode (sdom:get-attribute-node bodyelem "id"))
	   (baseuri (sdom:base-uri attrnode)))
      (if baseuri 
	  (raise (make-message-condition
		  "base URI for attr should be null"))
	  #t))))

(define nodegetbaseuri-17
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo-base #t))
	   (plist (sdom:get-elements-by-tag-name doc "p"))
	   (pelem (list-ref plist 0))
	   (textnode (sdom:first-child pelem))
	   (baseuri (sdom:base-uri textnode)))
      (if baseuri 
	  (raise (make-message-condition
		  "base URI for text should be null"))
	  #t))))

(define nodegetbaseuri-18
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo-base #t))
	   (blist (sdom:get-elements-by-tag-name doc "body"))
	   (belem (list-ref blist 0))
	   (comment (sdom:create-comment doc "BLAH")))
      (sdom:append-child! belem comment)
      (if (sdom:base-uri comment)
	  (raise (make-message-condition
		  "base URI for comment should be null"))
	  #t))))

(define nodegettextcontent-1
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t)))
      (if (sdom:text-content doc)
	  (raise (make-message-condition
		  "text content for document node should be null"))
	  #t))))

(define nodegettextcontent-2
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc))
	   (rootns (sdom:namespace-uri docelem))
	   (rootname (sdom:tag-name docelem))
	   (newdoc (sdom:create-document rootname #f rootns)))
      (if (sdom:text-content newdoc)
	  (raise (make-message-condition
		  "text content for document node should be null"))
	  #t))))

(define nodegettextcontent-6
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elemlist (sdom:get-elements-by-tag-name doc "acronym"))
	   (elem (list-ref elemlist 0))
	   (attr (sdom:get-attribute-node elem "title")))
      (if (not (equal? "Yes" (sdom:text-content attr)))
	  (raise (make-message-condition
		  "text content should equal attribute value"))
	  #t))))

(define nodegettextcontent-7
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elemlist (sdom:get-elements-by-tag-name doc "p"))
	   (elem (list-ref elemlist 0)))
      (sdom:set-attribute! elem "xml:lang" "en-US" xmlns)
      (let ((attr (sdom:get-attribute-node elem "lang" xmlns)))
	(if (not (equal? "en-US" (sdom:text-content attr)))
	    (raise (make-message-condition
		    "text content should equal attribute value"))
	    #t)))))

(define nodegettextcontent-8
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo #t))
	   (elem (sdom:create-element doc "p" xhtmlns))
	   (att (sdom:create-attribute doc "xml:lang" xmlns)))
      (sdom:set-attribute-node! elem att)
      (if (not (equal? (sdom:text-content att) ""))
	  (raise (make-message-condition
		  "text content should be nothing on childless attr"))
	  #t))))

(define nodegettextcontent-9
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo #t))
	   (elem (sdom:create-element doc "p" xhtmlns))
	   (txt (sdom:create-text-node doc "Replacement Text")))
      (sdom:append-child! elem txt)
      (if (not (equal? (sdom:text-content txt) "Replacement Text"))
	  (raise (make-message-condition
		  "text content should be value of text node"))
	  #t))))

(define nodegettextcontent-10
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elemlist (sdom:get-elements-by-tag-name doc "em"))
	   (elem (list-ref elemlist 0))
	   (txt (sdom:first-child elem)))
      (if (not (equal? (sdom:text-content txt) "EMP0001"))
	  (raise (make-message-condition
		  "text content should be value of text node"))
	  #t))))

(define nodegettextcontent-11
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (cdata (sdom:create-cdata-section doc "FOO")))
      (if (not (equal? (sdom:text-content cdata) "FOO"))
	  (raise (make-message-condition
		  "text content should be value of cdata section"))
	  #t))))

(define nodegettextcontent-12
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo #t))
	   (elem (sdom:create-element doc "body" xhtmlns))
	   (comment (sdom:create-comment doc "Comment")))
      (sdom:append-child! elem comment)
      (if (not (equal? (sdom:text-content comment) "Comment"))
	  (raise (make-message-condition
		  "text content should be value of comment"))
	  #t))))

(define nodegettextcontent-13
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elemlist (sdom:get-elements-by-tag-name doc "strong"))
	   (elem (list-ref elemlist 0))
	   (cdata (sdom:create-cdata-section doc "FOO")))
      (sdom:append-child! elem cdata)
      (if (not (equal? (sdom:text-content elem) "Margaret MartinFOO"))
	  (raise (make-message-condition
		  "text content should be concat of children"))
	  #t))))

(define nodegettextcontent-14
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elemlist (sdom:get-elements-by-tag-name doc "p"))
	   (elem (list-ref elemlist 0)))
      (if (not (equal? (sdom:text-content elem)
		       (string-append "EMP0001Margaret MartinAccountant56,000"
				      "Female1230 North Ave. Dallas, Texas "
				      "98551")))
	  (raise (make-message-condition
		  "text content should be concat of children"))
	  #t))))

(define nodegettextcontent-19
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo #t))
	   (elemlist (sdom:get-elements-by-tag-name doc "body"))
	   (elem (list-ref elemlist 0)))
      (if (not (equal? (sdom:text-content elem) "bar"))
	  (raise (make-message-condition
		  "text content should ignore element whitespace"))
	  #t))))

(define nodegetuserdata-1
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t)))
      (if (sdom:get-user-data doc "key1")
	  (raise (make-message-condition "unset userdata key should be null"))
	  #t))))

(define nodegetuserdata-3
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo #t))
	   (elem (sdom:create-element doc "body" xhtmlns)))
      (sdom:set-user-data! doc "something" elem '())
      (if (not (sdom:same-node? elem (sdom:get-user-data doc "something")))
	  (raise (make-message-condition "user data should preserve node data"))
	  #t))))

(define nodegetuserdata-6
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (txt (sdom:create-text-node doc "TEXT")))
      (if (sdom:get-user-data txt "")
	  (raise (make-message-condition
		  "unset userdata should be null on new text node"))
	  #t))))

(define nodegetuserdata-7
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (pi (sdom:create-processing-instruction doc "PITARGET" "PIDATA")))
      (sdom:set-user-data! pi "key" pi '())
      (if (not (sdom:same-node? pi (sdom:get-user-data pi "key")))
	  (raise (make-message-condition
		  "should be able to set node as its own user data"))
	  #t))))

(define nodeinsertbefore-1
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc))
	   (newcomment (sdom:create-comment doc "Comment"))
	   (newpi (sdom:create-processing-instruction doc "PITarget" "PIData")))
      (sdom:insert-before! doc newcomment docelem)
      (if (not (equal? (sdom:data (sdom:insert-before! doc newcomment docelem))
		       "Comment"))
	  (raise (make-message-condition
		  "inserting node should not change value")))
      (sdom:insert-before! doc newpi newcomment)
      (if (not (equal? (sdom:target (sdom:insert-before! doc newpi docelem))
		       "PITarget"))
	  (raise (make-message-condition
		  "inserting node should not change value"))
	  #t))))

(define nodeinsertbefore-6
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc))
	   (rootns (sdom:namespace-uri docelem))
	   (roottagname (sdom:tag-name docelem))
	   (newelem (sdom:create-element doc roottagname rootns)))
      (guard (ex ((sdom:exception? ex)
		  (if (not (or (eqv? (sdom:exception-code ex) 3)
			       (eqv? (sdom:exception-code ex) 9)))
		      (raise (make-message-condition 
			      "exception did not have correct code")))))
	     (begin 
	       (sdom:insert-before! doc newelem docelem)
	       (raise (make-message-condition
		       "exception should be thrown on previous line"))))
      #t)))

(define nodeinsertbefore-7
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docalt (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc))
	   (newcomment (sdom:create-comment docalt "Comment"))
	   (success #f))
      (guard (ex ((sdom:exception? ex)
		  (if (eqv? (sdom:exception-code ex)
			    sdom:exception-code-wrong-document-err)
		      (set! success #t))))
	     (sdom:insert-before! doc newcomment docelem))
      (if (not success)
	  (raise (make-message-condition
		  "exception should be thrown when documents do not match"))
	  #t))))

(define nodeinsertbefore-8
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docalt (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element docalt))
	   (newcomment (sdom:create-comment doc "Comment")))
      (guard 
       (ex ((sdom:exception? ex)
	    (if (not (eqv? (sdom:exception-code ex)
			   sdom:exception-code-not-found-err))
		(raise (make-message-condition 
			"not-found-err should be thrown on mismatch")))))
       (sdom:insert-before! doc newcomment docelem))
      #t)))

(define nodeinsertbefore-9
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (newcomment (sdom:create-comment doc "Comment")))
      (sdom:append-child! doc newcomment)
      (let* ((docfrag (sdom:create-document-fragment doc))
	     (insertcomment (sdom:create-comment doc "insertComment")))
	(sdom:append-child! docfrag insertcomment)
	(sdom:insert-before! doc docfrag newcomment)
	(let* ((comment (sdom:previous-sibling newcomment))
	       (data (sdom:data comment)))
	  (if (not (equal? data "insertComment"))
	      (raise (make-message-condition
		      "sequential insertions should produce desired order"))
	      #t))))))

(define nodeinsertbefore-10
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc))
	   (rootns (sdom:namespace-uri docelem))
	   (tagname (sdom:tag-name docelem))
	   (elemlist (sdom:get-elements-by-tag-name doc "p"))
	   (elem (list-ref elemlist 0))
	   (newelem (sdom:create-element doc tagname rootns)))
      (guard (ex ((sdom:exception? ex)
		  (if (not (memv (sdom:exception-code ex) '(3 8 9)))
		      (raise (make-message-condition 
			      "wrong exception type thrown")))))
	     (begin
	       (sdom:insert-before! doc newelem elem)
	       (raise (make-message-condition
		       "exception should have been thrown"))))
      #t)))

(define nodeinsertbefore-11
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docfrag (sdom:create-document-fragment doc))
	   (elem (sdom:create-element doc "body" xhtmlns))
	   (pi (sdom:create-processing-instruction doc "PITarget" "PIData"))
	   (comment (sdom:create-comment doc "Comment"))
	   (txt (sdom:create-text-node doc "Text"))
	   (cdata (sdom:create-cdata-section doc "CDATA"))
	   (eref (sdom:create-entity-reference doc "alpha")))
      (sdom:append-child! docfrag elem)
      (sdom:append-child! docfrag pi)
      (sdom:append-child! docfrag comment)
      (sdom:append-child! docfrag txt)
      (sdom:append-child! docfrag cdata)
      (sdom:append-child! docfrag eref)
      (if (not (equal? (sdom:data (sdom:insert-before! docfrag comment pi))
		       "Comment"))
	  (raise (make-message-condition
		  "should be able to reinsert comment before PI")))
      (if (not (equal? (sdom:data (sdom:insert-before! docfrag txt comment))
		       "Text"))
	  (raise (make-message-condition
		  "should be able to reinsert text node before comment")))
      (if (not (equal? (sdom:data (sdom:insert-before! docfrag cdata txt))
		       "CDATA"))
	  (raise (make-message-condition
		  "should be able to reinsert cdata before text node")))
      (if (not (equal? (sdom:node-name (sdom:insert-before! docfrag eref cdata))
		       "alpha"))
	  (raise (make-message-condition
		  "should be able to reinsert entity ref before cdata")))
      #t)))

(define nodeinsertbefore-12
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docfrag (sdom:create-document-fragment doc))
	   (docfragnew (sdom:create-document-fragment doc))
	   (elem (sdom:create-element 
		  doc "dom3:elem" "http://www.w3.org/DOM/Test")))
      (sdom:append-child! docfrag elem)
      (sdom:insert-before! docfrag docfragnew elem)
      (if (not (equal? (sdom:node-name (sdom:last-child docfrag)) "dom3:elem"))
	  (raise (make-message-condition
		  "inserting empty doc fragment should have no effect"))
	  #t))))

(define nodeinsertbefore-13
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc))
	   (rootns (sdom:namespace-uri docelem))
	   (tagname (sdom:tag-name docelem))
	   (docalt (sdom:clone-node hc-staff #t))
	   (docfrag (sdom:create-document-fragment doc))
	   (elem (sdom:create-element doc tagname rootns))
	   (elemalt (sdom:create-element docalt tagname rootns)))
      (sdom:append-child! docfrag elem)
      (guard (ex ((sdom:exception? ex)
		  (if (not (eqv? (sdom:exception-code ex)
				 sdom:exception-code-wrong-document-err))
		      (raise (make-message-condition
			      "wrong exception type thrown")))))
	     (begin 
	       (sdom:insert-before! docfrag elemalt elem)
	       (raise (make-message-condition
		       "exception should have been thrown"))))
      #t)))

(define nodeinsertbefore-16
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (childlist (sdom:get-elements-by-tag-name doc "p"))
	   (elem (list-ref childlist 0))
	   (first-child (sdom:first-child elem))
	   (ref-elem (sdom:next-sibling first-child))
	   (newelem (sdom:create-element doc "xhtml:br" xhtmlns)))
      (sdom:insert-before! elem newelem ref-elem)
      (let* ((childlist (sdom:get-elements-by-tag-name doc "p"))
	     (elem (list-ref childlist 0))
	     (first-child (sdom:first-child elem)))
	(if (not (equal? (sdom:node-name (sdom:next-sibling first-child))
			 "xhtml:br"))
	    (raise (make-message-condition
		    "should be able to insert new node before first child"))
	    #t)))))

(define nodeinsertbefore-17
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (childlist (sdom:get-elements-by-tag-name doc "p" "*"))
	   (elem (list-ref childlist 0))
	   (refnode (sdom:first-child elem))
	   (newtext (sdom:create-text-node doc "newText")))
      (sdom:insert-before! elem newtext refnode)
      (if (not (equal? (sdom:node-name (sdom:first-child elem)) "#text"))
	  (raise (make-message-condition
		  "should be able to insert text node before first child"))
	  #t))))

(define nodeinsertbefore-18
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (element (sdom:create-element doc "element"))
	   (newelem (sdom:create-element doc "dom3:elem"
					 "http://www.w3.org/DOM"))
	   (newcomment (sdom:create-comment doc "Comment"))
	   (newcdata (sdom:create-cdata-section doc "CDATASection"))
	   (newpi (sdom:create-processing-instruction doc "target" "data")))
      (sdom:append-child! element newelem)
      (sdom:append-child! element newcomment)
      (sdom:append-child! element newpi)
      (sdom:append-child! element newcdata)
      (sdom:insert-before! element newcomment newelem)
      (if (not (equal? (sdom:data (sdom:first-child element)) "Comment"))
	  (raise (make-message-condition
		  "should be able to reinsert comment node"))
	  #t))))

(define nodeinsertbefore-19
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (childlist (sdom:get-elements-by-tag-name doc "var"))
	   (elem (list-ref childlist 0))
	   (refnode (sdom:first-child elem))
	   (newnode (sdom:create-entity-reference doc "alpha")))
      (if (not (equal? (sdom:node-name 
			(sdom:insert-before! elem newnode refnode))
		       "alpha"))
	  (raise (make-message-condition
		  "should be able to insert entity ref before node"))
	  #t))))

(define nodeinsertbefore-20
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (childlist (sdom:get-elements-by-tag-name doc "var"))
	   (elem (list-ref childlist 0))
	   (refnode (sdom:first-child elem))
	   (newnode (sdom:create-attribute doc "xml:lang" xmlns)))
      (guard (ex ((sdom:exception? ex)
		  (if (not (eqv? (sdom:exception-code ex)
				 sdom:exception-code-hierarchy-request-err))
		      (raise (make-message-condition
			      "wrong exception type thrown")))))
	     (begin
	       (sdom:insert-before! elem newnode refnode)
	       (raise (make-message-condition
		       "exception should have been thrown"))))
      #t)))

(define nodeinsertbefore-21
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (childlist (sdom:get-elements-by-tag-name doc "var"))
	   (elem (list-ref childlist 0))
	   (refnode (sdom:first-child elem))
	   (newnode (sdom:parent-node elem)))
      (guard (ex ((sdom:exception? ex)
		  (if (not (eqv? (sdom:exception-code ex)
				 sdom:exception-code-hierarchy-request-err))
		      (raise (make-message-condition 
			      "wrong exception type thrown")))))
	     (begin
	       (sdom:insert-before! elem newnode refnode)
	       (raise (make-message-condition
		       "exception should have been thrown"))))
      #t)))

(define nodeinsertbefore-22
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo #t))
	   (elem (sdom:create-element doc "xhtml:body" xhtmlns))
	   (refnode (sdom:create-element doc "xhtml:a" xhtmlns))
	   (ancestor (sdom:create-element doc "xhtml:p" xhtmlns)))
      (sdom:append-child! elem refnode)
      (sdom:append-child! ancestor elem)
      (guard (ex ((sdom:exception? ex)
		  (if (not (eqv? (sdom:exception-code ex)
				 sdom:exception-code-hierarchy-request-err))
		      (raise (make-message-condition 
			      "wrong exception type thrown")))))
	     (begin 
	       (sdom:insert-before! elem ancestor refnode)
	       (raise (make-message-condition
		       "exception should have been thrown"))))
      #t)))

(define nodeinsertbefore-23
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doc2 (sdom:clone-node hc-staff #t))
	   (elem (sdom:create-element doc "xhtml:body" xhtmlns))
	   (refnode (sdom:create-element doc "xhtml:p" xhtmlns))
	   (newnode (sdom:create-text-node doc2 "Text")))
      (sdom:append-child! elem refnode)
      (guard (ex ((sdom:exception? ex)
		  (if (not (eqv? (sdom:exception-code ex)
				 sdom:exception-code-wrong-document-err))
		      (raise (make-message-condition
			      "wrong exception type thrown")))))
	     (begin
	       (sdom:insert-before! elem newnode refnode)
	       (raise (make-message-condition
		       "exception should have been thrown"))))
      #t)))

(define nodeinsertbefore-24
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elem (sdom:create-element doc "xhtml:p" xhtmlns))
	   (refnode (sdom:create-cdata-section doc "CDATASection"))
	   (newnode (sdom:create-comment doc "Comment")))
      (guard (ex ((sdom:exception? ex)
		  (if (not (eqv? (sdom:exception-code ex)
				 sdom:exception-code-not-found-err))
		      (raise (make-message-condition
			      "wrong exception type thrown")))))
	     (begin 
	       (sdom:insert-before! elem newnode refnode)
	       (raise (make-message-condition
		       "exception should have been thrown"))))
      #t)))

(define nodeisequalnode-1
  (lambda ()
    (let* ((doc1 (sdom:clone-node hc-staff #t))
	   (doc2 (sdom:clone-node hc-staff #t)))
      (if (not (sdom:equal-node? doc1 doc2))
	  (raise (make-message-condition
		  "equal documents should be evaluated as equal nodes"))
	  #t))))

(define nodeisequalnode-2
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc))
	   (rootns (sdom:namespace-uri docelem))
	   (rootname (sdom:tag-name docelem))
	   (doc1 (sdom:create-document rootname #f rootns))
	   (doc2 (sdom:create-document rootname #f rootns)))
      (if (not (sdom:equal-node? doc1 doc2))
	  (raise (make-message-condition
		  "newly-created equal documents should be evaluated equal"))
	  #t))))

(define nodeisequalnode-4
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elem (sdom:create-element doc "xhtml:p" xhtmlns))
	   (ownerdoc (sdom:owner-document elem)))
      (if (not (sdom:equal-node? doc ownerdoc))
	  (raise (make-message-condition
		  "owner document and document should be equal nodes"))
	  #t))))

(define nodeisequalnode-6
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elem1 (sdom:create-element doc "xhtml:html" xhtmlns))
	   (elem2 (sdom:create-element doc "xhtml:html" xhtmlns)))
      (if (not (sdom:equal-node? elem1 elem2))
	  (raise (make-message-condition
		  "newly-created equal elements should be evaluated equal"))
	  #t))))

(define nodeisequalnode-7
  (lambda ()
    (let* ((doc1 (sdom:clone-node hc-staff #t))
	   (doc2 (sdom:clone-node hc-staff #t))
	   (elem1 (sdom:create-element doc1 "xhtml:html" xhtmlns))
	   (elem2 (sdom:create-element doc2 "xhtml:html" xhtmlns)))
      (if (not (sdom:equal-node? elem1 elem2))
	  (raise (make-message-condition
		  "equal elements should be equal across equal documents"))
	  #t))))

(define nodeisequalnode-8
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elist (sdom:get-elements-by-tag-name doc "em"))
	   (elem1 (list-ref elist 0))
	   (elem2 (sdom:create-element doc "em" xhtmlns))
	   (text (sdom:create-text-node doc "EMP0001")))
      (sdom:append-child! elem2 text)
      (if (not (sdom:equal-node? elem1 elem2))
	  (raise (make-message-condition
		  "equal node hierarchies should be evaluated equal"))
	  #t))))

(define nodeisequalnode-9
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc))
	   (rootns (sdom:namespace-uri docelem))
	   (rootname (sdom:tag-name docelem))
	   (newdoc (sdom:create-document rootname #f rootns))
	   (elist (sdom:get-elements-by-tag-name doc "em"))
	   (elem1 (list-ref elist 0))
	   (elem2 (sdom:create-element doc "em" xhtmlns))
	   (text (sdom:create-text-node doc "EMP0001")))
      (sdom:append-child! elem2 text)
      (if (not (sdom:equal-node? elem1 elem2))
	  (raise (make-message-condition
		  "equal node hierarchies should be evaluated equal"))
	  #t))))

(define nodeisequalnode-11
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc))
	   (rootns (sdom:namespace-uri docelem))
	   (rootname (sdom:tag-name docelem))
	   (newdoc (sdom:create-document rootname #f rootns))
	   (elist (sdom:get-elements-by-tag-name doc "p"))
	   (elem1 (list-ref elist 0))
	   (elem2 (sdom:import-node newdoc elem1 #f)))
      (if (sdom:equal-node? elem1 elem2)
	  (raise (make-message-condition
		  "unequal node hierarchies should not be evaluated equal")))
      (let* ((dupdoc (sdom:clone-node hc-staff #t))
	     (elem4 (sdom:import-node dupdoc elem1 #t)))
	(if (not (sdom:equal-node? elem1 elem4))
	    (raise (make-message-condition
		    "equal node hierarchies should be evaluated equal"))
	    #t)))))

(define nodeisequalnode-12
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elem1 (sdom:document-element doc))
	   (elem2 (sdom:document-element doc)))
      (if (not (sdom:equal-node? elem1 elem2))
	  (raise (make-message-condition
		  "document elements of same node should be equal"))
	  #t))))

(define nodeisequalnode-13
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc))
	   (rootns (sdom:namespace-uri docelem))
	   (rootname (sdom:tag-name docelem))
	   (newdoc (sdom:create-document rootname #f rootns))
	   (elist (sdom:get-elements-by-tag-name doc "p"))
	   (elem1 (list-ref elist 0))
	   (elem2 (sdom:clone-node elem1 #f)))
      (if (sdom:equal-node? elem1 elem2)
	  (raise (make-message-condition
		  "unequal node hierarchies should not be evaluated equal")))
      (let ((elem3 (sdom:clone-node elem1 #t)))
	(if (not (sdom:equal-node? elem1 elem3))
	    (raise (make-message-condition
		    "equal node hierarchies should be evaluated equal"))
	    #t)))))

(define nodeisequalnode-15
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elist (sdom:get-elements-by-tag-name doc "acronym"))
	   (addrelement (list-ref elist 0))
	   (attr1 (sdom:get-attribute-node addrelement "title"))
	   (attr2 (sdom:create-attribute doc "title" #f)))
      (sdom:set-value! attr2 "Yes")
      (if (not (sdom:equal-node? attr1 attr2))
	  (raise (make-message-condition
		  "equal attrs should be evaluated equal"))
	  #t))))

(define nodeisequalnode-17
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc))
	   (rootns (sdom:namespace-uri docelem))
	   (rootname (sdom:tag-name docelem))
	   (newdoc (sdom:create-document rootname #f rootns))
	   (attr1 (sdom:create-attribute doc "root" #f))
	   (attr2 (sdom:import-node newdoc attr1 #t)))
      (if (not (sdom:equal-node? attr1 attr2))
	  (raise (make-message-condition
		  "equal attrs should be evaluated equal across documents"))
	  #t))))

(define nodeisequalnode-18
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc))
	   (rootname (sdom:tag-name docelem))
	   (rootns (sdom:namespace-uri docelem))
	   (newdoc (sdom:create-document rootname #f rootns))
	   (attr1 (sdom:create-attribute doc "title" #f))
	   (attr2 (sdom:adopt-node! newdoc attr1)))
      (if (not (sdom:equal-node? attr1 attr2))
	  (raise (make-message-condition
		  "equal attrs should be evaluated equal across documents"))
	  #t))))

(define nodeisequalnode-19
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (attr1 (sdom:create-attribute doc "lang" xmlns))
	   (attr2 (sdom:create-attribute doc "lang" #f)))
      (if (sdom:equal-node? attr1 attr2)
	  (raise (make-message-condition
		  "namespace inequality should preclude node equality"))
	  #t))))

(define nodeisequalnode-20
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elem1 (sdom:create-element doc "xhtml:html" xhtmlns))
	   (attr1 (sdom:create-attribute doc "xhtml:html" xhtmlns)))
      (if (sdom:equal-node? elem1 attr1)
	  (raise (make-message-condition
		  "node type inequality should preclude node equality"))
	  #t))))

(define nodeisequalnode-28
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (text1 (sdom:create-text-node doc ""))
	   (text2 (sdom:create-text-node doc ""))
	   (text3 (sdom:create-text-node doc "#Text")))
      (if (not (sdom:equal-node? text1 text2))
	  (raise (make-message-condition
		  "null text node text should not preclude equality")))
      (if (sdom:equal-node? text1 text3)
	  (raise (make-message-condition
		  "different text node text should preclude equality"))
	  #t))))

(define nodeisequalnode-29
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (comment1 (sdom:create-comment doc "comment"))
	   (comment2 (sdom:create-comment doc "comment"))
	   (comment3 (sdom:create-comment doc "#Comment")))
      (if (not (sdom:equal-node? comment1 comment2))
	  (raise (make-message-condition
		  "comments with equal comment text should be equal")))
      (if (sdom:equal-node? comment1 comment3)
	  (raise (make-message-condition
		  "comments with unequal comment text should be unequal"))
	  #t))))

(define nodeisequalnode-31
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (cdata1 (sdom:create-cdata-section doc "cdata"))
	   (cdata2 (sdom:create-cdata-section doc "cdata"))
	   (cdata3 (sdom:create-cdata-section doc "#CDATASection")))
      (if (not (sdom:equal-node? cdata1 cdata2))
	  (raise (make-message-condition
		  "cdata nodes with same content should be evaluated equal")))
      (if (sdom:equal-node? cdata1 cdata3)
	  (raise (make-message-condition
		  "cdata nodes with different content should not be equal"))
	  #t))))

(define nodeisequalnode-32
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (pi1 (sdom:create-processing-instruction doc "Target1" "pi"))
	   (pi2 (sdom:create-processing-instruction doc "Target1" "pi"))
	   (pi3 (sdom:create-processing-instruction 
		 doc "Target1" "#ProcessingInstruction")))
      (if (not (sdom:equal-node? pi1 pi2))
	  (raise (make-message-condition
		  "PIs with same target and data should be evaluated equal")))
      (if (sdom:equal-node? pi1 pi3)
	  (raise (make-message-condition
		  "PIs with different target and data should not be equal"))
	  #t))))

(define nodelookupnamespaceuri-1
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo-nodefaultns #t)))
      (if (sdom:namespace-uri doc)
	  (raise (make-message-condition
		  "namespace uri should be null when there is no default"))
	  #t))))

(define nodelookupnamespaceuri-2
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc))
	   (rootns (sdom:namespace-uri docelem))
	   (rootname (sdom:local-name docelem))
	   (qname (string-append "dom3:" rootname))
	   (newdoc (sdom:create-document qname #f rootns))
	   (namespaceuri (sdom:lookup-namespace-uri newdoc "dom3")))
      (if (not (equal? rootns namespaceuri))
	  (raise (make-message-condition
		  "namespace URIs should be portable across prefixes"))
	  #t))))

(define nodelookupnamespaceuri-5
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc))
	   (rootns (sdom:namespace-uri docelem))
	   (rootname (sdom:local-name docelem))
	   (qname (string-append "dom3:" rootname))
	   (newdoc (sdom:create-document qname #f rootns))
	   (elem (sdom:document-element newdoc)))
      (if (not (equal? (sdom:lookup-namespace-uri elem "dom3") rootns))
	  (raise (make-message-condition
		  "namespace URIs should be portable across prefixes"))
	  #t))))

(define noderemovechild-1
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t)))
      (guard (ex ((sdom:exception? ex)
		  (if (not (eqv? (sdom:exception-code ex)))
		      (raise (make-message-condition
			      "wrong exception type thrown")))))
	     (begin
	       (sdom:remove-child! doc doc)
	       (raise (make-message-condition
		       "exception should have been thrown"))))
      #t)))

(define noderemovechild-2
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc))
	   (rootns (sdom:namespace-uri docelem))
	   (rootname (sdom:tag-name docelem))
	   (newdoc (sdom:create-document rootname #f rootns)))
      (guard (ex ((sdom:exception? ex)
		  (if (not (eqv? (sdom:exception-code ex)
				 sdom:exception-code-not-found-err))
		      (raise (make-message-condition
			      "wrong exception type thrown")))))
	     (begin
	       (sdom:remove-child! doc newdoc)
	       (raise (make-message-condition
		       "exception should have been thrown"))))
      #t)))

(define noderemovechild-3
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc)))
      (sdom:remove-child! doc docelem)
      (if (sdom:document-element doc)
	  (raise (make-message-condition
		  "removing child should remove it from document")))
      (guard (ex ((sdom:exception? ex)
		  (if (not (eqv? (sdom:exception-code ex)
				 sdom:exception-code-not-found-err))
		      (raise (make-message-condition
			      "wrong exception type thrown")))))
	     (begin
	       (sdom:remove-child! docelem doc)
	       (raise (make-message-condition
		       "exception should have been thrown"))))
      #t)))

(define noderemovechild-8
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (comment (sdom:create-comment doc "Comment")))
      (sdom:append-child! doc comment)
      (if (not (equal? (sdom:data (sdom:remove-child! doc comment)) "Comment"))
	  (raise (make-message-condition
		  "removing child should not affect data value"))
	  #t))))

(define noderemovechild-9
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (pi (sdom:create-processing-instruction doc "PIT" "PID")))
      (sdom:append-child! doc pi)
      (if (not (equal? (sdom:target (sdom:remove-child! doc pi)) "PIT"))
	  (raise (make-message-condition
		  "removing child should not affect PI target"))
	  #t))))

(define noderemovechild-10
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo #t))
	   (docfrag (sdom:create-document-fragment doc))
	   (elem (sdom:create-element doc "dom3:br" xhtmlns)))
      (sdom:append-child! docfrag elem)
      (if (not (equal? (sdom:node-name (sdom:remove-child! docfrag elem))
		       "dom3:br"))
	  (raise (make-message-condition
		  "removing child should not affect element name"))
	  #t))))

(define noderemovechild-11
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docfrag (sdom:create-document-fragment doc))
	   (txt (sdom:create-text-node doc "TEXT")))
      (sdom:append-child! docfrag txt)
      (sdom:remove-child! docfrag txt)
      (if (sdom:first-child docfrag)
	  (raise (make-message-condition
		  "should be able to remove text node from fragment"))
	  #t))))

(define noderemovechild-12
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docfrag (sdom:create-document-fragment doc))
	   (eref (sdom:create-entity-reference doc "ent1")))
      (sdom:append-child! docfrag eref)
      (sdom:remove-child! docfrag eref)
      (if (sdom:first-child docfrag)
	  (raise (make-message-condition
		  "should be able to remove entity reference from fragment")))
      (guard 
       (ex ((sdom:exception? ex)
	    (if (not 
		 (or (eqv? (sdom:exception-code ex)
			   sdom:exception-code-not-found-err)
		     (eqv? (sdom:exception-code ex)
			   sdom:exception-code-no-modification-allowed-err)))
		(raise (make-message-condition
			"wrong exception type thrown")))))

       (begin 
	 (sdom:remove-child! eref docfrag)
	 (raise (make-message-condition "should have thrown exception"))))
      #t)))

(define noderemovechild-16
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (parentlist (sdom:get-elements-by-tag-name doc "em"))
	   (child (list-ref parentlist 0))
	   (parent (sdom:parent-node child)))
      (if (not (equal? (sdom:node-name (sdom:remove-child! parent child)) "em"))
	  (raise (make-message-condition
		  "removing child should not affect node name")))
      (guard (ex ((sdom:exception? ex)
		  (or (eqv? (sdom:exception-code ex)
			     sdom:exception-code-not-found-err)
		      (raise (make-message-condition 
			      "wrong exception type thrown")))))
	     (begin
	       (sdom:remove-child! child parent)
	       (raise (make-message-condition 
		       "should have thrown exception"))))
      #t)))

(define noderemovechild-17
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (parentlist (sdom:get-elements-by-tag-name doc "em"))
	   (parent (list-ref parentlist 0))
	   (child (sdom:first-child parent)))
      (if (not (equal? (sdom:node-value (sdom:remove-child! parent child))
		       "EMP0001"))
	  (raise (make-message-condition
		  "removing text node should not affect node value")))
      (guard (ex ((sdom:exception? ex)
		  (or (eqv? (sdom:exception-code ex)
			    sdom:exception-code-not-found-err)
		      (raise (make-message-condition
			      "wrong exception type thrown")))))
	     (begin
	       (sdom:remove-child! child parent)
	       (raise (make-message-condition
		       "should have thrown exception"))))
      #t)))

(define noderemovechild-20
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (parentlist (sdom:get-elements-by-tag-name doc "p"))
	   (parent (list-ref parentlist 0))
	   (child (sdom:create-element doc "dom3:br" xhtmlns)))
      (sdom:append-child! parent child)
      (if (not (equal? (sdom:node-name (sdom:remove-child! parent child))
		       "dom3:br"))
	  (raise (make-message-condition
		  "removing new node should not affect node name")))
      (let ((clonedchild (sdom:clone-node child #t)))
	(guard (ex ((sdom:exception? ex)
		    (or (eqv? (sdom:exception-code ex)
			      sdom:exception-code-not-found-err)
			(raise (make-message-condition
				"wrong exception type thrown")))))
	       (begin 
		 (sdom:remove-child! parent clonedchild)
		 (raise (make-message-condition
			 "exception should be thrown"))))
	#t))))

(define noderemovechild-21
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc))
	   (parent (sdom:create-element doc "dom3:p" xhtmlns))
	   (child (sdom:create-element doc "dom3:br" xhtmlns)))
      (sdom:append-child! parent child)
      (sdom:append-child! docelem parent)
      (if (not (equal? (sdom:node-name (sdom:remove-child! parent child))
		       "dom3:br"))
	  (raise (make-message-condition
		  "removing document element should not affect node name")))
      (guard (ex ((sdom:exception? ex)
		  (if (not (eqv? (sdom:exception-code ex)
				 sdom:exception-code-not-found-err))
		      (raise (make-message-condition
			      "wrong exception type thrown")))))
	     (begin
	       (sdom:remove-child! docelem child)
	       (raise (make-message-condition
		       "should have thrown exception"))))
      #t)))

(define noderemovechild-22
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (parent (sdom:create-element doc "dom3:p" xhtmlns))
	   (child (sdom:create-comment doc "DATA")))
      (sdom:append-child! parent child)
      (if (not (equal? (sdom:node-value (sdom:remove-child! parent child))
		       "DATA"))
	  (raise (make-message-condition
		  "removing comment node should not affect node value")))
      (guard (ex ((sdom:exception? ex)
		  (if (not (eqv? (sdom:exception-code ex)
				 sdom:exception-code-not-found-err))
		      (raise (make-message-condition
			      "wrong exception type thrown")))))

	     (begin
	       (sdom:remove-child! parent child)
	       (raise (make-message-condition "should have thrown exception"))))
      #t)))

(define noderemovechild-23
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (parent (sdom:create-element doc "dom3:p" xhtmlns))
	   (child (sdom:create-processing-instruction doc "TARGET" "DATA")))
      (sdom:append-child! parent child)
      (if (not (equal? (sdom:target (sdom:remove-child! parent child))
		       "TARGET"))
	  (raise (make-message-condition
		  "removing PI should not affect PI target")))
      (guard (ex ((sdom:exception? ex)
		  (if (not (eqv? (sdom:exception-code ex)
				 sdom:exception-code-not-found-err))
		      (raise (make-message-condition
			      "wrong exception type thrown")))))
	     (begin
	       (sdom:remove-child! parent child)
	       (raise (make-message-condition "should have thrown exception"))))
      #t)))

(define noderemovechild-28
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (parentlist (sdom:get-elements-by-tag-name doc "acronym"))
	   (elem (list-ref parentlist 0))
	   (parent (sdom:get-attribute-node elem "title"))
	   (child (sdom:first-child parent)))
      (if (not (equal? (sdom:node-value (sdom:remove-child! parent child))
		       "Yes"))
	  (raise (make-message-condition
		  "removing attr text should not affect node value")))
      (guard (ex ((sdom:exception? ex)
		  (if (not (eqv? (sdom:exception-code ex)
				 sdom:exception-code-not-found-err))
		      (raise (make-message-condition
			      "wrong exception type thrown")))))
	     (begin
	       (sdom:remove-child! child parent)
	       (raise (make-message-condition "should have thrown exception"))))
      #t)))

(define nodereplacechild-1
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t)))
      (guard (ex ((sdom:exception? ex)
		  (if (not (memv (sdom:exception-code ex) '(3 8)))
		      (raise (make-message-condition
			      "wrong exception type thrown")))))
	     (begin
	       (sdom:replace-child! doc doc doc)
	       (raise (make-message-condition "should have thrown exception"))))
      #t)))

(define nodereplacechild-3
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (newdoc (sdom:create-document "dom3:doc" #f 
					 "http://www.w3.org/DOM")))
      (guard (ex ((sdom:exception? ex)
		  (if (not (memv (sdom:exception-code ex) '(3 4 8)))
		      (raise (make-message-condition 
			      "wrong exception type thrown")))))
	     (begin
	       (sdom:replace-child! doc newdoc doc)
	       (raise (make-message-condition "should have thrown exception"))))
      #t)))

(define nodereplacechild-4
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc)))
      (guard (ex ((sdom:exception? ex)
		  (if (not (memv (sdom:exception-code ex) '(3 8)))
		      (raise (make-message-condition
			      "wrong exception type thrown")))))
	     (begin
	       (sdom:replace-child! doc doc docelem)
	       (raise (make-message-condition "should have thrown exception"))))
      #t)))

(define nodereplacechild-8
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo #t))
	   (docelem (sdom:document-element doc))
	   (rootname (sdom:tag-name docelem))
	   (rootns (sdom:namespace-uri docelem))
	   (doc2 (sdom:create-document rootname #f rootns))
	   (elem (sdom:create-element doc2 rootname rootns)))
      (guard (ex ((sdom:exception? ex)
		  (if (not (memv (sdom:exception-code ex) '(4 9)))
		      (raise (make-message-condition
			      "wrong exception type thrown")))))
	     (begin 
	       (sdom:replace-child! doc elem docelem)
	       (raise (make-message-condition "should have thrown exception"))))
      #t)))

(define nodereplacechild-12
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo #t))
	   (comment (sdom:create-comment doc "dom3:doc"))
	   (pi (sdom:create-processing-instruction doc "PITarget" "PIData")))
      (sdom:append-child! doc comment)
      (sdom:append-child! doc pi)
      (let ((rc (sdom:replace-child! doc comment pi)))
	(if (not rc) (raise (make-message-condition
			     "value of replaced child should not be null")))
	(if (not (equal? (sdom:node-name rc) "PITarget"))
	    (raise (make-message-condition
		    "replacement should not affect node name"))))
      (let ((lc (sdom:last-child doc)))
	(if (not lc) (raise (make-message-condition
			     "replacement should succeed")))
	(if (not (equal? (sdom:node-name lc) "#comment"))
	    (raise (make-message-condition
		    "replacement should not affect node name"))
	    #t)))))

(define nodereplacechild-14
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elem (sdom:create-element doc "dom3:doc1elem"
				      "http://www.w3.org/DOM/Test"))
	   (newdoc (sdom:create-document "dom3:doc" #f 
					 "http://www.w3.org/DOM/Test"))
	   (elem2 (sdom:create-element newdoc "dom3:doc2elem" 
				    "http://www.w3.org/DOM/Test"))
	   (docelem (sdom:document-element newdoc)))
      (let* ((elem3 (sdom:import-node newdoc elem #t)))
	(sdom:append-child! docelem elem3)
	(sdom:append-child! docelem elem2)
	(if (not (equal? (sdom:node-name
			  (sdom:replace-child! docelem elem3 elem2))
			 "dom3:doc2elem"))
	    (raise (make-message-condition
		    "replacement should not affect node name"))
	    #t)))))

(define nodereplacechild-15
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc))
	   (rootname (sdom:tag-name docelem))
	   (rootns (sdom:namespace-uri docelem))
	   (elem (sdom:create-element doc rootname rootns))
	   (docfrag (sdom:create-document-fragment doc))
	   (elem2 (sdom:create-element doc rootname rootns)))
      (sdom:set-attribute! elem2 "title" "new element")
      (sdom:append-child! docfrag elem2)
      (if (not (equal? (sdom:get-attribute (sdom:replace-child! 
					    docfrag elem elem2)
					   "title")
		       "new element"))
	  (raise (make-message-condition
		  "replacement should not affect attribute content"))
	  #t))))

(define nodereplacechild-16
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elem (sdom:create-element doc "dom3:p" xhtmlns))
	   (docfrag (sdom:create-document-fragment doc))
	   (txt (sdom:create-text-node doc "Comment")))
      (sdom:append-child! docfrag txt)
      (sdom:append-child! docfrag elem)
      (if (not (equal? (sdom:node-name (sdom:replace-child! docfrag txt elem))
		       "dom3:p"))
	  (raise (make-message-condition
		  "replacement should not affect node name"))
	  #t))))

(define nodereplacechild-17
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docfrag (sdom:create-document-fragment doc))
	   (cmt (sdom:create-comment doc "Comment"))
	   (pi (sdom:create-processing-instruction doc "target" "Comment")))
      (sdom:append-child! docfrag pi)
      (sdom:append-child! docfrag cmt)
      (if (not (equal? (sdom:data (sdom:replace-child! docfrag pi cmt))
		       "Comment"))
	  (raise (make-message-condition
		  "replacement should not affect node data")))
      (if (not (equal? (sdom:target (sdom:replace-child! docfrag cmt pi))
		       "target"))
	  (raise (make-message-condition
		  "replacement should not affect PI target"))
	  #t))))

(define nodereplacechild-18
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docfrag (sdom:create-document-fragment doc))
	   (cdata (sdom:create-cdata-section doc "CDATASection"))
	   (entref (sdom:create-entity-reference doc "alpha")))
      (sdom:append-child! docfrag entref)
      (sdom:append-child! docfrag cdata)
      (if (not (equal? (sdom:node-value (sdom:replace-child! 
					 docfrag entref cdata))
		       "CDATASection"))
	  (raise (make-message-condition
		  "replacement should not affect node value")))
      (if (not (equal? (sdom:node-name (sdom:replace-child!
					docfrag cdata entref))
		       "alpha"))
	  (raise (make-message-condition
		  "replacement should not affect node name"))
	  #t))))

(define nodereplacechild-19
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docfrag (sdom:create-document-fragment doc))
	   (elem (sdom:create-element doc "dom3:p" xhtmlns))
	   (entref (sdom:create-entity-reference doc "alpha")))
      (sdom:append-child! elem entref)
      (sdom:append-child! docfrag elem)
      (if (not (equal? (sdom:node-name (sdom:replace-child! 
					docfrag entref elem))
		       "dom3:p"))
	  (raise (make-message-condition
		  "replacement should not affect node name"))
	  #t))))

(define nodereplacechild-20
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docfrag (sdom:create-document-fragment doc))
	   (elem (sdom:create-element doc "dom3:p" xhtmlns))
	   (attr (sdom:create-attribute doc "xml:lang" xmlns)))
      (sdom:append-child! docfrag elem)
      (guard (ex ((sdom:exception? ex)
		  (if (not (eqv? (sdom:exception-code ex)
				 sdom:exception-code-hierarchy-request-err))
		      (raise (make-message-condition
			      "wrong exception type thrown")))))
	     (begin
	       (sdom:replace-child! docfrag attr elem)
	       (raise (make-message-condition
		       "exception should have been thrown"))))
      #t)))

(define nodereplacechild-22
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elem (sdom:create-element doc "dom3:p" xhtmlns))
	   (entrefmain (sdom:create-entity-reference doc "delta"))
	   (entref (sdom:create-entity-reference doc "beta")))
      (sdom:append-child! elem entref)
      (guard 
       (ex ((sdom:exception? ex)
	    (if (not (eqv? (sdom:exception-code ex)
			   sdom:exception-code-no-modification-allowed-err))
		(raise (make-message-condition 
			"wrong exception type thrown")))))
       (begin 
	 (sdom:replace-child! entrefmain elem entref)
	 (raise (make-message-condition "exception should have been thrown"))))
      (guard 
       (ex ((sdom:exception? ex)
	    (if (not (eqv? (sdom:exception-code ex)
			   sdom:exception-code-no-modification-allowed-err))
		(raise (make-message-condition 
			"wrong exception type thrown")))))
       
       (begin
	 (sdom:replace-child! entrefmain entref elem)
	 (raise (make-message-condition
		 "exception should have been thrown"))))
      (guard (ex ((sdom:exception? ex)
		  (if (not (eqv? (sdom:exception-code ex)
				 sdom:exception-code-hierarchy-request-err))
		      (raise (make-message-condition 
			      "wrong exception type thrown")))))
	     (begin
	       (sdom:replace-child! entrefmain entrefmain entref)
	       (raise (make-message-condition
		       "exception should have been thrown"))))
      #t)))

(define nodereplacechild-23
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elem (sdom:create-element doc "dom3:p" xhtmlns))
	   (entref (sdom:create-entity-reference doc "delta"))
	   (txt (sdom:create-text-node doc "Text"))
	   (comment (sdom:create-comment doc "Comment"))
	   (cdata (sdom:create-cdata-section doc "CDATASection"))
	   (pi (sdom:create-processing-instruction doc "target" "data")))
      (sdom:append-child! elem entref)
      (sdom:append-child! elem txt)
      (sdom:append-child! elem comment)
      (sdom:append-child! elem pi)
      (sdom:append-child! elem cdata)
      (guard
       (ex ((sdom:exception? ex)
	    (if (not (eqv? (sdom:exception-code ex)
			   sdom:exception-code-no-modification-allowed-err))
		(raise (make-message-condition
			"wrong exception type thrown")))))
       (begin
	 (sdom:replace-child! entref cdata elem)
	 (raise (make-message-condition "exception should have been thrown"))))
      (guard
       (ex ((sdom:exception? ex)
	    (if (not (eqv? (sdom:exception-code ex)
			   sdom:exception-code-no-modification-allowed-err))
		(raise (make-message-condition
			"wrong exception type thrown")))))
       (begin
	 (sdom:replace-child! entref pi cdata)
	 (raise (make-message-condition "exception should have been thrown"))))
      (guard
       (ex ((sdom:exception? ex)
	    (if (not (eqv? (sdom:exception-code ex)
			   sdom:exception-code-no-modification-allowed-err))
		(raise (make-message-condition
			"wrong exception type thrown")))))
       (begin
	 (sdom:replace-child! entref comment pi)
	 (raise (make-message-condition "exception should have been thrown"))))
      (guard
       (ex ((sdom:exception? ex)
	    (if (not (eqv? (sdom:exception-code ex)
			   sdom:exception-code-no-modification-allowed-err))
		(raise (make-message-condition 
			"wrong exception type thrown")))))
       (begin
	 (sdom:replace-child! entref txt comment)
	 (raise (make-message-condition "exception should have been thrown"))))
      (guard
       (ex ((sdom:exception? ex)
	    (if (not (eqv? (sdom:exception-code ex)
			   sdom:exception-code-no-modification-allowed-err))
		(raise (make-message-condition
			"wrong exception type thrown")))))
       (begin
	 (sdom:replace-child! entref elem txt)
	 (raise (make-message-condition "exception should have been thrown"))))
      #t)))

(define nodereplacechild-26
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc))
	   (childlist (sdom:get-elements-by-tag-name doc "p"))
	   (elem (list-ref childlist 0))
	   (firstchild (sdom:first-child elem)))
      (guard (ex ((sdom:exception? ex)
		  (if (not (eqv? (sdom:exception-code ex)
				 sdom:exception-code-hierarchy-request-err))
		      (raise (make-message-condition
			      "wrong exception type thrown")))))
	     (begin
	       (sdom:replace-child! elem docelem firstchild)
	       (raise (make-message-condition
		       "exception should have been thrown"))))
      #t)))

(define nodereplacechild-27
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (childlist (sdom:get-elements-by-tag-name doc "p" "*"))
	   (elem (list-ref childlist 0))
	   (firstchild (sdom:first-child elem))
	   (doc2 (sdom:clone-node hc-staff #t))
	   (childlist2 (sdom:get-elements-by-tag-name doc2 "p" "*"))
	   (elem2 (list-ref childlist2 0)))
      (guard (ex ((sdom:exception? ex)
		  (if (not (eqv? (sdom:exception-code ex)
				 sdom:exception-code-wrong-document-err))
		      (raise (make-message-condition
			      "wrong exception type thrown")))))
	     (begin
	       (sdom:replace-child! elem elem2 firstchild)
	       (raise (make-message-condition
		       "exception should have been thrown"))))
      #t)))

(define nodereplacechild-29
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (childlist (sdom:get-elements-by-tag-name doc "p"))
	   (elem (list-ref childlist 0))
	   (oldchild (sdom:create-element doc "dom3:br" xhtmlns))
	   (newchild (sdom:create-element doc "dom3:span" xhtmlns)))
      (guard (ex ((sdom:exception? ex)
		  (if (not (eqv? (sdom:exception-code ex)
				 sdom:exception-code-not-found-err))
		      (raise (make-message-condition
			      "wrong exception type thrown")))))
	     (begin
	       (sdom:replace-child! elem newchild oldchild)
	       (raise (make-message-condition
		       "exception should have been thrown"))))
      #t)))

(define nodereplacechild-30
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (parent (sdom:create-element doc "xhtml:html" xhtmlns))
	   (oldchild (sdom:create-element doc "xhtml:head" xhtmlns))
	   (newelement (sdom:create-element doc "xhtml:body" xhtmlns))
	   (newtext (sdom:create-text-node doc "Text"))
	   (newcomment (sdom:create-comment doc "Comment"))
	   (newpi (sdom:create-processing-instruction doc "target" "data"))
	   (newcdata (sdom:create-cdata-section doc "Cdata"))
	   (neweref (sdom:create-entity-reference doc "delta")))
      (sdom:append-child! parent oldchild)
      (sdom:append-child! parent newelement)
      (sdom:append-child! parent newcomment)
      (sdom:append-child! parent newpi)
      (sdom:append-child! parent newcdata)
      (sdom:append-child! parent neweref)
      (if (not (equal? (sdom:node-name (sdom:replace-child! 
					parent newelement oldchild))
		       "xhtml:head"))
	  (raise (make-message-condition
		  "elem replacement should preserve node name on elem")))
      (if (not (equal? (sdom:node-name (sdom:replace-child!
					parent oldchild newelement))
		       "xhtml:body"))
	  (raise (make-message-condition
		  "elem replacement should preserve node name on elem")))
      (if (not (equal? (sdom:node-name (sdom:replace-child!
					parent newtext oldchild))
		       "xhtml:head"))
	  (raise (make-message-condition
		  "text replacement should preserve node name on elem")))
      (if (not (equal? (sdom:node-name (sdom:replace-child!
					parent oldchild newtext))
		       "#text"))
	  (raise (make-message-condition
		  "elem replacement should preserve node name on text")))
      (if (not (equal? (sdom:node-name (sdom:replace-child!
					parent newcomment oldchild))
		       "xhtml:head"))
	  (raise (make-message-condition
		  "comment replacement should preserve node name on elem")))
      (if (not (equal? (sdom:node-name (sdom:replace-child!
					parent oldchild newcomment))
		       "#comment"))
	  (raise (make-message-condition
		  "elem replacement should preserve node name on comment")))
      (if (not (equal? (sdom:node-name (sdom:replace-child!
					parent oldchild newpi))
		       "target"))
	  (raise (make-message-condition
		  "elem replacement should preserve node name on PI")))
      (if (not (equal? (sdom:node-name (sdom:replace-child!
					parent oldchild newcdata))
		       "#cdata-section"))
	  (raise (make-message-condition
		  "elem replacement should preserve node name on CDATA")))
      (if (not (equal? (sdom:node-name (sdom:replace-child!
					parent oldchild neweref))
		       "delta"))
	  (raise (make-message-condition
		  "elem replacement should preserve node name on ent-ref"))
	  #t))))

(define nodereplacechild-32
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (newchild (sdom:create-text-node doc "Text"))
	   (childlist (sdom:get-elements-by-tag-name doc "acronym" "*"))
	   (elem (list-ref childlist 0))
	   (parent (sdom:get-attribute-node elem "title"))
	   (entref (sdom:create-entity-reference doc "entity1")))
      (sdom:append-child! parent entref)
      (if (not (equal? (sdom:node-name (sdom:replace-child!
					parent newchild entref))
		       "entity1"))
	  (raise (make-message-condition
		  "replacement should preserve entity reference name"))
	  #t))))

(define nodereplacechild-34
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (parent (sdom:create-attribute doc "xml:lang" xmlns))
	   (oldchild (sdom:create-entity-reference doc "delta"))
	   (newchild (sdom:create-text-node doc "Text")))
      (sdom:append-child! parent oldchild)
      (sdom:replace-child! parent newchild oldchild)
      (if (not (equal? (sdom:value parent) "Text"))
	  (raise (make-message-condition
		  "replaced node should have correct value"))
	  #t))))

(define nodereplacechild-35
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (parent (sdom:create-attribute doc "xml:lang" xmlns))
	   (oldchild (sdom:create-entity-reference doc "delta"))
	   (newchild (sdom:create-attribute doc "xml:lang" xmlns)))
      (sdom:append-child! parent oldchild)
      (guard (ex ((sdom:exception? ex)
		  (if (not (eqv? (sdom:exception-code ex)
				 sdom:exception-code-hierarchy-request-err))
		      (raise (make-message-condition
			      "wrong exception type thrown")))))

	     (begin
	       (sdom:replace-child! parent newchild oldchild)
	       (raise (make-message-condition
		       "exception should have been thrown"))))
      #t)))

(define nodereplacechild-36
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (parent (sdom:create-attribute doc "xml:lang" xmlns))
	   (oldchild (sdom:create-entity-reference doc "delta"))
	   (newchild (sdom:create-text-node doc "Text")))
      (guard (ex ((sdom:exception? ex)
		  (if (not (eqv? (sdom:exception-code ex)
				 sdom:exception-code-not-found-err))
		      (raise (make-message-condition
			      "wrong exception type thrown")))))

	     (begin
	       (sdom:replace-child! parent newchild oldchild)
	       (raise (make-message-condition
		       "exception should have been thrown"))))
      #t)))

(define nodereplacechild-37
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doc2 (sdom:clone-node hc-staff #t))
	   (parent (sdom:create-attribute doc "xml:lang" xmlns))
	   (oldchild (sdom:create-text-node doc "Text"))
	   (newchild (sdom:create-entity-reference doc2 "delta")))
      (sdom:append-child! parent oldchild)
      (guard (ex ((sdom:exception? ex)
		  (if (not (eqv? (sdom:exception-code ex)
				 sdom:exception-code-wrong-document-err))
		      (raise (make-message-condition
			      "wrong exception type thrown")))))
	     (begin
	       (sdom:replace-child! parent newchild oldchild)
	       (raise (make-message-condition
		       "exception should have been thrown"))))
      #t)))

(define nodereplacechild-39
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo #t))
	   (docelem (sdom:document-element doc))
	   (rootname (sdom:tag-name docelem))
	   (rootns (sdom:namespace-uri docelem))
	   (newelement (sdom:create-element doc rootname rootns))
	   (newcomment (sdom:create-comment doc "second element goes here")))
      (sdom:append-child! doc newcomment)
      (guard (ex ((sdom:exception? ex)
		  (if (not (memv (sdom:exception-code ex) '(3 9)))
		      (raise (make-message-condition
			      "wrong exception type thrown")))))
	     (begin
	       (sdom:replace-child! doc newelement newcomment)
	       (raise (make-message-condition
		       "exception should have been thrown"))))
      #t)))

(define nodesettextcontent-1
  (lambda ()
    (let ((doc (sdom:clone-node hc-staff #t)))
      (sdom:set-text-content! doc "textContent")
      (let* ((elemlist (sdom:get-elements-by-tag-name doc "acronym"))
	     (elem (list-ref elemlist 0)))
	(if (null? elemlist)
	    (raise (make-message-condition
		    "setting text content should preserve structure")))
	(if (not (equal? (sdom:node-name elem) "acronym"))
	    (raise (make-message-condition
		    "setting text content should preserve node name"))
	    #t)))))

(define nodesettextcontent-2
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (newdoc (sdom:create-document "dom3:elem" #f 
					 "http://www.w3.org/DOM/Test"))
	   (newelem (sdom:create-element newdoc "dom3:childElem"
					 "http://www.w3.org/DOM/Test"))
	   (docelem (sdom:document-element newdoc)))
      (sdom:append-child! docelem newelem)
      (sdom:set-text-content! newdoc "textContent")
      (let* ((elemlist (sdom:get-elements-by-tag-name newdoc "childElem" "*"))
	     (elemchild (list-ref elemlist 0)))
	(if (not (equal? (sdom:node-name elemchild) "dom3:childElem"))
	    (raise (make-message-condition
		    "setting text content should not affect node name"))
	    #t)))))

(define nodesettextcontent-6
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elem (sdom:create-element doc "p" xhtmlns))
	   (attr (sdom:create-attribute doc "xml:lang" xmlns)))
      (sdom:set-attribute-node! elem attr)
      (sdom:set-text-content! attr "NA")
      (if (not (equal? (sdom:text-content attr) "NA"))
	  (raise (make-message-condition
		  "setting text content should change text content"))
	  #t))))

(define nodesettextcontent-7
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elemlist (sdom:get-elements-by-tag-name doc "em"))
	   (elem (list-ref elemlist 0))
	   (txt (sdom:first-child elem)))
      (sdom:set-text-content! elem "Text")
      (if (not (equal? (sdom:text-content elem) "Text"))
	  (raise (make-message-condition
		  "setting text content should change text content"))
	  #t))))

(define nodesettextcontent-8
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elem (sdom:create-element doc "xhtml:p" xhtmlns))
	   (pi (sdom:create-processing-instruction doc "PIT" "PID")))
      (sdom:append-child! elem pi)
      (sdom:set-text-content! pi "PID")
      (if (not (equal? (sdom:text-content pi) "PID"))
	  (raise (make-message-condition
		  "setting text content should change text content"))
	  #t))))

(define nodesettextcontent-10
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elem (sdom:create-element doc "dom3:elem"
				      "http://www.w3.org/DOM/Test"))
	   (txt (sdom:create-text-node doc "Text"))
	   (comment (sdom:create-comment doc "Comment"))
	   (entref (sdom:create-entity-reference doc "ent1"))
	   (pi (sdom:create-processing-instruction doc "PIT" "PIData"))
	   (cdata (sdom:create-cdata-section doc "CData")))
      (sdom:append-child! elem txt)
      (sdom:append-child! elem comment)
      (sdom:append-child! elem entref)
      (sdom:append-child! elem pi)
      (sdom:append-child! elem cdata)
      (sdom:set-text-content! elem "ELEMENT")
      (if (not (equal? (sdom:text-content elem) "ELEMENT"))
	  (raise (make-message-condition
		  "setting text content should change text content"))
	  #t))))

(define nodesettextcontent-11
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docfrag (sdom:create-document-fragment doc))
	   (elem (sdom:create-element doc "xhtml:p" xhtmlns))
	   (txt (sdom:create-text-node doc "Text "))
	   (comment (sdom:create-comment doc "Comment"))
	   (entref (sdom:create-entity-reference doc "alpha"))
	   (pi (sdom:create-processing-instruction doc "PIT" "PIData "))
	   (cdata (sdom:create-cdata-section doc "CData")))
      (sdom:append-child! elem txt)
      (sdom:append-child! elem comment)
      (sdom:append-child! elem entref)
      (sdom:append-child! elem pi)
      (sdom:append-child! elem cdata)
      (sdom:append-child! docfrag elem)
      (sdom:set-text-content! docfrag "DOCUMENTFRAGMENT")
      (if (not (equal? (sdom:text-content (sdom:last-child docfrag))
		       "DOCUMENTFRAGMENT"))
	  (raise (make-message-condition
		  "setting text content should change text content"))
	  #t))))

(define nodesettextcontent-12
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elem (sdom:document-element doc))
	   (entref (sdom:create-entity-reference doc "beta")))
      (sdom:append-child! elem entref)
      (guard 
       (ex ((sdom:exception? ex) 
	    (if (not (eqv? (sdom:exception-code ex)
			   sdom:exception-code-no-modification-allowed-err))
		(raise (make-message-condition 
			"wrong exception type thrown")))))
       (begin
	 (sdom:set-text-content! entref "NA")
	 (raise (make-message-condition "exception should have been thrown"))))
      #t)))

(define nodesetuserdata-1
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t)))
      (if (sdom:set-user-data! doc "something" '() '())
	  (raise (make-message-condition "user data should be null initially"))
	  #t))))

(define nodesetuserdata-3
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elem (sdom:create-element doc "xhtml:p" xhtmlns))
	   (txt (sdom:create-text-node doc "TEXT")))
      (sdom:set-user-data! doc "Key1" elem '())
      (if (not (sdom:equal-node? (sdom:set-user-data! doc "Key1" txt '()) 
				 elem))
	  (raise (make-message-condition
		  "setting user data should return previous value"))
	  #t))))

(define nodesetuserdata-4
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elem (sdom:create-element doc "p" xhtmlns))
	   (txt (sdom:create-text-node doc "TEXT")))
      (sdom:set-user-data! elem "Key1" txt '())
      (sdom:set-user-data! elem "Key2" txt '())
      (let* ((ret1 (sdom:get-user-data elem "Key1"))
	     (ret2 (sdom:get-user-data elem "Key2")))
	(if (not (sdom:equal-node? ret1 ret2))
	    (raise (make-message-condition
		    "setting user data should not affect node equality"))
	    #t)))))

(define nodesetuserdata-5
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (doc2 (sdom:clone-node hc-staff #t))
	   (attr (sdom:create-attribute doc "lang" xmlns)))
      (sdom:set-user-data! attr "Key1" doc '())
      (sdom:set-user-data! attr "Key2" doc2 '())
      (if (not (sdom:equal-node? (sdom:get-user-data attr "Key1")
				 (sdom:get-user-data attr "Key2")))
	  (raise (make-message-condition
		  "setting user data should not affect node equality"))
	  #t))))

(define nodesetuserdata-8
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc))
	   (entref (sdom:create-entity-reference doc "delta"))
	   (cdata (sdom:create-cdata-section doc "CDATASection")))
      (sdom:set-user-data! entref "Key1" doc '())
      (sdom:set-user-data! cdata "Key2" docelem '())
      (if (sdom:equal-node? (sdom:get-user-data entref "Key1")
			    (sdom:get-user-data cdata "Key2"))
	  (raise (make-message-condition
		  "setting user data should not affect node inequality"))
	  #t))))

(define nodesetuserdata-9
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc)))
      (sdom:set-user-data! docelem "Key1" doc '())
      (if (sdom:get-user-data doc "Key1")
	  (raise (make-message-condition
		  "setting user data should affectly only target node"))
	  #t))))

(define nodesetuserdata-10
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (docelem (sdom:document-element doc))
	   (entref (sdom:create-entity-reference doc "delta"))
	   (cdata (sdom:create-cdata-section doc "CDATASection")))
      (sdom:set-user-data! entref "Key1" doc '())
      (sdom:set-user-data! cdata "Key2" docelem '())
      (if (sdom:equal-node? (sdom:get-user-data entref "Key1")
			    (sdom:get-user-data cdata "Key2"))
	  (raise (make-message-condition
		  "setting user data should not affect node inequality"))
	  #t))))

(define textreplacewholetext-1
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (itemlist (sdom:get-elements-by-tag-name doc "strong"))
	   (elem (list-ref itemlist 0))
	   (textnode (sdom:first-child elem))
	   (replaced (sdom:replace-whole-text! textnode "New Content")))
      (if (not (equal? (sdom:whole-text replaced) "New Content"))
	  (raise (make-message-condition
		  "replacing whole text should affect node text"))
	  #t))))

(define textreplacewholetext-2
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (itemlist (sdom:get-elements-by-tag-name doc "strong"))
	   (elem (list-ref itemlist 0))
	   (textnode (sdom:first-child elem)))
      (if (sdom:replace-whole-text! textnode "")
	  (raise (make-message-condition
		  "replacing whole text with empty string should be null"))
	  #t))))

(define textreplacewholetext-3
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (textnode (sdom:create-text-node doc "New Text"))
	   (replacedtext (sdom:replace-whole-text! textnode " a b c b ")))
      (if (not (equal? (sdom:whole-text replacedtext) " a b c b "))
	  (raise (make-message-condition
		  "replacing whole text should affect disconnected nodes"))
	  #t))))

(define textreplacewholetext-4
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (textnode (sdom:create-text-node doc "New Text")))
      (if (sdom:replace-whole-text! textnode "")
	  (raise (make-message-condition
		  "replacing whole text with empty string should be null"))
	  #t))))

(define textreplacewholetext-5
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (itemlist (sdom:get-elements-by-tag-name doc "strong"))
	   (elemname (list-ref itemlist 0))
	   (textnode (sdom:create-text-node doc "New Text"))
	   (cdatasection (sdom:create-cdata-section doc "New CDATA")))
      (sdom:append-child! elemname textnode)
      (sdom:append-child! elemname cdatasection)
      (let* ((textnode (sdom:first-child elemname))
	     (replacedtext (sdom:replace-whole-text! 
			    textnode "New Text and Cdata")))
	(if (not (equal? (sdom:whole-text replacedtext) "New Text and Cdata"))
	    (raise (make-message-condition
		    "replacing whole text should affect node text"))
	    #t)))))

(define textreplacewholetext-6
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (itemlist (sdom:get-elements-by-tag-name doc "strong"))
	   (elemstrong (list-ref itemlist 0))
	   (textnode (sdom:create-text-node doc "New Text"))
	   (erefnode (sdom:create-entity-reference doc "beta")))
      (sdom:append-child! elemstrong textnode)
      (sdom:append-child! elemstrong erefnode)
      (let* ((textnode (sdom:first-child elemstrong))
	     (replacedtext (sdom:replace-whole-text! textnode 
						     "New Text and Cdata")))
	(if (not (equal? (sdom:node-value replacedtext) "New Text and Cdata"))
	    (raise (make-message-condition
		    "replacing whole text should affect node text"))
	    #t)))))

(define textreplacewholetext-7
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (itemlist (sdom:get-elements-by-tag-name doc "strong"))
	   (elemname (list-ref itemlist 0))
	   (erefnode (sdom:create-entity-reference doc "ent4"))
	   (textnode (sdom:create-text-node doc "New Text")))
      (sdom:append-child! elemname erefnode)
      (sdom:append-child! elemname textnode)
      (let* ((textnode (sdom:first-child elemname))
	     (replacedtext (sdom:replace-whole-text! textnode 
						     "New Text and Cdata"))
	     (textnode (sdom:first-child elemname)))
	(if (not (sdom:same-node? textnode replacedtext))
	    (raise (make-message-condition
		    "replaced text should be same as new first child")))
	(if (not (equal? (sdom:node-value textnode) "New Text and Cdata"))
	    (raise (make-message-condition
		    "replacing whole text should affect node value")))
	(let ((node (sdom:next-sibling textnode)))
	  (if (not node)
	      (raise (make-message-condition
		      "replacing text should not affect unrelated nodes")))
	  (if (not (eqv? (sdom:node-type node) 
			 sdom:node-type-entity-reference))
	      (raise (make-message-condition
		      "replacing text should not affect unrelated node type"))
	      #t))))))

(define textwholetext-1
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (itemlist (sdom:get-elements-by-tag-name doc "strong"))
	   (elemname (list-ref itemlist 0))
	   (textnode (sdom:first-child elemname))
	   (nametext (sdom:whole-text textnode)))
      (if (not (equal? nametext "Margaret Martin"))
	  (raise (make-message-condition
		  "retrieving whole text should retrieve all adjacent text"))
	  #t))))

(define textwholetext-2
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (itemlist (sdom:get-elements-by-tag-name doc "strong"))
	   (elemname (list-ref itemlist 0))
	   (newtext (sdom:create-text-node doc "New Text")))
      (sdom:append-child! elemname newtext)
      (if (not (equal? (sdom:whole-text (sdom:first-child elemname))
		       "Margaret MartinNew Text"))
	  (raise (make-message-condition
		  "retrieving whole text should retrieve all adjacent text"))
	  #t))))

(define textwholetext-3
  (lambda ()
    (let* ((doc (sdom:clone-node hc-staff #t))
	   (elem (sdom:create-element doc "p" xhtmlns))
	   (text1 (sdom:create-text-node doc "Text I"))
	   (text2 (sdom:create-text-node doc " Text II")))
      (sdom:append-child! elem text1)
      (sdom:append-child! elem text2)
      (if (not (equal? (sdom:whole-text text1) "Text I Text II"))
	  (raise (make-message-condition
		  "retrieving whole text should retrieve all adjacent text"))
	  #t))))

(define userdatahandler-1
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo #t))
	   (notifications (list))
	   (handler (lambda (op key data src dst)
		      (let ((newnot (list op key data src dst)))
			(set! notifications
			      (append notifications (list newnot))))))
	   (hello "Hello")
	   (mister "Mr.")
	   (plist (sdom:get-elements-by-tag-name doc "p"))
	   (node (list-ref plist 0)))
      (sdom:set-user-data! node "greeting" hello handler)
      (sdom:set-user-data! node "salutation" mister handler)
      (let* ((elemns (sdom:namespace-uri node))
	     (newnode (sdom:rename-node! node "div" elemns))
	     (greetingcount 0)
	     (salutationcount 0))
	(if (not (eqv? (length notifications) 2))
	    (raise (make-message-condition
		    "handler should receive both notifications")))
	(for-each (lambda (x)
		    (if (not (eqv? (car x) sdom:user-data-event-node-renamed))
			(raise (make-message-condition
				"wrong operation for notification")))
		    (cond ((equal? (list-ref x 1) "greeting")
			   (set! greetingcount (+ greetingcount 1))
			   (if (not (equal? (list-ref x 2) hello))
			       (raise (make-message-condition
				       "wrong data for greeting"))))
			  ((equal? (list-ref x 1) "salutation")
			   (set! salutationcount (+ salutationcount 1))
			   (if (not (equal? (list-ref x 2) mister))
			       (raise (make-message-condition
				       "wrong data for salutation"))))
			  (else (raise (make-message-condition
					"wrong key for notification"))))
		    (if (not (sdom:same-node? node (list-ref x 3)))
			(raise (make-message-condition
				"source node should be old node")))
		    (let ((dst (list-ref x 4)))
		      (if (null? dst)
			  (if (not (sdom:same-node? node newnode))
			      (raise (make-message-condition
				      "node should be reused under rename")))
			  (if (not (sdom:same-node? newnode dst))
			      (raise (make-message-condition
				      "wrong destination node"))))))
		  notifications)
	(if (not (eqv? greetingcount 1)) 
	    (raise (make-message-condition
		    "wrong number of greetings received")))
	(if (not (eqv? salutationcount 1))
	    (raise (make-message-condition
		    "wrong number of salutations received"))
	    #t)))))

(define userdatahandler-2
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo #t))
	   (notifications (list))
	   (handler (lambda (op key data src dst)
		      (let ((newnot (list op key data src dst)))
			(set! notifications
			      (append notifications (list newnot))))))
	   (hello "Hello")
	   (mister "Mr.")
	   (plist (sdom:get-elements-by-tag-name doc "p"))
	   (node (list-ref plist 0)))
      (sdom:set-user-data! node "greeting" hello handler)
      (sdom:set-user-data! node "salutation" mister handler)
      (let* ((elemns (sdom:namespace-uri node))
	     (newnode (sdom:clone-node node #t))
	     (greetingcount 0)
	     (salutationcount 0))
	(if (not (eqv? (length notifications) 2))
	    (raise (make-message-condition
		    "handler should receive both notifications")))
	(for-each (lambda (x)
		    (if (not (eqv? (car x) sdom:user-data-event-node-cloned))
			(raise (make-message-condition
				"wrong operation for notification")))
		    (cond ((equal? (list-ref x 1) "greeting")
			   (set! greetingcount (+ greetingcount 1))
			   (if (not (equal? (list-ref x 2) hello))
			       (raise (make-message-condition
				       "wrong data for greeting"))))
			  ((equal? (list-ref x 1) "salutation")
			   (set! salutationcount (+ salutationcount 1))
			   (if (not (equal? (list-ref x 2) mister))
			       (raise (make-message-condition
				       "wrong data for salutation"))))
			  (else (raise (make-message-condition
					"wrong key for notification"))))
		    (if (not (sdom:same-node? node (list-ref x 3)))
			(raise (make-message-condition
				"source node should be old node")))
		    (if (not (sdom:same-node? newnode (list-ref x 4)))
			(raise (make-message-condition
				"destination node should be new node"))))
		  notifications)
	(if (not (eqv? greetingcount 1)) 
	    (raise (make-message-condition
		    "wrong number of greetings received")))
	(if (not (eqv? salutationcount 1))
	    (raise (make-message-condition
		    "wrong number of salutations received"))
	    #t)))))

(define userdatahandler-3
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo #t))
	   (docelem (sdom:document-element doc))
	   (rootns (sdom:namespace-uri docelem))
	   (rootname (sdom:tag-name docelem))
	   (newdoc (sdom:create-document rootname #f rootns))
	   (notifications (list))
	   (handler (lambda (op key data src dst)
		      (let ((newnot (list op key data src dst)))
			(set! notifications
			      (append notifications (list newnot))))))
	   (hello "Hello")
	   (mister "Mr.")
	   (plist (sdom:get-elements-by-tag-name doc "p"))
	   (node (list-ref plist 0)))
      (sdom:set-user-data! node "greeting" hello handler)
      (sdom:set-user-data! node "salutation" mister handler)
      (let* ((elemns (sdom:namespace-uri node))
	     (newnode (sdom:import-node doc node #t))
	     (greetingcount 0)
	     (salutationcount 0))
	(if (not (eqv? (length notifications) 2))
	    (raise (make-message-condition
		    "handler should receive both notifications")))
	(for-each (lambda (x)
		    (if (not (eqv? (car x) sdom:user-data-event-node-imported))
			(raise (make-message-condition
				"wrong operation for notification")))
		    (cond ((equal? (list-ref x 1) "greeting")
			   (set! greetingcount (+ greetingcount 1))
			   (if (not (equal? (list-ref x 2) hello))
			       (raise (make-message-condition
				       "wrong data for greeting"))))
			  ((equal? (list-ref x 1) "salutation")
			   (set! salutationcount (+ salutationcount 1))
			   (if (not (equal? (list-ref x 2) mister))
			       (raise (make-message-condition
				       "wrong data for salutation"))))
			  (else (raise (make-message-condition
					"wrong key for notification"))))
		    (if (not (sdom:same-node? node (list-ref x 3)))
			(raise (make-message-condition
				"source node should be old node")))
		    (if (not (sdom:same-node? newnode (list-ref x 4)))
			(raise (make-message-condition
				"destination node should be new node"))))
		  notifications)
	(if (not (eqv? greetingcount 1)) 
	    (raise (make-message-condition
		    "wrong number of greetings received")))
	(if (not (eqv? salutationcount 1))
	    (raise (make-message-condition
		    "wrong number of salutations received"))
	    #t)))))

(define userdatahandler-4
  (lambda ()
    (let* ((doc (sdom:clone-node barfoo #t))
	   (docelem (sdom:document-element doc))
	   (rootns (sdom:namespace-uri docelem))
	   (rootname (sdom:tag-name docelem))
	   (newdoc (sdom:create-document rootname #f rootns))
	   (notifications (list))
	   (handler (lambda (op key data src dst)
		      (let ((newnot (list op key data src dst)))
			(set! notifications
			      (append notifications (list newnot))))))
	   (hello "Hello")
	   (mister "Mr.")
	   (plist (sdom:get-elements-by-tag-name doc "p"))
	   (node (list-ref plist 0)))
      (sdom:set-user-data! node "greeting" hello handler)
      (sdom:set-user-data! node "salutation" mister handler)
      (let* ((elemns (sdom:namespace-uri node))
	     (newnode (sdom:adopt-node! doc node))
	     (greetingcount 0)
	     (salutationcount 0))
	(if (not (eqv? (length notifications) 2))
	    (raise (make-message-condition
		    "handler should receive both notifications")))
	(for-each (lambda (x)
		    (if (not (eqv? (car x) sdom:user-data-event-node-adopted))
			(raise (make-message-condition
				"wrong operation for notification")))
		    (cond ((equal? (list-ref x 1) "greeting")
			   (set! greetingcount (+ greetingcount 1))
			   (if (not (equal? (list-ref x 2) hello))
			       (raise (make-message-condition
				       "wrong data for greeting"))))
			  ((equal? (list-ref x 1) "salutation")
			   (set! salutationcount (+ salutationcount 1))
			   (if (not (equal? (list-ref x 2) mister))
			       (raise (make-message-condition
				       "wrong data for salutation"))))
			  (else (raise (make-message-condition
					"wrong key for notification"))))
		    (if (not (sdom:same-node? node (list-ref x 3)))
			(raise (make-message-condition
				"source node should be old node")))
		    (if (not (null? (list-ref x 4)))
			(raise (make-message-condition
				"destination node should be null"))))
		  notifications)
	(if (not (eqv? greetingcount 1)) 
	    (raise (make-message-condition
		    "wrong number of greetings received")))
	(if (not (eqv? salutationcount 1))
	    (raise (make-message-condition
		    "wrong number of salutations received"))
	    #t)))))

(define tests 
  `((attrisid-1)
    (attrisid-2)
    (attrisid-3)
    (attrisid-6)
    (attrisid-7)
    (canonicalform-1)
    (canonicalform-2)
    (canonicalform-3)
    (canonicalform-4)
    (cdatasections-1)
    (comments-1)
    (documentadoptnode-3)
    (documentadoptnode-4)
    (documentadoptnode-7)
    (documentadoptnode-8)
    (documentadoptnode-9)
    (documentadoptnode-11)
    (documentadoptnode-12)
    (documentadoptnode-13)
    (documentadoptnode-14)
    (documentadoptnode-15)
    (documentadoptnode-21)
    (documentadoptnode-22)
    (documentadoptnode-23)
    (documentadoptnode-24)
    (documentadoptnode-25)
    (documentadoptnode-26)
    (documentadoptnode-27)
    (documentadoptnode-30)
    (documentadoptnode-31)
    (documentadoptnode-32)
    (documentadoptnode-33)
    (documentadoptnode-34)
    (documentadoptnode-35)
    (documentadoptnode-36)
    (documentgetstricterrorchecking-1)
    (documentgetstricterrorchecking-2)
    (documentnormalizedocument-1)
    (documentnormalizedocument-3)
    (documentnormalizedocument-4)
    (documentnormalizedocument-8)
    (documentnormalizedocument-11)
    (documentrenamenode-1)
    (documentrenamenode-2)
    (documentrenamenode-3)
    (documentrenamenode-4)
    (documentrenamenode-5)
    (documentrenamenode-7)
    (documentrenamenode-10)
    (documentrenamenode-11)
    (documentrenamenode-12)
    (documentrenamenode-13)
    (documentrenamenode-14)
    (documentrenamenode-15)
    (documentrenamenode-16)
    (documentrenamenode-17)
    (documentrenamenode-19)
    (documentrenamenode-20)
    (documentrenamenode-21)
    (documentrenamenode-22)
    (documentrenamenode-23)
    (documentrenamenode-24)
    (documentrenamenode-26)
    (documentrenamenode-27)
    (documentrenamenode-29)
    (domconfigcdatasections-1)
    (domconfigcomments-1)
    (domconfigentities-1)
    (domconfigerrorhandler-1)
    (domconfiginfoset-1)
    (domconfignamespacedeclarations-1)
    (domconfigparameternames-1)
    (domconfigsplitcdatasections-1)
    (domconfigurationcansetparameter-1)
    (domconfigurationgetparameter-1)
    (domconfigurationgetparameter-2)
    (elementsetidattribute-3)
    (elementsetidattribute-5)
    (elementsetidattribute-9)
    (elementsetidattribute-10)
    (elementsetidattributenode-2)
    (elementsetidattributenode-3)
    (entities-1)
    (entities-2)
    (entities-3)
    (entities-4)    
    (handleerror-1)
    (infoset-2)
    (infoset-4)
    (infoset-8)
    (infoset-9)
    (nodeappendchild-2)
    (nodecomparedocumentposition-3)
    (nodecomparedocumentposition-4)
    (nodecomparedocumentposition-5)
    (nodecomparedocumentposition-6)
    (nodecomparedocumentposition-7)
    (nodecomparedocumentposition-8)
    (nodecomparedocumentposition-9)
    (nodecomparedocumentposition-11)
    (nodecomparedocumentposition-12)
    (nodecomparedocumentposition-13)
    (nodecomparedocumentposition-14)
    (nodecomparedocumentposition-15)
    (nodecomparedocumentposition-16)
    (nodecomparedocumentposition-17)
    (nodecomparedocumentposition-18)
    (nodecomparedocumentposition-19)
    (nodecomparedocumentposition-20)
    (nodecomparedocumentposition-21)
    (nodecomparedocumentposition-25)
    (nodecomparedocumentposition-30)
    (nodecomparedocumentposition-31)
    (nodecomparedocumentposition-33)
    (nodecomparedocumentposition-34)
    (nodecomparedocumentposition-37)
    (nodecomparedocumentposition-38)
    (nodecomparedocumentposition-39)
    (nodecomparedocumentposition-40)
    (nodegetbaseuri-2)
    (nodegetbaseuri-4)
    (nodegetbaseuri-5)
    (nodegetbaseuri-7)
    (nodegetbaseuri-9)
    (nodegetbaseuri-10)
    (nodegetbaseuri-15)
    (nodegetbaseuri-17)
    (nodegetbaseuri-18)
    (nodegettextcontent-1)
    (nodegettextcontent-2)
    (nodegettextcontent-6)
    (nodegettextcontent-7)
    (nodegettextcontent-8)
    (nodegettextcontent-9)
    (nodegettextcontent-10)
    (nodegettextcontent-11)
    (nodegettextcontent-12)
    (nodegettextcontent-13)
    (nodegettextcontent-14)
    (nodegettextcontent-19)
    (nodegetuserdata-1)
    (nodegetuserdata-3)
    (nodegetuserdata-6)
    (nodegetuserdata-7)
    (nodeinsertbefore-1)
    (nodeinsertbefore-6)
    (nodeinsertbefore-7)
    (nodeinsertbefore-8)
    (nodeinsertbefore-9)
    (nodeinsertbefore-10)
    (nodeinsertbefore-11)
    (nodeinsertbefore-12)
    (nodeinsertbefore-13)
    (nodeinsertbefore-16)
    (nodeinsertbefore-17)
    (nodeinsertbefore-18)
    (nodeinsertbefore-19)
    (nodeinsertbefore-20)
    (nodeinsertbefore-21)
    (nodeinsertbefore-22)
    (nodeinsertbefore-23)
    (nodeinsertbefore-24)
    (nodeisequalnode-1)
    (nodeisequalnode-2)
    (nodeisequalnode-4)
    (nodeisequalnode-6)
    (nodeisequalnode-7)
    (nodeisequalnode-8)
    (nodeisequalnode-9)
    (nodeisequalnode-11)
    (nodeisequalnode-12)
    (nodeisequalnode-13)
    (nodeisequalnode-15)
    (nodeisequalnode-17)
    (nodeisequalnode-18)
    (nodeisequalnode-19)
    (nodeisequalnode-20)
    (nodeisequalnode-28)
    (nodeisequalnode-29)
    (nodeisequalnode-31)
    (nodeisequalnode-31)
    (nodelookupnamespaceuri-1)
    (nodelookupnamespaceuri-2)
    (nodelookupnamespaceuri-5)
    (noderemovechild-1)
    (noderemovechild-2)
    (noderemovechild-3)
    (noderemovechild-8)
    (noderemovechild-9)
    (noderemovechild-10)
    (noderemovechild-11)
    (noderemovechild-12)
    (noderemovechild-16)
    (noderemovechild-17)
    (noderemovechild-20)
    (noderemovechild-21)
    (noderemovechild-22)
    (noderemovechild-23)
    (noderemovechild-28)
    (nodereplacechild-1)
    (nodereplacechild-3)
    (nodereplacechild-4)
    (nodereplacechild-8)
    (nodereplacechild-12)
    (nodereplacechild-14)
    (nodereplacechild-15)
    (nodereplacechild-16)
    (nodereplacechild-17)
    (nodereplacechild-18)
    (nodereplacechild-19)
    (nodereplacechild-20)
    (nodereplacechild-22)
    (nodereplacechild-23)
    (nodereplacechild-26)
    (nodereplacechild-27)
    (nodereplacechild-29)
    (nodereplacechild-30)
    (nodereplacechild-32)
    (nodereplacechild-34)
    (nodereplacechild-35)
    (nodereplacechild-36)
    (nodereplacechild-37)
    (nodereplacechild-39)
;    (nodesettextcontent-1)
    (nodesettextcontent-2)
    (nodesettextcontent-6)
    (nodesettextcontent-7)
    (nodesettextcontent-8)
    (nodesettextcontent-10)
    (nodesettextcontent-11)
    (nodesettextcontent-12)
    (nodesetuserdata-1)
    (nodesetuserdata-3)
    (nodesetuserdata-4)
    (nodesetuserdata-5)
    (nodesetuserdata-8)
    (nodesetuserdata-9)
    (nodesetuserdata-10)
    (textreplacewholetext-1)
    (textreplacewholetext-2)
    (textreplacewholetext-3)
    (textreplacewholetext-4)
    (textreplacewholetext-5)
    (textreplacewholetext-6)
    (textreplacewholetext-7)
    (textwholetext-1)
    (textwholetext-2)
    (textwholetext-3)
    (userdatahandler-1)
    (userdatahandler-2)
    (userdatahandler-3)
    (userdatahandler-4)))

(define run-tests
  (lambda ()
    (let ((failures 0))
      (for-each (lambda (x)
		  (let* ((name (symbol->string (car x)))
			 (dots (make-string (- 38 (string-length name)) #\.))
			 (s (current-milliseconds)))
		    (display name)
		    (display dots)
		    (call/cc 
		     (lambda (cc) 
		       (display (with-exception-handler 
				 (lambda (exn)
				   (begin
				     (if (message-condition? exn)
					 (display (condition-message exn))
					 (display exn))
				     (newline)
				     (set! failures (+ failures 1)) 
				     (cc)))
				 (lambda () 
				   (begin 
				     (apply (eval-string name) '())))))))
		    (display " (")
		    (display (number->string (- (current-milliseconds) s)))
		    (display ")") 
		    (newline)))
		tests))))
