;; core.scm: main module exports and implementations for SDOM
;; Copyright (C) 2011 Julian Graham

;; SDOM free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; SDOM is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SDOM.  If not, see <http://www.gnu.org/licenses/>.

#!r6rs

(library (sdom core (0 5 1))
  (export sdom:sxml->sdom 
	  sdom:xml->sdom 

	  sdom:make-exception
	  sdom:exception?
	  sdom:exception-code

	  sdom:node?
	  sdom:node-type 
	  sdom:node-name
	  sdom:node-value
	  sdom:owner-document
	  sdom:base-uri
	  sdom:namespace-uri
	  sdom:prefix
	  sdom:local-name
	  sdom:text-content
	  sdom:set-text-content!
	  sdom:parent-node
	  sdom:next-sibling
	  sdom:previous-sibling
	  sdom:read-only
	  sdom:event-listeners

	  sdom:create-element
	  sdom:node-type-element 
	  sdom:element?
	  sdom:tag-name

	  sdom:create-attribute
	  sdom:node-type-attr 
	  sdom:attr?
	  sdom:owner-element
	  sdom:specified
	  sdom:name
	  sdom:value
	  sdom:set-value!

	  sdom:create-text-node
	  sdom:node-type-text 
	  sdom:text-node?
	  sdom:whole-text

	  sdom:create-cdata-section
	  sdom:node-type-cdata-section 
	  sdom:cdata-section?

	  sdom:create-entity-reference
	  sdom:node-type-entity-reference 
	  sdom:entity-reference?
	  
	  sdom:create-entity
	  sdom:node-type-entity 
	  sdom:entity?

	  sdom:create-processing-instruction
	  sdom:node-type-processing-instruction 
	  sdom:processing-instruction?
	  sdom:target
	  sdom:data

	  sdom:create-comment
	  sdom:node-type-comment 
	  sdom:comment?

	  sdom:node-type-document 
	  sdom:document?
	  sdom:document-element
	  sdom:document-uri
	  sdom:set-document-uri!
	  sdom:doctype
	  sdom:set-doctype!

	  sdom:node-type-document-type 
	  sdom:document-type?
	  sdom:entities

	  sdom:create-document-fragment
	  sdom:node-type-document-fragment 
	  sdom:document-fragment?

	  sdom:create-notation
	  sdom:node-type-notation 
	  sdom:notation?
	  
	  sdom:name
	  sdom:child-nodes
	  sdom:first-child
	  sdom:last-child

	  sdom:attributes

	  sdom:is-id

	  sdom:exception-code-index-size-err 
	  sdom:exception-code-domstring-size-err 
	  sdom:exception-code-hierarchy-request-err 
	  sdom:exception-code-wrong-document-err 
	  sdom:exception-code-invalid-character-err 
	  sdom:exception-code-no-data-allowed-err 
	  sdom:exception-code-no-modification-allowed-err 
	  sdom:exception-code-not-found-err 
	  sdom:exception-code-not-supported-err 
	  sdom:exception-code-inuse-attribute-err 
	  sdom:exception-code-invalid-state-err 
	  sdom:exception-code-syntax-err 
	  sdom:exception-code-invalid-modification-err 
	  sdom:exception-code-namespace-err 
	  sdom:exception-code-invalid-access-err 
	  sdom:exception-code-validation-err 
	  sdom:exception-code-type-mismatch-err 
 
	  sdom:error-severity-warning 
	  sdom:error-severity-error 
	  sdom:error-severity-fatal-error 
	   
	  sdom:document-position-disconnected 
	  sdom:document-position-contained-by 
	  sdom:document-position-contains 
	  sdom:document-position-following 
	  sdom:document-position-preceding 
	  sdom:document-position-implementation-specific 
 
	  sdom:user-data-event-node-cloned 
	  sdom:user-data-event-node-imported 
	  sdom:user-data-event-node-deleted 
	  sdom:user-data-event-node-renamed 
	  sdom:user-data-event-node-adopted 

	  sdom:dispatch-event
	  sdom:register-event-dispatcher!

	  sdom:register-feature! 
	  sdom:has-feature? 
 
	  sdom:config-parameter-names 
	  sdom:add-dom-config-parameter! 
	  sdom:get-dom-config-parameter 
	  sdom:set-dom-config-parameter! 
	  sdom:can-set-dom-config-parameter? 
	  sdom:signal-error 
 
	  sdom:insert-before! 
	  sdom:insert-after! 
	  sdom:remove-child! 
	  sdom:replace-child! 
	  sdom:replace-whole-text! 
	  sdom:append-child! 
	  sdom:clone-node 
	  sdom:normalize! 
	  sdom:normalize-document! 
	  sdom:adopt-node! 
	  sdom:import-node 
	  sdom:rename-node! 
	   
	  sdom:same-node? 
	  sdom:equal-node? 
	  sdom:has-child-nodes? 
	  sdom:supported? 
	   
	  sdom:default-namespace? 
	  sdom:lookup-prefix 
	  sdom:lookup-namespace-uri 
	   
	  sdom:set-user-data! 
	  sdom:get-user-data 
	  
	  sdom:create-element
	  sdom:create-document 
	  sdom:create-document-type 
	   
	  sdom:get-elements-by-tag-name 
	  sdom:get-element-by-id 
	  sdom:set-id-attribute! 
	  sdom:set-id-attribute-node! 
	  sdom:get-attribute-node 
	  sdom:set-attribute-node! 
	  sdom:remove-attribute-node! 
	  sdom:get-attribute 
	  sdom:set-attribute! 
	  sdom:remove-attribute! 
	   
	  sdom:compare-document-position

	  root
	  document-order)
  (import (rnrs)
          (only (srfi :1) every lset-union take-while list= list-index)
	  (srfi :13)
	  (srfi :39)
	  (only (srfi :69) hash-by-identity)
	  (sdom common)
	  (sdom parser))

  (define sdom:node-type-node 0)
  (define sdom:node-type-character-data 100)
  
  (define sdom:node-type-element 1)
  (define sdom:node-type-attr 2)
  (define sdom:node-type-text 3)
  (define sdom:node-type-cdata-section 4)
  (define sdom:node-type-entity-reference 5)
  (define sdom:node-type-entity 6)
  (define sdom:node-type-processing-instruction 7)
  (define sdom:node-type-comment 8)
  (define sdom:node-type-document 9)
  (define sdom:node-type-document-type 10)
  (define sdom:node-type-document-fragment 11)
  (define sdom:node-type-notation 12)
  
  (define sdom:exception-code-index-size-err 1)
  (define sdom:exception-code-domstring-size-err 2)
  (define sdom:exception-code-hierarchy-request-err 3)
  (define sdom:exception-code-wrong-document-err 4)
  (define sdom:exception-code-invalid-character-err 5)
  (define sdom:exception-code-no-data-allowed-err 6)
  (define sdom:exception-code-no-modification-allowed-err 7)
  (define sdom:exception-code-not-found-err 8)
  (define sdom:exception-code-not-supported-err 9)
  (define sdom:exception-code-inuse-attribute-err 10)
  (define sdom:exception-code-invalid-state-err 11)
  (define sdom:exception-code-syntax-err 12)
  (define sdom:exception-code-invalid-modification-err 13)
  (define sdom:exception-code-namespace-err 14)
  (define sdom:exception-code-invalid-access-err 15)
  (define sdom:exception-code-validation-err 16)
  (define sdom:exception-code-type-mismatch-err 17)
  
  (define sdom:error-severity-warning 1)
  (define sdom:error-severity-error 2)
  (define sdom:error-severity-fatal-error 3)
  
  (define sdom:document-position-disconnected 1)
  (define sdom:document-position-preceding 2)
  (define sdom:document-position-following 4)
  (define sdom:document-position-contains 8)
  (define sdom:document-position-contained-by 16)
  (define sdom:document-position-implementation-specific 32)
  
  (define sdom:user-data-event-node-cloned 1)
  (define sdom:user-data-event-node-imported 2)
  (define sdom:user-data-event-node-deleted 3)
  (define sdom:user-data-event-node-renamed 4)
  (define sdom:user-data-event-node-adopted 5)
  
  (define sdom:event-exception-code-unspecified-event-type-err	0)
  (define sdom:event-exception-code-dispatch-request-err 1)
  
  (define sdom:event-phase-capturing 1)
  (define sdom:event-phase-target 2)
  (define sdom:event-phase-bubbling 3)
  
  (define xmlns-ns-uri "http://www.w3.org/2000/xmlns")

  (define pending-synchronizations (make-parameter (make-eq-hashtable)))
  (define (register-synchronization! s)
    (let* ((t (dom-synchronization-target s))
	   (d (if (sdom:document? t) t (sdom:owner-document t))))
      (hashtable-set!
       (pending-synchronizations) d
       (cons s (hashtable-ref (pending-synchronizations) d (list))))))
  (define (process-synchronizations)
    (define text-content-roots '())
    (define (process-synchronizations-inner lst)
      (or (null? lst)
	  (let ((cl (car lst)))
	    (case (dom-synchronization-type cl)
	      ((text-content)
	       (let ((target (root (dom-synchronization-target cl))))
		 (if (not (memq target text-content-roots))
		     (begin
		       (update-text-content! target)
		       (set! text-content-roots 
			     (cons target text-content-roots))))))
		     
	      ((whole-text) 
	       (update-whole-text! (dom-synchronization-target cl))))
	    (process-synchronizations-inner (cdr lst)))))
    (define (process-document-synchronizations docs)
      (or (null? docs)
	  (begin
	    (process-synchronizations-inner 
	     (hashtable-ref (pending-synchronizations) (car docs) '()))
	    (process-document-synchronizations (cdr docs)))))
    (process-document-synchronizations 
     (vector->list (hashtable-keys (pending-synchronizations))))
    (pending-synchronizations (make-eq-hashtable)))
  
  (define-record-type dom-synchronization
    (fields type target args)
    (protocol (lambda (n) (lambda (type target . args) (n type target args)))))
  
  (define-syntax with-synchronizations
    (lambda (stx)
      (syntax-case stx ()
	((_ expr ...)
	 #'(dynamic-wind
	       (lambda () (pending-synchronizations (make-eq-hashtable)))
	       (lambda () expr ...)
	       process-synchronizations)))))

  (define (default-dom-error-handler severity msg type excep data loc)
    (display (string-append "SDOM: "
			    (cond ((eqv? severity sdom:error-severity-warning)
				   "warning: ")
				  ((eqv? severity sdom:error-severity-error)
				   "error: ")
				  ((eqv? severity 
					 sdom:error-severity-fatal-error)
				   "fatal error: "))
			    msg))
    (newline)
    #t)
  
  (define (sdom:signal-error doc severity msg type excep data loc)
    (let* ((handler (sdom:get-dom-config-parameter doc "error-handler")))
      (apply handler `(,severity ,msg ,type ,excep ,data ,loc))))
  
;;---------------------------------------------------------------------------;;
;;                                                                           ;;
;; DOM feature management functions                                          ;;
;;                                                                           ;;
;;---------------------------------------------------------------------------;;

  (define sdom-feature-registry '(("Core" . "3.0") ("XML" . "1.0")))
  (define (sdom:register-feature! feature version)
    (if (not (sdom:has-feature? feature version))
	(set! sdom-feature-registry
	      (append sdom-feature-registry `((,feature . ,version))))))
  (define (sdom:has-feature? feature version) 
    (let ((f (find (lambda (x) (equal? x `(,feature . ,version))) 
		   sdom-feature-registry))) 
      (and f #t)))

;;---------------------------------------------------------------------------;;
;;                                                                           ;;
;; DOM configuration functions                                               ;;
;;                                                                           ;;
;;---------------------------------------------------------------------------;;

  ;; The values after the option name are required supported values; the first
  ;; is the default.

  (define sdom:config-parameter-names
    `("canonical-form" "cdata-sections" "check-character-normalization"
      "comments" "datatype-normalization" "element-content-whitespace" 
      "entities" "error-handler" "infoset" "namespaces" "namespace-declarations"
      "normalize-characters" "split-cdata-sections" "strict-error-checking"
      "validate" "validate-if-schema" "well-formed" "sdom:prefer-orig-prefix"
      "sdom:resolve-new-prefixes"))

  (define-record-type sdom-config-parameter
    (fields name default predicate side-effect-proc))

  (define sdom:config-parameters (make-hashtable string-ci-hash string-ci=?))
  (for-each 
   (lambda (x) (hashtable-set! sdom:config-parameters 
			       (sdom-config-parameter-name x) 
			       x))    
   (list (make-sdom-config-parameter 
	  "canonical-form" #f boolean?
	  (lambda (d x . old-x)
	    (if x (begin (sdom:set-dom-config-parameter! d "entities" #f)
			 (sdom:set-dom-config-parameter! 
			  d "normalize-characters" #f)
			 (sdom:set-dom-config-parameter! d "cdata-sections" #f)
			 (sdom:set-dom-config-parameter! d "namespaces" #t)
			 (sdom:set-dom-config-parameter! 
			  d "namespace-declarations" #t)
			 (sdom:set-dom-config-parameter! d "well-formed" #t)
			 (sdom:set-dom-config-parameter!
			  d "element-content-whitespace" #t)))))
	 (make-sdom-config-parameter "cdata-sections" #t boolean? #f)
	 (make-sdom-config-parameter "check-character-normalization" #f not #f)
	 (make-sdom-config-parameter "comments" #t boolean? #f)
	 (make-sdom-config-parameter "datatype-normalization" #f not #f)
	 (make-sdom-config-parameter "element-content-whitespace" 
				     #t (lambda (x) x) #f)
	 (make-sdom-config-parameter "entities" #t boolean? #f)
	 (make-sdom-config-parameter 
	  "error-handler" default-dom-error-handler procedure? #f)
	 (make-sdom-config-parameter 
	  "infoset" #f boolean? 
	  (lambda (d x . old-x) 
	    (if x (begin (sdom:set-dom-config-parameter! 
			  d "validate-if-schema" #f)
			 (sdom:set-dom-config-parameter! d "entities" #f)
			 (sdom:set-dom-config-parameter!
			  d "datatype-normalization" #f)
			 (sdom:set-dom-config-parameter! d "cdata-sections" #f)
			 (sdom:set-dom-config-parameter! 
			  d "namespace-declarations" #t)
			 (sdom:set-dom-config-parameter! d "well-formed" #t)
			 (sdom:set-dom-config-parameter! 
			  d "element-content-whitespace" #t)
			 (sdom:set-dom-config-parameter! d "comments" #t)
			 (sdom:set-dom-config-parameter! d "namespaces" #t))
		(if (and (not (null? old-x)) (car old-x))
		    (hashtable-set! (sdom:dom-config d) "infoset" #t)))))
	 (make-sdom-config-parameter "namespaces" #t (lambda (x) x) #f)
	 (make-sdom-config-parameter "namespace-declarations" #t boolean? #f)
	 (make-sdom-config-parameter "normalize-characters" #f not #f)
	 (make-sdom-config-parameter "split-cdata-sections" #t boolean? #f)
	 (make-sdom-config-parameter "strict-error-checking" #t boolean? #f)
	 (make-sdom-config-parameter "validate" #f not #f)
	 (make-sdom-config-parameter "validate-if-schema" #f not #f)
	 (make-sdom-config-parameter "well-formed" #t (lambda (x) x) #f)
	 (make-sdom-config-parameter "sdom:prefer-orig-prefix" #f boolean? #f)
	 (make-sdom-config-parameter 
	  "sdom:resolve-new-prefixes" #t boolean? #f)))
  
  (define (get-sdom-config-default str)
    (let ((param (hashtable-ref sdom:config-parameters str #f)))
      (if param 
	  (sdom-config-parameter-default param)
	  (raise (sdom:make-exception sdom:exception-code-not-found-err #f)))))

  (define (sdom:get-dom-config-parameter doc str) 
    (if (not (sdom:document? doc))
	(raise (sdom:make-exception sdom:exception-code-type-mismatch-err #f)))
    (hashtable-ref (sdom:dom-config doc) str (get-sdom-config-default str)))
  
  (define (internal-get-dom-config-entry str)
    (hashtable-ref sdom:config-parameters str #f))
  
  (define (internal-check-dom-config-parameter doc str val)
    (if (not (eqv? (sdom:node-type doc) sdom:node-type-document))
	(raise (sdom:make-exception sdom:exception-code-type-mismatch-err #f)))
    (let ((foo (internal-get-dom-config-entry str)))
      (if (not foo) 
	  (raise (sdom:make-exception sdom:exception-code-not-found-err #f)))
      (if (not (apply (sdom-config-parameter-predicate foo) val '()))
	  (raise (sdom:make-exception sdom:exception-code-not-supported-err 
				      #f)))))
  
  (define (sdom:add-dom-config-parameter! str . vals) 
    (hashtable-set! sdom:config-parameters 
		    str (apply make-sdom-config-parameter (cons str vals))))
  
  (define (sdom:set-dom-config-parameter! doc str val)
    (internal-check-dom-config-parameter doc str val)
    (let* ((dc (sdom:dom-config doc))
	   (old-val (hashtable-ref dc str '())))
      (hashtable-set! dc str val)
      (let ((entry (internal-get-dom-config-entry str)))
	(if (sdom-config-parameter-side-effect-proc entry)
	    (apply (sdom-config-parameter-side-effect-proc entry) 
		   (if (null? old-val)
		       (list doc val) 
		       (list doc val old-val)))))))

  (define (sdom:can-set-dom-config-parameter? doc str val)
    (guard (ex ((sdom:exception? ex) #f))
	   (internal-check-dom-config-parameter doc str val)))
  
  (define sdom:event-dispatchers (list))
  (define (sdom:register-event-dispatcher! proc)
    (set! sdom:event-dispatchers (cons proc sdom:event-dispatchers)))
  
  (define (sdom:dispatch-event . args)
    (for-each (lambda (dispatcher) (apply dispatcher args))
	      sdom:event-dispatchers))
  
  (define (prefix-and-local-name str)
    (let ((colon-index (string-index-right str #\:)))
      (if colon-index 
	  (values (substring str 0 colon-index)
		  (substring str (+ colon-index 1)))
	  (values #f str))))

  (define (prefix str)
    (let-values (((prefix local-name) (prefix-and-local-name str)))
      prefix))

  (define (local-name str)
    (let-values (((prefix local-name) (prefix-and-local-name str)))
      local-name))

  (define (qname? q)
    (let ((ci (string-index q #\:)))
      (if ci
	  (let ((prefix (substring q 0 ci))
		(local-part (substring q (+ ci 1))))
	    (and (ncname? prefix) (ncname? local-part)))
	  (ncname? q))))

  (define (valid-namespace-combo? doc qname uri)
    (let ((resolve (sdom:get-dom-config-parameter 
		    doc "sdom:resolve-new-prefixes")))
      (let-values 
	  (((prefix local-name) (prefix-and-local-name qname)))
	(cond ((not (qname? qname)) #f)
	      ((and prefix (not uri)) #f)
	      ((and (or (equal? qname "xml")
			(equal? prefix "xml"))
		    (not (equal? uri xml-ns-uri))) #f)
	      ((and (or (equal? qname "xmlns")
			(equal? prefix "xmlns"))
		    (not (equal? uri xmlns-ns-uri))) #f)
	      ((and (equal? uri xmlns-ns-uri)
		    (not (equal? qname "xmlns"))
		    (not (equal? prefix "xmlns"))) #f)
	      (else #t)))))
    
  (define-condition-type &sdom:exception &condition 
    sdom:make-exception sdom:exception?
    (exception-code sdom:exception-code)
    (error-severity sdom:exception-severity))

  (define-record-type (sdom:node sdom:make-node sdom:node?)
    (fields (mutable read-only sdom:read-only sdom:set-read-only!)
	    (immutable node-type sdom:node-type)
            (mutable node-name sdom:node-name sdom:set-node-name!)
	    (mutable node-value sdom:node-value sdom:set-node-value!)
	    (mutable parent-node sdom:parent-node sdom:set-parent-node!)
	    (mutable child-nodes sdom:child-nodes sdom:set-child-nodes!)
	    (mutable first-child sdom:first-child sdom:set-first-child!)
	    (mutable last-child sdom:last-child sdom:set-last-child!)
	    (mutable previous-sibling 
		     sdom:previous-sibling 
		     sdom:set-previous-sibling!)
	    (mutable next-sibling sdom:next-sibling sdom:set-next-sibling!)
	    (mutable attributes sdom:attributes sdom:set-attributes!)
	    (mutable owner-document 
		     sdom:owner-document 
		     sdom:set-owner-document!)
	    (mutable namespace-uri sdom:namespace-uri sdom:set-namespace-uri!)
	    (mutable prefix sdom:prefix sdom:set-prefix!)
	    (mutable local-name sdom:local-name sdom:set-local-name!)
	    (mutable text-content 
		     sdom:text-content 
		     sdom:set-text-content-internal!)
	    (immutable user-data sdom:user-data)
	    (immutable event-listeners sdom:event-listeners))
    (protocol 
     (lambda (n) 
       (lambda (read-only node-type node-name node-value namespace-uri 
		text-content)
	 (let-values (((prefix local-name) 
		       (if node-name 
			   (prefix-and-local-name node-name)
			   (values #f #f))))
	   (n read-only node-type node-name node-value #f '() #f #f #f #f
	      '() #f namespace-uri prefix local-name text-content
	      (make-hashtable string-hash equal?)
	      (make-hashtable string-hash equal?)))))))
  (define-record-type (sdom:character-data
		       sdom:make-character-data 
		       sdom:character-data?)
    (parent sdom:node)
    (fields (mutable data
		     sdom:character-data-data 
		     sdom:set-character-data-data!) 
	    (mutable length sdom:length sdom:set-length!))
    (protocol (lambda (n)
		(lambda (type data length node-name)
		  (let ((p (n #f type node-name data #f data)))
		    (p data length))))))
  (define-record-type (sdom:text-node sdom:make-text-node sdom:text-node?)
    (parent sdom:character-data)
    (fields is-element-content-whitespace 
	    (mutable whole-text sdom:whole-text sdom:set-whole-text!))
    (protocol (lambda (n)
		(lambda (type text . name)
		  (let ((p (n type text (string-length text) 
			      (if (null? name) "#text" (car name)))))
		    (p #f text))))))
  (define-record-type (sdom:cdata-section
		       sdom:make-cdata-section
		       sdom:cdata-section?)
    (parent sdom:text-node)
    (protocol (lambda (n)
		(lambda (text)
		  (let ((p (n sdom:node-type-cdata-section text 
			      "#cdata-section")))
		    (p))))))
  (define-record-type (sdom:comment sdom:make-comment sdom:comment?)
    (parent sdom:character-data)
    (protocol (lambda (n)
		(lambda (text)
		  (let ((p (n sdom:node-type-comment text 
			      (string-length text) "#comment")))
		    (p))))))
  (define-record-type (sdom:notation sdom:make-notation sdom:notation?)
    (parent sdom:node) (fields public-id system-id))
  (define-record-type (sdom:entity sdom:make-entity sdom:entity?)
    (parent sdom:node)
    (fields public-id
	    system-id
	    notation-name
	    input-encoding
	    xml-encoding
	    xml-version)
    (protocol (lambda (n)
		(lambda (name)
		  (let ((p (n #t sdom:node-type-entity name #f #f "")))
		    (p #f #f #f #f #f #f))))))
  (define-record-type (sdom:entity-reference 
		       sdom:make-entity-reference 
		       sdom:entity-reference?)
    (parent sdom:node)
    (protocol (lambda (n)
		(lambda (name)
		  (let ((p (n #t sdom:node-type-entity-reference name #f #f 
			      "")))
		    (p))))))
  (define-record-type (sdom:processing-instruction
		       sdom:make-processing-instruction
		       sdom:processing-instruction?)
    (parent sdom:node) 
    (fields (immutable target sdom:target) 
	    (mutable data 
		     sdom:processing-instruction-data
		     sdom:set-processing-instruction-data!))
    (protocol (lambda (n)
		(lambda (target data)
		  (let ((p (n #f sdom:node-type-processing-instruction target 
			      data #f data)))
		    (p target data))))))
  (define-record-type (sdom:attr sdom:make-attr sdom:attr?)
    (parent sdom:node)
    (fields (mutable name sdom:name sdom:set-name!)
	    (mutable specified sdom:specified sdom:set-specified!)
	    (mutable value sdom:value sdom:set-value-internal!) 
	    (mutable owner-element sdom:owner-element sdom:set-owner-element!)
	    schema-type-info 
	    (mutable is-id sdom:is-id sdom:set-is-id!))
    (protocol (lambda (n)
		(lambda (name value schema-type-info is-id ns)
		  (let ((p (n #f sdom:node-type-attr name value ns "")))
		    (p name (and value #t) value #f schema-type-info is-id))))))
  (define-record-type (sdom:element sdom:make-element sdom:element?)
    (parent sdom:node)
    (fields (mutable tag-name sdom:tag-name sdom:set-tag-name!) 
	    schema-type-info)
    (protocol (lambda (n) 
		(lambda (tag-name schema-type-info ns)
		  (let ((p (n #f sdom:node-type-element tag-name #f ns "")))
		    (p tag-name schema-type-info))))))
  (define-record-type (sdom:document-type 
		       sdom:make-document-type 
		       sdom:document-type?)
    (fields name 
	    (mutable entities sdom:entities sdom:set-entities!)
	    notations 
	    public-id 
	    system-id 
	    internal-subset)
    (protocol (lambda (n)
		(lambda (qname public-id system-id internal-subset)
		  (let ((p (n #t sdom:node-type-document-type qname #f #f #f)))
		    (p qname '() #f public-id system-id internal-subset)))))
    (parent sdom:node))
  (define-record-type (sdom:document-fragment 
		       sdom:make-document-fragment 
		       sdom:document-fragment?)
    (parent sdom:node)
    (protocol (lambda (n)
		(lambda ()
		  (let ((p (n #f sdom:node-type-document-fragment 
			      "#document-fragment" #f #f "")))
		    (p))))))
  (define-record-type (sdom:document sdom:make-document sdom:document?)
    (parent sdom:node)
    (fields (mutable doctype sdom:doctype sdom:set-doctype!)
	    implementation 
	    (mutable document-element 
		     sdom:document-element 
		     sdom:set-document-element!)
	    input-encoding
	    xml-encoding 
	    xml-standalone
	    xml-version
	    (mutable document-uri sdom:document-uri sdom:set-document-uri!)
	    (immutable dom-config sdom:dom-config))
    (protocol (lambda (n)
		(lambda ()
		  (let ((p (n #f sdom:node-type-document "#document" #f #f #f)))
		    (p #f #f #f #f #f #f #f #f 
		       (make-hashtable string-ci-hash string-ci=?)))))))

  (define (sdom:base-uri node)
    (cond ((sdom:document? node) (sdom:document-uri node))
	  ((sdom:element? node) 
	   (or (sdom:get-attribute 
		node "base" "http://www.w3.org/XML/1998/namespace")
	       (sdom:base-uri (sdom:parent-node node))))
	  ((sdom:processing-instruction? node)
	   (if (sdom:parent-node node)
	       (sdom:base-uri (sdom:parent-node node))
	       (sdom:base-uri (sdom:owner-document node))))
	  (else #f)))

  (define (sdom:data node)
    (cond ((sdom:character-data? node) (sdom:character-data-data node))
	  ((sdom:processing-instruction? node)
	   (sdom:processing-instruction-data node))
	  (else (raise (sdom:make-exception
			sdom:exception-code-type-mismatch-err #f)))))

  (define (sdom:set-text-content! node text)
    (cond ((sdom:read-only node) 
	   (raise (sdom:make-exception 
		   sdom:exception-code-no-modification-allowed-err #f)))
	  ((or (sdom:element? node)
	       (sdom:attr? node)
	       (sdom:entity-reference? node)
	       (sdom:document-fragment? node))
	   (for-each (lambda (c) (sdom:remove-child! node c)) 
		     (sdom:child-nodes node))
	   (if (and text (> (string-length text) 0))
	       (sdom:append-child!
		node (sdom:create-text-node (sdom:owner-document node) text))))

	  ((sdom:text-node? node))
	  ((sdom:cdata-section? node))
	  ((sdom:processing-instruction? node))))

  (define (ancestors node)
    (cond ((not node) '())
	  ((or (sdom:document? node) (sdom:document-fragment? node))
	   (list node))
	  ((sdom:attr? node) 
	   (cons node (ancestors (sdom:owner-element node))))
	  (else (cons node (ancestors (sdom:parent-node node))))))

  (define (root n) (let ((p (sdom:parent-node n))) (if p (root p) n)))

  (define allowed-child-types
    (lambda (node-type)
      (cond ((eqv? node-type sdom:node-type-attr)
	     `(,sdom:node-type-text
	       ,sdom:node-type-entity-reference))
	    ((or (eqv? node-type sdom:node-type-cdata-section)
		 (eqv? node-type sdom:node-type-comment)
		 (eqv? node-type sdom:node-type-document-type)
		 (eqv? node-type sdom:node-type-notation)
		 (eqv? node-type sdom:node-type-processing-instruction)) '())
	    ((eqv? node-type sdom:node-type-document)
	     `(,sdom:node-type-element
	       ,sdom:node-type-processing-instruction
	       ,sdom:node-type-comment
	       ,sdom:node-type-document-type))
	    ((or (eqv? node-type sdom:node-type-document-fragment)
		 (eqv? node-type sdom:node-type-element)
		 (eqv? node-type sdom:node-type-entity)
		 (eqv? node-type sdom:node-type-entity-reference))
	     `(,sdom:node-type-element
	       ,sdom:node-type-processing-instruction
	       ,sdom:node-type-comment
	       ,sdom:node-type-text
	       ,sdom:node-type-cdata-section
	       ,sdom:node-type-entity-reference))
	    (else (raise (sdom:make-exception 
			  sdom:exception-code-type-mismatch-err #f))))))

  (define (type-allowed-as-child parent-type child-type)
    (not (eq? (memv child-type (allowed-child-types parent-type)) #f)))

  (define (list-head lst k)
    (if (or (null? lst) (<= k 0))
	'()
	(cons (car lst) (list-head (cdr lst) (- k 1)))))

  (define (last-pair lst)
    (define (last-pair-inner c lst)
      (if (pair? lst) (last-pair-inner (car lst) (cdr lst)) (cons c lst)))
    (last-pair-inner (car lst) (cdr lst)))  

  (define (insert-child! node child pos)
    (let ((old-parent (sdom:parent-node child)))
      (if old-parent (remove-child! old-parent (child-index old-parent child)))
      (let* ((children (sdom:child-nodes node))
	     (num-children (length children))
	     (pos (min num-children pos))
	     (prev-sibling (and (> num-children 0)
				(> pos 0)
				(list-ref children (- pos 1))))
	     (next-sibling (and (< pos num-children)
				(list-ref children pos)))
	     (head (list-head children pos))
	     (tail (list-tail children pos)))
	(sdom:set-parent-node! child node)
      
	(and prev-sibling 
	     (sdom:set-next-sibling! prev-sibling child)
	     (sdom:set-previous-sibling! child prev-sibling))
	(and next-sibling
	     (sdom:set-previous-sibling! next-sibling child)
	     (sdom:set-next-sibling! child next-sibling))
	
	(sdom:set-child-nodes! node (append head (list child) tail))
	(sdom:set-first-child! node (if (null? head) child (car head)))
	(sdom:set-last-child!
	 node (if (null? tail) child (car (last-pair tail))))

	(if (and (sdom:attr? node) (sdom:text-node? child))
	    (begin (sdom:set-node-value! node (sdom:data child))
		   (sdom:set-value-internal! node (sdom:data child))))
	    
	(if (not (or (sdom:comment? child)
		     (sdom:processing-instruction? child)))
	    (register-synchronization!
	     (make-dom-synchronization 'text-content node)))
	(if (sdom:text-node? child)
	    (register-synchronization! 
	     (make-dom-synchronization 'whole-text child))))))
  
  (define (sdom:insert-before! node new-node . ref-node)    
    (let ((s (if (or (null? ref-node) (not (car ref-node)))
		 1
		 (let* ((cr (car ref-node))
			(pos (list-index (lambda (x) (sdom:same-node? x cr))
					 (sdom:child-nodes node))))
		   (or pos (raise (sdom:make-exception 
				   sdom:exception-code-not-found-err #f)))))))
      (with-synchronizations
       (if (sdom:document-fragment? new-node)
	   (for-each (lambda (x)
		       (check-insertion-error node x)
		       (insert-child! node x s)
		       (set! s (+ s 1)))
		     (sdom:child-nodes new-node))
	   (begin
	     (check-insertion-error node new-node)
	     (insert-child! node new-node s))))
      new-node))

  (define (sdom:insert-after! node new-node . ref-node)
    (let ((s (if (or (null? ref-node) (not (car ref-node)))
		 1
		 (let* ((cr (car ref-node))
			(pos (list-index (lambda (x) (sdom:same-node? x cr))
					 (sdom:child-nodes node))))
		   (if pos 
		       (+ pos 1)
		       (raise (sdom:make-exception 
			       sdom:exception-code-not-found-err #f)))))))
      (check-insertion-error node new-node)
      (with-synchronizations (insert-child! node new-node s))))

  (define (remove-child! node pos)
    (let* ((children (sdom:child-nodes node))
	   (child (list-ref children pos))
	   (children (append (list-head children pos)
			     (list-tail children (+ pos 1)))))
      (make-children! node children)
      (orphan-node! child)

      (cond ((sdom:document? node) 
	     (cond ((sdom:element? child) (sdom:set-document-element! node #f))
		   ((sdom:document-type? child) (sdom:set-doctype! node #f))))
	    ((sdom:attr? node) (sdom:set-specified! node #f)))

      (if (not (or (sdom:comment? child)
		   (sdom:processing-instruction? child)))
	  (register-synchronization!
	   (make-dom-synchronization 'text-content node)))
      (if (sdom:text-node? child)
	  (register-synchronization! 
	   (make-dom-synchronization 'whole-text node)))
      child))

  (define (sdom:remove-child! node oc)
    (let ((parent (sdom:parent-node oc)))
      (if (and parent (sdom:same-node? node parent))
	  (with-synchronizations
	   (remove-child! node (list-index (lambda (x) (sdom:same-node? oc x))
					   (sdom:child-nodes node))))
	  (raise (sdom:make-exception sdom:exception-code-not-found-err #f)))))

  (define (sdom:replace-child! node new-child old-child)
    (if (sdom:same-node? node new-child)
	(raise (sdom:make-exception sdom:exception-code-hierarchy-request-err 
				    #f)))
    (let* ((parentold (sdom:parent-node old-child))
	   (parentnew (sdom:parent-node new-child)))
      (if (or (sdom:read-only node)
	      (if (not parentold) #f (sdom:read-only parentold)))
	  (raise (sdom:make-exception 
		  sdom:exception-code-no-modification-allowed-err #f)))
      (if (and parentold (sdom:same-node? node parentold))
	  (let ((pos (list-index (lambda (x) (sdom:same-node? x old-child))
				 (sdom:child-nodes node))))
	    (check-insertion-error node new-child)
	    (with-synchronizations
	     (remove-child! node pos)
	     (insert-child! node new-child pos))
	    old-child)
	  (raise (sdom:make-exception sdom:exception-code-not-found-err #f)))))

  (define (check-insertion-error x y)
    (let ((od1 (sdom:owner-document x))
	  (od2 (sdom:owner-document y)))
      (if (not (or (and (not (sdom:document? x)) (sdom:same-node? od1 od2))
		   (and (sdom:document? x)
			(or (sdom:document-type? y) (sdom:same-node? x od2)))))
	  (raise (sdom:make-exception sdom:exception-code-wrong-document-err 
				      #f)))
      (if (or (not (type-allowed-as-child (sdom:node-type x) 
					  (sdom:node-type y)))
	      (sdom:same-node? x y)
	      (memp (lambda (z) (sdom:same-node? z y))
		    (ancestors x))
	      (and (sdom:document? x) 
		   (sdom:element? y)
		   (sdom:document-element x)))
	  (raise (sdom:make-exception sdom:exception-code-hierarchy-request-err
				      #f)))))
 
  (define (sdom:append-child! node new-child)
    (check-insertion-error node new-child)
    (with-synchronizations
     (insert-child! node new-child (length (sdom:child-nodes node)))))

  (define (handle-user-data-event node op src dst)
    (let ((ht (sdom:user-data node)))
      (vector-for-each (lambda (key)
			 (let ((val (hashtable-ref ht key #f)))
			   (if (and val (procedure? (cdr val)))
			       (apply (cdr val) op key (car val) src dst '()))))
		       (hashtable-keys ht))))

  (define (sdom:set-user-data! node key data . handler)
    (let* ((ht (sdom:user-data node))
	   (oldval (hashtable-ref ht key #f)))
      (hashtable-set! ht key (if (and (not (null? handler))
				      (procedure? (car handler)))
				 (cons data (car handler))
				 (cons data #f)))
      (and oldval (car oldval))))

  (define (sdom:get-user-data node key)
    (let ((user-data (hashtable-ref (sdom:user-data node) key #f)))
      (and user-data (car user-data))))

  (define (sdom:equal-node? n1 n2)
    (let* ((nsu1 (sdom:namespace-uri n1))
	   (nsu2 (sdom:namespace-uri n2))
	   (nm1 (sdom:node-name n1))
	   (nm2 (sdom:node-name n2)))
      (and (eq? (sdom:node-type n1) (sdom:node-type n2))
	   (equal? nm1 nm2)
	   (if (or (and nsu1 (not nsu2)) (and (not nsu1) nsu2))
	       #f
	       (if nsu1 (equal? (local-name nm1) (local-name nm2)) #t))
	   (equal? nsu1 nsu2)
	   (equal? (prefix nm1) (prefix nm2))
	   (equal? (sdom:node-value n1) (sdom:node-value n2))
	   (list= sdom:equal-node? (sdom:attributes n1) (sdom:attributes n2))
	   (list= sdom:equal-node? 
		  (sdom:child-nodes n1) 
		  (sdom:child-nodes n2)))))
  
  (define (sdom:has-child-nodes? node) (not (null? (sdom:child-nodes node))))

  (define sdom:same-node? eq?)  
  (define (sdom:supported? node feature version) #f)
  (define (child-index parent child)
    (list-index (lambda (x) (sdom:same-node? x child))
		(sdom:child-nodes parent)))

  (define (get-adjacent-text-nodes node)
    (define (filter-nodes lst)
      (filter sdom:text-node? 
	      (take-while (lambda (n) (not (stop-node? n))) lst)))

    (define (stop-node? node)
      (or (sdom:attr? node)
	  (sdom:element? node)
	  (sdom:comment? node)
	  (sdom:processing-instruction? node)))

    (define (find-root node)
      (let ((parent (if (sdom:attr? node) 
			(sdom:owner-element node) 
			(sdom:parent-node node))))
      (if (and parent (not (stop-node? node))) (find-root parent) node)))

    (let* ((root (find-root node))
	   (docorder (document-order root))
	   (rdocorder (reverse docorder)))
      (lset-union sdom:same-node? 
		  (filter-nodes docorder) 
		  (reverse (filter-nodes rdocorder)))))

  (define (synchronize-whole-text! nodes)
    (let ((whole-text 
	   (apply string-append (map sdom:character-data-data nodes))))
      (for-each (lambda (n) (sdom:set-whole-text! n whole-text)) nodes)))

  (define (update-whole-text! node)
    (synchronize-whole-text! (get-adjacent-text-nodes node)))

  (define (sdom:replace-whole-text! node txt)
    (if (not (sdom:text-node? node))
	(raise (sdom:make-exception sdom:exception-code-type-mismatch-err #f)))
    (let ((span (get-adjacent-text-nodes node)))
      (for-each (lambda (x)
		  (let ((parent (sdom:parent-node x)))
		    (if (and parent 
			     (or (equal? txt "") 
				 (not (sdom:same-node? x node))))
			(sdom:remove-child! (sdom:parent-node x) x))))
		span)
      (if (equal? txt "") 
	  #f 
	  (begin (sdom:set-character-data-data! node txt)
		 (sdom:set-node-value! node txt)
		 (register-synchronization!
		  (make-dom-synchronization 'text-content node))
		 (sdom:set-whole-text! node txt)
		 node))))

  (define (internal-lookup-scoped-namespace node)
    (letrec ((aef (lambda (x) (find sdom:element? (ancestors x))))
	     (ns-prop (sdom:namespace-uri node))
	     (f (lambda (n) #f)))
      (or ns-prop
	  (cond ((sdom:element? node) (f node))
		((sdom:attr? node) (f (sdom:owner-element node)))
		(else #f)))))

  (define (internal-ns-lookup node str sym)
    (let ((type (sdom:node-type node))
	  (aef (lambda (x) 
		 (find (lambda (y) (sdom:element? y))
		       (ancestors x)))))
      (cond ((eqv? type sdom:node-type-element)
	     (let* ((ns (sdom:namespace-uri node))
		    (prefix (sdom:prefix node)))

	       ;; We're doing a prefix lookup.
	       
	       (cond ((eq? sym 'prefix)
		      (let ((f (lambda (elt uri orig)
				 (let ((eltns (sdom:namespace-uri elt))
				       (eltprefix (sdom:prefix elt)))
				   (if (and eltns
					    (equal? eltns uri)
					    eltprefix
					    (let ((r (internal-ns-lookup 
						      orig eltprefix 'ns)))
					      (and r (equal? r uri))))
				       eltprefix
				       (let ((ae (aef elt)))
					 (if ae 
					     (internal-ns-lookup 
					      ae eltprefix sym)
					     #f)))))))
			(f node str node)))
		     
		     ;; We're doing a namespaceURI lookup.
		     
		     ((eq? sym 'ns)
		      (if (and ns (equal? str prefix)) 
			  ns
			  (let ((decl (sdom:lookup-namespace-uri node str)))
			    (if decl
				(cadr decl)
				(let ((ae (aef node)))
				  (if ae 
				      (internal-ns-lookup ae str sym) 
				      #f))))))
		     
		     ;; We're doing a default namespace lookup.
		     
		     ((eq? sym 'default)
		      (if prefix
			  (let ((ae (aef node)))
			    (if ae (internal-ns-lookup ae str sym) #f))
			  (equal? str ns))))))
	    
	    ((eqv? type sdom:node-type-document)	     
	     (let ((de (sdom:document-element node)))
	       (if de (internal-ns-lookup de str sym) #f)))
	    ((or (eqv? type sdom:node-type-entity)
		 (eqv? type sdom:node-type-notation)
		 (eqv? type sdom:node-type-document-type)
		 (eqv? type sdom:node-type-document-fragment)) #f)
	    ((eqv? type sdom:node-type-attr)
	     (let ((p (sdom:parent-node node)))
	       (if p (internal-ns-lookup p str sym) #f)))
	    (else (let ((ae (aef node)))
		    (if ae (internal-ns-lookup ae str sym) #f))))))

  (define (sdom:default-namespace? node namespace-uri)
    (let ((ns (internal-ns-lookup node namespace-uri 'default)))
      (not (eq? ns #f))))

  ;; I don't think we support DOM level 2 namespace-declarations as attributes

  (define (match-ancestor pred node)
    (define (match-ancestor-inner node)
      (and node (or (pred node) 
		    (match-ancestor-inner (sdom:parent-node node)))))
    (match-ancestor-inner (sdom:parent-node node)))

  (define (sdom:lookup-namespace-uri node prefix)
    (define (lookup-namespace-uri node)
      (cond ((not node) #f)
	    ((sdom:element? node)
	     (if (and (sdom:namespace-uri node) 
		      (equal? (sdom:prefix node) prefix))
		 (sdom:namespace-uri node)
		 (let loop ((as (sdom:attributes node)))
		   (if (null? as) 
		       (lookup-namespace-uri 
			(match-ancestor sdom:element? node))
		       (let ((a (car as)))
			 (cond ((and (equal? (sdom:prefix a) "xmlns")
				     (equal? (sdom:local-name a) prefix))
				(sdom:value a))
			       ((and (equal? (sdom:local-name a) "xmlns")
				     (not (sdom:prefix a)))
				(sdom:value a))
			       (else (loop (cdr as)))))))))
	    ((sdom:document? node)
	     (lookup-namespace-uri (sdom:document-element node)))
	    ((or (sdom:entity? node)
		 (sdom:notation? node)
		 (sdom:document-type? node)
		 (sdom:document-fragment? node))
	     #f)
	    ((sdom:attr? node)
	     (and (sdom:owner-element node) 
		  (lookup-namespace-uri (sdom:owner-element node))))
	    (else (lookup-namespace-uri (match-ancestor sdom:element? node)))))

    (if (not (sdom:node? node))
	(raise (sdom:make-exception sdom:exception-code-type-mismatch-err #f)))
    (if (not (string? prefix))
	(raise (sdom:make-exception sdom:exception-code-type-mismatch-err #f)))
    (lookup-namespace-uri node))

  (define (sdom:lookup-prefix node namespace-uri)
    (internal-ns-lookup node namespace-uri 'prefix))

  (define (clone-node node deep)    
    (define (clone-document old-doc)
      (let ((doc (sdom:create-document #f #f)))
	(if deep
	    (let ((children (map (lambda (c) (import-node doc c #t))
				 (sdom:child-nodes old-doc))))
	      (make-children! doc children)
	      (for-each (lambda (c)
			  (cond ((sdom:element? c) 
				 (sdom:set-document-element! doc c))
				((sdom:document-type? c)
				 (sdom:set-doctype! doc c))))
			children)))
	doc))
    (let*
	((doc (sdom:owner-document node))
	 (new-node 
	  (cond ((sdom:document? node) (clone-document node))
		((sdom:element? node)
		 (let* ((x (if (sdom:namespace-uri node)
			       (sdom:create-element doc 
						    (sdom:tag-name node) 
						    (sdom:namespace-uri node))
			       (sdom:create-element doc (sdom:tag-name node))))
			(od (sdom:owner-document x)))
		   (sdom:set-attributes!
		    x (make-attributes x (map (lambda (y) (clone-node y #t)) 
					      (sdom:attributes node))))
		   (if deep
		       (make-children! x (map (lambda (y) (import-node od y #t))
					      (sdom:child-nodes node))))
		   x))
		((sdom:cdata-section? node)
		 (sdom:create-cdata-section 
		  doc (string-copy (sdom:character-data-data node))))
		((sdom:text-node? node) 
		 (sdom:create-text-node 
		  doc (string-copy (sdom:character-data-data node))))
		((sdom:attr? node)
		 (let ((new-node 
			(sdom:create-attribute 
			 doc (sdom:name node) (sdom:namespace-uri node))))
		   (if (sdom:specified node)
		       (set-value! new-node (sdom:value node)))
		   new-node))
		((sdom:processing-instruction? node)
		 (sdom:create-processing-instruction 
		  doc (sdom:target node) (sdom:data node)))
		((sdom:comment? node)
		 (sdom:create-comment doc (string-copy (sdom:data node))))
		((sdom:document-type? node)
		 (let ((new-node (sdom:create-document-type
				  (sdom:document-type-name node)
				  (sdom:document-type-public-id node)
				  (sdom:document-type-system-id node))))
		   (if deep
		       (let ((entities (map (lambda (x) (clone-node x #t))
					    (sdom:entities node))))
			 (make-children! new-node entities)
			 (sdom:set-entities! new-node entities)))
		   new-node))
				       
		((sdom:entity? node)
		 (let* ((new-node (sdom:make-entity (sdom:node-name node)))
			(cs (sdom:child-nodes node))
			(len (length cs)))
		   (if deep (for-each (lambda (x) 
					(insert-child! new-node x len))
				      cs))
		   new-node))))
	 (ns (sdom:namespace-uri node)))
      (if ns (sdom:set-namespace-uri! node ns))
      new-node))

  (define (sdom:clone-node node deep)
    (let* ((d (if (sdom:document? node) node (sdom:owner-document node)))      
	   (new-node (with-synchronizations (clone-node node deep))))
      (if (and (not (sdom:document-type? node)) (not (sdom:document? node)))
	  (sdom:set-owner-document! node d))
      (handle-user-data-event node sdom:user-data-event-node-cloned node 
			      new-node)
      new-node))
  
  ;;--------------------------------------------------------------------------;;
  ;;                                                                          ;;
  ;; Normalization functions for nodes and documents                          ;;
  ;;                                                                          ;;
  ;;--------------------------------------------------------------------------;;

  (define (orphan-node! node)
    (sdom:set-previous-sibling! node #f)
    (sdom:set-next-sibling! node #f)
    (sdom:set-parent-node! node #f))

  (define (normalize! node)
    (define adjacent-text-nodes '())
    (define (normalization-visitor continuation queue lst)
      (define (merge-text!)
	(if (> (length adjacent-text-nodes) 1)
	    (let* ((nodes (reverse adjacent-text-nodes))
		   (node (car nodes))
		   (text (apply string-append (map sdom:node-value nodes))))
	      (sdom:set-node-value! node text)
	      (sdom:set-character-data-data! node text)
	      (for-each (lambda (x) 
			  (let ((p (sdom:parent-node x)))
			    (remove-child! p (child-index p x))))
			(cdr nodes))))
	(set! adjacent-text-nodes '()))
      (define (stop-node? node)
	(or (sdom:attr? node)
	    (sdom:cdata-section? node)
	    (sdom:element? node)
	    (sdom:entity-reference? node)
	    (sdom:comment? node)
	    (sdom:processing-instruction? node)))
      (if (null? queue) (begin (merge-text!) lst)
	  (let ((q (car queue)))
	    (case (car q)
	      ((enter)
	       (let ((node (cadr q)))
		 (cond ((stop-node? node) (merge-text!))
		       ((sdom:text-node? node)
			(set! adjacent-text-nodes 
			      (cons node adjacent-text-nodes))))
		 (continuation queue lst)))
	      ((exit) 
	       (let ((node (cadr q)))
		 (if (stop-node? node) (merge-text!))
		 (continuation queue lst)))
	      (else (raise (make-assertion-violation)))))))

    (document-order node normalization-visitor))

  (define (sdom:normalize! node)
    (or (sdom:node? node)
	(raise (sdom:make-exception sdom:exception-code-type-mismatch-err #f)))	
    (with-synchronizations (normalize! node)))
  
  (define (sdom:normalize-document! node)
    (define (normalize-document! doc)
      (define split-cdata-sections? 
	(sdom:get-dom-config-parameter doc "split-cdata-sections"))
      (define cdata-sections? 
	(sdom:get-dom-config-parameter doc "cdata-sections"))
      (define comments? (sdom:get-dom-config-parameter doc "comments"))
      (define entities? (sdom:get-dom-config-parameter doc "entities"))
      (define aborted? #f)
      (define (document-normalization-visitor continuation queue lst)
	(define (update-entity-reference! n)
	  (let* ((dtd (sdom:doctype doc))
		 (refname (sdom:node-name n))
		 (match (find (lambda (x) (equal? (sdom:node-name x) refname)) 
			      (if dtd (sdom:entities dtd) '())))
		 (refkids (sdom:child-nodes n))
		 (entkids (if match (sdom:child-nodes match) '())))
	    (if (and match (not (list= sdom:equal-node? refkids entkids)))
		(let* ((len (length entkids))
		       (clones (map (lambda (x) (clone-node x #t)) entkids))
		       (p (sdom:parent-node n))
		       (ci (child-index p n)))
		  (remove-child! p (child-index p n))
		  (let loop ((clones clones) (ci ci))
		    (if (not (null? clones))
			(begin
			  (insert-child! p (car clones) ci)
			  (loop (cdr clones) (+ ci 1)))))
		  clones)
		(list n))))

	(define (split-cdata-section! cdata-section)
	  (define (string-split str tok)
	    (if (equal? str "") 
		(list)
		(let ((pos (string-contains str tok)))
		  (if pos
		      (if (eqv? pos 0)
			  (string-split (substring str (+ pos 3)) tok)
			  (cons (substring str 0 pos) 
				(string-split (substring str (+ pos 3)) tok)))
		      (list str)))))

	  (let* ((parent (sdom:parent-node cdata-section))
		 (ci (child-index parent cdata-section))
		 (val (sdom:data cdata-section))
		 (sections (string-split val "]]>"))
		 (split-nodes (map (lambda (s) 
				     (sdom:create-cdata-section doc s))
				   sections)))
	    (remove-child! parent ci)
	    (let loop ((nodes split-nodes) (ci ci))
	      (if (not (null? nodes))
		  (begin
		    (insert-child! parent (car nodes) ci)
		    (loop (cdr nodes) (+ ci 1)))))

	    (or (sdom:signal-error
		 doc sdom:error-severity-warning
		 "splitting cdata section" "cdata-sections-splitted" #f val #f)
		(set! aborted? #t))

	    split-nodes))

	(if (or aborted? (null? queue))
	    (continuation queue lst)
	    (let ((q (car queue)))
	      (case (car q)
		((enter)
		 (let* ((node (cadr q))
			(parent (sdom:parent-node node)))

		   (cond ((sdom:cdata-section? node)
			  (cond ((and split-cdata-sections?
				      (string-contains (sdom:data node) "]]>"))
				 (let ((sections (split-cdata-section! node)))
				   (continuation (append (enter-exit sections)
							 (cdr queue))
						 lst)))
				((not cdata-sections?)
				 (let ((pos (child-index parent node))
				       (txt (sdom:create-text-node 
					     doc (sdom:data node))))
				   (remove-child! parent pos)
				   (insert-child! parent txt pos)
				   (continuation (append (enter-exit (list txt))
							 (cdr queue))
						 lst)))
				(else (continuation queue lst))))
			 ((and (sdom:entity-reference? node)
			       (not entities?))
			  (continuation
			   (append (enter-exit (update-entity-reference! node))
				   (cdr queue))
			   lst))
			 ((sdom:comment? node) 
			  (if comments?
			      (continuation queue lst)
			      (let ((parent (sdom:parent-node node)))
				(remove-child! parent (child-index parent node))
				(continuation (cdr queue) lst))))
			 (else (continuation queue lst)))))
		((exit) (continuation queue lst))
		(else (raise (make-assertion-violation)))))))

      (document-order doc document-normalization-visitor))

    (or (sdom:document? node)
	(raise (sdom:make-exception sdom:exception-code-type-mismatch-err #f)))
    (with-synchronizations 
     (normalize-document! node)
     (normalize! node)))

  (define (sdom:compare-document-position node1 node2)
    (let* ((ownerdoc1 (if (sdom:document? node1)
			  node1 (sdom:owner-document node1)))
	   (ownerdoc2 (if (sdom:document? node2)
			  node2 (sdom:owner-document node2))) 
	   (ancestors1 (ancestors node1))
	   (ancestors2 (ancestors node2)))
      (cond ((sdom:same-node? node1 node2) 0)
	    ((or (not (sdom:same-node? ownerdoc1 ownerdoc2))
		 (and (not (and (sdom:document? node1) (sdom:document? node2)))
		      (let ((lpa1 (if (null? ancestors1)
				      ancestors1
				      (last-pair ancestors1)))
			    (lpa2 (if (null? ancestors2)
				      ancestors2
				      (last-pair ancestors2))))
			(or (and (null? lpa1) (null? lpa2))
			    (and (not (and (null? lpa1)
					   (sdom:same-node? node1 (car lpa2))))
				 (not (and (null? lpa2) 
					   (sdom:same-node? node2 (car lpa1))))
				 (not (and (not (null? lpa1))
					   (not (null? lpa2))
					   (sdom:same-node? (car lpa1) 
							    (car lpa2)))))))))
	     (bitwise-ior (if (> (hash-by-identity node1) 
				 (hash-by-identity node2))
			      sdom:document-position-following
			      sdom:document-position-preceding)
			  sdom:document-position-disconnected
			  sdom:document-position-implementation-specific))
	    ((sdom:same-node? node1 node2) 0)
	    ((find (lambda (x) (sdom:same-node? x node1)) ancestors2)
	     (bitwise-ior sdom:document-position-contained-by
			  sdom:document-position-following))
	    ((find (lambda (x) (sdom:same-node? x node2)) ancestors1)
	     (bitwise-ior sdom:document-position-contains
			  sdom:document-position-preceding))
	    (else (let* ((commonroot (find (lambda (x) 
					     (find (lambda (y)
						     (sdom:same-node? x y)) 
						   ancestors2))
					   ancestors1))
			 (rootlist1
			  (reverse (take-while (lambda (x) 
						 (not (sdom:same-node? 
						       x commonroot))) 
					       ancestors1)))
			 (rootlist1 (if (null? rootlist1) 
					node1 
					(car rootlist1)))
			 (rootlist2 
			  (reverse (take-while (lambda (x)
						 (not (sdom:same-node?
						       x commonroot)))
					       ancestors2)))
			 (rootlist2 (if (null? rootlist2) 
					node2 
					(car rootlist2)))
			 (typer1 (sdom:attr? rootlist1))
			 (typer2 (sdom:attr? rootlist2)))
		    (if (eqv? (sdom:attr? rootlist1) (sdom:attr? rootlist2))
			(if (sdom:attr? rootlist1)
			    (bitwise-ior
			     32 (let ((attrs (sdom:attributes commonroot)))
				  (if (> (list-index (lambda (x)
						       (sdom:same-node? 
							x rootlist1))
						     attrs)
					 (list-index (lambda (x) 
						       (sdom:same-node? 
							x rootlist2))
						     attrs))
				      sdom:document-position-following
				      sdom:document-position-preceding)))
			    (let ((children (sdom:child-nodes commonroot)))
			      (if (< (list-index (lambda (x)
						   (sdom:same-node? 
						    x rootlist1))
						 children)
				     (list-index (lambda (x)
						   (sdom:same-node?
						    x rootlist2))
						 children))
				  sdom:document-position-following
				  sdom:document-position-preceding)))
			(if (sdom:attr? rootlist1)
			    sdom:document-position-following
			    sdom:document-position-preceding)))))))

  (define (sdom:create-document root-name doctype . namespace-uri)
    (let* ((newdoc (sdom:make-document))
	   (docelem (and root-name 
			 (let ((e (if (null? namespace-uri)
				      (sdom:create-element newdoc root-name #f)
				      (sdom:create-element
				       newdoc root-name (car namespace-uri)))))
			   (sdom:set-owner-document! e newdoc)
			   (sdom:set-document-element! newdoc e)
			   e)))
	   (doctype (and doctype 
			 (begin
			   (if (not (sdom:document-type? doctype))
			       (raise (sdom:make-exception 
				       sdom:exception-code-type-mismatch-err 
				       #f)))
			   (if (sdom:owner-document doctype)
			       (raise (sdom:make-exception 
				       sdom:exception-code-wrong-document-err 
				       #f)))
			   (sdom:set-owner-document! doctype newdoc)
			   (sdom:set-doctype! newdoc doctype)
			   doctype))))
      (make-children! newdoc (append (if docelem (list docelem) '())
				     (if doctype (list doctype) '())))
      newdoc))

  (define (sdom:create-document-fragment doc)
    (let ((docfrag (sdom:make-document-fragment)))
      (sdom:set-owner-document! docfrag doc)
      docfrag))

  (define (sdom:create-document-type qname public-id system-id)
    (sdom:make-document-type qname public-id system-id #f))

  (define (sdom:create-attribute doc name . namespace-uri)
    (let* ((ns (and (not (null? namespace-uri)) (car namespace-uri)))
	   (attr (sdom:make-attr name #f #f #f ns)))
      (sdom:set-owner-document! attr doc)
      attr))

  (define (sdom:create-cdata-section doc cdata)
    (let ((cdata-section (sdom:make-cdata-section cdata)))
      (sdom:set-owner-document! cdata-section doc)
      (sdom:set-text-content-internal! cdata-section cdata)
      cdata-section))

  (define (sdom:create-text-node doc text)
    (let ((text-node (sdom:make-text-node sdom:node-type-text text)))
      (sdom:set-owner-document! text-node doc)
      (sdom:set-text-content-internal! text-node text)
      text-node))
  
  (define (sdom:create-comment doc comment)
    (let ((comment-node (sdom:make-comment comment)))
      (sdom:set-owner-document! comment-node doc)
      (sdom:set-text-content-internal! comment-node comment)
      comment-node))

  (define (sdom:create-entity-reference doc entity)
    (let ((entity-reference (sdom:make-entity-reference entity)))
      (sdom:set-owner-document! entity-reference doc)
      entity-reference))

  (define (sdom:create-element doc tag-name . namespace-uri)
    (let ((namespace-uri (and (not (null? namespace-uri))
			      (car namespace-uri))))
      (let ((element (sdom:make-element tag-name #f namespace-uri)))
	(sdom:set-owner-document! element doc)
	element)))

  (define (sdom:create-processing-instruction doc target data)
    (let ((node (sdom:make-processing-instruction target data)))
      (sdom:set-owner-document! node doc)
      (sdom:set-text-content-internal! node data)
      node))

  (define (enter-exit nodes)
    (if (null? nodes) (list)
	(let ((node (car nodes)))
	  (cons (list 'enter node) 
		(cons (list 'exit node) (enter-exit (cdr nodes)))))))

  (define (document-order start . args)
    (define (document-order-default-visitor visitor queue lst)
      (visitor queue lst))

    (define configured-visitor 
      (if (null? args) document-order-default-visitor (car args)))

    (define (document-order-visitor queue lst)

      (if (null? queue) lst
	  (let ((q (car queue)))
	    (case (car q)
	      ((enter)
	       (let ((node (cadr q)))
		 (cond ((sdom:document? node)
			(document-order-recursive 
			 (append 
			  (enter-exit (list (sdom:document-element node)))
			  (cdr queue)) 
			 lst))
		       ((sdom:element? node)
			(document-order-recursive
			 (append (enter-exit (sdom:attributes node))
				 (enter-exit (sdom:child-nodes node))
				 (cdr queue))
			 (cons node lst)))
		       ((sdom:attr? node)
			(document-order-recursive
			 (append (enter-exit (sdom:child-nodes node)) 
				 (cdr queue))
			 (cons node lst)))
		       ((or (sdom:document-fragment? start)
			    (sdom:entity? start)
			    (sdom:entity-reference? start))
			(document-order-recursive
			 (append (enter-exit (sdom:child-nodes node))
				 (cdr queue)) 
			 lst))
		       (else (document-order-recursive 
			      (cdr queue) (cons node lst))))))
	      ((exit) (document-order-recursive (cdr queue) lst))
	      (else (raise (make-assertion-violation)))))))
            
    (define (document-order-recursive queue lst)
      (configured-visitor document-order-visitor queue lst))
    
    (reverse (document-order-recursive (enter-exit (list start)) '())))

  (define (sdom:get-elements-by-tag-name doc name . args)
    (let ((ns (and (not (null? args))
		   (let ((ca (car args)))
		     (if (string? ca) 
			 ca 
			 (raise (sdom:make-exception 
				 sdom:exception-code-type-mismatch-err #f)))))))
      (if (not (sdom:document? doc))
	  (raise (sdom:make-exception sdom:exception-code-type-mismatch-err 
				      #f)))
      (filter (lambda (x)
		(and (sdom:element? x)
		     (if ns
			 (and (or (equal? ns "*")
				  (equal? (sdom:namespace-uri x) ns))
			      (or (equal? name "*")
				  (equal? (sdom:local-name x) name)))
			 (or (equal? name "*")
			     (equal? (sdom:tag-name x) name)))))
	      (document-order doc))))

  (define (make-children! parent nodes)
    (let loop ((cs nodes))
      (if (not (null? cs))
	  (let ((c (car cs)))
	    (sdom:set-parent-node! c parent)
	    (if (not (null? (cdr cs)))
		(let ((nc (cadr cs)))
		  (sdom:set-next-sibling! c nc)
		  (sdom:set-previous-sibling! nc c)))
	    (loop (cdr cs)))))
    
    (sdom:set-child-nodes! parent nodes)
    (register-synchronization! (make-dom-synchronization 'text-content parent))

    (if (null? nodes)
	(begin 
	  (sdom:set-first-child! parent #f)
	  (sdom:set-last-child! parent #f))
	(begin
	  (sdom:set-first-child! parent (car nodes))
	  (sdom:set-last-child! parent (car (last-pair nodes))))))

  (define (make-attributes parent nodes)
    (for-each (lambda (x) (sdom:set-owner-element! x parent)) nodes) nodes)

  (define (import-node doc node deep)
    (let ((new-node (clone-node node #f)))
      (if (sdom:attr? new-node)
	  (begin 
	    (sdom:set-specified! new-node #t)
	    (sdom:set-is-id! new-node #f)))
      (sdom:set-owner-document! new-node doc)

      (if (sdom:element? node)
	  (let ((child-attrs (map (lambda (x) (import-node doc x #t))
				  (sdom:attributes node))))
	    (sdom:set-attributes! 
	     new-node (make-attributes new-node child-attrs))))

      (if deep
	  (let ((child-nodes (map (lambda (x) (import-node doc x #t)) 
				  (sdom:child-nodes node))))
	    (make-children! new-node child-nodes)
	    (cond ((sdom:document-type? node)
		   (sdom:set-entities! new-node 
				       (filter sdom:entity? child-nodes))))))
      new-node))

  (define (sdom:import-node doc node deep)
    (if (or (sdom:document? node) (sdom:document-type? node))
	(raise (sdom:make-exception sdom:exception-code-not-supported-err #f))
	(with-synchronizations 
	 (let ((new-node (import-node doc node deep)))
	   (handle-user-data-event 
	    node sdom:user-data-event-node-imported node new-node)
	   new-node))))

  ;; INCOMPLETE -- NEED TO ADD NEW NAMESPACE DECL IF NECESSARY!

  (define (adopt-node! doc node orphan?)
    (if (sdom:attr? node)
	(begin
	  (sdom:set-owner-element! node #f)
	  (sdom:set-specified! node #t)
	  (sdom:set-is-id! node #f)
	  (for-each (lambda (x) (adopt-node! doc x #f)) 
		    (sdom:child-nodes node))))
	  
    (let ((parent (sdom:parent-node node)))
      (if (and parent orphan?) (sdom:remove-child! parent node)))
    (sdom:set-owner-document! node doc)
    node)

  (define (sdom:adopt-node! doc node)
    (if (not (sdom:document? doc))
	(raise (sdom:make-exception sdom:exception-code-type-mismatch-err #f)))
    (if (or (sdom:document? node) (sdom:document-type? node))
	(raise (sdom:make-exception sdom:exception-code-not-supported-err #f)))
    (if (or (sdom:notation? node)
	    (sdom:entity? node))
	(raise (sdom:make-exception sdom:exception-code-type-mismatch-err #f)))
    (if (sdom:read-only node)
	(raise (sdom:make-exception 
		sdom:exception-code-no-modification-allowed-err #f)))
    (adopt-node! doc node #t)
    (handle-user-data-event node sdom:user-data-event-node-adopted node '())
    node)      

  (define (sdom:rename-node! node qname ns)
    (let ((old-name (string-copy (sdom:node-name node)))
	  (old-ns (sdom:namespace-uri node))
	  (doc (sdom:owner-document node)))
      (if (not (or (sdom:attr? node) (sdom:element? node)))
	  (raise (sdom:make-exception sdom:exception-code-not-supported-err 
				      #f)))
      (if (not (qname? qname))
	  (raise (sdom:make-exception sdom:exception-code-invalid-character-err
				      #f)))
      (if (not (valid-namespace-combo? doc qname ns))
	  (raise (sdom:make-exception sdom:exception-code-namespace-err #f)))

      (sdom:set-node-name! node qname)
      (sdom:set-local-name! node qname)
      (sdom:set-namespace-uri! node ns)
      (sdom:set-prefix! node qname)

      (if (sdom:element? node)
	  (begin
	    (sdom:set-tag-name! node qname)
	    (sdom:dispatch-event node "DOMElementNameChanged" node
				 #f #f #f #f #f old-ns old-name))
	  (begin
	    (sdom:set-name! node qname)
	    (let ((parent (sdom:parent-node node)))
	      (if parent
		  (sdom:dispatch-event 
		   parent "DOMAttributeNameChanged" node #f #f #f
		   old-name #f old-ns old-name)))))
      (handle-user-data-event 
       node sdom:user-data-event-node-renamed node node)
      node))
  
  ;; Here are some attribute-mangling functions -- the ones that don't deal with
  ;; nodes explicitly will call into the ones that do.

  ;;--------------------------------------------------------------------------;;
  ;;                                                                          ;;
  ;; Attribute management functions                                           ;;
  ;;                                                                          ;;
  ;;--------------------------------------------------------------------------;;

  (define (sdom:get-attribute-node elt name . namespace-uri) 
    (if (not (sdom:element? elt))
	(raise (sdom:make-exception sdom:exception-code-type-mismatch-err #f)))
    (let* ((finder (if (null? namespace-uri)
		       (lambda (x) (equal? name (sdom:name x)))
		       (lambda (x) (and (equal? name (sdom:local-name x))
					(equal? (sdom:namespace-uri x)
						(car namespace-uri)))))))
      (find finder (sdom:attributes elt))))

  (define (sdom:set-attribute-node-internal! elt node old-node)
    (let ((attrs (sdom:attributes elt)))
      (sdom:set-owner-element! node elt)
      (sdom:set-attributes!
       elt (cons node (if old-node (remq old-node attrs) attrs)))))

  (define (sdom:set-attribute-node! elt node) 
    (if (not (and (sdom:element? elt) (sdom:attr? node)))
	(raise (sdom:make-exception sdom:exception-code-type-mismatch-err #f)))
    (if (not (sdom:same-node? (sdom:owner-document elt)
			      (sdom:owner-document node)))
	(raise (sdom:make-exception sdom:exception-code-wrong-document-err #f)))
    (if (sdom:owner-element node)
	(raise (sdom:make-exception sdom:exception-code-inuse-attribute-err 
				    #f)))

    (let* ((name (sdom:name node))
	   (old-node (sdom:get-attribute-node elt name))
	   (old-value (sdom:get-attribute elt name))
	   (new-value (sdom:value node)))
      (if old-value (sdom:dispatch-event 
		     elt "DOMAttrModified" old-node old-value new-value name 3))
      (with-synchronizations
       (sdom:set-attribute-node-internal! elt node old-node))
      (sdom:dispatch-event elt "DOMAttrModified" node #f new-value name 2)))

  (define (remove-attribute! elt node)
    (let* ((attributes (sdom:attributes elt))
	   (attributes (remq node attributes)))

      (sdom:set-attributes! elt node)
      (orphan-node! node)
      
      node))

  (define (sdom:remove-attribute-node! elt node) 
    (if (not (and (sdom:element? elt) (sdom:attr? node)))
	(raise (sdom:make-exception sdom:exception-code-type-mismatch-err #f)))
    (if (not (sdom:same-node? elt (sdom:owner-element node)))
	(raise (sdom:make-exception sdom:exception-code-not-found-err #f)))

    (remove-attribute! elt node))

  (define (sdom:get-attribute elt name . namespace-uri)
    (let ((node (if (not (null? namespace-uri))
		    (sdom:get-attribute-node elt name (car namespace-uri))
		    (sdom:get-attribute-node elt name))))
      (if node (sdom:node-value node) #f)))

  (define (sdom:set-attribute! elt name value . namespace-uri)
    (if (not (sdom:element? elt))
	(raise (sdom:make-exception sdom:exception-code-type-mismatch-err #f)))
    (let ((ns (if (not (null? namespace-uri)) (car namespace-uri) #f))
	  (doc (sdom:owner-document elt)))
      (if (not (valid-namespace-combo? doc name ns))
	  (raise (sdom:make-exception sdom:exception-code-namespace-err #f)))
      (let* ((old-attr (if ns 
			   (sdom:get-attribute-node elt (local-name name) ns)
			   (sdom:get-attribute-node elt name)))
	     (attr (or old-attr 
		       (let ((attr (if ns
				       (sdom:create-attribute doc name ns)
				       (sdom:create-attribute doc name))))
			 (sdom:set-attribute-node-internal! elt attr #f)
			 attr)))
	     (old-value (and old-attr (sdom:value old-attr))))
	(sdom:set-value! attr value)
	(sdom:dispatch-event 
	 elt "DOMAttrModified" attr old-value value name (if old-value 2 1)))))

  (define (update-text-content! root)
    (define (update-text-content-visitor continuation queue lst)
      (define (update-node! node)
	(sdom:set-text-content-internal! 
	 node (apply string-append
		     (map (lambda (n) (or (sdom:text-content n) "")) 
			  (filter (lambda (n)
				    (not (or (sdom:comment? n)
					     (sdom:processing-instruction? n))))
				  (sdom:child-nodes node))))))
      (if (null? queue) lst
	  (let ((q (car queue)))
	    (if (eq? (car q) 'exit)
		(let ((node (cadr q)))
		  (cond ((or (sdom:element? node)
			     (sdom:attr? node)
			     (sdom:entity? node)
			     (sdom:entity-reference? node)
			     (sdom:document-fragment? node))
			 (update-node! node))
			((or (sdom:text-node? node)
			     (sdom:cdata-section? node)
			     (sdom:comment? node)
			     (sdom:processing-instruction? node))
			 (sdom:set-text-content-internal! 
			  node (sdom:node-value node))))))
	    (continuation queue lst))))

    (document-order root update-text-content-visitor))

  (define (set-value! attr value)
    (sdom:set-node-value! attr value)
    (sdom:set-value-internal! attr value)
    (sdom:set-specified! attr #t)
    
    (for-each (lambda (c) (remove-child! attr (child-index attr c)))
	      (sdom:child-nodes attr))
    (insert-child! 
     attr (sdom:create-text-node (sdom:owner-document attr) value) 0))

  (define (sdom:set-value! attr value)
    (or (sdom:attr? attr)
	(raise (sdom:make-exception sdom:exception-code-type-mismatch-err #f)))
    (with-synchronizations (set-value! attr value)))

  (define (sdom:remove-attribute! elt name . namespace-uri)
    (let ((attr (if (not (null? namespace-uri))
		    (sdom:get-attribute-node elt name (car namespace-uri))
		    (sdom:get-attribute-node elt name))))
      (if attr (sdom:remove-attribute-node! elt attr))))

  (define (sdom:get-element-by-id doc id)
    (define (pred x)
      (and (sdom:element? x)
	   (find (lambda (y) (and (sdom:is-id y) (equal? id (sdom:value y))))
		 (sdom:attributes x))))
    (if (not (sdom:document? doc))
	(raise (sdom:make-exception sdom:exception-code-type-mismatch-err #f)))
    (find pred (document-order doc)))

  (define (sdom:set-id-attribute! node name is-id . args)
    (let ((attr (if (null? args)
		    (sdom:get-attribute-node node name)
		    (sdom:get-attribute-node node name (car args)))))
      (if attr
	  (sdom:set-id-attribute-node! attr is-id)
	  (raise (sdom:make-exception sdom:exception-code-not-found-err #f)))))

  (define (sdom:set-id-attribute-node! attr is-id)
    (sdom:set-is-id! attr is-id))

  (define (sdom:xml->sdom port . extras)
    (let ((ns (if (not (null? extras)) (car extras) '()))
	  (parser (if (> (length extras) 1) (cadr extras) sdom:xml-parser)))
      (sdom:sxml->sdom (parser port ns))))

  (define-record-type (sdom:attr-group sdom:make-attr-group sdom:attr-group?)
    (parent sdom:node)
    (fields (mutable element 
		     sdom:attr-group-element 
		     sdom:set-attr-group-element!))
    (protocol (lambda (n) 
		(lambda ()
		  (let ((p (n #f #f #f #f #f #f))) 
		    (p #f))))))

  (define (sdom:sxml->sdom sxml-tree)

    (define (make-prepender n)
      (lambda (c)
	(or (sdom:document? n) 
	    (sdom:set-owner-document! c (sdom:owner-document n)))
	(sdom:set-child-nodes! n (cons c (sdom:child-nodes n)))))

    (define (make-element-prepender e)
      (define p (make-prepender e))
      (lambda (c)
	(if (sdom:attr-group? c) 
	    (sdom:set-attr-group-element! c e)
	    (p c))))
  
    (define (make-document-prepender d)
      (define p (make-prepender d))
      (lambda (c)
	(sdom:set-owner-document! c d)
	(cond ((sdom:element? c) (sdom:set-document-element! d c))
	      ((sdom:document-type? c) (sdom:set-doctype! d c)))
	(p c)))

    (define (make-document-type-prepender dtd)
      (define p (make-prepender dtd))
      (lambda (e)
	(if (sdom:entity? e) 
	    (sdom:set-entities! dtd (cons e (sdom:entities dtd))))
	(p e)))

    (define (postprocess-attr-group a)
      (postprocess-node a)
      (let* ((e (sdom:attr-group-element a))
	     (cs (sdom:child-nodes a))
	     (op (find (lambda (attr) 
			 (equal? (sdom:name attr) "*SDOM:ORIG-PREFIX*"))
		       cs))
	     (cs (if op (remq op cs) cs)))
	
	(if op
	    (let ((new-name (string-append 
			     (sdom:value op) ":" (sdom:tag-name e))))
	      (sdom:set-node-name! e new-name)
	      (sdom:set-tag-name! e new-name)
	      (sdom:set-prefix! e (sdom:value op))))
	(sdom:set-attributes! e (make-attributes e cs))))

    (define (postprocess-node n)
      (make-children! n (reverse (sdom:child-nodes n))))

    (define (handle-attrs attr-node)
      (let* ((attr-group (sdom:make-attr-group))
	     (proc (make-prepender attr-group))
	     (attrs (map (lambda (a) (list 'process (cons '^^ a) proc))
			 (filter (lambda (a)
				   (and (list? a)  (not (memq (car a) '(^)))))
				 (cdr attr-node)))))
	(apply values 
	       (append (list attr-group) attrs `((postprocess 
						  ,postprocess-attr-group 
						  ,attr-group))))))

    (define (handle-attr attr)
      (define (find-attr-attr name)
	(let* ((aalist
		(find (lambda (a) (and (list? a) (eq? (car a) '^))) attr))
	       (aa (and aalist (find (lambda (aa) 
				       (and (pair? aa) (eq? (car aa) name)))
				     (cdr aalist)))))
	  (and aa (cadr aa))))
	  
      (let-values (((prefix local-name) 
		    (prefix-and-local-name (symbol->string (cadr attr)))))
	(let* ((orig-prefix (find-attr-attr '*SDOM:ORIG-PREFIX*))
	       (attr-name (cond ((equal? prefix xml-ns-uri)
				 (string-append "xml:" local-name))
				(orig-prefix
				 (string-append orig-prefix ":" local-name))
				(prefix (string-append prefix ":" local-name))
				(else local-name)))
	       (a (sdom:make-attr attr-name (caddr attr) #f #f prefix))
	       (proc (make-prepender a)))
	  (values a
		  `(process ,(caddr attr) ,proc) 
		  `(postprocess ,postprocess-node ,a)))))

    (define (process-node node)
      (cond 
       ((string? node) (values (sdom:make-text-node sdom:node-type-text node)))
       ((list? node)
	(case (car node)
	  ((*TOP*)
	   
	   (let* ((d (sdom:make-document))
		  (proc (make-document-prepender d)))
	     (apply values (append (list d) 
				   (map (lambda (c) (list 'process c proc))
					(cdr node))
				   `((postprocess ,postprocess-node ,d))))))
	  
	  ((*COMMENT*) (values (sdom:make-comment)))
	  ((*ENTITY*) 
	   (let* ((entity (sdom:make-entity (symbol->string (cadr node))))
		  (proc (make-prepender entity)))
	     (apply values (append `(,entity)
				   `((process ,(caddr node) ,proc))))))
	  ((*ENTITY-REF*) (values (sdom:make-entity-reference)))
	  ((*FRAGMENT*) (values (sdom:make-document-fragment)))
	  ((*DOCTYPE*) 
	   (let ((dt (sdom:make-document-type 
		      (list-ref node 1)
		      (list-ref node 4)
		      (list-ref node 5)
		      (list-ref node 6))))
	     (apply values
		    (append (list dt)
			    (let ((ents (list-ref node 2))
				  (proc (make-document-type-prepender dt)))
			      (if ents
				  (map (lambda (e) (list 'process e proc)) ents)
				  '()))))))

	  ((*PI*) (values (sdom:make-processing-instruction 
			   (symbol->string (cadr node)) (caddr node))))
	  ((*NOTATION*) (values (sdom:make-notation)))
	  
	  ((^) (handle-attrs node))
	  ((^^) (handle-attr node))
	  
	  (else (if (symbol? (car node))
		    (let* ((hs (symbol->string (car node))))
		      (let-values (((prefix local-name) 
				    (prefix-and-local-name hs)))
			(let* ((e (sdom:make-element 
				   (or local-name hs) #f prefix))
			       (proc (make-element-prepender e)))
			  (apply values 
				 (append (list e)
					 (map (lambda (c) 
						(list 'process c proc))
					      (cdr node))
					 `((postprocess ,postprocess-node 
							,e)))))))
		    (raise (sdom:make-exception 
			    sdom:exception-code-type-mismatch-err 
			    #f))))))
       (else (raise (sdom:make-exception
		     sdom:exception-code-type-mismatch-err #f)))))

    (define (process-tree operations)
      (let ((op (car operations)))
	(case (car op)
	  ((done) (cadr op))
	  ((process) (let*-values
		      (((node . child-ops) (process-node (cadr op)))
		       ((proc) (values (caddr op))))
		      (proc node)
		      (process-tree (append child-ops (cdr operations)))))
	  ((postprocess) (let-values (((proc node) (apply values (cdr op))))
			   (proc node)
			   (process-tree (cdr operations))))
	  (else (raise (condition 
			(make-assertion-violation)
			(make-message-condition 
			 "Unknown parsing operation.")))))))

    (with-synchronizations
     (let-values (((doc . child-ops) (process-node sxml-tree)))
       (process-tree (append child-ops `((done ,doc)))))))
)
