;; sdom.scm: main module exports and implementations for SDOM
;; Copyright (C) 2004 Julian Graham

;; SDOM is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2 of the License, or
;; (at your option) any later version.
;;
;; SDOM is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with SDOM; if not, write to the Free Software
;; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

(define-module (sdom core)
  #:use-module (ice-9 regex)
  #:use-module (ice-9 slib)
  #:use-module (srfi srfi-1)
  #:use-module (srfi srfi-13)
  #:use-module (sxml ssax)
  #:export     (sdom:sxml->sdom 
		sdom:set-dom-property!
		sdom:get-dom-property
		sdom:node?
		sdom:node-attributes
		sdom:node-name
		sdom:node-type

		sdom:node-type-element
		sdom:node-type-attr
		sdom:node-type-text
		sdom:node-type-cdata-section
		sdom:node-type-entity-reference
		sdom:node-type-entity
		sdom:node-type-processing-instruction
		sdom:node-type-comment
		sdom:node-type-document
		sdom:node-type-document-type
		sdom:node-type-document-fragment
		sdom:node-type-notation

		sdom:exception-code-index-size-err
		sdom:exception-code-domstring-size-err
		sdom:exception-code-hierarchy-request-err
		sdom:exception-code-wrong-document-err
		sdom:exception-code-invalid-character-err
		sdom:exception-code-no-data-allowed-err
		sdom:exception-code-no-modification-allowed-err
		sdom:exception-code-not-found-err
		sdom:exception-code-not-supported-err
		sdom:exception-code-inuse-attribute-err
		sdom:exception-code-invalid-state-err
		sdom:exception-code-syntax-err
		sdom:exception-code-invalid-modification-err
		sdom:exception-code-namespace-err
		sdom:exception-code-invalid-access-err
		sdom:exception-code-validation-err
		sdom:exception-code-type-mismatch-err

		sdom:error-severity-warning
		sdom:error-severity-error
		sdom:error-severity-fatal-error

		sdom:document-position-disconnected
		sdom:document-position-contained-by
		sdom:document-position-contains
		sdom:document-position-following
		sdom:document-position-preceding
		sdom:document-position-implementation-specific
		
		sdom:user-data-event-node-cloned
		sdom:user-data-event-node-imported
		sdom:user-data-event-node-deleted
		sdom:user-data-event-node-renamed
		sdom:user-data-event-node-adopted

		sdom:register-feature!
		sdom:has-feature?

		sdom:config-parameter-names
		sdom:add-dom-config-parameter!
		sdom:get-dom-config-parameter
		sdom:set-dom-config-parameter!
		sdom:can-set-dom-config-parameter?
		sdom:signal-error

		sdom:insert-before!
		sdom:insert-after!
		sdom:remove-child!
		sdom:replace-child!
		sdom:replace-whole-text!
		sdom:append-child!
		sdom:clone-node
		sdom:normalize!
		sdom:normalize-document!
		sdom:adopt-node!
		sdom:import-node
		sdom:rename-node!

		sdom:same-node?
		sdom:equal-node?
		sdom:has-child-nodes?
		sdom:supported?

		sdom:default-namespace?		
		sdom:lookup-prefix
		sdom:lookup-namespace-uri

		sdom:set-user-data!
		sdom:get-user-data

		sdom:create-node
		sdom:create-document
		sdom:create-document-type
		
		sdom:get-elements-by-tag-name
		sdom:get-element-by-id
		sdom:set-id-attribute!
		sdom:set-id-attribute-node!
		sdom:get-attribute-node
		sdom:set-attribute-node!
		sdom:remove-attribute-node!
		sdom:get-attribute
		sdom:set-attribute!
		sdom:remove-attribute!

		sdom:compare-document-position
		
		sdom:dispatch-event))


;; The following constructs get added to the SXML tree to make it into a DOM-
;; compliant document tree. !!! We're going to use sub @-annotations to store
;; our data!

(define sdom:node-type-node 0)
(define sdom:node-type-character-data 100)

(define sdom:node-type-element 			1)
(define sdom:node-type-attr	 		2)
(define sdom:node-type-text 			3)
(define sdom:node-type-cdata-section 		4)
(define sdom:node-type-entity-reference 	5)
(define sdom:node-type-entity 			6)
(define sdom:node-type-processing-instruction 	7)
(define sdom:node-type-comment 			8)
(define sdom:node-type-document			9)
(define sdom:node-type-document-type 		10)
(define sdom:node-type-document-fragment 	11)
(define sdom:node-type-notation 		12)

(define sdom:exception-code-index-size-err 		1)
(define sdom:exception-code-domstring-size-err 		2)
(define sdom:exception-code-hierarchy-request-err 	3)
(define sdom:exception-code-wrong-document-err 		4)
(define sdom:exception-code-invalid-character-err 	5)
(define sdom:exception-code-no-data-allowed-err 	6)
(define sdom:exception-code-no-modification-allowed-err 7)
(define sdom:exception-code-not-found-err 		8)
(define sdom:exception-code-not-supported-err 		9)
(define sdom:exception-code-inuse-attribute-err 	10)
(define sdom:exception-code-invalid-state-err 		11)
(define sdom:exception-code-syntax-err 			12)
(define sdom:exception-code-invalid-modification-err 	13)
(define sdom:exception-code-namespace-err 		14)
(define sdom:exception-code-invalid-access-err 		15)
(define sdom:exception-code-validation-err 		16)
(define sdom:exception-code-type-mismatch-err 		17)

(define sdom:error-severity-warning		1)
(define sdom:error-severity-error		2)
(define sdom:error-severity-fatal-error		3)

(define sdom:document-position-disconnected 		1)
(define sdom:document-position-preceding 		2)
(define sdom:document-position-following 		4)
(define sdom:document-position-contains 		8)
(define sdom:document-position-contained-by 		16)
(define sdom:document-position-implementation-specific 	32)

(define sdom:user-data-event-node-cloned 	1)
(define sdom:user-data-event-node-imported	2)
(define sdom:user-data-event-node-deleted	3)
(define sdom:user-data-event-node-renamed	4)
(define sdom:user-data-event-node-adopted	5)

(define sdom:event-exception-code-unspecified-event-type-err	0)
(define sdom:event-exception-code-dispatch-request-err		1)

(define sdom:event-phase-capturing	1)
(define sdom:event-phase-target		2)
(define sdom:event-phase-bubbling	3)

(define xml-ns-uri "http://www.w3.org/XML/1998/namespace")
(define xmlns-ns-uri "http://www.w3.org/2000/xmlns")

;; This guy stores the user data across all nodes / documents.  We're not going
;; to export it, because we have a pair of functions that handle access to it.

(define initial-user-data-hash-size 16)
(define user-data-hash (make-weak-key-hash-table initial-user-data-hash-size))

(define default-dom-error-handler
  (lambda (severity msg type excep data loc)
    (display (string-append "SDOM: "
			    (cond ((eqv? severity sdom:error-severity-warning)
				   "warning: ")
				  ((eqv? severity sdom:error-severity-error)
				   "error: ")
				  ((eqv? severity 
					 sdom:error-severity-fatal-error)
				   "fatal error: "))
			    msg))
    (newline)
    #t))

(define sdom:signal-error
  (lambda (doc severity msg type excep data loc)
    (let* ((handler (sdom:get-dom-config-parameter doc "error-handler")))
      (apply handler `(,severity ,msg ,type ,excep ,data ,loc)))))

;; How to extract the annotations from a particular node?
;; If the node is an element node, annotations are stored in sub attributes
;; like so '(@ (@ ...)).  If the node is an attribute, then the sub-annotation
;; looks like an attribute itself -- but we know it's not, because attributes
;; don't have attributes.

;; (attr (@ (sdom:name "value")))
;; (node (@ (attribute "value")))
;; (node (@ (attribute "value") (@ (sdom:name "value"))))

(define fold
  (lambda (_kons_38 _knil_39 _lis1_40)
    ((letrec ((_lp_41 (lambda (_lis_42 _ans_43)
                        (if (null? _lis_42)
			    _ans_43
			    (_lp_41 (cdr _lis_42)
				    (_kons_38 (car _lis_42) _ans_43))))))
       _lp_41)
     _lis1_40
     _knil_39)))

(define xor (lambda (x y) (if x (if y #f #t) (if y #t #f))))

(define extended-char-regex "\\/\\:")
(define qname-char-regex "([[:alnum:]._-]|\\:)+")
(define qname-char-regex-extended "([[:alnum:]._-]|\\:|\\/)+")
(define qname-regex 
  "([[:alpha:]_][[:alnum:]._-]*\\:)?[[:alpha:]_][[:alnum:]._-]*")
(define qname-regex-extended
  "([[:alpha:]_]([[:alnum:]._-]|\\:|\\/)*)?[[:alpha:]_][[:alnum:]._-]*")

(define regex-match?
  (lambda (pattern str)
    (let ((match (string-match pattern str)))
      (and match (equal? (match:substring match) str)))))

;; I feel like this will be useful -- but I don't feel like writing it just yet

(define sdom:node?
  (lambda (x) #t))

(define single-at-finder 
  (lambda (node)
    (find (lambda (item) (and (list? item) (eq? (car item) '@))) (cdr node))))

(define multi-at-finder
  (lambda (node)
    (filter (lambda (item) (eq? (car item) '@)) (cdr node))))

(define annotations
  (lambda (node)
    (let* ((type (sdom:node-type node))
	   (at-list (single-at-finder node)))
      (if at-list
	  (if (eqv? type sdom:node-type-element)
	      (let ((sub-at-list (single-at-finder at-list)))
		(if sub-at-list (cdr sub-at-list) '()))
	      (cdr at-list))
	  '()))))

(define whole-annotation
  (lambda (node annotation-name)
    (let ((result (find (lambda (item) (eq? (car item) annotation-name))
			(annotations node))))
      (if result result '()))))

(define annotation
  (lambda (node annotation-name)
    (let ((whole (whole-annotation node annotation-name)))
      (if (null? whole) whole (cadr whole)))))

(define annotate!
  (lambda (node new-annotation)
    (let ((anntns (annotations node))
	  (cna (car new-annotation)))
      (if (null? anntns)
	  (let ((type (sdom:node-type node)))
	    (if (eqv? type sdom:node-type-element)
		(let ((attrs (single-at-finder node)))
		  (if (eq? attrs #f)
		      (append! node (list (list '@ (list '@ new-annotation))))
		      (let ((sub-attrs (single-at-finder attrs)))
			(if sub-attrs
			    (let ((x (find (lambda (y) (eq? (car y) cna))
					   (cdr sub-attrs))))
			      (if x 
				  (set-cdr! x (cdr new-annotation))
				  (append! sub-attrs `(,new-annotation))))
			    (append! attrs (list (list '@ new-annotation)))))))
		(set-cdr! (last-pair node) (list (list '@ new-annotation)))))
	  (let ((x (find (lambda (item) (eq? (car item) cna))
			 anntns)))
	    (if x (set-cdr! x (cdr new-annotation))
		(append! anntns (list new-annotation))))))))

;; This not only removes the annotation itself, but may also remove the entire
;; annotative node if there are no more annotations left in it.

(define remove-annotation!
  (lambda (node sym)
    (let ((type (sdom:node-type node))
	  (annt (find (lambda (x) (eq? (car x) sym)) (annotations node)))
	  (at-list (single-at-finder node)))
      (if (eqv? type sdom:node-type-element)
	  (for-each (lambda (item)
		      (begin (if (eq? (car item) '@)
				 (delq! annt item))
			     (if (= (length item) 1)
				 (delq! item at-list))))
		    (cdr at-list))
	  (if at-list (begin 
			(delq! annt at-list)
			(if (= (length at-list) 1)
			    (delq! at-list node))))))))

;;---------------------------------------------------------------------------;;
;;                                                                           ;;
;; Internal functions for manipulating namespaces                            ;;
;;                                                                           ;;
;;---------------------------------------------------------------------------;;

(define add-namespace
  (lambda (node prefix uri) 
    (if (null? prefix)
	(annotate! node (list 'sdom:default-namespace uri))
	(let* ((doc (if (eqv? (sdom:node-type node) sdom:node-type-document)
			node 
			(sdom:get-dom-property node "sdom:owner-document")))
	       (new-decl (if (sdom:get-dom-config-parameter 
			      doc "sdom:resolve-new-prefixes")
			     `(,(string->symbol uri) ,uri ,prefix)
			     `(,prefix ,uri ,prefix)))
	       (w (whole-annotation node '*NAMESPACES*)))
	  (if (null? w)
	      (annotate! node `(*NAMESPACES* ,new-decl))
	      (let ((n (lookup-prefix-at-node node prefix)))
		(if (null? n) 
		    (append! w `(,new-decl))
		    (begin (if (= (length n) 3) 
			       (set-car! (caddr n) prefix) 
			       (append! n `(,prefix)))
			   (set-car! (cadr n) uri)))))))))

(define remove-namespace-by-prefix
  (lambda (node prefix) ()))

(define remove-namespace-by-namespace
  (lambda (node uri) ()))

(define lookup-prefix-at-node
  (lambda (node prefix) 
    (let* ((w (whole-annotation node '*NAMESPACES*))
	   (doc (if (eqv? (sdom:node-type node) sdom:node-type-document)
		    node
		    (sdom:get-dom-property node "sdom:owner-document")))
	   (p (sdom:get-dom-config-parameter doc "sdom:prefer-orig-prefix"))
	   (pref-sym (string->symbol prefix)))
      (if (null? w) 
	  '() 
	  (let ((ns (find (if p 
			      (lambda (x) (if (= (length x) 3)
					      (eq? (caddr x) pref-sym)
					      (eq? (car x) pref-sym)))
			      (lambda (x) (eq? (car x) pref-sym)))
			  (cdr w))))
	    (if ns ns '()))))))

(define lookup-namespace-at-node
  (lambda (node uri) 
    (let ((w (whole-annotation node '*NAMESPACES*)))
      (if (null? w) 
	  '() 
	  (let ((ns (find (lambda (x) (equal? (cadr x) uri)) (cdr w))))
	    (if ns ns '()))))))

;;---------------------------------------------------------------------------;;
;;                                                                           ;;
;; DOM feature management functions                                          ;;
;;                                                                           ;;
;;---------------------------------------------------------------------------;;

(define sdom-feature-registry '(("Core" . "3.0") ("XML" . "1.0")))
(define sdom:register-feature! 
  (lambda (feature version)
      (if (not (sdom:has-feature? feature version)) 
	  (append! sdom-feature-registry `((,feature . ,version))))))
(define sdom:has-feature? 
  (lambda (feature version) 
    (let ((f (find (lambda (x) (equal? x `(,feature . ,version))) 
		   sdom-feature-registry))) 
      (if (not f) #f #t))))

;;---------------------------------------------------------------------------;;
;;                                                                           ;;
;; DOM configuration functions                                               ;;
;;                                                                           ;;
;;---------------------------------------------------------------------------;;

;; The values after the option name are required supported values; the first
;; is the default.

(define sdom:config-parameter-names
  `("canonical-form" "cdata-sections" "check-character-normalization"
    "comments" "datatype-normalization" "element-content-whitespace" "entities"
    "error-handler" "infoset" "namespaces" "namespace-declarations"
    "normalize-characters" "split-cdata-sections" "strict-error-checking"
    "validate" "validate-if-schema" "well-formed" "sdom:prefer-orig-prefix"
    "sdom:resolve-new-prefixes"))

(define sdom-config-defaults
  `(("canonical-form" #f ,(lambda (x) (not x)))
    ("cdata-sections" #t ,boolean?)
    ("check-character-normalization" #f ,(lambda (x) (not x)))
    ("comments" #t ,boolean?)
    ("datatype-normalization" #f ,(lambda (x) (not x)))
    ("element-content-whitespace" #t ,(lambda (x) (eq? #t x)))
    ("entities" #t ,boolean?)
    ("error-handler" ,default-dom-error-handler ,procedure?)
    ("infoset" #f ,boolean? 
     ,(lambda (d x) 
	(if x (begin (sdom:set-dom-config-parameter! d "validate-if-schema" #f)
		     (sdom:set-dom-config-parameter! d "entities" #f)
		     (sdom:set-dom-config-parameter! 
		      d "datatype-normalization" #f)
		     (sdom:set-dom-config-parameter! d "cdata-sections" #f)
		     (sdom:set-dom-config-parameter! 
		      d "namespace-declarations" #t)
		     (sdom:set-dom-config-parameter! d "well-formed" #t)
		     (sdom:set-dom-config-parameter! 
		      d "element-content-whitespace" #t)
		     (sdom:set-dom-config-parameter! d "comments" #t)
		     (sdom:set-dom-config-parameter! d "namespaces" #t))
	    (set-car! (cdr (find (lambda (x) 
				   (and (list? x) (equal? "infoset" (car x))))
				 (whole-annotation d '*CONFIG*)))
		      #t))))
    ("namespaces" #t ,(lambda (x) (eq? #t x)))
    ("namespace-declarations" #t ,boolean?)
    ("normalize-characters" #f ,(lambda (x) (not x)))
    ("split-cdata-sections" #t ,boolean?)
    ("strict-error-checking" #t ,boolean?)
    ("validate" #f ,(lambda (x) (not x)))
    ("validate-if-schema" #f ,(lambda (x) (not x)))
    ("well-formed" #t ,(lambda (x) (eq? #t x)))

    ("sdom:prefer-orig-prefix" #f ,boolean?)
    ("sdom:resolve-new-prefixes" #t ,boolean?)))

(define get-sdom-config-default 
  (lambda (str) 
    (let ((def (find (lambda (x) (equal? (car x) (string-downcase str)))
		     sdom-config-defaults)))
      (if (not def)
	  (throw 'sdom:exception sdom:exception-code-not-found-err)
	  (cadr def)))))

(define sdom:get-dom-config-parameter
  (lambda (doc str) 
    (if (not (eqv? (sdom:node-type doc) sdom:node-type-document))
	(throw 'sdom:exception sdom:exception-code-type-mismatch-err))
    (let* ((config (whole-annotation doc '*CONFIG*))
	   (match (find (lambda (x) 
			  (and (list? x) (equal? (car x) 
						 (string-downcase str))))
			config)))
      (if (not match) (get-sdom-config-default str) (cadr match)))))

(define internal-get-dom-config-entry
  (lambda (str)
    (find (lambda (x) (equal? (car x) (string-downcase str))) 
	  sdom-config-defaults)))

(define internal-check-dom-config-parameter
  (lambda (doc str val)
    (if (not (eqv? (sdom:node-type doc) sdom:node-type-document))
	(throw 'sdom:exception sdom:exception-code-type-mismatch-err))
    (let ((foo (internal-get-dom-config-entry str)))
      (if (not foo) 
	  (throw 'sdom:exception sdom:exception-code-not-found-err))
      (if (not (apply (caddr foo) val '()))
	  (throw 'sdom:exception sdom:exception-code-not-supported-err)))))

(define sdom:add-dom-config-parameter! 
  (lambda (str . vals) 
    (append! sdom-config-defaults `(,(append `(,str) vals)))))

(define sdom:set-dom-config-parameter!
  (lambda (doc str val)
    (internal-check-dom-config-parameter doc str val)
    (let* ((config (begin (if (null? (annotation doc '*CONFIG*))
			      (annotate! doc (list '*CONFIG*)))
			  (whole-annotation doc '*CONFIG*)))
	   (match (find (lambda (x) 
			  (and (list? x) 
			       (equal? (string-downcase str) (car x))))
			config)))
      (if match 
	  (set-car! (cdr match) val)
	  (append! config `((,(string-downcase str) ,val))))
      (let ((entry (internal-get-dom-config-entry str)))
	(if (= (length entry) 4)
	    (apply (cadddr entry) doc val '()))))))

(define sdom:can-set-dom-config-parameter?
  (lambda (doc str val)
    (false-if-exception (internal-check-dom-config-parameter doc str val))))

;;---------------------------------------------------------------------------;;
;;                                                                           ;;
;; Internal functions for managing entities and entity references            ;;
;;                                                                           ;;
;;---------------------------------------------------------------------------;;

(define dereference-entity-reference
  (lambda (node)
    (let ((a (ancestors node))
	  (f (lambda (x) 
	       (let ((g (lambda (y) 
			  (if (and (eqv? (sdom:node-type y)
					 sdom:node-type-entity)
				   (equal? (derive-name y)
					   (derive-name node)))
			      (extract-children y) #f))))
		 (if (find g (extract-children x)) #t #f)))))
      (find f (reverse a)))))

(define update-entity-reference!
  (lambda (node)
    (let ((vals (dereference-entity-reference node)))
      (for-each (lambda (x) (sdom:remove-child! node x)) 
		(extract-children node))
      (if vals (for-each (lambda (x) (sdom:append-child! node x)) vals)))))

;;---------------------------------------------------------------------------;;
;;                                                                           ;;
;; Internal functions for managing internal and external node                ;;
;; representations                                                           ;;
;;                                                                           ;;
;;---------------------------------------------------------------------------;;

;; The motivation for this pair of functions is that according to the SXML
;; grammar, character data, entities, and comments can't have annotations.
;; The solution is to store the annotations in the node's parent and just
;; attach them to the node when we extract it.

(define internal-rep-cache (make-weak-key-hash-table 16))

(define pos-symbol 
  (lambda (pos) (string->symbol (string-append "@-" (number->string pos)))))

(define ext->int
  (lambda (node)
    (let* ((type (sdom:node-type node))
	   (parent (sdom:get-dom-property node "sdom:parent-node"))
	   (oldrep (sdom:get-dom-property node "sdom:sxml-representation"))
	   (val (if (eqv? type sdom:node-type-comment)
		    oldrep
		    (sdom:get-dom-property node "sdom:node-value"))))
      (if (or (eqv? type sdom:node-type-text)
	      (eqv? type sdom:node-type-cdata-section)
	      (eqv? type sdom:node-type-comment)
	      (eqv? type sdom:node-type-entity))
	  (begin 
	    (if (not (null? parent))
		(let* ((rep `(sdom:sxml-representation ,oldrep))
		       (pos (letrec 
				((f (lambda (x count)
				      (let ((cx (car x))
					    (dx (cdr x)))
					(cond ((null? x) #f)
					      ((and (list? cx) 
						    (eq? (car cx) '@))
					       (f dx count))
					      ((eq? cx (cadr rep)) count)
					      (else (f dx (+ count 1))))))))
			      (f (cdr parent) 
				 (if (eqv? (sdom:node-type parent) 
					   sdom:node-type-attr) 0 1))))
		       (a-pos (list-index (lambda (x) (and (list? x) 
							   (eq? (car x) '@)))
					  parent))
		       (r-pos (+ (if (and a-pos (< a-pos pos)) (+ pos 1) pos)
				 (if (eqv? (sdom:node-type parent)
					   sdom:node-type-attr)
				     1 0))))
		  (if pos 
		      (begin 
			(annotate! parent 
				   (cons (pos-symbol pos)
					 `(,(append '(@) (annotations node)))))
			(cond ((or (eqv? type sdom:node-type-text)
				   (eqv? type sdom:node-type-cdata-section))
			       (list-set! parent r-pos val))))
		      (throw 'sdom:exception 
			     sdom:exception-code-not-found-err))))
	    (annotate! node `(sdom:sxml-representation ,val)))))))


(define int->ext
  (lambda (parent child-pos)
    (letrec ((annts (annotation parent (pos-symbol child-pos)))
	     (create-rep (lambda (x)
			   (append (if (list? x) x `(,x)) `(,annts))))
	     (f (lambda (item count)
		  (let ((ci (car item)))
		    (cond ((eq? item '()) '())
			  ((and (list? ci)
				(eq? (car ci) '@)) (f (cdr item) count))
			  ((eqv? count child-pos) 
			   (if (null? annts)
			       ci
			       (let ((cached-ref (hashq-ref internal-rep-cache 
							    ci)))
				 (if (not cached-ref)
				     (let ((c (create-rep ci)))
				       (hashq-set! internal-rep-cache ci c)
				       c)
				     cached-ref))))
			  (else (f (cdr item) (+ count 1))))))))
      (if (eqv? (sdom:node-type parent) sdom:node-type-attr)
	  (f (cddr parent) 1)
	  (f (cdr parent) 1)))))

;; Unless the events module is loaded, this is a no-op.

(define sdom:dispatch-event
  (lambda args
    (if (defined? 'sdom:dispatch-event-internal)
	(apply (module-ref (resolve-module '(sdom events))
			   'sdom:dispatch-event-internal) args)
	#f)))

(define sdom:dom-implementation-create-document-type
  (lambda (q-name public-id system-id) ()))

(define sdom:dom-implementation-get-feature
  (lambda (feature version) ()))

(define sdom:node-type
  (lambda (node) 
    (if (or (not (list? node)) (null? node))
	(throw 'sdom:exception sdom:exception-code-type-mismatch-err))
    (case (car node)
      ((*TOP*) sdom:node-type-document)
      ((*COMMENT*) sdom:node-type-comment)
      ((*ENTITY*) sdom:node-type-entity)
      ((*ENTITY-REF*) sdom:node-type-entity-reference)
      ((*FRAGMENT*) sdom:node-type-document-fragment)
      ((*DOCTYPE*) sdom:node-type-document-type)
      ((*PI*) sdom:node-type-processing-instruction)
      ((*NOTATION*) sdom:node-type-notation)
      ((@) sdom:node-type-attr) 
      (else (cond ((string? (car node))
		   (let ((annts (single-at-finder node)))
		     (if (or (not annts) 
			     (null? annts)
			     (not (find (lambda (x) 
					  (equal? x '(sdom:is-cdata #t)))
					(cdr annts))))
			 sdom:node-type-text
			 sdom:node-type-cdata-section)))
		  ((symbol? (car node)) sdom:node-type-element)
		  (else throw 'sdom:exception 
			sdom:exception-code-type-mismatch-err))))))

(define derive-name
  (lambda (node)
    (let ((type (sdom:node-type node)))
      (case type
	((2) (symbol->string (cadr node)))
	((4) "#cdata-section")
	((8) "#comment")
	((9) "#document")
	((11) "#document-fragment")
	((10) "???")
	((1) (symbol->string (car node)))
	((6) "???")
	((5) (symbol->string (cadr node)))
	((12) "???")
	((7) (string-copy (cadr node)))
	((3) "#text")))))

(define derive-value
  (lambda (node)
    (let ((type (sdom:node-type node)))
      (case type
	((2)
	 (if (sdom:get-dom-property node "sdom:specified")
	     (if (sdom:has-child-nodes? node)
		 (sdom:get-dom-property (sdom:get-dom-property 
					 node "sdom:first-child")
					"sdom:node-value") "")))
	
	;;		 '())) ;; HAVE TO HANDLE ENTITY-REFS, LOOKUPS, ETC. !!!
	((4) (car node))
	((3) (car node))
	((8) (cadr node))
	((7) (caddr node))
	(else '())))))

(define set-value!
  (lambda (node value)
    (let ((type (sdom:node-type node)))
      (if (not (equal? (sdom:get-dom-property node "sdom:node-value") value))
	  (begin
	    (cond ((eqv? type sdom:node-type-attr)
		   (let* ((d (sdom:get-dom-property node 
						    "sdom:owner-document"))
			  (t (sdom:create-node d sdom:node-type-text value))
			  (x (sdom:get-dom-property node "sdom:first-child")))
		     (if (null? x)
			 (sdom:append-child! node t) 
			 (sdom:replace-child! node t x))))
		  ((or (eqv? type sdom:node-type-cdata-section)
		       (eqv? type sdom:node-type-text))
		   (let ((old-value (sdom:get-dom-property 
				     node "sdom:node-value")))
		     (set-car! node value)
		     (ext->int node)
		     (sdom:dispatch-event
		      node 'sdom:event-dom-character-data-modified 
		      node old-value value "" 0)))
		  ((eqv? type sdom:node-type-comment) 
		   (set-car! (cdr node) value)
		   (ext->int node))
		  ((eqv? type sdom:node-type-processing-instruction) 
		   (let ((old-value (sdom:get-dom-property node
							   "sdom:node-value")))
		     (set-car! (cddr node) value)
		     (sdom:dispatch-event 
		      node 'sdom:event-dom-character-data-modified
		      node old-value value "" 0)))))))))

(define get-prefix 
  (lambda (str) 
    (let ((i (string-rindex str #\:))) 
      (if (and i (> i 0)) (substring str 0 i) '()))))

(define valid-qname-chars?
  (lambda (doc qname)
    (let ((resolve (sdom:get-dom-config-parameter 
		    doc "sdom:resolve-new-prefixes")))
      (regex-match? (if resolve qname-char-regex-extended qname-char-regex) 
		    qname))))

(define valid-namespace-combo? 
  (lambda (doc qname uri)
    (let ((resolve (sdom:get-dom-config-parameter 
		    doc "sdom:resolve-new-prefixes"))
	  (prefix (get-prefix qname)))
    (cond ((not (regex-match? (if resolve qname-regex-extended qname-regex)
			      qname)) #f)
	  ((and (not (null? prefix)) (null? uri)) #f)
	  ((and (or (equal? qname "xml")
		    (equal? prefix "xml"))
		(not (equal? uri xml-ns-uri))) #f)
	  ((and (or (equal? qname "xmlns")
		    (equal? prefix "xmlns"))
		(not (equal? uri xmlns-ns-uri))) #f)
	  ((and (equal? uri xmlns-ns-uri)
		(not (equal? qname "xmlns"))
		(not (equal? prefix "xmlns"))) #f)
	  (else #t)))))

(define get-local-name 
  (lambda (str) 
    (let ((i (string-rindex str #\:))) 
      (if (and i (not (eqv? i (- (string-length str) 1)))) 
	  (substring str (+ i 1)) 
	  str))))

(define set-prefix!
  (lambda (node p)
    (let ((type (sdom:node-type node))
	  (namespace-uri (sdom:get-dom-property node "sdom:namespace-uri"))
	  (name (sdom:get-dom-property node "sdom:local-name")))
      (if (or (null? namespace-uri)
	      (and (equal? p "xml")
		   (not (equal? namespace-uri xml-ns-uri)))
	      (and (eqv? type sdom:node-type-attr)
		   (or (and (equal? p "xmlns")
			    (not (equal? namespace-uri xmlns-ns-uri)))
		       (equal? (sdom:get-dom-property
				node "sdom:qualified-name")
			       "xmlns"))))
	  (throw 'sdom:exception sdom:exception-code-namespace-err)
	  (cond ((eqv? type sdom:node-type-attr)
		 (set-car! (cdr node) 
			   (string->symbol (string-append (p ":" name)))))
		((eqv? type sdom:node-type-element)
		 (set-car! (cdr node)
			   (string->symbol (string-append (p ":" name))))))))))

(define extract-attributes
  (lambda (node)
    (let ((type (sdom:node-type node)))
      (cond ((eqv? type sdom:node-type-element)
	     (let ((node-list (single-at-finder node)))
	       (if (not (eq? node-list #f))
		   (map (lambda (item) (cons '@ item))
			(filter (lambda (item) (not (eq? (car item) '@)))
				(cdr node-list)))
		   '())))
	    (else '())))))

(define extract-children
  (lambda (node)
    (let ((type (sdom:node-type node))
	  (parent-fn (lambda () node))
	  (not-annt-fn (lambda (item) 
			 (or (and (list? item) (not (eq? (car item) '@)))
			     (string? item)))))
      (cond ((or (eqv? type sdom:node-type-element)
		 (eqv? type sdom:node-type-attr))
	     (let ((counter 0)
		   (x (filter not-annt-fn (if (eqv? type sdom:node-type-attr)
					      (cddr node)
					      (cdr node)))))
	       (map (lambda (item) 
		      (set! counter (+ counter 1))
		      (int->ext node counter))
		    x)))
	    ((or (eqv? type sdom:node-type-document)
		 (eqv? type sdom:node-type-document-fragment))
	     (let ((counter 0))
	       (map (lambda (item)
		      (set! counter (+ counter 1))
		      (int->ext node counter))
		    (filter not-annt-fn (cdr node)))))
	    ((eqv? type sdom:node-type-attr)
	     (let ((child (caddr node))) 
	       (if (not (and (list? child) (eq? (car child) '@))) 
		   `(,(int->ext node 1))
		   '())))
	    (else '())))))

(define first-child 
  (lambda (node)
    (let ((child-list (extract-children node)))
      (if (not (null? child-list))
	  (car child-list)
	  '()))))

(define last-child
  (lambda (node)
    (let ((child-list (extract-children node)))
      (if (not (null? child-list))
	  (car (last-pair child-list))
	  '()))))

(define owner-document
  (lambda (node)
    (letrec 
	((type (sdom:node-type node))
	 (top-finder 
	  (lambda (item)
	    (if (null? item)
		'()
		(let ((owner (annotation item 'sdom:owner-document))
		      (item-type (sdom:node-type item)))
		  (cond ((not (null? owner)) (owner))
			((eqv? item-type sdom:node-type-document) item)
			((eqv? item-type sdom:node-type-attr)
			 (top-finder 
			  (sdom:get-dom-property item "sdom:owner-element")))
			((or (eqv? item-type sdom:node-type-element)
			     (eqv? item-type sdom:node-type-entity-reference)
			     (eqv? item-type sdom:node-type-document-type)
			     (eqv? item-type 
				   sdom:node-type-processing-instruction)
			     (eqv? item-type sdom:node-type-text)
			     (eqv? item-type sdom:node-type-cdata-section)
			     (eqv? item-type sdom:node-type-comment))
			 (top-finder 
			  (sdom:get-dom-property item "sdom:parent-node")))
			(else '())))))))
      (if (eqv? type sdom:node-type-document)
	  '()
	  (top-finder node)))))

(define sdom:dom-structure
  `(,sdom:node-type-node 
    (@ (sdom:read-only 
	,(lambda (x) 
	   (let ((r (find (lambda (y) 
			    (eq? (annotation y 'sdom:read-only) #t))
			  (ancestors x))))
	     (if r #t #f))))
       (sdom:node-type ,sdom:node-type)
       (sdom:node-name ,derive-name)
       (sdom:node-value ,derive-value ,set-value!)
       (sdom:parent-node ,(lambda (node)
			    (let ((p (annotation node 'sdom:parent-node)))
			      (if (procedure? p) (p) '()))))
       (sdom:child-nodes ,extract-children)
       (sdom:first-child ,(lambda (node) 
			       (let ((x (extract-children node)))
				 (if (not (null? x)) (car x) '()))))
       (sdom:last-child ,(lambda (node)
			      (let ((x (extract-children node)))
				(if (not (null? x)) 
				    (car (last-pair x))
				    '()))))
       (sdom:previous-sibling 
	  ,(lambda (node) 
	     (let* ((p (sdom:get-dom-property node "sdom:parent-node"))
		    (r (if (null? p) '() (reverse (extract-children p)))))
	       (letrec ((f (lambda (x)
			     (if (or (null? x) (sdom:same-node? (car x) node))
				 x (f (cdr x))))))
		 (if (not (null? p))
		     (let ((fr (f r)))
		       (if (> (length fr) 1) (cadr fr) '()))
		     '())))))
       (sdom:next-sibling
	,(lambda (node) 
	     (let* ((p (sdom:get-dom-property node "sdom:parent-node"))
		    (r (if (null? p) '() (extract-children p))))
	       (letrec ((f (lambda (x)
			     (if (or (null? x) (sdom:same-node? (car x) node))
				 x (f (cdr x))))))
		 (if (not (null? p))
		     (let ((fr (f r)))
		       (if (> (length fr) 1) (cadr fr) '()))
		     '())))))
       (sdom:attributes ,extract-attributes)
       (sdom:owner-document ,owner-document)
       (sdom:namespace-uri #f)
       (sdom:prefix ,(lambda (x) (get-prefix (derive-name x)))
		    ,set-prefix!)
       (sdom:local-name
	,(lambda (x) 
	   (if (not (null? (sdom:get-dom-property x "sdom:namespace-uri")))
	       (get-local-name (derive-name x)) '())))
       (sdom:base-uri
	,(lambda (x)
	   (let* ((type (sdom:node-type x)))
	     (cond ((memv type `(,sdom:node-type-element
				 ,sdom:node-type-processing-instruction))
		    (let ((attr (if (eqv? type sdom:node-type-element)
				    (sdom:get-attribute x "xml:base")
				    '())))
		      (if (null? attr)
			  (let ((p (sdom:get-dom-property 
				    x "sdom:parent-node")))
			    (if (null? p) 
				(sdom:get-dom-property 
				 (sdom:get-dom-property 
				  x "sdom:owner-document") 
				 "sdom:document-uri")
				(sdom:get-dom-property p "sdom:base-uri")))
			  attr)))
		   ((eqv? type sdom:node-type-document)
		    (sdom:get-dom-property x "sdom:document-uri"))
		   (else '())))))
       (sdom:text-content 
	,(lambda (x)
	   (let ((type (sdom:node-type x)))
	     (cond ((memv type `(,sdom:node-type-element
				 ,sdom:node-type-attr
				 ,sdom:node-type-entity
				 ,sdom:node-type-entity-reference
				 ,sdom:node-type-document-fragment))
		    (let ((nodes 
			   (filter 
			    (lambda (y)
			      (not 
			       (memv y
				`(,sdom:node-type-comment
				  ,sdom:node-type-processing-instruction))))
			    (extract-children x))))
		      (if (null? nodes) 
			  "" 
			  (apply string-append 
				 (map (lambda (y) 
					(sdom:get-dom-property 
					 y "sdom:text-content")) 
				      nodes)))))		      
		   ((memv type `(,sdom:node-type-text
				 ,sdom:node-type-cdata-section
				 ,sdom:node-type-comment
				 ,sdom:node-type-processing-instruction))
		    (derive-value x))
		   (else '()))))
	,(lambda (x y)
	   (let ((children (extract-children x))
		 (type (sdom:node-type x)))
	     (if (is-readonly? x) 
		 (throw 'sdom:exception 
			sdom:exception-code-no-modification-allowed-err))
	     (if (not (memv type `(,sdom:node-type-document
				   ,sdom:node-type-document-type
				   ,sdom:node-type-notation)))
		 (begin 
		   (for-each (lambda (z) (sdom:remove-child! x z)) children)
		   (if (not (null? y))
		       (if (memv type 
				 `(,sdom:node-type-text
				   ,sdom:node-type-cdata-section
				   ,sdom:node-type-comment
				   ,sdom:node-type-processing-instruction))
			   (set-value! x y)
			   (sdom:append-child! 
			    x (sdom:create-node (sdom:get-dom-property 
						 x "sdom:owner-document") 
						sdom:node-type-text y)))))))))
       (sdom:sxml-representation #f #f))
    (,sdom:node-type-character-data 
     (@ (sdom:data ,derive-value ,set-value!)
	(sdom:length ,(lambda (node) 
			   (let ((v (derive-value node)))
			     (if (not (null? v))
				 (string-length (derive-value node))
				 0)))))
     (,sdom:node-type-text 
      (@ (sdom:is-element-content-whitespace #f)
	 (sdom:whole-text 
	  ,(lambda (x) 
	     (let ((span (get-adjacent-text-nodes x)))
	       (letrec ((f (lambda (y)
			     (cond ((null? y) "")
				   ((memv (sdom:node-type (car y))
					  `(,sdom:node-type-text
					    ,sdom:node-type-cdata-section))
				    (string-append (derive-value (car y)) 
						   (f (cdr y))))
				   (else (f (cdr y)))))))
		 (f span))))))
      (,sdom:node-type-cdata-section))
     (,sdom:node-type-comment))
    (,sdom:node-type-notation (@ (sdom:public-id #f)
				 (sdom:system-id #f)))
    (,sdom:node-type-entity (@ (sdom:public-id #f)
			       (sdom:system-id #f)
			       (sdom:notation-name #f)
			       (sdom:input-encoding #f)
			       (sdom:xml-encoding #f)
			       (sdom:xml-version #f)))
    (,sdom:node-type-entity-reference)
    (,sdom:node-type-processing-instruction 
     (@ (sdom:target ,cadr)
	(sdom:data ,caddr
		   ,set-value!)))
    (,sdom:node-type-attr (@ (sdom:name ,derive-name)
			     (sdom:specified
				,(lambda (node)
				    (if (or (annotation node 'sdom:specified)
					    (find (lambda (item) 
						    (eqv? (sdom:node-type item)
							  sdom:node-type-text))
						  (extract-children node)))
					#t 
					#f)))
			     (sdom:value ,derive-value ,set-value!)
			     (sdom:owner-element
				,(lambda (x)
				    (let ((p (annotation x 
							 'sdom:owner-element)))
				      (if (procedure? p) (p) '()))))
			     (sdom:schema-type-info ())
			     (sdom:is-id
				,(lambda (x)
				    (not (null? (annotation 
						 x 'sdom:is-id)))))))
    (,sdom:node-type-element (@ (sdom:tag-name ,derive-name)
				(sdom:schema-type-info ())))
    (,sdom:node-type-document-type (@ (sdom:name #f)
				      (sdom:entities #f)
				      (sdom:notations #f)
				      (sdom:public-id #f)
				      (sdom:system-id #f)
				      (sdom:internal-subset #f)))
    (,sdom:node-type-document-fragment)
    (,sdom:node-type-document 
     (@ (sdom:doc-type #f)
	(sdom:implementation #f)
	(sdom:document-element
	   ,(lambda (x) 
	       (let ((r (find (lambda (y) 
				(eqv? (sdom:node-type y) 
				      sdom:node-type-element))
			      (extract-children x))))
		 (if r r '()))))
	(sdom:input-encoding #f)
	(sdom:xml-encoding #f)
	(sdom:xml-standalone #f #f)
	(sdom:xml-version #f #f)
	(sdom:document-uri #f #f)
	(sdom:dom-config #f)))))


(define get-property-info
  (lambda (node-type name)
    (letrec ((matching-path 
	      (lambda (subtree)
		(if (null? subtree)
		    #f
		    (let ((cs (car subtree)))
		      (cond
		       ((eqv? cs node-type)
			(let ((r (single-at-finder subtree)))
			  (if r (cdr r) '())))
		       ((and (list? cs)
			     (not (eq? (car cs) '@)))
			(let ((sub-result (matching-path cs)))
			  (if (not sub-result)
			      (matching-path (cdr subtree))
			      (append (let ((r (single-at-finder cs)))
					(if r (cdr r) '()))
				      sub-result))))
		       (else (matching-path (cdr subtree)))))))))
      (find (lambda (item) (eq? (car item) name))
	    (let ((r (matching-path `(,sdom:dom-structure)))) 
	      (if (not r) '() r))))))

(define sdom:get-dom-property
  (lambda (node name)
    (if (not (sdom:node? node))
	(throw 'sdom:exception sdom:exception-code-type-mismatch-err))
    (if (not (string? name))
	(throw 'sdom:exception sdom:exception-code-type-mismatch-err))
    (let ((prop-spec (get-property-info (sdom:node-type node) 
					(string->symbol name))))
      (if prop-spec (let ((f (cadr prop-spec))) 
		      (if f 
			  (apply f (list node)) 
			  (annotation node (string->symbol name))))
	  (throw 'sdom:exception sdom:exception-code-not-found-err)))))

(define sdom:set-dom-property!
  (lambda (node name value) 
    (let ((x (get-property-info (sdom:node-type node) (string->symbol name))))
      (cond ((not x)
	     (throw 'sdom:exception sdom:exception-code-not-found-err))
	    ((eqv? (length x) 2)
	     (throw 'sdom:exception 
		    sdom:exception-code-no-modification-allowed-err))
	    (else (let ((f (caddr x)))
		    (if f (apply f (list node value)) 
			(annotate! node `(,(string->symbol name) 
					  ,value)))))))))

(define ancestors
  (lambda (node)
    (letrec 
	((f (lambda (item)
	      (let ((type (sdom:node-type item)))
		(cons item
		      (cond ((or (eqv? type sdom:node-type-document)
				 (eqv? type 
				       sdom:node-type-document-fragment))
			     '())
			    ((eqv? type sdom:node-type-attr)
			     (let ((g (sdom:get-dom-property 
				       item "sdom:owner-element")))
			       (if (not (null? g)) (f g) '())))
			    (else
			     (let ((g (sdom:get-dom-property
				       item "sdom:parent-node")))
			       (if (not (null? g)) (f g) '())))))))))
      (cdr (f node)))))

(define allowed-child-types
  (lambda (node-type)
    (cond ((eqv? node-type sdom:node-type-attr)
	   `(,sdom:node-type-text
	     ,sdom:node-type-entity-reference))
	  ((or (eqv? node-type sdom:node-type-cdata-section)
	       (eqv? node-type sdom:node-type-comment)
	       (eqv? node-type sdom:node-type-document-type)
	       (eqv? node-type sdom:node-type-notation)
	       (eqv? node-type sdom:node-type-processing-instruction)) '())
	  ((eqv? node-type sdom:node-type-document)
	   `(,sdom:node-type-element
	     ,sdom:node-type-processing-instruction
	     ,sdom:node-type-comment
	     ,sdom:node-type-document-type))
	  ((or (eqv? node-type sdom:node-type-document-fragment)
	       (eqv? node-type sdom:node-type-element)
	       (eqv? node-type sdom:node-type-entity)
	       (eqv? node-type sdom:node-type-entity-reference))
	   `(,sdom:node-type-element
	     ,sdom:node-type-processing-instruction
	     ,sdom:node-type-comment
	     ,sdom:node-type-text
	     ,sdom:node-type-cdata-section
	     ,sdom:node-type-entity-reference))
	  (else (throw 'sdom:exception 
		       sdom:exception-code-type-mismatch-err)))))

(define type-allowed-as-child
  (lambda (parent-type child-type) 
    (not (eq? (memv child-type (allowed-child-types parent-type)) #f))))

(define remove-child!
  (lambda (node pos)
    (letrec ((type (sdom:node-type node))
	     (pos-sym (pos-symbol pos))
	     (f (lambda (item count)
		  (cond ((null? item) '())
			((and (list? (car item)) (eq? (caar item) '@)) 
			 (f (cdr item) count))
			((eqv? count pos) 
			 ;; (sdom:dispatch-event 
			 ;;  `(,(car item) sdom:event-dom-node-removed))
			 (delq! (car item) node))
			(else (f (cdr item) (+ count 1)))))))
      (f node (if (eqv? type sdom:node-type-attr) -1 0))
      (remove-annotation! node pos-sym)
      (for-each 
       (lambda (item) 
	 (let ((str (symbol->string (car item))))
	   (if (and (equal? (substring str 0 1) "@")
		    (>= (string->number (substring str 2)) pos))
	       (set-car! item (pos-symbol (- (string->number (substring str 2))
					     1))))))
       (annotations node)))))

;; pos starts at 0

(define insert-child!
  (lambda (node child pos)
    (let ((type (sdom:node-type node))
	  (new-type (sdom:node-type child))
	  (children (extract-children node))
	  (od1 (owner-document node))
	  (od2 (owner-document child)))
      (if (not (or (and (not (eqv? type sdom:node-type-document)) 
			(eq? od1 od2))
		   (and (eqv? type sdom:node-type-document)
			(eq? node od2)))) 
	  (throw 'sdom:exception sdom:exception-code-wrong-document-err))
      (if (or (not (type-allowed-as-child type new-type))
	      (or (eq? node child)
		  (not (eq? (memv child (ancestors node)) #f)))
	      (and (eqv? type sdom:node-type-document)
		   (eqv? new-type sdom:node-type-element)
		   (not (eq? children '()))))
	  (throw 'sdom:exception sdom:exception-code-hierarchy-request-err))

      (let ((old-parent (sdom:get-dom-property child "sdom:parent-node")))
	(if (not (null? old-parent))
	    (sdom:remove-child! old-parent child)
	    (remove-annotation! child 'sdom:owner-document)))
      
      (annotate! child `(sdom:parent-node ,(lambda () node)))

      ;; Here we need to actually insert the sxml representation of the node,
      ;; plus change the special annotations for any requisite nodes that
      ;; fall after the insertion.  First, push all numbered annotations up by
      ;; one.

      (for-each 
       (lambda (item) 
	 (let ((str (symbol->string (car item))))
	   (if (and (equal? (substring str 0 1) "@")
		    (>= (string->number (substring str 2)) pos))
	       (set-car! item (pos-symbol (+ (string->number (substring str 2))
					     1))))))
       (annotations node))

      (let* ((base-pos (cond ((eqv? type sdom:node-type-attr) (+ pos 1))
			     (else pos)))
	     (a-pos (list-index (lambda (x) (and (list? x) (eq? (car x) '@)))
				node))
	     (real-pos (if (and a-pos (< a-pos base-pos))
			   (+ base-pos 1) base-pos))

	     ;; This is how we decide whether or not the child needs to be
	     ;; represented by its sxml-representation.

	     (rep (let ((annt (annotation child 'sdom:sxml-representation)))
		    (if (not (null? annt)) annt child))))
	(if (< (length node) real-pos) 
	    (append! node (list rep))
	    (set-cdr! (list-tail node (- real-pos 1))
		      (let ((b (take-right (cdr node) 
					   (- (length node) real-pos))))
			(append (list rep) b))))))
    (ext->int child)))
      ;; (sdom:dispatch-event (child 'sdom:event-dom-node-inserted)))))

(define list-pos
  (lambda (lst item pred) 
    (letrec ((f (lambda (x y z)
		  (cond ((null? x) #f) 
			((pred (car x) y) z) 
			(else (f (cdr x) y (+ z 1)))))))
      (f lst item 1))))

(define sdom:insert-before!
  (lambda (node new-node . ref-node)
    (if (and (not (null? ref-node))
	     (not (null? (car ref-node))))
	(if (eqv? (sdom:node-type new-node) sdom:node-type-document-fragment)
	    (for-each (lambda (x) (sdom:insert-before! node x (car ref-node)))
		      (extract-children new-node))
	    (let ((pos (list-pos (extract-children node) (car ref-node) 
				 sdom:same-node?)))
	      (if pos 
		  (insert-child! node new-node pos)
		  (throw 'sdom:exception sdom:exception-code-not-found-err))))
	(if (eqv? (sdom:node-type new-node) sdom:node-type-document-fragment)
	    (for-each (lambda (x)
			(insert-child! node x 
				       (+ (length (extract-children node)) 1)))
		      (extract-children new-node))
	    (insert-child! node new-node 
			   (+ (length (extract-children node)) 1))))
    new-node))

(define sdom:insert-after!
  (lambda (node new-node . ref-node)
    (if (and (not (null? ref-node))
	     (not (null? (car ref-node))))
	(let ((pos (list-pos (extract-children node) (car ref-node) 
			     sdom:same-node?)))
	  (if pos
	      (insert-child! node new-node (+ pos 1))
	      (throw 'sdom:exception sdom:exception-code-not-found-err)))
	(insert-child! node new-node (+ (length (extract-children node)) 1)))))

(define sdom:remove-child!
  (lambda (node old-child)
    (let ((parent (sdom:get-dom-property old-child "sdom:parent-node")))
      (if (and (not (null? parent)) (sdom:same-node? node parent))
	  (begin (remove-child! node (list-pos (extract-children node) 
					       old-child 
					       sdom:same-node?))
		 (remove-annotation! old-child 'sdom:parent-node)
		 (if (eqv? (sdom:node-type node) sdom:node-type-document)
		     (annotate! old-child `(sdom:owner-document 
					    ,(lambda () node)))
		     (annotate! old-child `(sdom:owner-document
					    ,(lambda ()
					       (sdom:get-dom-property
						node "sdom:owner-document")))))
		 old-child)
	  (throw 'sdom:exception sdom:exception-code-not-found-err)))))

;; FOR MORE-THAN-ONE-STEP MODIFICATIONS, NEED TO CHECK TO SEE WHETHER BOTH
;; STEPS CAN COMPLETE BEFORE ACTUALLY PERFORMING MODIFICATIONS.  NO, YOU
;; HAVEN'T DONE THIS YET!

(define sdom:replace-child!
  (lambda (node new-child old-child)
    (if (sdom:same-node? node new-child)
	(throw 'sdom:exception sdom:exception-code-hierarchy-request-err))
    (let ((parentold (sdom:get-dom-property old-child "sdom:parent-node"))
	  (parentnew (sdom:get-dom-property new-child "sdom:parent-node")))
      (if (or (is-readonly? node) 
	      (if (null? parentold) #f (is-readonly? parentold)))
	  (throw 'sdom:exception 
		 sdom:exception-code-no-modification-allowed-err))
      (if (and (not (null? parentold)) (sdom:same-node? node parentold))
	  (let ((pos (list-pos (extract-children node) old-child 
			       sdom:same-node?)))
	    (remove-child! node pos)
	    (insert-child! node new-child pos)
	    (remove-annotation! old-child 'sdom:parent-node)
	    (if (eqv? (sdom:node-type node) sdom:node-type-document)
		(annotate! old-child `(sdom:owner-document ,(lambda () node)))
		(annotate! old-child `(sdom:owner-document
				       ,(lambda ()
					  (sdom:get-dom-property
					   node "sdom:owner-document")))))
	    old-child)
	  (throw 'sdom:exception sdom:exception-code-not-found-err)))))

(define sdom:append-child!
  (lambda (node new-child)
    (insert-child! node new-child (+ (length (extract-children node)) 1))))

;; I THINK THIS IS A STUPID PIECE OF THE API.  SHOULDN'T STORE THE DATA IN THE
;; TREE ITSELF, SINCE THE SPEC DOESN'T SAY IT'S A DOM PROPERTY.  EXTERNAL HASH
;; MAYBE?

(define handle-user-data-event
  (lambda (node op src dst)
    (let ((node-hash (hash-ref user-data-hash node)))
      (if node-hash
	  (hash-fold (lambda (key val foo) 
		       (if (and (pair? val) (procedure? (cdr val)))
			   (apply (cdr val) op key (car val) src dst '()))
		       '())
		     '()
		     node-hash)))))
    
(define sdom:set-user-data!
  (lambda (node key data . handler)
    (let ((node-hash (hash-ref user-data-hash node)))
      (if node-hash
	  (let ((oldval (hash-ref node-hash key (list))))
	    (hash-set! node-hash key (if (and (not (null? handler))
					      (procedure? (car handler)))
					 (cons data (car handler))
					 (cons data '())))
	    (if (null? oldval) oldval (car oldval)))
	  (let ((new-hash-table (make-hash-table initial-user-data-hash-size)))
	    (hash-set! user-data-hash node new-hash-table)
	    (hash-set! new-hash-table key (if (and (not (null? handler))
						   (procedure? (car handler)))
					      (cons data (car handler))
					      (cons data '())))
	    (list))))))

(define sdom:get-user-data
  (lambda (node key) 
    (let ((node-hash (hash-ref user-data-hash node)))
      (if node-hash 
	  (let ((data (hash-ref node-hash key)))
	    (if data (car data) '()))
	  '()))))

(define sdom:equal-node? 
  (lambda (node1 node2)
    (let ((node1-type (sdom:node-type node1))
	  (node2-type (sdom:node-type node2)))
      (letrec ((rec-equal? (lambda (list1 list2)
			    (cond ((xor (null? list1) (null? list2)) #f)
				  ((null? list1) #t)
				  ((sdom:equal-node? (car list1) (car list2))
				   (rec-equal? (cdr list1) (cdr list2)))
				  (else #f)))))
	(cond ((not (eqv? node1-type node2-type)) #f)
	      ((not (and (equal? (sdom:get-dom-property node1 "sdom:node-name")
				 (sdom:get-dom-property node2 
							"sdom:node-name"))
			 (equal? (sdom:get-dom-property node1 
							"sdom:local-name")
				 (sdom:get-dom-property node2 
							"sdom:local-name"))
			 (equal? (sdom:get-dom-property node1 
							"sdom:namespace-uri")
				 (sdom:get-dom-property node2
							"sdom:namespace-uri"))
			 (equal? (sdom:get-dom-property node1 "sdom:prefix")
				 (sdom:get-dom-property node2 "sdom:prefix"))
			 (equal? (sdom:get-dom-property node1 
							"sdom:node-value")
				 (sdom:get-dom-property 
				  node2 "sdom:node-value")))) #f)
	      ((not (rec-equal? (extract-attributes node1)
				(extract-attributes node2))) #f)
	      ((not (rec-equal? (extract-children node1)
				(extract-children node2))) #f)
	      (else #t))))))

(define sdom:has-child-nodes? 
  (lambda (node) (> (length (extract-children node)) 0)))

(define sdom:same-node?
  (lambda (node1 node2) 
    (let ((type1 (sdom:node-type node1))
	  (type2 (sdom:node-type node2))
	  (annotation1 (annotation node1 'sdom:sxml-representation))
	  (annotation2 (annotation node2 'sdom:sxml-representation)))
      (cond ((and (not (null? annotation1)) (eq? annotation1 annotation2)) #t)
	    ((eqv? type1 sdom:node-type-attr) (eq? (cdr node1) (cdr node2)))
	    (else (eq? node1 node2))))))
      
(define sdom:supported? (lambda (node feature version) ()))

(define get-adjacent-text-nodes
  (lambda (node)
    (letrec ((seekend 
	      (lambda (lst counter)
		(if (or (null? lst)
			(memv (sdom:node-type (car lst))
			      `(,sdom:node-type-element
				,sdom:node-type-comment
				,sdom:node-type-processing-instruction)))
		    counter
		    (seekend (cdr lst) (+ counter 1))))))
    (let* ((docorder 
	    (document-order (car (last-pair (cons node (ancestors node))))))
	   (ldocorder (length docorder))
	   (rdocorder (reverse docorder))
	   (fpos (list-index (lambda (x) (sdom:same-node? x node)) docorder))
	   (rpos (list-index (lambda (x) (sdom:same-node? x node)) rdocorder))
	   (start (- ldocorder (seekend (list-tail rdocorder rpos) rpos)))
	   (end (seekend (list-tail docorder fpos) fpos)))
      (list-head (list-tail docorder start) (- end start))))))

(define sdom:replace-whole-text! 
  (lambda (node txt)
    (if (not (eqv? (sdom:node-type node) sdom:node-type-text))
	(throw 'sdom:exception sdom:exception-code-type-mismatch-err))
    (let* ((span (get-adjacent-text-nodes node))
	   (cspan (if (equal? txt "") span (cdr span))))
      (for-each (lambda (x) 
		  (let* ((isattr (eqv? (sdom:node-type x) sdom:node-type-attr))
			 (parent (sdom:get-dom-property
				  x (if isattr
					"sdom:owner-element"
					"sdom:parent-node"))))
		    (if (not (null? parent))
			(if isattr
			    (sdom:remove-attribute-node! parent x)
			    (sdom:remove-child! parent x)))))
		cspan)
      (if (equal? txt "")
	  (list)
	  (begin (set-value! (car span) txt) (car span))))))

(define is-readonly?
  (lambda (node)
    (let ((type (sdom:node-type node)))
      (if (or (eqv? type sdom:node-type-entity)
	      (eqv? type sdom:node-type-entity-reference)
	      (eqv? type sdom:node-type-notation))
	  #t
	  (let ((parent (if (eqv? type sdom:node-type-attr)
			    (sdom:get-dom-property node "sdom:owner-element")
			    (sdom:get-dom-property node "sdom:parent-node"))))
	    (if (null? parent) #f (is-readonly? parent)))))))
  
(define internal-lookup-scoped-namespace
  (lambda (node)
    (letrec ((aef (lambda (x) 
		    (find (lambda (y) 
			    (eqv? (sdom:node-type y) 
				  sdom:node-type-element))
			  (ancestors x))))
	     (type (sdom:node-type node))
	     (ns-prop (sdom:get-dom-property node "sdom:namespace-uri"))
	     (f (lambda (n)
		  (let ((ns-decl (annotation n 'sdom:default-namespace)))
		    (if (not (null? ns-decl))
			ns-decl
			(let ((a (aef n)))
			  (if a (f a) '())))))))
      (if (null? ns-prop)
	  (cond ((eqv? type sdom:node-type-element) (f node))
		((eqv? type sdom:node-type-attr) 
		 (f (sdom:get-dom-property node "sdom:owner-element")))
		(else '()))
	  ns-prop))))

(define internal-ns-lookup
  (lambda (node str sym)
    (let ((type (sdom:node-type node))
	  (aef (lambda (x) 
		 (find (lambda (y) 
			 (eqv? (sdom:node-type y) sdom:node-type-element))
		       (ancestors x)))))
      (cond ((eqv? type sdom:node-type-element)
	     (let ((ns (sdom:get-dom-property node "sdom:namespace-uri"))
		   (prefix (sdom:get-dom-property node "sdom:prefix"))
		   (decls (whole-annotation node '*NAMESPACES*)))

	       ;; We're doing a prefix lookup.
	       
	       (cond ((eq? sym 'prefix)
		      (let ((f (lambda (elt uri orig)
				 (let ((eltns (sdom:get-dom-property 
					       elt "sdom:namespace-uri"))
				       (eltprefix (sdom:get-dom-property
						   elt "sdom:prefix")))
				   (if (and (not (null? eltns))
					    (equal? eltns uri)
					    (not (null? eltprefix))
					    (let ((r (internal-ns-lookup 
						      orig eltprefix 'ns)))
					      (and r (equal? r uri))))
				       eltprefix
				       (let ((ae (aef elt)))
					 (if ae 
					     (internal-ns-lookup 
					      ae eltprefix sym)
					     '())))))))
			(f node str node)))

		     ;; We're doing a namespaceURI lookup.

		     ((eq? sym 'ns)
		      (if (and (not (null? ns)) (equal? str prefix)) 
			  ns
			  (let ((decl (lookup-namespace-at-node node str)))
			    (if (null? decl)
				(let ((ae (aef node)))
				  (if ae (internal-ns-lookup ae str sym) #f))
				(cadr decl)))))
				
		     ;; We're doing a default namespace lookup.

		     ((eq? sym 'default)
		      (if (null? prefix)
			  (equal? str ns)
			  (let ((ae (aef node)))
			    (if ae (internal-ns-lookup ae str sym) #f)))))))
	    ((eqv? type sdom:node-type-document)	     
	     (let ((de (sdom:get-dom-property node "sdom:document-element")))
	       (if de (internal-ns-lookup de str sym) #f)))
	    ((or (eqv? type sdom:node-type-entity)
		 (eqv? type sdom:node-type-notation)
		 (eqv? type sdom:node-type-document-type)
		 (eqv? type sdom:node-type-document-fragment)) #f)
	    ((eqv? type sdom:node-type-attr)
	     (let ((p (sdom:get-dom-property node "sdom:owner-element")))
	       (if (not (null? p)) (internal-ns-lookup p str sym) #f)))
	    (else (let ((ae (aef node)))
		    (if ae (internal-ns-lookup ae str sym) #f)))))))

(define sdom:default-namespace?
  (lambda (node namespace-uri)
    (let ((ns (internal-ns-lookup node namespace-uri 'default)))
      (not (eq? ns #f)))))

;; I don't think we support DOM level 2 namespace-declarations as attributes

(define sdom:lookup-namespace-uri
  (lambda (node prefix)
    (if (not (sdom:node? node))
	(throw 'sdom:exception sdom:exception-code-type-mismatch-err))
    (if (not (string? prefix))
	(throw 'sdom:exception sdom:exception-code-type-mismatch-err))
    (let ((ns (internal-ns-lookup node prefix 'ns)))
      (if ns ns '()))))

(define sdom:lookup-prefix
  (lambda (node namespace-uri)
    (let ((prefix (internal-ns-lookup node namespace-uri 'prefix)))
      (if prefix prefix '()))))

(define clone-node
  (lambda (node deep)
    (let* ((type (sdom:node-type node))
	   (doc (sdom:get-dom-property node "sdom:owner-document"))
	   (val (sdom:get-dom-property node "sdom:node-value"))
	   (name (sdom:get-dom-property node "sdom:node-name"))
	   (new-node (cond ((eqv? type sdom:node-type-element)
			    (let ((x (sdom:create-node 
				      doc type name)))
			      (for-each (lambda (y) 
					  (let ((z (clone-node y #t)))
					    (sdom:set-attribute-node! x z)))
					(extract-attributes node))
			      (if deep 
				  (for-each (lambda (y) 
					      (let ((z (clone-node y #t)))
						(sdom:append-child! x z)))
					    (extract-children node)))
			      x))
			   ((or (eqv? type sdom:node-type-cdata-section)
				(eqv? type sdom:node-type-text))
			    (sdom:create-node 
			     doc type (string-copy val)))
			   ((eqv? type sdom:node-type-attr)
			    (let ((new-node (sdom:create-node 
					     doc type name)))
			      (if (sdom:has-child-nodes? node)
				  (let ((x (clone-node (first-child node) #t)))
				    (sdom:append-child! new-node x)))
			      new-node))))
	   (ns (sdom:get-dom-property node "sdom:namespace-uri")))
      (annotate! new-node `(sdom:owner-document ,(lambda () doc)))
      (if (not (null? ns))
	  (annotate! new-node `(sdom:namespace-uri ,ns)))
      new-node)))

(define sdom:clone-node
  (lambda (node deep)
    (let ((new-node (clone-node node deep)))
      (handle-user-data-event node sdom:user-data-event-node-cloned node 
			      new-node)
      new-node)))
    
;;---------------------------------------------------------------------------;;
;;                                                                           ;;
;; Normalization functions for nodes and documents                           ;;
;;                                                                           ;;
;;---------------------------------------------------------------------------;;

(define internal-document-normalize!
  (lambda (doc w x)
    (if (not (null? x))
	(let* ((node (car x))
	       (type (sdom:node-type node)))
	  (cond ((eqv? type sdom:node-type-entity-reference)
		 (let ((c (dereference-entity-reference node)))
		   (if c 
		       (begin 
			 (update-entity-reference! node)
			 (if (not (sdom:get-dom-config-paramete 
				   doc "entities"))
			     (begin
			       (sdom:remove-child! w (car x))
			       (for-each (lambda (y) (sdom:append-child! w y))
					 c)))
			 (internal-document-normalize! 
			  doc w (extract-children w)))
		       (internal-document-normalize! doc w (cdr x)))))
		((eqv? type sdom:node-type-cdata-section)
		 (let ((val (sdom:get-dom-property (car x) "sdom:node-value")))
		   (if (sdom:get-dom-config-parameter doc "cdata-sections")
		       (let ((pos (string-contains val "]]>")))
			 (if pos 
			     (if (sdom:get-dom-config-parameter 
				  doc "split-cdata-sections")
				 (let ((pre (substring val 0 pos))
				       (post (substring val (+ pos 3))))
				   (begin
				     (if (> (string-length pre) 0)
					 (begin
					   (sdom:set-dom-property! 
					    (car x) "sdom:node-value" pre)
					   (if (> (string-length post) 0)
					       (sdom:insert-before! 
						w (sdom:create-node 
						   doc 
						   sdom:node-type-cdata-section
						   post) (if (null? (cdr x))
							     '()
							     (cadr x)))))
					 (if (> (string-length post) 0)
					     (sdom:set-dom-property!
					      (car x) "sdom:node-value" post)
					     (sdom:remove-child! w (car x))))
				     
				     (if (sdom:signal-error
					  doc sdom:error-severity-warning
					  "splitting cdata section" 
					  "cdata-sections-splitted" '() val 
					  '())
					 (internal-document-normalize! 
					  doc w (extract-children w)))))
				 (sdom:signal-error 
				  doc sdom:error-severity-error 
				  "unrepresentable character data"
				  "character data" '() val '()))
			     (internal-document-normalize! 
			      doc w (cdr x))))
		       (begin 
			 (sdom:replace-child! 
			  w (sdom:create-node doc sdom:node-type-text val) 
			  (car x))
			 (internal-document-normalize! 
			  doc w (extract-children w))))))

		((and (eqv? type sdom:node-type-comment)
		      (not (sdom:get-dom-config-parameter doc "comments")))
		 (sdom:remove-child! w (car x)) 
		 (internal-document-normalize! doc w (extract-children w)))
		(else (internal-document-normalize! doc w (cdr x))))))))

(define internal-normalize-node!
  (lambda (w x)
    (if (not (null? x))
	(begin 
	  (if (and (eqv? (sdom:node-type (car x)) sdom:node-type-text)
		   (not (null? (cdr x)))
		   (eqv? (sdom:node-type (cadr x)) sdom:node-type-text))
	      (begin
		(sdom:set-dom-property! 
		 (car x) "sdom:node-value"
		 (string-append 
		  (sdom:get-dom-property (car x) "sdom:node-value")
		  (sdom:get-dom-property (cadr x) "sdom:node-value")))
		(sdom:remove-child! w (cadr x))
		(internal-normalize-node! w (extract-children w))))
	  (internal-normalize-node! w (cdr x))))))

(define internal-normalize!
  (lambda (node doc)
    (if (not (null? doc)) 
	(internal-document-normalize! doc node (extract-children node)))
    (internal-normalize-node! node (extract-children node))
    (for-each (lambda (x) (internal-normalize! x doc)) 
	      (extract-children node))))
	     
(define sdom:normalize!
  (lambda (node) (internal-normalize! node '())))

(define sdom:normalize-document!
  (lambda (node) 
    (if (not (eqv? (sdom:node-type node) sdom:node-type-document))
	(throw 'sdom:exception sdom:exception-code-type-mismatch-err)
	(internal-normalize! node node))))

(define sdom:compare-document-position
  (lambda (node1 node2)
    (let* ((type1 (sdom:node-type node1))
	   (type2 (sdom:node-type node2))
	   (ownerdoc1 (if (eqv? type1 sdom:node-type-document)
			  node1 (sdom:get-dom-property 
				 node1 "sdom:owner-document")))
	   (ownerdoc2 (if (eqv? type2 sdom:node-type-document)
			  node2 (sdom:get-dom-property 
				 node2 "sdom:owner-document")))
	   (ancestors1 (ancestors node1))
	   (ancestors2 (ancestors node2)))
      (cond ((eq? node1 node2) 0)
	    ((or (not (eq? ownerdoc1 ownerdoc2)) ;; Different owners?
		 (and (not (and (eqv? type1 sdom:node-type-document)
				(eqv? type2 sdom:node-type-document)))
		      (let ((lpa1 (last-pair ancestors1))
			    (lpa2 (last-pair ancestors2)))
			(or (and (null? lpa1) (null? lpa2))
			    (and (not (and (null? lpa1) 
					   (sdom:same-node? 
					    node1 (car lpa2))))
				 (not (and (null? lpa2) 
					   (sdom:same-node? 
					    node2 (car lpa1))))
				 (not (sdom:same-node? (car lpa1) 
						       (car lpa2))))))))
	     (logior (if (> (hashq node1 most-positive-fixnum)
			    (hashq node2 most-positive-fixnum))
			 sdom:document-position-following
			 sdom:document-position-preceding)
		     sdom:document-position-disconnected
		     sdom:document-position-implementation-specific))
	    ((sdom:same-node? node1 node2) 0)
	    ((find (lambda (x) (sdom:same-node? x node1)) ancestors2)
	     (logior sdom:document-position-contained-by
		     sdom:document-position-following))
	    ((find (lambda (x) (sdom:same-node? x node2)) ancestors1)
	     (logior sdom:document-position-contains
		     sdom:document-position-preceding))
	    (else (let* ((commonroot (find (lambda (x) 
					     (find (lambda (y)
						     (sdom:same-node? x y)) 
						   ancestors2))
					   ancestors1))
			 (rootlist1
			  (reverse (take-while (lambda (x) 
						 (not (sdom:same-node? 
						       x commonroot))) 
					       ancestors1)))
			 (rootlist1 (if (null? rootlist1) 
					node1 
					(car rootlist1)))
			 (rootlist2 
			  (reverse (take-while (lambda (x)
						 (not (sdom:same-node?
						       x commonroot)))
					       ancestors2)))
			 (rootlist2 (if (null? rootlist2) 
					node2 
					(car rootlist2)))
			 (typer1 (eqv? (sdom:node-type rootlist1)
				       sdom:node-type-attr))
			 (typer2 (eqv? (sdom:node-type rootlist2)
				       sdom:node-type-attr)))
		    (if (eqv? typer1 typer2)
			(if typer1
			    (logior 
			     32 (let ((attrs (extract-attributes commonroot)))
				  (if (> (list-index (lambda (x)
						       (sdom:same-node? 
							x rootlist1))
						     attrs)
					 (list-index (lambda (x) 
						       (sdom:same-node? 
							x rootlist2))
						     attrs))
				      sdom:document-position-following
				      sdom:document-position-preceding)))
			    (let ((children (extract-children commonroot)))
			      (if (< (list-index (lambda (x)
						   (sdom:same-node? 
						    x rootlist1))
						 children)
				     (list-index (lambda (x)
						   (sdom:same-node?
						    x rootlist2))
						 children))
				  sdom:document-position-following
				  sdom:document-position-preceding)))
			(if typer1
			    sdom:document-position-following
			    sdom:document-position-preceding))))))))

(define sdom:create-node
  (lambda (document type . args)
    (let ((newnode 
	   (cond ((eqv? type sdom:node-type-attr)
		  (list '@ (string->symbol (car args))))
		 ((eqv? type sdom:node-type-cdata-section)
		  (list (string-copy (car args)) 
			(list '@ (list 'sdom:is-cdata #t)))) 
		 ((eqv? type sdom:node-type-comment)
		  (list '*COMMENT* (car args)))
		 ((eqv? type sdom:node-type-document-fragment)
		  (list '*FRAGMENT*))
		 ((eqv? type sdom:node-type-element)
		  (list (string->symbol (car args))))
		 ((eqv? type sdom:node-type-entity) (list '*ENTITY))
		 ((eqv? type sdom:node-type-entity-reference)
		  (list '*ENTITY-REF* (string->symbol (car args))))
		 ((eqv? type sdom:node-type-processing-instruction)
		  (list '*PI* (car args) (cadr args)))		 
		 ((eqv? type sdom:node-type-text)
		  (list (string-copy (car args))))
		 (else (throw 'sdom:exception 
			      sdom:exception-code-type-mismatch-err)))))
      (cond ((and (= (length args) 2)
		  (or (eqv? type sdom:node-type-attr)
		      (eqv? type sdom:node-type-element)))

	     ;; Need to check the qname / namespace combo here!!!

	     (annotate! newnode `(sdom:namespace-uri ,(cadr args))))
	    ((eqv? type sdom:node-type-comment)
	     (annotate! newnode 
			`(sdom:sxml-representation ,(list-copy newnode))))
	    ((or (eqv? type sdom:node-type-text)
		 (eqv? type sdom:node-type-cdata-section))
	     (annotate! newnode `(sdom:sxml-representation 
				  ,(string-copy (car args))))))
      (if (not (eqv? type sdom:node-type-document-type))
	  (annotate! newnode `(sdom:owner-document ,(lambda () document))))
      newnode)))

(define sdom:create-document
  (lambda (root-name doctype . namespace-uri)
    (letrec ((newdoc (list '*TOP* 
			   (list (string->symbol root-name)
				 (list '@ 
				  (list '@ (list 'sdom:parent-node 
						 (lambda () newdoc))))))))
      (if (not (null? doctype))
	  (begin
	    (if (not (eqv? (sdom:node-type doctype) 
			   sdom:node-type-document-type))
		(throw 'sdom:exception sdom:exception-code-type-mismatch-err))
	    (if (not (null? (sdom:get-dom-property doctype 
						   "sdom:owner-document")))
		(throw 'sdom:exception sdom:exception-code-wrong-document-err))
	    (annotate! doctype `(sdom:owner-document `(lambda () ,newdoc)))
	    (annotate! newdoc `(sdom:doctype ,doctype))))
      (if (and (not (null? namespace-uri))
	       (not (null? (car namespace-uri))))
	  (annotate! (sdom:get-dom-property newdoc "sdom:document-element")
		     `(sdom:namespace-uri ,(car namespace-uri))))
      newdoc)))

(define sdom:create-document-type
  (lambda (qname public-id system-id)
    (let ((new-dtd `(*DOCTYPE* ,qname)))
      (if (not (null? public-id))
	  (annotate! new-dtd `(sdom:public-id ,public-id)))
      (if (not (null? system-id))
	  (annotate! new-dtd `(sdom:system-id ,system-id)))
      new-dtd)))

;; I *think* this is correct -- DOM core isn't formally precise on the
;; definition of document order for non-element/attr nodes             - julian

(define document-order
  (lambda (start)
    (if (not (null? start))
	(let ((type (sdom:node-type start))
	      (f (lambda (x y) (append (document-order x) y))))
	  (cond ((eqv? type sdom:node-type-document)
		 (document-order (sdom:get-dom-property 
				  start "sdom:document-element")))
		((eqv? type sdom:node-type-element)
		 (cons start (fold-right 
			      f '() (append (extract-attributes start)
					    (extract-children start)))))
		((eqv? type sdom:node-type-attr)
		 (cons start (fold-right 
			      f '() (extract-children start))))
		((or (eqv? type sdom:node-type-document-fragment)
		     (eqv? type sdom:node-type-entity)
		     (eqv? type sdom:node-type-entity-reference))
		 (fold-right 
		  f '() (extract-children start)))
		(else `(,start))))
	'())))

(define sdom:get-elements-by-tag-name
  (lambda (doc name . args)
    (let ((type (sdom:node-type doc)))
      (if (not (eqv? type sdom:node-type-document))
	  (throw 'sdom:exception sdom:exception-code-type-mismatch-err))
      (filter (lambda (x) 
		(and (eqv? (sdom:node-type x) sdom:node-type-element)
		     (or (equal? name "*")
			 (equal? (get-local-name (symbol->string (car x))) 
				 name))
		     (if (not (null? args))
			 (let ((ns (car args)))
			   (if (not (string? ns))
			       (throw 'sdom:exception 
				      sdom:exception-code-type-mismatch-err))
			   (or (equal? ns "*")
			       (let ((y (sdom:lookup-prefix 
					 x (get-prefix (car x)))))
				 (if (null? y) #f (equal? ns y)))))
			 #t)))
	      (document-order doc)))))

;; NOT DONE, OBVIOUSLY

(define sdom:import-node
  (lambda (doc node deep)
    (let* ((type (sdom:node-type node))
	   (new-node (cond ((memv type `(,sdom:node-type-document 
					 ,sdom:node-type-document-type))
			    (throw 'sdom:exception 
				   sdom:exception-code-not-supported-err))
			   ((eqv? type sdom:node-type-notation))
			   ((memv type `(,sdom:node-type-attr 
					 ,sdom:node-type-element))
			    (clone-node node deep)))))
      (annotate! new-node `(sdom:owner-document ,(lambda () doc)))
      (handle-user-data-event 
       node sdom:user-data-event-node-imported node new-node)
      new-node)))

;; INCOMPLETE -- NEED TO ADD NEW NAMESPACE DECL IF NECESSARY!

(define sdom:adopt-node!
  (lambda (doc node)
    (if (not (eqv? (sdom:node-type doc) sdom:node-type-document))
	(throw 'sdom:exception sdom:exception-code-type-mismatch-err))
    (let ((type (sdom:node-type node)))
      (if (memv type `(,sdom:node-type-document
		       ,sdom:node-type-document-type))
	  (throw 'sdom:exception sdom:exception-code-not-supported-err))
      (if (memv type `(,sdom:node-type-notation
		       ,sdom:node-type-entity))
	  (throw 'sdom:exception sdom:exception-code-type-mismatch-err))
      (if (sdom:get-dom-property node "sdom:read-only")
	  (throw 'sdom:exception 
		 sdom:exception-code-no-modification-allowed-err))

      (if (eqv? type sdom:node-type-attr)
	  (begin 
	    (remove-annotation! node 'sdom:owner-element)
	    (if (not (sdom:get-dom-property node "sdom:specified"))
		(begin
		  (annotate! node `(sdom:value 
				    ,(sdom:get-dom-property node 
							    "sdom:value")))
		  (annotate! node '(sdom:specified #t))))))
      (let ((parent (sdom:get-dom-property node "sdom:parent-node")))
	(if (not (null? parent)) (sdom:remove-child! parent node))
	(annotate! node `(sdom:owner-document ,(lambda () doc)))))
    (handle-user-data-event node sdom:user-data-event-node-adopted node '())
    node))

;; This needs to handle user data events and real event handlers 
;; (DOESN'T, YET) -- for all intents and purposes, this is a new node.  
;; The difficulty is that for elements, the name is the head of the list, which
;; we can't modify.

(define sdom:rename-node!
  (lambda (node qname ns)
    (let ((type (sdom:node-type node))
	  (old-name (string-copy (derive-name node)))
	  (old-ns (let ((x (sdom:get-dom-property node "sdom:namespace-uri")))
		    (if (not (null? x)) (string-copy x) '())))
	  (doc (sdom:get-dom-property node "sdom:owner-document")))
      (if (not (memv type `(,sdom:node-type-attr ,sdom:node-type-element)))
	  (throw 'sdom:exception sdom:exception-code-not-supported-err))
      (if (not (valid-qname-chars? doc qname))
	  (throw 'sdom:exception sdom:exception-code-invalid-character-err))
      (if (not (valid-namespace-combo? doc qname ns))
	  (throw 'sdom:exception sdom:exception-code-namespace-err))
      (if (or (null? ns) (string-null? ns))
	  (remove-annotation! node 'sdom:namespace-uri)
	  (annotate! node (list 'sdom:namespace-uri ns)))
      (if (eqv? type sdom:node-type-element)
	  (begin

	    ;; Need to update any user-data hash keys that might be using this
	    ;; node as a key. This is the only instance in which we need to do
	    ;; this, since the ptr at the head of the list is changing...

	    (let ((oldtable (hash-ref user-data-hash node)))
	      (if oldtable (hash-remove! user-data-hash node))
	      (set-car! node (string->symbol qname))
	      (if oldtable (hash-set! user-data-hash node oldtable)))
	    (sdom:dispatch-event node 'sdom:event-dom-element-name-changed node
				 '() '() '() '() old-name old-ns))
	  (let ((parent (sdom:get-dom-property node "sdom:parent-node")))
	    (set-car! (cdr node) (string->symbol qname))
	    (if (not (null? parent))
		(sdom:dispatch-event 
		 parent 'sdom:event-dom-attribute-name-changed node '() '() 
		 old-name '() old-name old-ns))))
      (handle-user-data-event 
       node sdom:user-data-event-node-renamed node '())
      node)))
		   
;; Here are some attribute-mangling functions -- the ones that don't deal with
;; nodes explicitly will call into the ones that do.

;;---------------------------------------------------------------------------;;
;;                                                                           ;;
;; Attribute management functions                                            ;;
;;                                                                           ;;
;;---------------------------------------------------------------------------;;

(define sdom:get-attribute-node
  (lambda (elt name . namespace-uri) 
    (if (not (eqv? (sdom:node-type elt) sdom:node-type-element))
	(throw 'sdom:exception sdom:exception-code-type-mismatch-err))
    (let* ((finder (if (null? namespace-uri) 
		       (lambda (x) (equal? name (sdom:get-dom-property 
						 x "sdom:name")))
		       (lambda (x) (and (equal? name 
						(sdom:get-dom-property 
						 x "sdom:local-name"))
					(equal? (sdom:get-dom-property 
						 x "sdom:namespace-uri")
						(car namespace-uri))))))
	   (match (find finder (extract-attributes elt))))
      (if match match '()))))

(define sdom:set-attribute-node-internal!
  (lambda (elt node)
    (let ((x (find (lambda (y) (equal? (sdom:get-dom-property node "sdom:name")
				       (sdom:get-dom-property y "sdom:name")))
		   (extract-attributes elt))))
      (annotate! node `(sdom:owner-element ,(lambda () elt)))
      (remove-annotation! node 'sdom:owner-document)
      (if x (delq! x (single-at-finder elt)))
      (append! (single-at-finder elt) `(,(cdr node))))))

(define sdom:set-attribute-node!
  (lambda (elt node) 
    (if (not (and (eqv? (sdom:node-type elt) sdom:node-type-element)
		  (eqv? (sdom:node-type node) sdom:node-type-attr)))
	(throw 'sdom:exception sdom:exception-code-type-mismatch-err))
    (if (not (eq? (sdom:get-dom-property elt "sdom:owner-document")
		  (sdom:get-dom-property node "sdom:owner-document")))
	(throw 'sdom:exception sdom:exception-code-wrong-document-err))
    (if (not (null? (sdom:get-dom-property node "sdom:owner-element")))
	(throw 'sdom:exception sdom:exception-code-inuse-attribute-err))

    (let* ((name (sdom:get-dom-property node "sdom:name"))
	   (old-node (sdom:get-attribute-node elt name))
	   (old-value (sdom:get-attribute elt name))
	   (new-value (sdom:get-dom-property node "sdom:value")))
      (if (not (null? old-value))
	  (sdom:dispatch-event elt 'sdom:event-dom-attr-modified 
			       old-node old-value old-value name 3))
      (sdom:set-attribute-node-internal! elt node)
      (sdom:dispatch-event elt 'sdom:event-dom-attr-modified 
			   node new-value new-value name 2))))

(define sdom:remove-attribute-node!
  (lambda (elt node) 
    (if (not (and (eqv? (sdom:node-type elt) sdom:node-type-element)
		  (eqv? (sdom:node-type node) sdom:node-type-attr)))
	(throw 'sdom:exception sdom:exception-code-type-mismatch-err))
    (if (not (eq? elt (sdom:get-dom-property node "sdom:owner-element")))
	(throw 'sdom:exception sdom:exception-code-not-found-err))
    (delq! (cdr node) (single-at-finder elt))
    (let ((doc (sdom:get-dom-property node "sdom:owner-document")))
      (annotate! node `(sdom:owner-document ,(lambda () doc))))
    (remove-annotation! node 'sdom:owner-element)))

(define sdom:get-attribute
  (lambda (elt name . namespace-uri)
    (let ((node (if (not (null? namespace-uri))
		    (sdom:get-attribute-node elt name (car namespace-uri))
		    (sdom:get-attribute-node elt name))))
      (if (null? node) '() (derive-value node)))))

(define sdom:set-attribute!
  (lambda (elt name value . namespace-uri)
    (if (not (and (sdom:node? elt)
		  (eqv? (sdom:node-type elt) sdom:node-type-element)))
	(throw 'sdom:exceptiom sdom:exception-code-type-mismatch-err))
    (let ((ns (if (not (null? namespace-uri)) (car namespace-uri) #f))
	  (doc (sdom:get-dom-property elt "sdom:owner-document")))
      (if (not (valid-namespace-combo? doc name (if ns ns '())))
	  (throw 'sdom:exception sdom:exception-code-namespace-err))
      (let ((attr (if ns 
		      (sdom:get-attribute-node elt name ns)
		      (sdom:get-attribute-node elt name))))
	(if (null? attr)
	    (let* ((attr (if ns
			     (sdom:create-node doc sdom:node-type-attr name ns)
			     (sdom:create-node doc sdom:node-type-attr name))))
	      (sdom:set-dom-property! attr "sdom:value" value)
	      (sdom:set-attribute-node-internal! elt attr)
	      (sdom:dispatch-event elt 'sdom:event-dom-attr-modified 
				   attr value value name 1))

	    (let ((old-value (sdom:get-dom-property attr "sdom:value")))
	      (sdom:set-dom-property! attr "sdom:value" value)
	      (sdom:dispatch-event elt 'sdom:event-dom-attr-modified 
				   attr old-value value name 2)))))))

(define sdom:remove-attribute!
  (lambda (elt name . namespace-uri) 
    (let ((attr (if (not (null? namespace-uri))
		    (sdom:get-attribute-node elt name (car namespace-uri))
		    (sdom:get-attribute-node elt name))))
      (if (not (null? attr)) (sdom:remove-attribute-node! elt attr)))))

(define sdom:get-element-by-id
  (lambda (doc id)
    (let ((pred (lambda (x) 
		  (and (eqv? (sdom:node-type x)
			     sdom:node-type-element)
		       (find (lambda (y) 
			       (and (sdom:get-dom-property y "sdom:is-id")
				    (equal? id (derive-value y))))
			     (extract-attributes x))))))
      (if (not (eqv? (sdom:node-type doc) sdom:node-type-document))
	  (throw 'sdom:exception sdom:exception-code-type-mismatch-err))
      (find pred (document-order doc)))))

(define sdom:set-id-attribute!
  (lambda (node name is-id . args)
    (let ((attr (if (null? args)
		    (sdom:get-attribute-node node name)
		    (sdom:get-attribute-node node name (car args)))))
      (if (null? attr)
	  (throw 'sdom:exception sdom:exception-code-not-found-err))
      (if is-id 
	  (annotate! attr '(sdom:is-id #t)) 
	  (remove-annotation! attr 'sdom:is-id)))))

(define sdom:set-id-attribute-node!
  (lambda (attr is-id)
    (if (not (eqv? (sdom:node-type attr) sdom:node-type-attr))
	(throw 'sdom:exception sdom:exception-code-type-mismatch-err))
    (if is-id
	(annotate! attr '(sdom:is-id #t))
	(remove-annotation! attr 'sdom:is-id))))

(define sdom:sxml->sdom
  (lambda (sxml-tree)
    (letrec 
	((found-url-prefix #f)
	 (sdom-tree (copy-tree sxml-tree))	 
	 (tag-sibs!
	  (lambda (node-head parent-node pos)
	    (if (string? node-head)
		(annotate! parent-node 
			   `(,(pos-symbol pos)
			     (@ (sdom:parent-node ,(lambda () parent-node))
				(sdom:sxml-representation ,node-head))))
		(let ((type (sdom:node-type node-head))
		      (attr-fn (lambda (attr-item) 
				 (if (not (eq? (car attr-item) '@))
				     (tag-sibs! (append '(@) attr-item) 
						node-head 1))))
		      (string-fn (lambda (item)
				   (if (string? (car item))
				       (set-car! item 
						 (string-copy (car item))))))
		      (parent-fn (lambda () parent-node))
		      (counter 1))
		  (cond ((eqv? type sdom:node-type-attr)
			 (annotate! node-head 
				    `(sdom:owner-element ,parent-fn)))
			((or (eqv? type sdom:node-type-comment)
			     (eqv? type sdom:node-type-entity))
			 (annotate! parent-node 
				    `(,(pos-symbol pos) 
				      ((sdom:parent-node ,parent-fn)))))
			((not (eqv? type sdom:node-type-document))
			 (annotate! node-head `(sdom:parent-node 
						,parent-fn))))
		  (if (memv type `(,sdom:node-type-attr 
				   ,sdom:node-type-element))
		      (let* ((name (derive-name node-head))
			     (p (get-prefix name))
			     (l (get-local-name name)))
			(if (not (null? p))
			    (let ((ns (sdom:lookup-namespace-uri node-head p))
				  (scope-ns (internal-lookup-scoped-namespace
					     node-head)))

			      ;; If the prefix has a slash or colon in it, it
			      ;; must have been resolved to a namespace URI
			      ;; beforehand by SXML, so we need to adapt the
			      ;; behavior of our parser for this document.

			      (if (and (not found-url-prefix)
				       (string-match extended-char-regex p))
				  (begin
				    (sdom:set-dom-config-parameter 
				     sdom-tree "sdom:resolve-new-prefixes" #t)
				    (set! found-url-prefix #t)))
				  
			      (if (null? ns)
				  (begin 
				    (annotate! node-head 
					       `(sdom:namespace-uri ,p))
				    (if (eqv? type sdom:node-type-element)
					(begin (add-namespace node-head p p) 
					       (annotate! 
						node-head
						`(sdom:default-namespace 
						  ,p)))))
				  (begin 
				    (annotate! node-head `(sdom:namespace-uri
							   ,ns))
				    (if (or (and (null? ns)
						 (null? scope-ns))
					    (not (equal? ns scope-ns)))
					(annotate! node-head 
						   `(sdom:default-namespace 
						     ,ns))
					(if (eqv? type sdom:node-type-attr)
					    (set-car! (cdr node-head) 
						      (string->symbol l))
					    (set-car! node-head 
						      (string->symbol 
						       l))))))))))

		  (pair-for-each string-fn (cdr node-head))
		  (for-each (cond ((eqv? type sdom:node-type-element)
				   (lambda (item)
				     (if (and (list? item) 
					      (eq? (car item) '@))
					 (for-each attr-fn (cdr item))
					 (begin
					   (tag-sibs! item node-head counter)
					   (set! counter (+ counter 1))))))
				  ((eqv? type sdom:node-type-document)
				   (lambda (item)
				     (if (and (list? item)
					      (not (eq? (car item) '@)))
					 (begin 
					   (tag-sibs! item node-head counter)
					   (set! counter (+ counter 1))))))
				  ((eqv? type sdom:node-type-attr)
				   (lambda (item)
				     (if (not (and (list? item)
						   (eq? (car item) '@)))
					 (begin
					   (tag-sibs! item node-head counter)
					   (set! counter (+ counter 1))))))
				  (else (lambda (item) '())))
			    (cond ((eqv? type sdom:node-type-attr)
				   (cddr node-head))
				  (else (cdr node-head)))))))))
      (tag-sibs! sdom-tree #f 1)
      sdom-tree)))


