;; tests.scm: test procedures for SDOM
;; Copyright (C) 2005 Julian Graham

;; SDOM is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2 of the License, or
;; (at your option) any later version.
;;
;; SDOM is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with SDOM; if not, write to the Free Software
;; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

(load "ls.scm")
(use-modules (ice-9 readline))
(use-modules (sxml ssax)
	     (sdom core)
	     (sdom events)
	     (sdom ls))
(activate-readline)

(define xmlns "http://www.w3.org/XML/1998/namespace")
(define xhtmlns "http://www.w3.org/1999/xhtml")

(define domns "http://www.w3.org/DOM")
(define xml-header "<?xml version=\"1.0\"?>")

(define test-string-1 (string-append xml-header "<a><b/></a>"))

(define hc-no-dtd-staff
  (ssax:xml->sxml (open-input-file "test-xml/hc-no-dtd-staff.xml") '()))
(define barfoo (ssax:xml->sxml (open-input-file "test-xml/barfoo.xml") '()))

(define test-error-handler 
  (lambda () (error #t "test-error-handler invoked!")))

(define attrisid-1 
  (lambda () 
    (let* ((hc-staff-dom (sdom:sxml->sdom hc-no-dtd-staff))
	   (elem-list (sdom:get-elements-by-tag-name hc-staff-dom "acronym"))
	   (acronym-elem (list-ref elem-list 0))
	   (attr (sdom:get-attribute-node acronym-elem "title")))
      (if (sdom:get-dom-property attr 'sdom:is-id) 
	  (error "attr is not id!") #t))))

(define attrisid-2
  (lambda ()
    (let* ((hc-staff-dom (sdom:sxml->sdom hc-no-dtd-staff))
	   (elem-list (sdom:get-elements-by-tag-name hc-staff-dom "acronym"))
	   (acronym-elem (list-ref elem-list 0)))
      (sdom:set-attribute! acronym-elem "xml:lang" "FR-fr" xmlns)
      (sdom:set-id-attribute! acronym-elem "lang" #t xmlns)
      (let ((attr (sdom:get-attribute-node acronym-elem "lang" xmlns)))
	(if (not (sdom:get-dom-property attr 'sdom:is-id))
	    (error "attr is id!") #t)))))

(define attrisid-3
  (lambda ()
    (let* ((hc-staff-dom (sdom:sxml->sdom hc-no-dtd-staff))
	   (elem-list (sdom:get-elements-by-tag-name hc-staff-dom "acronym"))
	   (acronym-elem (list-ref elem-list 0)))
      (sdom:set-attribute! acronym-elem "xml:lang" "FR-fr" xmlns)
      (sdom:set-id-attribute! acronym-elem "lang" #f xmlns)
      (let ((attr (sdom:get-attribute-node acronym-elem "lang" xmlns)))
	(if (sdom:get-dom-property attr 'sdom:is-id)
	    (error "attr is not id!") #t)))))

(define attrisid-6
  (lambda ()
    (let* ((hc-staff-dom (sdom:sxml->sdom hc-no-dtd-staff))
	   (attr (sdom:create-node 
		  hc-staff-dom sdom:node-type-attr "xml:lang" xmlns)))
      (if (sdom:get-dom-property attr 'sdom:is-id) 
	  (error "attr is not id!") #t))))

(define attrisid-7
  (lambda ()
    (let* ((hc-staff-dom (sdom:sxml->sdom hc-no-dtd-staff))
	   (elt-lst (sdom:get-elements-by-tag-name hc-staff-dom "acronym" "*"))
	   (acronym-elem (list-ref elt-lst 0)))
      (sdom:set-attribute! acronym-elem "dom3:newAttr" "null" domns)
      (sdom:set-id-attribute! acronym-elem "newAttr" #t domns)
      (let ((attr (sdom:get-attribute-node acronym-elem "newAttr" domns)))
	(if (not (sdom:get-dom-property attr 'sdom:is-id))
	    (error "attr is id!")
	    (let ((imported-attr (sdom:import-node hc-staff-dom attr #f)))
	      (if (sdom:get-dom-property attr 'sdom:is-id)
		  (error "attr is not id!") #t)))))))

(define cdatasections-1
  (lambda ()
    (let* ((barfoo-dom (sdom:sxml->sdom barfoo))
	   (elem-list (sdom:get-elements-by-tag-name barfoo-dom "p"))
	   (new-cdata (sdom:create-node 
		       barfoo-dom sdom:node-type-cdata-section "CDATA")))
      (sdom:append-child! (list-ref elem-list 0) new-cdata)
      (sdom:set-dom-config-parameter! barfoo-dom "cdata-sections" #f)
      (sdom:set-dom-config-parameter! 
       barfoo-dom "error-handler" test-error-handler)
      (sdom:normalize! barfoo-dom)
      (let* ((elem-list (sdom:get-elements-by-tag-name barfoo-dom "p"))
	     (cdata (sdom:get-dom-property (list-ref elem-list 0) 
					   'sdom:last-child)))
	(if (not (equal? (sdom:get-dom-property cdata 'sdom:node-name) 
			 "#cdata-section"))
	    (error #t "cdata section should be named \"#cdata-section\"") 
	    #t)))))

(define comments-1
  (lambda ()
    (let* ((barfoo-dom (sdom:sxml->sdom barfoo))
	   (elem-list (sdom:get-elements-by-tag-name barfoo-dom "p"))
	   (comment (sdom:create-node 
		     barfoo-dom sdom:node-type-comment "COMMENT_NODE")))
      (sdom:append-child! (list-ref elem-list 0) comment)
      (sdom:set-dom-config-parameter! barfoo-dom "cdata-sections" #f)
      (sdom:set-dom-config-parameter! 
       barfoo-dom "error-handler" test-error-handler)
      (sdom:normalize! barfoo-dom)
      (let* ((elem-list (sdom:get-elements-by-tag-name barfoo-dom "p"))
	     (cdata (sdom:get-dom-property (list-ref elem-list 0) 
					   'sdom:last-child)))
	(if (not (equal? (sdom:get-dom-property cdata 'sdom:node-name) 
			 "#comment"))
	    (error #t "cdata section should be named \"#comment\"") #t)))))

;; This test is a bit weird, since we're adopting the node into its own
;; document.

(define documentadoptnode-3
  (lambda ()
    (let* ((hc-staff-dom (sdom:sxml->sdom hc-no-dtd-staff))
	   (attr (sdom:create-node 
		  hc-staff-dom sdom:node-type-attr "xml:lang" xmlns)))
      (sdom:adopt-node! hc-staff-dom attr)
      (cond ((not (equal? (sdom:get-dom-property attr 'sdom:node-name)
			  "xml:lang"))
	     (error #t "attr name should be \"xml:lang\""))
	    ((not (equal? (sdom:get-dom-property attr 'sdom:namespace-uri)
			  xmlns))
	     (error #t "attr namespace-uri should be xmlns"))
	    ((not (equal? (sdom:get-dom-property attr 'sdom:prefix) "xml"))
	     (error #t "attr prefix should be \"xml\""))
	    ((not (eq? (sdom:get-dom-property attr 'sdom:owner-element) '()))
	     (error #t "attr owner-element should be null"))
	    ((not (sdom:get-dom-property attr 'sdom:specified))
	     (error #t "attr should be specified")))
      #t)))

(define documentadoptnode-4
  (lambda ()
    (let* ((doc (sdom:sxml->sdom hc-no-dtd-staff))
	   (doc-elem (sdom:get-dom-property doc 'sdom:document-element))
	   (root-name (sdom:get-dom-property doc-elem 'sdom:tag-name))
	   (root-ns (sdom:get-dom-property doc-elem 'sdom:namespace-uri))
	   (new-doc (sdom:create-document root-name '() root-ns))
	   (attr (sdom:create-node doc sdom:node-type-attr "xml:lang" xmlns)))
      (sdom:adopt-node! new-doc attr)
      (cond ((not (equal? (sdom:get-dom-property attr 'sdom:node-name)
			  "xml:lang"))
	     (error #t "attr name should be \"xml:lang\""))
	    ((not (equal? (sdom:get-dom-property attr 'sdom:namespace-uri)
			  xmlns))
	     (error #t "attr namespace-uri should be xmlns"))
	    ((not (equal? (sdom:get-dom-property attr 'sdom:prefix) "xml"))
	     (error #t "attr prefix should be \"xml\""))
	    ((not (eq? (sdom:get-dom-property attr 'sdom:owner-element) '()))
	     (error #t "attr owner-element should be null"))
	    ((not (sdom:get-dom-property attr 'sdom:specified))
	     (error #t "attr should be specified")))
      #t)))

(define documentadoptnode-7
  (lambda ()
    (let* ((doc (sdom:sxml->sdom hc-no-dtd-staff))
	   (success #f))
      (catch 'sdom:exception (lambda () (sdom:adopt-node! doc doc))
	     (lambda (key . args) 
	       (if (and (not (null? args)) 
			(eqv? (car args) 
			      sdom:exception-code-not-supported-err))
		   (set! success #t))))
      (if (not success)
	  (error #t "adopting self as node should not be supported!") #t))))

(define documentadoptnode-8
  (lambda ()
    (let* ((doc (sdom:sxml->sdom hc-no-dtd-staff))
	   (doc-elem (sdom:get-dom-property doc 'sdom:document-element))
	   (root-name (sdom:get-dom-property doc-elem 'sdom:tag-name))
	   (root-ns (sdom:get-dom-property doc-elem 'sdom:namespace-uri))
	   (new-doc (sdom:create-document root-name '() root-ns))
	   (success #f))
      (catch 'sdom:exception (lambda () (sdom:adopt-node! doc new-doc))
	     (lambda (key . args) 
	       (if (and (not (null? args)) 
			(eqv? (car args) 
			      sdom:exception-code-not-supported-err))
		   (set! success #t))))
      (if (not success)
	  (error #t "document adoption should not be supported!") #t))))

(define documentadoptnode-9
  (lambda ()
    (let* ((doc (sdom:sxml->sdom hc-no-dtd-staff))
	   (doc-elem (sdom:get-dom-property doc 'sdom:document-element))
	   (root-name (sdom:get-dom-property doc-elem 'sdom:tag-name))
	   (root-ns (sdom:get-dom-property doc-elem 'sdom:namespace-uri))
	   (new-doc (sdom:create-document root-name '() root-ns))
	   (success #f))
      (catch 'sdom:exception (lambda () (sdom:adopt-node! new-doc doc))
	     (lambda (key . args) 
	       (if (and (not (null? args)) 
			(eqv? (car args) 
			      sdom:exception-code-not-supported-err))
		   (set! success #t))))
      (if (not success)
	  (error #t "document adoption should not be supported!") #t))))

(define documentadoptnode-11
  (lambda ()
    (let* ((doc (sdom:sxml->sdom hc-no-dtd-staff))
	   (doc-elem (sdom:get-dom-property doc 'sdom:document-element))
	   (root-name (sdom:get-dom-property doc-elem 'sdom:tag-name))
	   (doc-type (sdom:create-document-type root-name '() '()))
	   (success #f))
      (catch 'sdom:exception (lambda () (sdom:adopt-node! doc doc-type))
	     (lambda (key . args) 
	       (if (and (not (null? args)) 
			(eqv? (car args) 
			      sdom:exception-code-not-supported-err))
		   (set! success #t))))
      (if (not success)
	  (error #t "document type adoption should not be supported!") #t))))

(define documentadoptnode-12
  (lambda ()
    (let* ((doc (sdom:sxml->sdom hc-no-dtd-staff))
	   (doc-elem (sdom:get-dom-property doc 'sdom:document-element))
	   (root-name (sdom:get-dom-property doc-elem 'sdom:tag-name))
	   (root-ns (sdom:get-dom-property doc-elem 'sdom:namespace-uri))
	   (doc-type (sdom:create-document-type root-name '() '()))
	   (new-doc (sdom:create-document root-name doc-type root-ns))
	   (success #f))
      (catch 'sdom:exception (lambda () (sdom:adopt-node! new-doc doc-type))
	     (lambda (key . args) 
	       (if (and (not (null? args)) 
			(eqv? (car args) 
			      sdom:exception-code-not-supported-err))
		   (set! success #t))))
      (if (not success)
	  (error #t "document type adoption should not be supported!") #t))))

(define documentadoptnode-13
  (lambda ()
    (let* ((doc (sdom:sxml->sdom hc-no-dtd-staff))
	   (doc-frag (sdom:create-node doc sdom:node-type-document-fragment))
	   (child-list (sdom:get-elements-by-tag-name doc "acronym"))
	   (acronym-node (list-ref child-list 0)))
      (sdom:append-child! doc-frag acronym-node)
      (sdom:adopt-node! doc doc-frag)
      (if (not (sdom:has-child-nodes? doc-frag))
	  (error #t "document fragment children should be adopted recursively")
	  #t))))

(define documentadoptnode-14
  (lambda ()
    (let* ((doc (sdom:sxml->sdom hc-no-dtd-staff))
	   (doc-elem (sdom:get-dom-property doc 'sdom:document-element))
	   (root-name (sdom:get-dom-property doc-elem 'sdom:tag-name))
	   (root-ns (sdom:get-dom-property doc-elem 'sdom:namespace-uri))
	   (new-doc (sdom:create-document root-name '() root-ns))
	   (doc-frag (sdom:create-node new-doc
				       sdom:node-type-document-fragment))
	   (imported (sdom:import-node new-doc doc-elem #t))
	   (doc-elem (sdom:get-dom-property new-doc 'sdom:document-element)))
      (sdom:append-child! doc-elem imported)
      (sdom:append-child! 
       doc-frag (list-ref (sdom:get-elements-by-tag-name new-doc "acronym") 0))
      (sdom:adopt-node! new-doc doc-frag)
      (if (not (sdom:has-child-nodes? doc-frag))
	  (error #t "document fragment children should be adopted recursively")
	  #t))))

(define documentadoptnode-15
  (lambda ()
    (let* ((doc (sdom:sxml->sdom hc-no-dtd-staff))
	   (doc-frag (sdom:create-node doc sdom:node-type-document-fragment)))
      (sdom:adopt-node! doc doc-frag)
      (if (sdom:has-child-nodes? doc-frag)
	  (error #t "newly-created document fragment should have no children")
	  #t))))
	   
(define documentadoptnode-21
  (lambda ()
    (let* ((doc (sdom:sxml->sdom hc-no-dtd-staff))
	   (child-list (sdom:get-elements-by-tag-name doc "acronym"))
	   (attr (sdom:get-attribute-node (list-ref child-list 0) "title")))
      (sdom:adopt-node! doc attr)
      (if (not (equal? (sdom:get-dom-property attr 'sdom:node-name) "title"))
	  (error #t "node adoption should preserve node-name"))
      (if (not (eqv? (sdom:get-dom-property attr 'sdom:node-type) 2))
	  (error #t "node adoption should preserve node-type"))
      (if (not (equal? (sdom:get-dom-property attr 'sdom:node-value) "Yes"))
	  (error #t "node adoption should preserve node-value"))
      (if (not (null? (sdom:get-dom-property attr 'sdom:owner-element)))
	  (error #t "owner-element should be null for newly-adopted nodes"))
      #t)))

(define documentadoptnode-22
  (lambda ()
    (let* ((doc (sdom:sxml->sdom hc-no-dtd-staff))
	   (doc-elem (sdom:get-dom-property doc 'sdom:document-element))
	   (adopted-node (sdom:adopt-node! doc doc-elem)))
      (if (not (sdom:has-child-nodes? adopted-node))
	  (error #t "children of adopted node should be recursively adopted"))
      (if (not (equal? (sdom:get-dom-property doc-elem 'sdom:node-name)
		       (sdom:get-dom-property adopted-node 'sdom:node-name)))
	  (error #t "node adoption should preserve node-name"))
      #t)))

(define documentadoptnode-23
  (lambda ()
    (let* ((doc (sdom:sxml->sdom hc-no-dtd-staff))
	   (child-list (sdom:get-elements-by-tag-name doc "acronym"))
	   (acronym-elem (list-ref child-list 0))
	   (adopted-node (sdom:adopt-node! doc acronym-elem)))
      (if (not (eqv? (length (sdom:get-dom-property acronym-elem 
						    'sdom:child-nodes))
		     (length (sdom:get-dom-property adopted-node
						    'sdom:child-nodes))))
	  (error #t "node adoption should preserve number of child nodes")
	  #t))))

(define documentadoptnode-24
  (lambda ()
    (let* ((doc (sdom:sxml->sdom hc-no-dtd-staff))
	   (new-doc (sdom:create-document 
		     "dom:test" '() "http://www.w3.org/DOM/Test"))
	   (child-list (sdom:get-elements-by-tag-name doc "code" "*"))
	   (code-elem (list-ref child-list 0))
	   (adopted-node (sdom:adopt-node! new-doc code-elem)))
      (if (not (eqv? (length (sdom:get-dom-property 
			      code-elem 'sdom:child-nodes))
		     (length (sdom:get-dom-property 
			      adopted-node 'sdom:child-nodes))))
	  (error #t "node adoption should preserve number of nodes")
	  #t))))

(define documentadoptnode-25
  (lambda ()
    (let* ((doc (sdom:sxml->sdom hc-no-dtd-staff))
	   (doc-elem (sdom:get-dom-property doc 'sdom:document-element))
	   (root-ns (sdom:get-dom-property doc-elem 'sdom:namespace-uri))
	   (root-name (sdom:get-dom-property doc-elem 'sdom:tag-name))
	   (new-elem (sdom:create-node doc sdom:node-type-element "th"
				       "http://www.w3.org/1999/xhtml"))
	   (new-doc (sdom:create-document root-name '() root-ns))
	   (adopted-node (sdom:adopt-node! new-doc new-elem)))
      (if (not (equal? "th" (sdom:get-dom-property adopted-node 
						   'sdom:node-name)))
	  (error #t "node adoption should preserve node-name"))
      (if (not (equal? "http://www.w3.org/1999/xhtml"
		       (sdom:get-dom-property adopted-node 
					      'sdom:namespace-uri)))
	  (error #t "node adoption should preserve namespace-uri")
	  #t))))

(define documentadoptnode-26
  (lambda ()
    (let* ((doc (sdom:sxml->sdom hc-no-dtd-staff))
	   (doc-elem (sdom:get-dom-property doc 'sdom:document-element))
	   (root-ns (sdom:get-dom-property doc-elem 'sdom:namespace-uri))
	   (root-tagname (sdom:get-dom-property doc-elem 'sdom:tag-name))
	   (new-doc (sdom:create-document root-tagname '() root-ns))
	   (new-elem (sdom:create-node new-doc sdom:node-type-element "head"
				       "http://www.w3.org/1999/xhtml")))
      (sdom:set-attribute! doc-elem "xml:lang" "en-US" xmlns)
      (let* ((doc-elem (sdom:get-dom-property new-doc 'sdom:document-element))
	     (appended-child (sdom:append-child! doc-elem new-elem))
	     (adopted-node (sdom:adopt-node! doc new-elem)))
	(if (not (equal? (sdom:get-dom-property adopted-node 'sdom:node-name)
			 "head"))
	    (error #t "node adoption should preserve node-name"))
	(if (not (equal? (sdom:get-dom-property adopted-node 
						'sdom:namespace-uri)
			 "head"))
	    (error #t "node adoption should preserve namespace-uri")
	    #t)))))

(define documentadoptnode-27
  (lambda ()
    (let* ((doc (sdom:sxml->sdom hc-no-dtd-staff))
	   (doc-elem (sdom:get-dom-property doc 'sdom:document-element))
	   (root-ns (sdom:get-dom-property doc-elem 'sdom:namespace-uri))
	   (root-tagname (sdom:get-dom-property doc-elem 'sdom:tag-name))
	   (new-doc (sdom:create-document root-tagname '() root-ns))
	   (new-elem (sdom:create-node new-doc sdom:node-type-element 
				       "xhtml:head" xhtmlns)))
      (sdom:set-attribute! new-elem "xml:lang" "en-US" xmlns)
      (let* ((doc-elem (sdom:get-dom-property new-doc 'sdom:document-element))
	     (appended-child (sdom:append-child! doc-elem new-elem))
	     (new-imp-elem (sdom:import-node doc new-elem #t))
	     (adopted-node (sdom:adopt-node! doc new-imp-elem)))
	(if (not (equal? (sdom:get-dom-property adopted-node 'sdom:node-name)
			 "xhtml:head"))
	    (error #t "node adoption should preserve node-name"))
	(if (not (equal? (sdom:get-dom-property adopted-node 
						'sdom:namespace-uri)
			 xhtmlns))
	    (error #t "node adoption should preserve namespace-uri")
	    #t)))))

(define documentadoptnode-30
  (lambda ()
    (let* ((doc (sdom:sxml->sdom hc-no-dtd-staff))
	   (new-text (sdom:create-node 
		      doc sdom:node-type-text
		      "sdom:adopt-node! test for a TEXT_NODE"))
	   (adopted-text (sdom:adopt-node! doc new-text)))
      (if (not (equal? (sdom:get-dom-property adopted-text 'sdom:node-value)
		       "sdom:adopt-node! test for a TEXT_NODE"))
	  (error #t "node adoption should preserve text node value") #t))))

(define documentadoptnode-31
  (lambda ()
    (let* ((doc (sdom:sxml->sdom hc-no-dtd-staff))
	   (doc-elem (sdom:get-dom-property doc 'sdom:document-element))
	   (root-name (sdom:get-dom-property doc-elem 'sdom:tag-name))
	   (root-ns (sdom:get-dom-property doc-elem 'sdom:namespace-uri))
	   (new-doc (sdom:create-document root-name '() root-ns))
	   (new-text (sdom:create-node 
		      new-doc sdom:node-type-text
		      "new sdom:adopt-node! test for a TEXT_NODE"))
	   (adopted-text (sdom:adopt-node! doc new-text)))
      (if (not (equal? (sdom:get-dom-property adopted-text 'sdom:node-value)
		       "new sdom:adopt-node! test for a TEXT_NODE"))
	  (error #t "node adoption should preserve text node value") #t))))

(define documentadoptnode-32
  (lambda ()
    (let* ((doc (sdom:sxml->sdom hc-no-dtd-staff))
	   (doc-adopter (sdom:sxml->sdom hc-no-dtd-staff))
	   (new-cdata (sdom:create-node 
		       doc sdom:node-type-cdata-section
		       "sdom:adopt-node! test for a CDATASECTION_NODE"))
	   (adopted-cdata (sdom:adopt-node! doc-adopter new-cdata)))
      (if (not (equal? (sdom:get-dom-property adopted-cdata 'sdom:node-value)
		       "sdom:adopt-node! test for a CDATASECTION_NODE"))
	  (error #t "node adoption should preserve cdata section value")
	  #t))))

(define documentadoptnode-33
  (lambda ()
    (let* ((doc (sdom:sxml->sdom hc-no-dtd-staff))
	   (doc-elem (sdom:get-dom-property doc 'sdom:document-element))
	   (root-ns (sdom:get-dom-property doc-elem 'sdom:namespace-uri))
	   (root-name (sdom:get-dom-property doc-elem 'sdom:tag-name))
	   (new-doc (sdom:create-document root-name '() root-ns))
	   (new-cdata (sdom:create-node 
		       new-doc sdom:node-type-cdata-section
		       "sdom:adopt-node! test for a CDATASECTION_NODE"))
	   (adopted-cdata (sdom:adopt-node! doc new-cdata)))
      (if (not (equal? (sdom:get-dom-property adopted-cdata 'sdom:node-value)
		       "sdom:adopt-node! test for a CDATASECTION_NODE"))
	  (error #t "node adoption should preserve cdata section value") #t))))

(define documentadoptnode-34
  (lambda ()
    (let* ((doc (sdom:sxml->sdom hc-no-dtd-staff))
	   (doc-elem (sdom:get-dom-property doc 'sdom:document-element))
	   (root-ns (sdom:get-dom-property doc-elem 'sdom:namespace-uri))
	   (root-name (sdom:get-dom-property doc-elem 'sdom:tag-name))
	   (new-doc (sdom:create-document root-name '() root-ns))
	   (new-comment (sdom:create-node 
			 new-doc sdom:node-type-comment
			 "sdom:adopt-node! test for a COMMENT_NODE"))
	   (adopted-comment (sdom:adopt-node! new-doc new-comment)))
      (if (not (equal? (sdom:get-dom-property adopted-comment 'sdom:node-value)
		       "sdom:adopt-node! test for a COMMENT_NODE"))
	  (error #t "node adoption should preserve comment value") #t))))

(define documentadoptnode-35
  (lambda ()
    (let* ((doc (sdom:sxml->sdom hc-no-dtd-staff))
	   (doc-elem (sdom:get-dom-property doc 'sdom:document-element))
	   (root-ns (sdom:get-dom-property doc-elem 'sdom:namespace-uri))
	   (root-name (sdom:get-dom-property doc-elem 'sdom:tag-name))
	   (new-doc (sdom:create-document root-name '() root-ns))
	   (new-pi (sdom:create-node 
		    new-doc 'sdom:node-type-processing-instruction
		    "PITarget" "PIData"))
	   (adopted-pi (sdom:adopt-node! doc new-pi)))
      (if (not (equal? (sdom:get-dom-property adopted-pi 'sdom:target) 
		       "PITarget"))
	  (error #t "node adoption should preserve PI target"))
      (if (not (equal? (sdom:get-dom-property adopted-pi 'sdom:data) "PIData"))
	  (error #t "node adoption should preserve PI data") #t))))
	   
(define documentadoptnode-36
  (lambda ()
    (let* ((doc (sdom:sxml->sdom hc-no-dtd-staff))
	   (doc-elem (sdom:get-dom-property doc 'sdom:document-element))
	   (root-ns (sdom:get-dom-property doc-elem 'sdom:namespace-uri))
	   (root-name (sdom:get-dom-property doc-elem 'sdom:node-name))
	   (new-doc (sdom:create-document root-name '() root-ns))
	   (new-pi-1 (sdom:create-node 
		      new-doc sdom:node-type-processing-instruction
		      "PITarget" "PIData"))
	   (new-pi-2 (sdom:create-node 
		      doc sdom:node-type-processing-instruction
		      "PITarget" "PIData"))
	   (adopted-pi-1 (sdom:adopt-node! new-doc new-pi-1))
	   (adopted-pi-2 (sdom:adopt-node! new-doc new-pi-2)))
      (if (or (not (equal? (sdom:get-dom-property adopted-pi-1 'sdom:target)
			   "PITarget"))
	      (not (equal? (sdom:get-dom-property adopted-pi-2 'sdom:target)
			   "PITarget")))
	  (error #t "node adoption should preserve PI target"))
      (if (or (not (equal? (sdom:get-dom-property adopted-pi-1 'sdom:data)
			   "PIData"))
	      (not (equal? (sdom:get-dom-property adopted-pi-2 'sdom:data)
			   "PIData")))
	  (error #t "node adoption should preserve PI data") #t))))

(define tests 
  `((,attrisid-1 "attrisid-1")
    (,attrisid-2 "attrisid-2")
    (,attrisid-3 "attrisid-3")
    (,attrisid-6 "attrisid-6")
    (,attrisid-7 "attrisid-7")
    (,cdatasections-1 "cdatasections-1")
    (,comments-1 "comments-1")
    (,documentadoptnode-3 "documentadoptnode-3")
    (,documentadoptnode-4 "documentadoptnode-4")
    (,documentadoptnode-7 "documentadoptnode-7")
    (,documentadoptnode-8 "documentadoptnode-8")
    (,documentadoptnode-9 "documentadoptnode-9")
    (,documentadoptnode-11 "documentadoptnode-11")
    (,documentadoptnode-12 "documentadoptnode-12")
    (,documentadoptnode-13 "documentadoptnode-13")
    (,documentadoptnode-14 "documentadoptnode-14")
    (,documentadoptnode-15 "documentadoptnode-15")
    (,documentadoptnode-21 "documentadoptnode-21")
    (,documentadoptnode-22 "documentadoptnode-22")
    (,documentadoptnode-23 "documentadoptnode-23")
    (,documentadoptnode-24 "documentadoptnode-24")
    (,documentadoptnode-25 "documentadoptnode-25")
    (,documentadoptnode-26 "documentadoptnode-26")
    (,documentadoptnode-27 "documentadoptnode-27")
    (,documentadoptnode-30 "documentadoptnode-30")
    (,documentadoptnode-31 "documentadoptnode-31")
    (,documentadoptnode-32 "documentadoptnode-32")
    (,documentadoptnode-33 "documentadoptnode-33")
    (,documentadoptnode-34 "documentadoptnode-34")
    (,documentadoptnode-35 "documentadoptnode-35")
    (,documentadoptnode-36 "documentadoptnode-36")))

(define run-tests
  (lambda () 
    (for-each (lambda (x)
		(let* ((name (cadr x))
		       (dots (make-string (- 38 (string-length name)) #\.)))
		  (display name)
		  (display dots)
		  (display (catch #t (lambda () (begin (apply (car x) '()) #t))
				  (lambda (key . args) #f)))
		  (newline)))
	      tests)))
