;; parser-tests.scm: Test suite for SCSS parsing
;; Copyright (C) 2010 Julian Graham

;; SCSS is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <http://www.gnu.org/licenses/>.

#!r6rs

(import (rnrs) (scss scss))

(define (t str) (scss:css->scss (open-string-input-port str)))

(define (test str expr)
  (let ((val (t str)))
    (if (equal? val expr)
	(display #t)
	(begin 
	  (display str) (display " => ") (display expr) (display ": ")
	  (display "#f ") 
	  (display val))))
  (newline))

(test "p { color: green; }" '(css (p (color "green"))))
(test "p { color: green }" '(css (p (color "green"))))
(test "p, { color: green }" '(css (p (color "green"))))
(test ",p { color: green } q { color: green }" '(css (q (color "green"))))
(test "p,q { color: green }" '(css ((p q) (color "green"))))
(test "p,q, { color: green }" '(css ((p q) (color "green"))))
(test "p { border: medium solid; }" '(css (p (border "medium solid"))))
(test "p.one { font-family: serif; }" '(css (p.one (font-family "serif"))))
(test "p.two { font-weight: bold; font: 1em/normal serif; }"
      '(css (p.two (font-weight "bold") (font "1em/normal serif"))))
(test "p { background: \"red\" }" '(css))
(test "p { bACkGRounD: gREen; }" '(css (p (background "gREen"))))
(test "@media all { p { color: green; } }" 
      `(css (,(string->symbol "@media") all (p (color "green")))))
(test "@MeDIa aLL { p { color: green; } }"
      `(css (,(string->symbol "@media") aLL (p (color "green")))))
(test "p.one:before { content: \"This \"; }" 
      '(css (p.one:before (content "\"This \""))))
(test "p.two:before { content: \"Th\\\nis \"; }" 
      '(css (p.two:before (content "\"This \""))))
(test "p { font-family: \"\\\"\", '\\'', serif; background: green }"
      '(css (p (font-family "\"\"\",''',serif") (background "green"))))
(test "p.class#id { background: green; color: white }"
      `(css (,(string->symbol "p.class|#id|")
	     (background "green") 
	     (color "white"))))
(test "p\\.class#id { background: red; }" 
      '(css ((id p.class id) (background "red"))))
(test "p.class\\#id { background: red; }" 
      `(css (,(string->symbol "p.class#id") (background "red"))))
(test "p.class#id { background\\: red; }" '(css))
(test "p.class#id { background: red\\; }" '(css))
(test "p.class#id \\{ background: red; \\}\n\np.class#id { background: red; }"
      '(css))
(test "p.c\\lass { bac\\kground: g\\reen; }" 
      '(css (p.class (background "green"))))
(test "p.c\\00006Cas\\000073 { back\\000067round: gr\\000065en; }"
      '(css (p.class (background "green"))))
(test "p.c\\00006Cas\\000073 { back\\000047round: gr\\000045en; }"
      '(css (p.class (background "grEen"))))
(test "p.cl\\ass { background: red; }" `(css (,(string->symbol "p.cl\nss")
					      (background "red"))))
(test "p.c\\06C ass { back\\67 round: gr\\000065 en; }"
      '(css (p.class (background "green"))))
(test "p.c\\06C  ass { back\\67round: r\\000065 ed; }" '(css))
(test "p.c\\06Cass { back\\67\n round: r\\000065ed; }" '(css))
(test "p.c\\06Cass { back\\67round: r\\000065\n ed; }" '(css))
(test "p {color: r\\ed}" '(css))
(test "p \\7Bcolor: red\\7D" '(css))
(test ".one { color: red; background: white; }" 
      `(css (,(string->symbol ".one") (color "red") (background "white"))))
(test "#ident, .two { color: green; }" 
      `(css ((,(string->symbol "|#ident|") ,(string->symbol ".two"))
	     (color "green"))))
(test ".IdE6n-3t0_6, .one { color: green; }" 
      `(css ((,(string->symbol ".IdE6n-3t0_6") ,(string->symbol ".one")) 
	     (color "green"))))
(test "#IdE6n-3t0_6, .two { color: green; }" 
      `(css ((,(string->symbol "|#IdE6n-3t0_6|") ,(string->symbol ".two"))
	     (color "green"))))
(test "._ident, .one { color: green; }" 
      `(css ((,(string->symbol "._ident") ,(string->symbol ".one")) 
	     (color "green"))))
(test "#_ident, .two { color: green; }" 
      `(css ((,(string->symbol "|#_ident|") ,(string->symbol ".two"))
	     (color "green"))))
(test ".-ident, .one { color: green; }" 
      `(css ((,(string->symbol ".-ident") ,(string->symbol ".one"))
	     (color "green"))))
(test ".-ident, .one { color: green; }" 
      `(css ((,(string->symbol ".-ident") ,(string->symbol ".one"))
	     (color "green"))))
(test "#-ident, .two { color: green; }" 
      `(css ((,(string->symbol "|#-ident|") ,(string->symbol ".two")) 
	     (color "green"))))
(test ".-1ident, .three { color: red; }" '(css))
(test "#-1ident, .four { color: red; }" '(css))
(test ".6ident, .one { color: red; }" '(css))
(test "#6ident, .two { color: red; }" '(css))
(test ".id4ent6, .one { color: green; }" 
      `(css ((,(string->symbol ".id4ent6") ,(string->symbol ".one")) 
	     (color "green"))))
(test "#id4ent6, .two { color: green; }" 
      `(css ((,(string->symbol "|#id4ent6|")
	      ,(string->symbol ".two")) (color "green"))))
(test ".\\ident, .one { color: green; }" 
      `(css ((,(string->symbol ".ident") ,(string->symbol ".one"))
	     (color "green"))))
(test "#\\ident, .two { color: green; }" 
      `(css ((,(string->symbol "|#ident|") ,(string->symbol ".two"))
	     (color "green"))))
(test ".ide\\n\\t, .one { color: green; }" 
      `(css ((,(string->symbol ".ident") ,(string->symbol ".one"))
	     (color "green"))))
(test "#ide\\n\\t, .two { color: green; }" 
      `(css ((,(string->symbol "|#ident|") ,(string->symbol ".two"))
	     (color "green"))))
(test ".\\36ident, .three { color: green; }"
      `(css ((,(string->symbol ".6ident") ,(string->symbol ".three"))
	     (color "green"))))
(test "#\\36ident, .four { color: green; }"
      `(css ((,(string->symbol "|#6ident|") ,(string->symbol ".four"))
	     (color "green"))))
(test ".\\-ident, .one { color: green; }" 
      `(css ((,(string->symbol ".-ident") ,(string->symbol ".one"))
	     (color "green"))))
(test "#\\-ident, .two { color: green; }"
      `(css ((,(string->symbol "|#-ident|") ,(string->symbol ".two"))
	     (color "green"))))
(test "@impor t \"support/import-red.css\";\n\np { color: green; }"
      '(css (p (color "green"))))
(test "p { color: red; } @foo{} p { color: green; }"
      '(css (p (color "red")) (p (color "green"))))
(test "p { color: red; } @foo; p { color: green; }"
      '(css (p (color "red")) (p (color "green"))))
(test "p { color: red; } @foo bar baz; p { color: green; }"
      '(css (p (color "red")) (p (color "green"))))
(test "p { color: green; } @foo{}; p { color: red; }" 
      '(css (p (color "green"))))
(test "p { color: green; } @import \"support/import-red.css\""
      '(css (p (color "green"))))
(test "@media all { @import \"support/import-red.css\"; p { color: green; } }"
      `(css (,(string->symbol "@media") all (p (color "green")))))

(test "p {color: red} <!-- p {color: green} -->"
      '(css (p (color "red")) (p (color "green"))))
(test (string-append "body { color: red; } "
		     "/* This is a CSS comment. */ "
		     ".one {color: green;} /* Another comment */ "
		     "/* The following should not be used: "
		     " .two {color: red;} */ "
		     ".three {color: green; /* color: red; */} "
		     "/** "
		     ".four {color: red;} */ "
		     ".five {color: green;} "
		     "/**/ "
		     ".six {color: green;} "
		     "/*********/ "
		     ".seven {color: green;} "
		     "/* a comment **/ "
		     ".eight {color: green;} ")
      `(css (body (color "red")) 
	    (,(string->symbol ".one") (color "green")) 
	    (,(string->symbol ".three") (color "green"))
	    (,(string->symbol ".five") (color "green"))
	    (,(string->symbol ".six") (color "green"))
	    (,(string->symbol ".seven") (color "green"))
	    (,(string->symbol ".eight") (color "green"))))

(test "p { color:green; color }" '(css (p (color "green"))))
(test "p { color:red;  color; color:green }"  
      '(css (p (color "red") (color "green"))))
(test "p { color:green; color; }" '(css (p (color "green"))))
(test "p { color:red; color:; color:green }" 
      '(css (p (color "red") (color "green"))))
(test "p { color:green; color{;color:maroon} }" '(css (p (color "green"))))
(test "p { color:red; color{;color:maroon}; color:green }" 
      '(css (p (color "red") (color "green"))))
(test (string-append "p.one {color: green; rotation: 70deg;} "
		     "p.oneb {color: green;} "
		     "p.oneb {color: invalidValue;} "
		     "div.twopc { background: white "
		     "url(support/swatch-red.png); color: green; } "
		     "p.two {background-color: inherit;} "
		     "p.eight {COLOR: GREEN;} "
		     "p.twentya {rotation-code: \"}\"; color: green;} " 
		     "p.twentyb {rotation-code: \"\\\"}\\\"\"; color: green;} "
		     "p.twentyonea {rotation-code: '}'; color: green;} "
		     "p.twentyoneb {rotation-code: '\\'}\\''; color: green;} "
		     "p.twentytwo { "
		     "type-display: @threedee {rotation-code: '}';}; "
		     "color: green; }")
      '(css (p.one (color "green"))
	    (p.oneb (color "green"))
	    (div.twopc (background "white url(support/swatch-red.png)") 
		       (color "green"))
	    (p.two (background-color "inherit"))
	    (p.eight (color "GREEN"))
	    (p.twentya (color "green"))
	    (p.twentyb (color "green"))
	    (p.twentyonea (color "green"))
	    (p.twentyoneb (color "green"))
	    (p.twentytwo (color "green"))))
(test "p.five {background-color: \"red\";}" '(css))
(test (string-append "@threedee { @background-lighting { "
		     "azimuth: 30deg; elevation: 190deg; } "
		     "p.seven { color: red } }")
      '(css))
(test (string-append "p.sixa {border-width: medium; border-style: solid;} "
		     "p.sixb {border-width: funny; border-style: solid;} "
		     "p.sixc {border-width: 50zu; border-style: solid;} "
		     "p.sixd {border-width: px; border-style: solid;}")
      '(css (p.sixa (border-width "medium") (border-style "solid"))
	    (p.sixb (border-style "solid"))
	    (p.sixc (border-style "solid"))
	    (p.sixd (border-style "solid"))))
(test (string-append "p.twentythree {text-indent: 0.5in;background:lime} "
		     "color: red "
		     "p.twentyfour {color: red;}")
      '(css (p.twentythree (text-indent "0.5in") (background "lime"))))
(test "ol:wait { color: red }" '(css))
(test "ul:lang(fr) {color:red;}" '(css))
(test "@media tty { h1 {color: red;} p.sixteen {color: red;} }"
      `(css (,(string->symbol "@media") tty (h1 (color "red")) 
	     (p.sixteen (color "red")))))
(test "@three-dee { p.seventeen { color: red } }"
      '(css))

(test "table { margin: 1em 5em; }" '(css (table (margin "1em 5em"))))
(test "td { padding: 4px; vertical-align: top; }"
      '(css (td (padding "4px") (vertical-align "top"))))
(test ".control { font: 40px Ahem; color: navy; }"
      `(css (,(string->symbol ".control") (font "40px Ahem") (color "navy"))))
(test ".test { font: 40px Ahem; border-top: 1ex solid navy; width: 1em; }"
      `(css (,(string->symbol ".test") 
	     (font "40px Ahem") 
	     (border-top "1ex solid navy") 
	     (width "1em"))))
(test (string-append ".four {margin-left:  3.37007874015748in;}"
		     ".five {margin-left:  8.56000000000000cm;}"
		     ".six {margin-left:   85.6000000000000mm;}"
		     ".seven {margin-left: 242.645669291339pt;}"
		     ".eight {margin-left: 20.2204724409449pc;}"
		     ".nine {margin-left: +20.2204724409449pc;}")
      `(css (,(string->symbol ".four") (margin-left "3.37007874015748in"))
	    (,(string->symbol ".five") (margin-left "8.56000000000000cm"))
	    (,(string->symbol ".six") (margin-left "85.6000000000000mm"))
	    (,(string->symbol ".seven") (margin-left "242.645669291339pt"))
	    (,(string->symbol ".eight") (margin-left "20.2204724409449pc"))
	    (,(string->symbol ".nine") (margin-left "+20.2204724409449pc"))))
(test (string-append ".container { "
		     "font: 15px/1 Ahem;"
		     "background: url(support/swatch-red.png) 45px 0 repeat-y;"
		     "color: green;"
		     "}")
      `(css (,(string->symbol ".container") 
	     (font "15px/1 Ahem")
	     (background "url(support/swatch-red.png) 45px 0 repeat-y")
	     (color "green"))))
(test ".four {padding-top: 20%}" `(css (,(string->symbol ".four")
					(padding-top "20%"))))
(test ".five {padding-top: -20px}" `(css (,(string->symbol ".five")
					  (padding-top "-20px"))))

(test "p { color: green" '(css (p (color "green"))))
(test "p { color: green; background: " '(css (p (color "green"))))
(test "p { " '(css))
(test "p { }" '(css))

;; 4.3.5 Counters

(test "#test span:before { content: counter(c); }"
      `(css ((// ,(string->symbol "|#test|") span:before) 
	     (content "counter(c)"))))

;; 4.3.6 Colors

(test "p#numnumpercent { color: rgb(255, 0, 0%) }" '(css))
(test "p#percentnumnum { color: rgb(100%, 0, 0) }" '(css))
(test "p#percentpercentnum { color: rgb(100%, 0%, 0) }" '(css))
(test "p#percentnumpercent { color: rgb(100%, 0, 0%) }" '(css))
(test "p#floatnumnum { color: rgb(255.0, 0, 0) }" '(css))
(test "p#numfloatnum { color: rgb(0, 128.0, 0) }" '(css))
(test "p#numnumnum { color: rgb(0, 128, 0) }"
      `(css (,(string->symbol "p|#numnumnum|") (color "rgb(0,128,0)"))))
(test "p#percentpercentpercent1 { color: rgb(0%, 50%, 0%) }"
      `(css (,(string->symbol "p|#percentpercentpercent1|")
	     (color "rgb(0%,50%,0%)"))))
(test "p#percentpercentpercent2 { color: rgb(0%, 49.99%, 0%) }"
      `(css (,(string->symbol "p|#percentpercentpercent2|")
	     (color "rgb(0%,49.99%,0%)"))))

(test "p { color: rgb(0,128,0" '(css (p (color "rgb(0,128,0)"))))


;; 5 Selectors

(test ".one, .two, .three { color: green; }" 
      `(css ((,(string->symbol ".one")
	      ,(string->symbol ".two")
	      ,(string->symbol ".three")) (color "green"))))
(test "html body div p {color: red;}" 
      '(css ((// html body div p) (color "red"))))
(test "span, ul li li {color: green;}"
      '(css ((span (// ul li li)) (color "green"))))
(test ".1 { color: red }" '(css))
(test "#test1 {color: green;}" `(css (,(string->symbol "|#test1|")
				      (color "green"))))
(test "p#test3 {color: green;}" `(css (,(string->symbol "p|#test3|")
				       (color "green"))))
(test "p:first-line { color: green; }" '(css (p:first-line (color "green"))))
(test "p:first-letter { color: green; }" 
      '(css (p:first-letter (color "green"))))
(test "p.test:first-line { color: white }" 
      '(css (p.test:first-line (color "white"))))
(test "p:first-line.two { color: yellow; background: red; }" '(css))
(test "a:link {color: green;}" '(css (a:link (color "green"))))
(test "a:visited {color: green;}" '(css (a:visited (color "green"))))
(test "a:hover {color: green;}" '(css (a:hover (color "green"))))
(test "a:focus {color: green;}" '(css (a:focus (color "green"))))
(test "a:active {color: green;}" '(css (a:active (color "green"))))

(test "body * { color: green; }" '(css ((// body *) (color "green"))))
(test "* { color: green; }" '(css (* (color "green"))))
(test "p { color: green ! important; }" '(css (p (! (color "green")))))

;; 5.6 Child selectors

;; 5.7 Adjacent sibling selectors

(test "a + b + c { color: green }" '(css ((+ a (+ b c)) (color "green"))))
(test "a b + c > d e { color: green }"
      '(css ((// a (+ b (> c d)) e) (color "green"))))

;; 5.8 Attribute selectors

(test "p[id] { color: green; }" `(css (,(string->symbol "p[id]")
				       (color "green"))))
(test "p[a=b] { color: green; }" `(css (,(string->symbol "p[a=b]")
					(color "green"))))
(test "p[a=\"b\"] { color: green; }"
      '(css ((attrib p (= a "b")) (color "green"))))
(test "p[a~=b] { color: green; }" `(css (,(string->symbol "p[a~=b]")
					 (color "green"))))
(test "p[a~=\"b\"] { color: green; }" 
      '(css ((attrib p (~= a "b")) (color "green"))))
(test "p[a|=b] { color: green; }" `(css (,(string->symbol "p[a|=b]")
					 (color "green"))))
(test "p[a|=\"b\"] { color: green; }" 
      `(css ((attrib p (,(string->symbol "|=") a "b")) (color "green"))))
(test "table[align=\"center\"] > caption[align=\"left\"] { color: green; }"
      '(css ((> (attrib table (= align "center")) 
		(attrib caption (= align "left"))) (color "green"))))
(test "a b[align=\"center\"] q { color: green }"
      '(css ((// a (attrib b (= align "center")) q) (color "green"))))

