;; scss.scm: main module exports and implementations for SCSS
;; Copyright (C) 2009 Julian Graham

;; SCSS is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <http://www.gnu.org/licenses/>.

(define-module (scss scss)
  #:export (scss:color->hex
	    
	    scss:set-sxml-parent-function!
	    scss:set-dot-handler!
	    scss:set-id-handler!
	    scss:set-pseudo-class-handler!
	    scss:set-pseudo-element-handler!
	    scss:set-uri-import-handler!

	    scss:set-author-stylesheet!
	    scss:set-agent-stylesheet!
	    scss:set-user-stylesheet!

	    scss:create-cascade

	    scss:stylesheet?
	    scss:cascade?
	    scss:inherited?
	    scss:get-default-value
  
	    scss:select-value
	    scss:select-value-at-node

	    scss:parser-debug-messages

	    scss:css->scss
	    scss:scss->css
	    
	    font-size?
	    line-height?)

  #:use-module (srfi srfi-1)
  #:use-module (srfi srfi-2)
  #:use-module (srfi srfi-9)
  #:use-module (srfi srfi-13)
  #:use-module (srfi srfi-14)
  #:use-module (srfi srfi-69))

(primitive-load-path "scss/lexer")
(primitive-load-path "scss/parser")
(primitive-load-path "scss/srfi-32")

(define-record-type lexical-token
  (make-lexical-token category source value)
  lexical-token?
  (category lexical-token-category)
  (source   lexical-token-source)
  (value    lexical-token-value))

(define-record-type source-location
  (make-source-location input line column offset length)
  source-location?
  (input   source-location-input)
  (line    source-location-line)
  (column  source-location-column)
  (offset  source-location-offset)
  (length  source-location-length))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Validation predicates
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (ident? x) (or (char-alphabetic? x) (memv x '(#\_ #\-))))
(define (css-string? x) (memv x '(#\" #\')))
(define (css-integer? x) (and=> (string->number x) exact?))
(define (pos-integer? x) (and=> (string->number x) exact?))
(define (css-number? x) (string->number x))
(define (pos-css-number? x) (string->number x))

(define (frequency? x)
  (define (f y) (string-suffix? y x))
  (and-let* ((s (find f  '("Hz" "kHz"))))
	    (pos-css-number? (substring x 0 (- (string-length x) s)))))

(define (angle? x)
  (define (f y) (string-suffix? y x)) 
  (and-let* ((s (find f '("deg" "grad" "rad"))))
	    (css-number? (substring x 0 (- (string-length x) s)))))

(define (percentage? x) 
  (and-let* ((p (string-index x #\%))) (string->number (substring x 0 p))))

(define (length? x) 
  (define (f y) (string-suffix? y x))
  (or (and (find f '("em" "ex" "px" "in" "cm" "mm" "pt" "pc"))
	   (css-number? (substring x 0 (- (string-length x) 2))))
      (equal? x "0")))

(define (cue? x) 
  (every? (lambda (x) (or (uri? x) (member x '("none" "inherit"))))
	  (string-tokenize x)))

(define color-table '(("maroon" "#800000") 
		      ("red" "#ff0000") 
		      ("orange" "#ffa500") 
		      ("yellow" "#ffff00") 
		      ("olive" "#808000") 
		      ("purple" "#800080")
		      ("fuchsia" "#ff00ff") 
		      ("white" "#ffffff") 
		      ("lime" "#00ff00") 
		      ("green" "#008000") 
		      ("navy" "#000080") 
		      ("blue" "#0000ff") 
		      ("aqua" "#00ffff") 
		      ("teal" "#008080") 
		      ("black" "#000000") 
		      ("silver" "#c0c0c0") 
		      ("gray" "#808080")))

(define (color? x)
  (define dx (string-downcase x))
  (define (pct? x) (string-suffix? "%" x))
  (define (int? x) (not (or (string-index x #\.) (pct? x))))
  (define ccs (char-set-adjoin char-set:digit #\. #\%))
  (define (f y) (equal? (car y) dx))
  (or (find f color-table)
      (and (string-prefix? "rgb(" x)
	   (let ((s (string-tokenize 
		     (substring x 4 (- (string-length x) 1)) ccs)))
	     (and (= (length s) 3) (or (every pct? s) (every int? s)))))
      (eqv? (string-ref x 0) #\#)))

(define border-styles
  '("none" 
    "hidden" 
    "dotted" 
    "dashed" 
    "solid" 
    "double"
    "groove" 
    "ridge" 
    "inset" 
    "outset"))

(define (border-*-style? x) (equal? x "inherit") (member x border-styles))
(define (border-*-width? x) 
  (or (length? x) (member x '("thin" "medium" "thick"))))

(define (svoice? x) (or (ident? x) (css-string? x)))
(define gvoices '("male" "female" "child"))
(define gfonts '("serif" "sans-serif" "cursive" "fantasy" "monospace"))

(define comma-charset (char-set #\,))
(define (font-family? x)
  (define (font-family-pred y)
    (let ((z (string-trim y)))
      (or (member z gfonts) (string-every char-set:graphic z))))
  (every font-family-pred (string-tokenize x comma-charset)))

(define (font-size? x) 
  (or (length? x) (percentage? x) (member x '("xx-small"
					      "x-small"
					      "small"
					      "medium"
					      "large"
					      "x-large"
					      "xx-large"
					      "larger"
					      "smaller"
					      "inherit"))))
(define (font-style? x) (member x '("normal" "italic" "oblique" "inherit")))
(define (font-variant? x) (member x '("normal" "small-caps" "inherit")))
(define (font-weight? x) (member x '("normal" 
				      "bold"
				      "bolder"
				      "lighter" 
				      "100" 
				      "200" 
				      "300" 
				      "400"
				      "500" 
				      "600" 
				      "700" 
				      "800" 
				      "900" 
				      "inherit")))

(define (line-height? x) (or (css-number? x) 
			     (length? x) 
			     (percentage? x)
			     (member x '("normal" "inherit"))))

;; This isn't 100% accurate, but...
(define (uri? x) (string-prefix? "url(" x))
(define (ls-image? x) (or (uri? x) (equal? x "none")))
(define ls-positions '("inside" "outside"))
(define ls-types '("disc" "circle" "square" "decimal" "decimal-leading-zero" 
		   "lower-roman" "upper-roman" "lower-greek" "lower-latin" 
		   "lower-alpha" "upper-latin" "upper-alpha" "armenian" 
		   "georgian" "none"))

(define (margin? x) (or (length? x) (percentage? x) (equal? x "auto")))

(define (counter? x) 
  (and (string-prefix? "counter(" x)
       (let* ((args (substring x 8 (- (string-length x) 1)))
	      (sx (string-tokenize args (char-set #\,))))
	 (every ident? sx))))
(define (ocolor? x) (or (color? x) (equal? x "invert")))

(define (css-time? x) (or (string-suffix? "ms" x)
			  (string-suffix? "s" x)))

(define (pcolor? x) (or (color? x) (member x '("transparent" "inherit"))))
(define (pimage? x) (or (uri? x) (member x '("none" "inherit"))))
(define prepeats '("repeat" "repeat-x" "repeat-y" "no-repeat" "inherit"))
(define pattaches '("scroll" "fixed" "inherit"))

(define lposition?
  (let ((hori '("left" "center" "right"))
	(vert '("top" "center" "bottom")))
    (lambda (x)
      (or (let ((sx (string-tokenize x)))
	    (and (= (length sx) 2)
		 (let ((casx (car sx))) 
		   (or (percentage? x) (length? x) (member x hori)))
		 (let ((cddx (cadr sx)))
		   (or (percentage? x) (length? x) (member x vert)))))
	  (aomf x `(,hori ,vert))
	  (equal? x "inherit")))))

(define (aomf x y)
  (define (f x y)
    (define (g z) (if (procedure? z) (z (car x)) (member (car x) z)))
    (and y (or (null? x) 
	       (and-let* ((a (find g y))) (f (cdr x) (delete a y))))))
  (let ((sx (string-tokenize x)))
    (and (<= (length sx) (length y)) (f sx y))))

(define (mf expr) (lambda (input) (member input expr)))

(define-record-type :css-property
  (css-property:new match-fn default inherited?)

  css-property?

  (match-fn css-property:match-fn)
  (default css-property:default)
  (inherited? css-property:inherited?))

(define (azimuth? x)
  (or (angle? x)
      (lset<= equal? 
	      (string-tokenize x)
	      ("left-side" 
	       "far-left" 
	       "left" 
	       "center-left" 
	       "center" 
	       "center-right" 
	       "right" 
	       "far-right"
	       "right-side"))
      (member x '("leftwards" "rightwards" "inherit"))))
(define (border-collapse? x) (member x '("collapse" "separate" "inherit")))
(define (border-color? x)
  (define (f x) (or (equal? x "transparent") (pcolor? x)))
  (or (equal? x "inherit") (every f (string-tokenize x))))
(define (border-spacing? x) (every length? (string-tokenize x)))
(define (border-style? x) (every border-*-style? (string-tokenize x)))
(define (border-width? x) (every border-*-width? (string-tokenize x)))
(define (bottom? x) 
  (or (length? x) (percentage? x) (member x '("auto" "inherit"))))
(define (caption-side? x) (member x '("top" "bottom" "inherit")))
(define (clip? x) 
  (or (member x '("auto" "inherit")) (string-prefix? "(rect" x)))
(define (content? x)
  (define (f x str-chr)
    (or (null? x)
	(let ((cx (car x)))
	  (if str-chr
	      (if (eqv? (string-ref cx (- (string-length cx) 1)) str-chr)
		  (f (cdr x) #f)
		  (f (cdr x) str-chr))
	      (let ((sc (string-ref cx 0)))
		(if (memv sc '(#\" #\'))
		    (f (cdr x) sc)
		    (and (or (uri? cx)
			     (counter? cx)
			     (member cx '("open-quote"
					  "close-quote"
					  "no-open-quote"
					  "no-close-quote")))
			 (f (cdr x) #f))))))))
  (or (member x '("normal" "inherit")) (f (string-tokenize x) #f)))
(define (counter-*? x) 
  (or (ident? x) (css-integer? x) (member x '("none" "inherit"))))
(define (cursor? x)
  (or (equal? x "inherit")
      (let* ((sx (string-tokenize x))
	     (lc (last sx)))
	(and (every uri? (delete lc sx))
	     (member lc '("auto"
			  "crosshair"
			  "default"
			  "pointer"
			  "move"
			  "e-resize"
			  "ne-resize"
			  "nw-resize"
			  "n-resize"
			  "se-resize"
			  "sw-resize"
			  "s-resize"
			  "w-resize"
			  "text"
			  "wait"
			  "help"
			  "progress"))))))
(define display-types
  '("inline" 
    "block" 
    "list-item" 
    "run-in" 
    "inline-block" 
    "table"
    "inline-table"
    "table-row-group"
    "table-header-group"
    "table-footer-group"
    "table-row"
    "table-column-group"
    "table-column"
    "table-cell"
    "table-caption"
    "none"))
(define (display? x) (or (member x display-types) (equal? x "inherit")))
(define (elevation? x)
  (or (angle? x)
      (member x '("below" "level" "above" "higher" "lower"))
      (equal? x "inherit")))	      
(define (height? x) 
  (or (length? x) (percentage? x) (member x '("auto" "inherit"))))
(define (letter-spacing? x) (or (length? x) (member x '("normal" "inherit")))) 
(define (line-height? x)
  (or (member x '("normal" "inherit"))
      (css-number? x) (length? x) (percentage? x)))
(define (list-style? x)
  (or (aomf x `(,ls-types ,ls-positions ,ls-image?)) (equal? x "inherit")))
(define (list-style-image? x) (or (equal? x "inherit") (ls-image? x)))
(define (list-style-position? x) 
  (or (equal? x "inherit") (member x ls-positions)))
(define (margin-*? x) 
  (or (length? x) (percentage? x) (member x '("auto" "inherit"))))
(define (margin? x) 
  (or (equal? x "inherit") (every margin-*? (string-tokenize x))))
(define (max-*? x) 
  (or (length? x) (percentage? x) (member x '("none" "inherit"))))
(define (min-*? x) (or (length? x) (percentage? x) (equal? x "inherit")))
(define (orphans? x) (or (css-integer? x) (equal? x "inherit")))
(define (outline? x)
  (or (aomf x `(,ocolor? 
		,border-style?
		,border-width?))
      (equal? x "inherit")))
(define (outline-color? x) (or (ocolor? x) (equal? x "inherit")))
(define (outline-style? x) (or (equal? x "inherit") (member x border-styles)))
(define (outline-width? x) 
  (or (border-width? x) (member x '("inherit" "medium"))))
(define (overflow? x)
  (member x '("visible" "hidden" "scroll" "auto" "inherit")))
(define (padding? x) 
  (or (equal? x "inherit") (every padding-*? (string-tokenize x))))
(define (padding-*? x) (or (length? x) (percentage? x) (equal? x "inherit")))
(define (page-break-*? x) 
  (member x '("auto" "always" "avoid" "left" "right" "inherit")))
(define (pause-*? x) (or (equal? x "inherit") (css-time? x) (percentage? x)))
(define (pause? x) (every pause-*? (string-tokenize x)))
(define (pitch? x) 
  (or (frequency? x) 
      (member x '("x-low" "low" "medium" "high" "x-high" "inherit"))))
(define (pitch-range? x) (or (css-number? x) (equal? x "inherit")))
(define (position? x) 
  (member x '("static" "relative" "absolute" "fixed" "inherit")))
(define (quotes? x)
  (or (member x '("none" "inherit"))
      (let ((sx (string-tokenize x))) ;; This is wrong...
	(and (even? (length sx)) (every css-string? sx)))))
(define (richness? x) (or (equal? x "inherit") (css-number? x)))
(define (speech-rate? x)
  (or (css-number? x) 
      (member x '("x-slow" "slow" "medium" "fast" "x-fast" "faster" "slower"))
      (equal? x "inherit")))
(define (system-font? x)
  (member x '("caption" 
	      "icon" 
	      "menu" 
	      "message-box" 
	      "small-caption" 
	      "status-bar")))
(define (text-align? x) 
  (member x '("left" "right" "center" "justify" "inherit")))
(define (text-decoration? x)
  (define (f x) (member x '("underline" "overline" "line-through" "blink")))
  (or (member x '("inherit" "none")) (every f (string-tokenize x))))
(define (text-indent? x) (or (length? x) (percentage? x) (equal? x "inherit")))
(define (text-transform? x) 
  (member x '("capitalize" "uppercase" "lowercase" "none" "inherit")))
(define (unicode-bidi? x) 
  (member x '("normal" "embed" "bidi-override" "inherit")))
(define (vertical-align? x)
  (or (length? x)
      (percentage? x)
      (member x '("baseline"
		  "sub"
		  "super"
		  "top"
		  "text-top" 
		  "middle" 
		  "bottom" 
		  "text-bottom" 
		  "inherit"))))
(define (white-space? x)
  (member x '("normal" "pre" "nowrap" "pre-wrap" "pre-line" "inherit")))
(define (widows? x) (or (equal? x "inherit") (css-integer? x)))
(define (word-spacing? x) (or (length? x) (member x '("normal" "inherit"))))
(define (visibility? x) (member x '("visible" "hidden" "collapse" "inherit")))
(define (voice-family? x)
  (define (f x) (or (svoice? x) (member x gvoices)))
  (or (equal? x "inherit") (every f (string-tokenizer ", "))))
(define (volume? x)
  (or (css-number? x)
      (percentage? x)
      (member x '("silent" "x-soft" "soft" "medium" "loud" "xloud"))
      (equal? x "inherit")))
(define (z-index? x) (or (css-integer? x) (member x '("auto" "inherit"))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; The property table
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define properties
  `(((azimuth) ,azimuth? ("center") #t)
    ((background-attachment) ,(mf '("scroll" "fixed" "inherit")) ("scroll") #f)
    ((background-color) ,pcolor? ("transparent") #f)
    ((background-image) ,pimage? ("none") #f)
    ((background-position) ,lposition? ("0%" "0%") #f)
    ((background-repeat) ,(mf prepeats) ("repeat") #f)
    ((background) ,(lambda (x) (or (aomf x (list pcolor?
						 pimage? 
						 prepeats
						 pattaches 
						 lposition?))
				   (equal? x "inherit")))
     ("black" "none" "no-repeat" "fixed" "0% 0%") 
     #f)
    ((border-collapse) ,border-collapse? ("separate") #t)
    ((border-color) ,border-color? (color color color color) #t)
    ((border-spacing) ,border-spacing? ("0") #t)
    ((border-style) ,border-style? ("none" "none" "none" "none") #f)
    ((border-top border-right border-bottom border-left)
     ,(lambda (x) (or (aomf x (list border-width? border-style? pcolor?))
		      (equal? "inherit" x)))
     ("medium" "none" color) 
     #f)
    ((border-top-color 
      border-right-color 
      border-left-color 	       
      border-bottom-color) ,pcolor? (color) #t)
    ((border-top-style 
      border-right-style 
      border-left-style 
      border-bottom-style) ,border-*-style? ("none") #f)
    ((border-top-width 
      border-right-width 
      border-bottom-width 
      border-left-width) ,border-*-width? ("medium") #f)
    ((border-width) ,border-width? ("medium" "medium" "medium" "medium") #f)
    ((border) ,(lambda (x) (or (equal? x "inherit")
			     (aomf x `(,border-width?
				       ,border-style? 
				       ,pcolor?))))
     ("medium" "none" color) #f)
    ((bottom) ,bottom? ("auto") #f)
    ((caption-side) ,caption-side? ("top") #t)
    ((clear) ,(mf '("none" "left" "right" "both" "inherit")) ("none") #f)
    ((clip) ,clip? ("auto") #f)
    ((color) ,pcolor? ("white") #t)
    ((content) ,content? ("normal") #f)
    ((counter-increment counter-reset) ,counter-*? ("none") #f)
    ((cue-after cue-before) ,cue? ("none") #f)
    ((cue) ,cue? ("none" "none") #f)
    ((cursor) ,cursor? ("auto") #t)
    ((direction) ,(mf '("ltr" "rtl" "inherit")) ("ltr") #t)
    ((display) ,display? ("inline") #f)
    ((elevation) ,elevation? ("level") #t)
    ((empty-cells) ,(mf '("show" "hide" "inherit")) ("show") #t)
    ((float) ,(mf '("left" "right" "none" "inherit")) ("none") #f)
    ((font-family) ,font-family? ("monospace") #t)
    ((font-size) ,font-size? ("medium") #t)
    ((font-style) ,font-style? ("normal") #t)
    ((font-variant) ,font-variant? ("normal") #t)
    ((font-weight) ,font-weight? ("normal") #t)
    ((font) ,(lambda (x)
	       (let ((f (lambda (x) (or (font-style? x)
					(font-variant? x)
					(font-weight? x))))
		     (g (lambda (x)
			  (let ((si (string-index x #\/)))
			    (if si 
				(let ((a (substring x 0 si))
				      (b (substring x (+ si 1))))
				  (and (font-size? a)
				       (line-height? b)))
				(font-size? x))))))
		 (or (system-font? x)
		     (equal? x "inherit")		     
		     (let ((sx (string-tokenize x)))
		       (and (>= (length sx) 2)
			    (g (car sx))
			    (font-family? (cadr sx)))))))
          ("normal" "normal" "normal" "medium" "monospace") 
   	  #t)
    ((height left right width bottom top) ,height? ("auto") #f)
    ((letter-spacing) ,letter-spacing? ("normal") #t)
    ((line-height) ,line-height? ("normal") #t)
    ((list-style-image) ,list-style-image? ("none") #t)
    ((list-style-position) ,list-style-position? ("outside") #t)
    ((list-style-type) ,(mf (cons "inherit" ls-types)) ("disc") #t)
    ((list-style) ,list-style? ("disc" "outside" "none") #t)
    ((margin-right margin-left margin-top margin-bottom) ,margin-*? ("0") #f)
    ((margin) ,margin? ("0" "0" "0" "0") #f)
    ((max-height max-width) ,max-*? ("none") #f)
    ((min-height min-width) ,min-*? ("0") #f)
    ((orphans) ,orphans? ("2") #t)
    ((outline-color) ,outline-color? ("invert") #f)
    ((outline-style) ,outline-style? ("none") #f)
    ((outline-width) ,outline-width? ("medium") #f)
    ((outline) ,outline? ("invert" "none" "medium") #f)
    ((overflow) ,overflow? ("visible") #f)
    ((padding-top 
      padding-right 
      padding-bottom 
      padding-left) ,padding-*? ("0") #f)
    ((padding) ,padding? ("0" "0" "0" "0") #f)
    ((page-break-after page-break-before) ,page-break-*? ("auto") #f)
    ((page-break-inside) ,(mf '("avoid" "auto" "inherit")) ("auto") #t)
    ((pause-after pause-before) ,pause-*? ("0") #f)
    ((pause) ,pause? ("0" "0") #f)
    ((pitch-range) ,pitch-range? ("50") #t)
    ((pitch) ,pitch? ("medium") #t)
    ((play-during) ,(lambda (x) (or (member x '("auto" "none" "inherit"))
				  (let ((sx (string-tokenize x)))
				    (and (uri? (car x))
					 (aomf (cdr x) '(("mix") 
							 ("repeat")))))))
		 ("auto") 
		 #f)
    ((position) ,position? ("static") #f)
    ((quotes) ,quotes? ("none") #t)
    ((richness) ,richness? ("50") #t)
    ((speak-header) ,(mf '("once" "always" "inherit")) ("once") #t)
    ((speak-numeral) ,(mf '("digits" "continuous" "inherit")) ("continuous") #t)
    ((speak-punctuation) ,(mf '("code" "none" "inherit")) ("none") #t)
    ((speak) ,(mf '("normal" "none" "spell-out" "inherit")) ("normal") #t)
    ((speech-rate) ,speech-rate? ("medium") #t)
    ((stress) ,(lambda (x) (or (equal? x "inherit") (number? x))) ("50") #t)
    ((table-layout) ,(mf '("auto" "fixed" "inherit")) ("auto") #f)
    ((text-align) ,text-align? ("left") #t)
    ((text-decoration) ,text-decoration? ("none") #f)
    ((text-indent) ,text-indent? ("0") #t)
    ((text-transform) ,text-transform? ("none") #t)
    ((unicode-bidi) ,unicode-bidi? ("normal") #f)
    ((vertical-align) ,vertical-align? ("baseline") #f)
    ((visibility) ,visibility? ("visible") #t)
    ((voice-family) ,voice-family? ("female") #t)
    ((volume) ,volume? ("medium") #t)
    ((white-space) ,white-space? ("normal") #t)
    ((widows) ,widows? ("2") #t)
    ((word-spacing) ,word-spacing? ("normal") #t)
    ((z-index) ,z-index? ("auto") #f)))

(define property-hash-table
  (let ((pht (make-hash-table eq? hash-by-identity)))
    (for-each (lambda (x)
		(let ((rec (apply css-property:new (cdr x))))		  
		  (for-each (lambda (y) (hash-table-set! pht y rec)) (car x))))
	      properties)
    pht))

(define (get-prop-entry p) (hash-table-ref/default property-hash-table p #f))

(define (get-default-prop-value pe)
  (define (foldfn x y)
    (let ((z (if (equal? y "") "" (string-append " " y))))
      (if (symbol? x)
	  (string-append (get-default-prop-value (get-prop-entry x)) z)
	  (string-append x z))))
  (and pe (fold foldfn "" (css-property:default pe))))


(define (chash cascade n)
  (modulo (+ (hash-by-identity (car cascade) n) 
	     (hash-by-identity (cadr cascade) n)
	     (hash-by-identity (caddr cascade) n))
	  n))
(define (cequal? x y) (list= eq? x y))

(define cht (make-hash-table cequal? chash)) ;; cascade -> hash table
;; doc * elt * cascade -> hash table
(define dht (make-hash-table eq? hash-by-identity)) 

(define (get-sxml-parent doc node)
  (define (f x) (get-sxml-parent x node))
  (if (memq node doc) doc (let ((c (filter list? (cdr doc))))
			    (if c (find f c) #f))))

(define (sxml-node-name node)
  (let* ((str (false-if-exception (symbol->string (car node))))
	 (ri (if str (string-rindex str #\:) #f)))
    (if ri (substring str (+ ri 1)) str)))

(define (sxml-attr-val node name)
  (define (find-attrs x) (and (list? x) (eq? (car x) '@)))
  (define (find-attr x) (and (list? x) (equal? (symbol->string (car x)) name)))
  (and (symbol? (car node))
       (and-let* ((attrs (find find-attrs (cdr node))))
		 (and=> (find find-attr (cdr attrs)) cadr))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Handler management functions
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (scss:set-sxml-parent-function! proc) 
  (verify-arg-types "scss:set-sxml-parent-function!" 
		    (list procedure?) 
		    (list proc)
		    1)
  (set! get-sxml-parent proc))

(define (internal-dot-handler sel doc node) #f)
(define (scss:set-dot-handler! p)
  (verify-arg-types "scss:set-dot-handler!" (list procedure?) (list p) 1)
  (set! internal-dot-handler p))

(define (internal-id-handler str doc node) #f)
(define (scss:set-id-handler! p) 
  (verify-arg-types "scss:set-id-handler!" (list procedure?) (list p) 1)
  (set! internal-id-handler p))

(define (internal-pseudo-class-handler str doc node) #f)
(define (scss:set-pseudo-class-handler! p)
  (verify-arg-types "scss:set-pseudo-class-handler!" 
		    (list procedure?) 
		    (list p)
		    1)
  (set! internal-pseudo-class-handler p))

(define (internal-pseudo-element-handler str doc node) #f)
(define (scss:set-pseudo-element-handler! p)
  (verify-arg-types "scss:set-pseudo-element-handler!"
		    (list procedure?) 
		    (list p)
		    1)
  (set! internal-pseudo-element-handler p))

(define (internal-uri-import-handler uri) (open-input-string ""))
(define (scss:set-uri-import-handler! p)
  (verify-arg-types "scss:set-uri-import-handler!"
		    (list procedure?)
		    (list p)
		    1)
  (set! internal-uri-import-handler p))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Parsing functions
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (combine sel combinator sels)
  (let ((csels (car sels)))
    (cons (list (string->symbol combinator) sel csels) (cdr sels))))

(define (parse-attrib-str str)
  (define (quote-pred x) (or (eqv? #\" x) (eqv? #\' x)))
  (define (tilde-pred x) (eqv? #\~ x))
  (define (pipe-pred x) (eqv? #\| x))
  (define (equal-pred x) (eqv? #\= x))

  (let* ((chrs (string->list str))
	 (quot (list-index quote-pred chrs))
	 (p (list-index tilde-pred chrs))
	 (p (or (and p (or (not quot) (< p quot)) p)
		(list-index pipe-pred chrs)))
	 (p (or (and p (or (not quot) (< p quot)) p)
		(list-index equal-pred chrs))))
    (if p
	(list (case (list-ref chrs p) ((#\~) '~=) ((#\|) '|=) ((#\=) '=))
	      (string->symbol (list->string (take (cdr chrs) (- p 1)))) 
	      (let* ((ccs (cdr chrs))
		     (val (list-tail ccs (if (eqv? (list-ref chrs p) #\=) 
					     p 
					     (+ p 1)))))
		(list->string (take (if quot (cdr val) val) 
				    (- (length val) (if quot 3 1))))))
	(string->symbol 
	 (list->string (take (cdr chrs) (- (length chrs) 2)))))))

(define (add-specifiers sel specifiers)
  (define (contains-special-chars? str)
    (define special-charset (list->char-set '(#\. #\: #\| #\[ #\' #\")))
    (define (contains-unescaped-special-chars? str start end)
      (let ((slash (string-index str #\\ start end)))
	(if (and slash (<= (+ slash 2) end))
	    (or (contains-unescaped-special-chars? str start slash)
		(contains-unescaped-special-chars? str (+ slash 2) end))
	    (string-index str special-charset start end))))
    (define (contains-special-chars-pivot? start x len end)
      (or (string-index str special-charset start x)
	  (string-index str special-charset (+ x len) end)))
    (let ((len (string-length str)))
      (case (string-ref str 0)
	((#\.) (string-index str special-charset 1 len))
	((#\:) (string-index str special-charset 1 len))
	((#\|) (string-index str special-charset 2 (- len 1)))
	((#\[) 
	 (cond ((string-contains str "|=") => 
		(lambda (x) (contains-special-chars-pivot? 1 x 2 (- len 1))))
	       ((string-contains str "~=") =>
		(lambda (x) (contains-special-chars-pivot? 1 x 2 (- len 1))))
	       ((string-contains str "=") =>
		(lambda (x) (contains-special-chars-pivot? 1 x 1 (- len 1))))
	       (else (string-index str special-charset 1 len))))
	(else (string-index str special-charset 0 len)))))

  (define (add-specifier specifier s)
    (let ((sm (substring specifier 1)))
      (case (string-ref specifier 0)
	((#\.) (list 'class s (string->symbol sm)))
	((#\:) (let ((x (string->symbol sm)))
		 (cond ((member x pseudo-classes) (list 'pclass s x))
		       ((member x pseudo-elements) (list 'pelement s x))
		       (else s))))
	((#\|) `(id ,s ,(string->symbol 
			 (substring sm 1 (string-index sm #\|)))))
	((#\[) `(attrib ,s ,(parse-attrib-str specifier)))
	(else s))))

  (define (valid-specifier? m)
    (if (eqv? (string-ref m 0) #\:)
	(let ((n (substring m 1)))
	  (or (member n pseudo-classes) (member n pseudo-elements)))
	#t))

  (let ((mods (if (list? specifiers) 
		  (map replace-escapes specifiers)
		  (map replace-escapes (list specifiers))))
	(sel (and sel (replace-escapes sel))))
    (and (every valid-specifier? mods)
	 (if (and (not (and sel (contains-special-chars? sel)))
		  (not (find contains-special-chars? mods)))
	     (string->symbol (string-append (or sel "") 
					    (apply string-append mods)))
	     (fold add-specifier (string->symbol (or sel "*")) mods)))))

(define pseudo-classes
  '("first-child" "link" "visited" "hover" "active" "focus" "lang"))

(define pseudo-elements '("first-line" "first-letter" "before" "after"))

(define (null-merge x y)
  (if (not (null? x)) 
      (if (null? y) `(,x) (if (list? y) (cons x y) `(,x ,y)))
      y))

(define (can-parse-unicode? num)
  (or (< num 256) 
      (let ((v (string-split (version) #\.)))
	(or (> 6 (string->number (cadr v)))
	    (> 1 (string->number (car v)))))))

(define (replace-escapes s)
  (define (list->char lst)
    (let ((n (string->number (list->string lst) 16)))      
       (if (can-parse-unicode? n)
	  (list (integer->char n))
	  lst)))

  ;; TODO: Tidy this up...

  (define (f l esc ws cs)
    (let ((c (and (not (null? l)) (car l))))
      (cond ((not c) (if (null? cs) '() (list->char cs)))
	    ((and (char-whitespace? c) (not (null? cs))) (f (cdr l) #f #t cs))
	    ((= (length cs) 6) (append (list->char cs) (f l #f #f '())))
	    (ws (if (char-whitespace? c) 
		    (f (cdr l) #f #t cs)
		    (append (list->char cs) (f l #f #f '()))))
	    ((or (char-numeric? c) (and (char-ci>=? c #\a) (char-ci<=? c #\f)))
	     (cond (esc (f (cdr l) #f #f `(,c)))
		   ((null? cs) (append `(,c) (f (cdr l) #f #f '())))
		   (else (f (cdr l) #f #f (append cs `(,c))))))
	    ((eqv? c #\\) (if esc
			      (append (list #\\) (f (cdr l) #f #f '()))
			      (if (null? cs) 
				  (f (cdr l) #t #f '())
				  (append (list->char cs)
					  (f (cdr l) #t #f '())))))
	    (else (cond (esc (append (list (if (char-whitespace? c) #\space c))
				     (f (cdr l) #f #f '())))
			((null? cs) (append `(,c) (f (cdr l) #f #f '())))
			(else (append (list->char cs)
				      `(,c) (f (cdr l) #f #f '()))))))))
  (if (symbol? s)
      (string->symbol 
       (list->string 
	(f (string->list (symbol->string s)) #f #f '())))
      (list->string (f (string->list s) #f #f '()))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Selector detection
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (end-index str start-char from)
  (define (index-filter x) (string-index str x from))
  (let* ((chrs (delete start-char (list #\. #\: #\| #\[)))
	 (indices (filter-map index-filter chrs)))
    (if (null? indices)
	(string-length str)
	(car (list-sort < indices)))))

(define (class? x recurse) 
  (define (f y z)
    (define (g a) (substring y (+ z 1) a))
    (or (and=> (end-index y #\. (+ z 1)) g)
	(substring y (+ z 1))))

  (if (list? x)
      (and (eq? (car x) 'class)
	   (case (length x)
	     ((2) (cadr x))
	     ((3) (and (if recurse (selector? (cadr x)) #t)
		       (symbol? (caddr x)) 
		       (caddr x)))
	     (else #f)))
      (let ((y (symbol->string x)))
	(let ((z (string-index y #\.)))
	  (if (and z (not (eqv? z (string-index-right y #\. (+ z 1)))))
	      (f y z)
	      #f)))))

(define (xor x y) (if x (and (not y) x) y))

(define (colon-match? x sym lst recurse)
  (if (list? x)
      (and (eq? (car x) sym)
	   (case (length x)
	     ((2) (and=> (member (cadr x) lst) car))
	     ((3) (and (if recurse (selector? (cadr x) recurse) #t) 
		       (and=> (member (caddr x) lst) car)))
	     (else #f)))
      (let ((y (symbol->string x)))
	(let* ((z1 (string-index y #\:))
	       (z2 (if z1 (string-index y #\: (+ z1 1)) #f))
	       (s1 (and z1 (if z2 
			       (substring y (+ z1 1) z2)
			       (substring y (+ z1 1)))))
	       (s2 (if z2 (substring y (+ z2 1)) #f)))
	  (xor (and=> (member s1 lst) car) (and=> (member s2 lst) car))))))

(define (pseudo-class? x recurse) 
  (colon-match? x 'pclass pseudo-classes recurse))
(define (pseudo-element? x recurse) 
  (colon-match? x 'pelement pseudo-elements recurse))

(define (id? x recurse) 
  (if (list? x)
      (and (eq? (car x) 'id)
	   (case (length x)
	     ((2) (cadr x))
	     ((3) (and (if recurse (selector? (cadr x) #t) #t)
		       (symbol? (caddr x)) 
		       (caddr x)))
	     (else #f)))
      (let* ((y (symbol->string x))
	     (ididx (string-index y #\|))
	     (endidx (and ididx (string-index y #\| (+ ididx 1)))))
	(and ididx
	     endidx
	     (eqv? (string-ref y (+ ididx 1)) #\#)
	     (substring y (+ ididx 2) endidx)))))

(define (attrib? x recurse)
  (if (list? x)
      (and (= (length x) 3) 
	   (eq? (car x) 'attrib) 
	   (if recurse (selector? (cadr x)) #t) 
	   (caddr x))
      (let* ((str (symbol->string x))
	     (start (string-index str #\[))
	     (end (string-index str #\] (if start start 0))))
	(and start end (parse-attrib-str (substring str start (+ end 1)))))))

(define (and-wrapper x y) (and x y))
(define (selector-recurse? x) (selector? x #t))
(define (contextual? x recurse)
  (and (list? x) 
       (eq? (car x) '//)
       (if recurse (fold and-wrapper #t (map selector-recurse? (cdr x))) #t)))
(define (simple? x recurse)
  (and (symbol? x)
       (let* ((str (symbol->string x))
	      (len (string-length str))
	      (end (min (or (string-index str #\.) len)
			(or (string-index str #\:) len)
			(or (string-index str #\|) len)
			(or (string-index str #\[) len))))
	 (if (eqv? end 0) "*" (substring str 0 end)))))
	 
(define (grouping? x recurse)
  (and (list? x) 
       (not (memq (car x) '(// + > class pclass pelement attrib id @import)))
       (if recurse (fold and-wrapper #t (map selector-recurse? x)) #t)
       x))

(define (combinator? x recurse)
  (and (list? x) (let ((y (car x)))
		   (and (memq y '(+ >))
			(= (length x) 3)
			(or (not recurse)
			    (and (selector? (cadr x) #t) 
				 (selector? (caddr x) #t)))
			y))))

(define (css-path? x recurse)
  (or (combinator? x recurse) (contextual? x recurse)))

(define (selector? x recurse)
  (or (and (list? x) (or (contextual? x recurse) (grouping? x recurse)))
      (class? x recurse)
      (pseudo-class? x recurse)
      (pseudo-element? x recurse)
      (id? x recurse)
      (attrib? x recurse)
      (combinator? x recurse)
      (simple? x recurse)))

;; Returns false or a cons of #t and a boolean indicating whether to cache the
;; value or not

(define (simple-selector-match? sel d node)
  (define (f preds tests c)
    (or (and (null? preds) (cons #t c))
	(let* ((x ((car preds) sel #f))
	       (y (if x ((car tests) x) #t)))
	  (and y
	       (f (cdr preds) 
		  (cdr tests)
		  (or c (not (and x (eq? (car preds) pseudo-element?)))))))))

  (define (imp? x y) (and (list? x) (eq? (car x) '@import)))

  (define (test-imp x) #f)
  (define (test-simple x) (or (equal? x "*") (equal? (sxml-node-name node) x)))
  (define (test-class x) (equal? (sxml-attr-val node "class") x)) 
  (define (test-pseudo-class x) (internal-pseudo-class-handler x d node))
  (define (test-pseudo-element x) (internal-pseudo-element-handler x d node))
  (define (test-id x) (internal-id-handler x d node))
  (define (test-attr x) 	    
    (if (list? x)
	(let ((y (sxml-attr-val node (symbol->string (cadr x)))))
	  (and y (case (car x)
		   (('~=) (member (caddr x) (string-tokenize y)))
		   (('|=) (member (caddr x) 
				  (string-tokenize y char-set:letter)))
		   (else (equal? y (caddr x))))))
	(sxml-attr-val node (symbol->string x))))

  (f (list imp? simple? class? pseudo-class? pseudo-element? id? attrib?)
     (list test-imp
	   test-simple
	   test-class
	   test-pseudo-class
	   test-pseudo-element
	   test-id
	   test-attr)
     #f))

;; Returns false or a cons of #t and a boolean indicating whether to cache the
;; value or not

(define (combinator-selector-match? sel d nodes combinator)
  (define (sxml-get-priv-sibling node doc)
    (define (f t)
      (and (not (null? t))
	   (not (null? (cdr t)))
	   (or (let ((ct (car t)))
		 (and (list? ct) 
		      (not (eq? (car ct) '@)) (eq? (cadr t) node) 
		      (car t)))
	       (f (cdr t)))))
  (f (get-sxml-parent doc node)))
  (if (eq? combinator '+)
      (let ((sib (sxml-get-priv-sibling (car nodes) d)))
	(and sib
	     (let ((m1 (simple-selector-match? (cadr sel) d sib))
	           (m2 (simple-selector-match? (caddr sel) d (car nodes))))
	       (and m1 m2 (cons #t (and (cdr m1) (cdr m2)))))))
      (let ((m1 (simple-selector-match? (cadr sel) d (car nodes)))
	    (m2 (simple-selector-match? (caddr sel) d (car nodes))))
	       (and m1 m2 (cons #t (and (cdr m1) (cdr m2)))))))

;; Returns false or a cons of #t and a boolean indicating whether to cache the
;; value or not

(define (path-selector-match? sel d nodes slen nlen)
  (define (ftail s x) 
    (define (f y) (equal? (sxml-node-name y) s))
    (find-tail f x))
  (and (<= slen nlen)
       (let* ((csel (car sel))
	      (r (if (combinator? csel #f)
		     (combinator-selector-match? csel d nodes)
		     (simple-selector-match? csel d (car nodes)))))
	 (if (null? (cdr sel)) 
	     r
	     (let* ((s (simple? (cadr sel) #f))
		    (t (or (and (equal? s "*") (cdr nodes))
			   (ftail s (cdr nodes))))
		    (sl1 (- slen 1))
		    (u (and t (path-selector-match? 
			       (cdr sel) d t sl1 (length t)))))
	       (and r u (cons #t (and (cdr r) (cdr u)))))))))

;; Returns false or a cons of #t and a boolean indicating whether to cache the
;; value or not

(define (selector-match? sel d n)
  (define (path-length sel)
    (define (f x y) (+ y (if (combinator? x #f) 2 1)))
    (fold f 0 sel))
  (let ((x (combinator? sel #f)))
    (if x 
	(combinator-selector-match? sel d n x)
	(if (contextual? sel #f)
	    (let* ((rsel (reverse (cdr sel)))
		   (psel (path-length rsel)))
	      (path-selector-match? rsel d n psel (length n)))
	    (simple-selector-match? sel d (car n))))))

(define (calc-precedence item)
  (case (car item)
    ((agent) 1)
    ((user) (if (and (= (length item) 6) (eq? (list-ref item 2) '!)) 5 2))
    (else (if (and (= (length item) 6) (eq? (list-ref item 2) '!)) 4 3))))

(define (calc-specificity sel)
  (define (count pred lst) (length (filter pred lst)))
  (define (id-pred x) (id? x #f))
  (define (alt-pred-1 x) 
    (or (pseudo-class? x #f) (attrib? x #f) (class? x #f)))
  (define (alt-pred-2 x) (or (pseudo-element? x #f) (simple? x #f)))
  (let* ((sel (cadr sel))
	 (lsel (if (contextual? sel #f) (cdr sel) (list sel))))
    (+ (* (count id-pred lsel) 100)
       (* (count alt-pred-1 lsel) 10)
       (count alt-pred-2 lsel))))

(define (validate-property property value)
  (let ((pe (hash-table-ref/default property-hash-table 
				    (symbol-downcase property)
				    #f)))
    (and pe (apply (css-property:match-fn pe) 
		   `(,(string-downcase value))))))

(define (verify-arg-types name predicates args startnum)
  (if (not (= (length predicates) (length args)))
      (error #t "predicate argument mismatch"))
  (letrec ((f (lambda (l1 l2 i)
		(or (null? l1) 
		    (if (apply (car l1) (list (car l2)))
			(f (cdr l1) (cdr l2) (+ i 1))
			(error #t (string-append 
				   name ": wrong argument type in position "
				   (number->string (+ startnum 1)))))))))
    (f predicates args 0)))

(define scss:parser-debug-messages #t)
(define (display-list . lst) (if scss:parser-debug-messages (display lst)))
 
(define (parse port)
  (let* ((lexer (make-css21-lexer port))
	 (parser (make-css21-parser lexer)))
    (parser)))

(define (scss:css->scss port . baseurl)
  (define (dirstr str) 
    (let ((s (string-index-right str #\/)))
      (if s (substring str 0 (+ s 1)) "")))
  (define (import-fn s i)
    (if (and (not (null? s)) (not (null? (cdr s))))
	(if (eq? (caar s) '@import) (import-fn (cdr s) (cons (cadar s) i)) i)
	i))

  (verify-arg-types "scss:css->scss" (list port?) (list port) 1)

  (if (not (null? baseurl)) 
      (verify-arg-types "scss:css->scss" (list string?) baseurl 1))

  (let* ((bu (if (null? baseurl)
		 (if (file-port? port) (dirstr (port-filename port)) (getcwd))
		 (car baseurl)))
	 (parsed-sheet (parse port))
	 (imports (import-fn (cdr parsed-sheet) (list)))
	 (clean-sheet (list-tail (cdr parsed-sheet) (length imports)))
	 (final (fold-right 
		 (lambda (s1 s2)
		   (let* ((s1 (if (string-prefix? "file://" s1)
				  (substring s1 7)
				  s1))
			  (bu (if (string-prefix? "file://" bu)
				  (substring bu 7)
				  bu))
			  (nbu (dirstr (if (string-prefix? "/" s1)
					   s1
					   (string-append bu "/" s1))))
			  (p (false-if-exception
			      (if (string-prefix? "/" s1)
				  (open-input-file s1)
				  (open-input-file (string-append 
						    bu "/" s1)))))
			  (n (false-if-exception (scss:css->scss p nbu))))
		     (if n (append (cdr n) s2) s2)))
		 clean-sheet
		 imports)))
    (cons 'css final)))

(define (scss:scss->css stylesheet port)
  (let* 
      ((ser-selector 
	(lambda (selector)
	  (display 
	   (fold (lambda (x y) 
		   (string-append 
		    (fold (lambda (q r)
			    (string-append 
			     q (if (or (ident? r)
				       (equal? r ">")
				       (equal? r "+")
				       (equal? r "*"))
				   " " "") r))
			  ""
			  (reverse x))
		    (if (null? y) "" (string-append ", " y))))
		 '()
		 (reverse selector))
	   port)))
       (ser-property (lambda (pair) 
		       (display "\t" port)
		       (display (car pair) port)
		       (display ": " port)
		       (display (cadr pair) port)
		       (display ";" port)
		       (newline port)))
       (ser-block (lambda (block)
		    (ser-selector (car block))
		    (display " {" port)
		    (newline port)
		    (for-each (lambda (x) (ser-property x)) (cadr block))
		    (display "}" port)
		    (newline port)
		    (newline port))))			  
    (verify-arg-types "scss:scss->css"
		      (list scss:stylesheet? port?) 
		      (list stylesheet port)
		      1)
    (for-each (lambda (block) (ser-block block)) stylesheet)))

(define (scss:create-cascade . arglist)
  (if (null? arglist) 
      (list (list) (list) (list))
      (if (= (length arglist) 3)
	  (begin (verify-arg-types "scss:create-cascade"
				   (make-list 3 scss:stylesheet?) 
				   arglist
				   1)
		 arglist)
	  (error #t "wrong number of arguments to scss:create-cascade"))))
 
(define (scss:set-author-stylesheet! cascade authorsheet) 
  (set-car! (list-tail cascade 2) authorsheet))

(define (scss:set-user-stylesheet! cascade usersheet) 
  (set-car! cascade usersheet))

(define (scss:set-agent-stylesheet! cascade agentsheet) 
  (set-car! (list-tail cascade 1) agentsheet))

(define (scss:cascade? cascade)
  (and (list? cascade) 
       (= (length cascade) 3) 
       (every (lambda (x) (or (null? x) (scss:stylesheet? x))) cascade)))

(define (scss:stylesheet? stylesheet)
  (and (list? stylesheet)
       (every (lambda (x) 
		(and (list? x)
		     (>= (length x) 2)
		     (list? (car x))
		     (list? (cdr x))
		     (every (lambda (y)
			      (and (list? y)
				   (or (= (length y) 2)
				       (and (= (length y) 3)
					    (eq? (list-ref y 2) '!)))
				   (string? (car y))
				   (string? (cadr y))))
			    (cdr x))))
	      stylesheet)))

(define (scss:color->hex color)
  (verify-arg-types "scss:color->hex" (list string?) (list color) 1)
  (let* ((dc (string-downcase color))
	 (c (find (lambda (x) (equal? (car x) dc)) color-table)))
    (if c (cadr c) (list))))

(define gpt
  (let* ((gps '((background-attachment background)
		(background-color background)
		(background-image background)
		(background-position background)
		(background-repeat background)

		(font-family font)
		(font-size font)
		(font-style font)
		(font-variant font)
		(font-weight font)
		
		(margin-top margin)
		(margin-left margin)
		(margin-bottom margin)
		(margin-right margin)
		
		(outline-top outline)
		(outline-left outline)
		(outline-bottom outline)
		(outline-right outline)
		
		(padding-top padding)
		(padding-left padding)
		(padding-bottom padding)
		(padding-right padding)
		
		(list-style-image list-style)
		(list-style-position list-style)
		(list-style-type list-style)
		
		(border-top-style border-top border-style border)
		(border-left-style border-left border-style border)
		(border-bottom-style border-bottom border-style border)
		(border-right-style border-right border-style border)
		
		(border-top-color color border-top border)
		(border-left-color color border-left border)
		(border-bottom-color color border-bottom border)
		(border-right-color color border-right border)
		
		(border-top-width border-top border-width border)
		(border-left-width border-left border-width border)
		(border-bottom-width border-bottom border-width border)
		(border-right-width border-right border-width border)))
		
	 (ht (make-hash-table eq? hash-by-identity)))
    (for-each (lambda (x) (hash-table-set! ht (car x) (cdr x))) gps) ht))

(define (symbol-downcase x)
  (string->symbol (string-downcase (symbol->string x))))

;; This function is a bit complicated.  The general idea is that for certain
;; properties, if they're not explicitly defined in the matched selector 
;; block, you can derive their value by looking at a more general property 
;; that is defined in that block.

(define (specify sprop val)
  (define (g pred d x)
    (let ((ml (filter pred (string-tokenize x))))
      (case (length ml)
	((1) (car ml)) 
	((2) (if (member d '("top" "bottom")) (car ml) (cadr ml)))
	((3) (cond ((equal? d "top") (car ml))
		   ((member d '("left" "right")) (cadr ml))
		   (else (caddr ml))))
	((4) (list-ref ml (list-index (lambda (y) (equal? y d))
				      '("top" "right" "bottom" "left"))))
	(else #f))))

  ;; If the color property is ready then apply it here, otherwise defer the
  ;; evaluation...

  (let* ((prop-bits (string-tokenize (symbol->string sprop) char-set:letter))
	 (c (car prop-bits))
	 (cc (cadr prop-bits)))
	 
	 ;; The way values are mapped to specific properties depends on the 
	 ;; number of values as per 
	 ;; http://www.w3.org/TR/CSS21/box.html#propdef-border-top-color
	 
    (and (> (length prop-bits) 1)
	 (cond 
	  ((equal? (car prop-bits) "border")
	   (if (= (length prop-bits) 2)
	       (cond ((equal? cc "color") (if (color? val) val 'color))
		     ((equal? cc "width") (and (border-width? val) val))
		     ((equal? cc "style") (border-style? val))
		     
		     ;; Else it's one of top, left, bottom, right
		     
		     (else #f))
	       (let* ((ccc (caddr prop-bits)))
		 (cond ((equal? ccc "color") (or (g color? cc val) 'color))
		       ((equal? ccc "width") (g border-width? cc val))
		       ((equal? ccc "style") (g border-style? cc val))))))
	  ((equal? c "padding") (g padding? cc val))
	  ((equal? c "margin") (g margin? cc val))

	  ;; The `background' property is different from some of the other
	  ;; shorthand properties in that it's possible to give it a value that
	  ;; doesn't give computed values to all of the more specific
	  ;; properties for which it is a shorthand.  In these cases, the
	  ;; specific properties should be explicitly set to their default
	  ;; values.

	  ((equal? c "background")
	   (cond ((equal? cc "color") (color? val) val)
		 ((equal? cc "attach") (member val pattaches))
		 ((equal? cc "image") (and (pimage? val) val))
		 ((equal? cc "repeat") (member val prepeats))
		 ;; FIX THIS!!!
		 ((equal? cc "position"))
		 (else #f)))
	  ((and (equal? c "list") (equal? cc "style"))
	   (let* ((ccc (caddr prop-bits)))
	     (cond ((equal? ccc "position")
		    (member val (cons "inherit" ls-positions)))
		   ((equal? ccc "image")
		    (or (ls-image? val) (equal? val "inherit")))
		   ((equal? ccc "type") (member val (cons "inherit" ls-types)))
		   (else #f))))
	  ((equal? c "outline")
	   (cond ((equal? cc "color") (and (color? val) val))
		 ((equal? cc "style") (and (border-style? val) val))
		 ((equal? cc "width") (and (border-width? val) val))
		 (else #f)))

	  ;; Like `background', the shorthand `font' property sets all values
	  ;; that can't be deduced from its computed value to their defaults.

	  ;; TODO: Actually parse the computed values!

	  ((equal? c "font")
	   (cond ((equal? cc "family") 
		  (or (and (font-family? val) val)
		      (get-default-prop-value (get-prop-entry 'font-family))))
		 ((equal? cc "size") 
		  (or (and (font-size? val) val)
		      (get-default-prop-value (get-prop-entry 'font-size))))
		 ((equal? cc "style") 
		  (or (and (font-style? val) val)
		      (get-default-prop-value (get-prop-entry 'font-style))))
		 ((equal? cc "variant") 
		  (or (and (font-variant? val) val)
		      (get-default-prop-value (get-prop-entry 'font-variant))))
		 ((equal? cc "weight") 
		  (or (and (font-weight? val) val)
		      (get-default-prop-value (get-prop-entry 'font-weight))))
		 (else #f)))
	  (else #f)))))

(define (filter-media medium cascade)
  (define (g x)
    (define (f y)
      (if (eq? (car y) '@media)
	  (let ((z (cadr y)))
	    (and (if (list? z)
		     (or (memq medium z) (memq 'all z))
		     (or (eq? medium z) (eq? 'all z)))
		 (cddr y)))
	  (list y)))
    (let ((r (fold-right append '() (filter-map f (if (null? x) x (cdr x))))))
      (if (null? r) r (cons 'css r))))
  (map g cascade))

(define (precedence-less x y) (< (calc-precedence x) (calc-precedence y)))
(define (specificity-less x y) (< (calc-specificity x) (calc-specificity y)))

;; Caching fn for text-based queries

(define (lookup-values cascade selector medium table)
  (define (find-matches sym sheet ss)
    (define (map-fn x)
      (define (filt-fn y)
	(define (match-pred x) (or (eq? x '*) (eq? x ss)))
	(define (inner-map-fn z) (cons* sym y z))
	(and (match-pred y) (map inner-map-fn (cdr x))))
      (filter-map filt-fn (let ((c (car x))) (or (grouping? c #f) (list c)))))
    (fold-right append '() (map map-fn (if (null? sheet) sheet (cdr sheet)))))
  (define (add! data order)
    (or (null? data) 
	(let* ((x (car data))
	       (imp (eq? (caddr x) '!))
	       (sdcy (symbol-downcase (list-ref x (if imp 3 2))))
	       (sdcv (string-downcase (list-ref x (if imp 4 3)))))
	  (if (hash-table-ref/default property-hash-table sdcy #f)
	      (hash-table-set! table sdcy `(,sdcy ,sdcv ,order)))
	  (add! (cdr data) (+ order 1)))))
  (let* ((cascade (filter-media medium cascade))
	 (ss (string->symbol selector))
	 (agent (apply append (find-matches 'agent (car cascade) ss)))
	 (user (apply append (find-matches 'user (cadr cascade) ss)))
	 (author (apply append (find-matches 'author (caddr cascade) ss))))
    (add! (list-stable-sort 
	   specificity-less
	   (list-stable-sort precedence-less (append '() agent user author)))
	  0)))

;; Caching fn for structure-based queries

(define (select-values cascade node doc medium)
  (define (get-ancestors doc node)
    (if (null? node)
	(list)
	(let ((p (get-sxml-parent doc node)))
	  (if (and p (not (eq? p doc))) 
	      (cons p (get-ancestors doc p)) 
	      (list)))))
  (define (collate table data order)
    (or (and (null? data) data)
	(let* ((x (car data))
	       (imp (eq? (caddr x) '!))
	       (sdcy (symbol-downcase (list-ref x (if imp 3 2))))
	       (sdcv (string-downcase (list-ref x (if imp 4 3)))))
	  (append (if (hash-table-ref/default property-hash-table sdcy #f)
		      (let ((r (hash-table-ref/default table sdcy #f)))
			(list (list sdcy sdcv order r)))
		      (list))
		  (collate table (cdr data) (- order 1))))))
  (define (find-matches sym sheet t ancestors)
    (define (map-fn x)
      (define (filt y)
	(define (inner-map r)
	  (define (inner-map-fn z)
	    (let ((imp (eq? (car z) '!)))
	      (if (cdr r) 
		  (hash-table-set! t 
				   (symbol-downcase (if imp (caadr z) (car z)))
				   (cdr r)))
	      (append (if imp (cons* sym y '! (cadr z)) (cons* sym y z))
		      (list (cdr r)))))
	  (and r (map inner-map-fn (cdr x))))
	(inner-map (selector-match? y doc ancestors)))
      (filter-map filt (let ((cx (car x))) (or (grouping? cx #f) (list cx)))))
    (fold-right append '() (map map-fn (if (null? sheet) sheet (cdr sheet)))))

  (let* ((cascade (filter-media medium cascade))
	 (a (cons node (get-ancestors doc node)))
	 (cc (make-hash-table eq? hash-by-identity))
	 (agent (apply append (find-matches 'agent (car cascade) cc a)))
	 (user (apply append (find-matches 'user (cadr cascade) cc a)))
	 (author (apply append (find-matches 'author (caddr cascade) cc a))))
    (collate cc
	     (list-stable-sort 
	      specificity-less
	      (list-stable-sort precedence-less 
				(append '() author user agent)))
	     0)))

(define (scss:get-default-value prop)
  (verify-arg-types "scss:get-default-value" (list symbol?) (list prop) 1)
  (get-default-prop-value (get-prop-entry prop)))

(define (scss:inherited? prop) 
  (and=> (get-prop-entry prop) css-property:inherited?))

(define default-lookup-table
  (let* ((dlt (make-hash-table eq? hash-by-identity))
	 (htwf (lambda (k v) 
		 (hash-table-set! dlt k (get-default-prop-value v)))))
    (hash-table-walk property-hash-table htwf)
    dlt))

;; lookup a value for a simple string, as opposed to a node
;;
;; @src the cascade
;; @sel the selector, a string
;; @rec a boolean value; should we return a default value if lookup fails?
;; @medium the media type, a string
;; @p the property, a symbol

(define (lookup-value src sel rec medium p)
  (define (cache x h)
    (and x (or (hash-table-ref/default h (car x) #f)
	       (and (hash-table-set! h (car x) (cadr x)) (cadr x)))))
  (define (lsf x y) (or (not y) (and x (> (caddr x) (caddr y)))))
  (let*	((p (string->symbol (string-downcase (symbol->string p))))
	 (d (or (hash-table-ref/default cht src #f)
		(let ((dh (make-hash-table equal? hash)))
		  (hash-table-set! cht src dh) dh))))	 
    (or (and=> (hash-table-ref/default d sel #f)
	       (lambda (x) (hash-table-ref/default x p #f)))
	(let* ((e (let ((x (hash-table-copy default-lookup-table)))
		    (hash-table-set! d sel x) x))
	       (eh (make-hash-table eq? hash-by-identity))
	       (vs (begin
		     (lookup-values src sel medium eh)
		     (list-stable-sort 
		      lsf 
		      (cons (hash-table-ref/default eh p #f)
			    (map (lambda (z) (hash-table-ref/default eh z #f))
				 (hash-table-ref/default gpt p '()))))))
	       (v (let ((cvs (car vs)))
		    (and cvs (and=> (if (eq? (car cvs) p)
					(cadr cvs)
					(specify p (cadr cvs)))
				    (lambda (x) (hash-table-set! e p x) x))))))
	  (for-each (lambda (x) (cache x e)) (cdr vs))
	  (or v (and rec (let* ((pe (get-prop-entry p))
				(d (get-default-prop-value pe)))
			   (hash-table-set! e p d) d)))))))

(define default-select-table
  (let* ((dst (make-hash-table eq? hash-by-identity))
	 (htwf (lambda (k v)
		 (if (not (css-property:inherited? v))
		     (hash-table-set! 
		      dst k `(,k ,(get-default-prop-value v) 0 #t))))))
    (hash-table-walk property-hash-table htwf)
    dst))

;; perform property selection on a node
;;
;; @src the cascade
;; @sel the selector, a node
;; @rec a boolean value; should we return a default value if lookup fails?
;; @doc the document to which sel belongs
;; @m the media type, a string
;; @p the property, a symbol

(define (select-value src sel rec doc m p)
  (define (select-hash x s)
    (modulo (+ (hashq (car x) s) (cdr x)) s))
  (define (select-eq? x y)
    (and (pair? x) (pair? y) (eq? (car x) (car y)) (eqv? (cdr x) (cdr y))))
  (define (active-pseudo-elements e)
    (cond ((eq? e doc) 0)
	  ((internal-pseudo-element-handler "before" doc e) 8)
	  ((internal-pseudo-element-handler "after" doc e) 4)
	  ((internal-pseudo-element-handler "first-letter" doc e) 3)
	  ((internal-pseudo-element-handler "first-line" doc e) 1)
	  (else (active-pseudo-elements (get-sxml-parent doc e)))))
  (define (get-element-hash e)
    (let* ((dh (or (hash-table-ref/default dht doc #f)
		   (let ((ht (make-hash-table select-eq? select-hash)))
		     (hash-table-set! dht doc ht) ht)))
	   (ase (active-pseudo-elements e)))
      (or (hash-table-ref/default dh (cons e ase) #f)
	  (let ((x (make-hash-table cequal? chash)))
	    (hash-table-set! dh (cons e ase) x) x))))
  (define (get-hash elt)
    (hash-table-ref/default (get-element-hash elt) src #f))
  (define (g x e)
    (and (not (null? x))
	 (or (hash-table-ref/default e (car x) #f) (g (cdr x) e))))
  (define (f sel)
    (define (cache x h)
      (and x (car (last-pair x)) (hash-table-set! h (car x) x) x))
    (define (s x p)
      (let ((y (specify p (cadr x))))
	(and y (cons* p y (cddr x)))))
    (let* 
	((p (string->symbol (string-downcase (symbol->string p))))
	 (pe (get-prop-entry p))
	 (eh (get-element-hash sel))
	 (e (hash-table-ref/default eh src #f))
	 (related? (lambda (x) 
		     (or (eq? (car x) p) 
			 (memq (car x) (hash-table-ref/default gpt p '()))))))

      ;; Is the value already cached explicitly?

      (or (and e (or (hash-table-ref/default e p #f)

		     ;; Maybe a more general form of the value is 
		     ;; cached...

		     (and=> (g (hash-table-ref/default gpt p '()) e)
			    (lambda (y) (cache (s y p) e)))))

	  ;; Okay, nothing is cached.  Do the lookup.
	  
	  (let* ((e (or e (let ((x (hash-table-copy default-select-table)))
			    (hash-table-set! eh src x) x)))
		 (vs (select-values src sel doc m))
		 (v (find related? vs)))		 
	    (for-each (lambda (x) 
			(hash-table-walk 
			 gpt (lambda (k v)
			       (and (find (lambda (z) (eq? z (car x))) v)
				    (cache (s x k) e))))
			(cache x e))
		      vs)
	    (or v
		(and rec
		     (or (and v (equal? (cadr v) "inherit"))
			 (and (not v) pe (css-property:inherited? pe)))
		     (let ((par (get-sxml-parent doc sel)))
		       (and rec 
			    (not (eq? par doc))
			    (let ((fp (f par))) (and fp (cache fp e)))))))))))
  (f sel)
  (and=> (let ((h (get-hash sel)))
	   (or (hash-table-ref/default h p #f)
	       (and rec (let ((x (list p (scss:get-default-value p))))
			  (hash-table-set! h p x) x))))
	 (lambda (x) (let ((result (cadr x)))
		       (if (symbol? result)
			   (select-value src sel rec doc m result)
			   result)))))

(define (scss:select-value source selector . r)
  (let ((r (if (string? selector)
	       (if (< (length r) 2) (cons 'all r) r)
	       (if (< (length r) 3) (cons* (car r) 'all (cdr r)) r))))
    (apply (if (string? selector) lookup-value select-value)
	   (append (list source selector #t) r))))

(define (scss:select-value-at-node source selector . r)
  (let ((r (if (string? selector)
	       (if (< (length r) 2) (cons 'all r) r)
	       (if (< (length r) 3) (cons* (car r) 'all (cdr r)) r))))
    (apply (if (string? selector) lookup-value select-value)
	   (append (list source selector #f) r))))
