(define (make-css21-lexer strm)
  (define (next-internal? strm str case-sensitive?)
    (define (unread c) (unread-char c strm))
    (define pred? (if case-sensitive? char=? char-ci=?))
    (define (rewind match) (for-each unread match) #f)
    (let loop ((chrs (string->list str))
	       (match (list)))
      (or (null? chrs)
	  (let ((c (read-char strm)))
	    (cond ((eof-object? c) (rewind match))
		  ((pred? c (car chrs)) (loop (cdr chrs) (cons c match)))
		  (else (unread-char c strm) (rewind match)))))))

  (define (next? strm str) (next-internal? strm str #t))
  (define (next-ci? strm str) (next-internal? strm str #f))
  
  (define (read-number strm seen-dot?)
    (let ((c (read-char strm)))
      (cond ((char-numeric? c) (cons c (read-number strm seen-dot?)))
	    ((char=? c #\.) (if seen-dot? 
				(begin (unread-char c strm) '())
				(cons c (read-number strm #t))))
	    (else (unread-char c strm) '()))))
  
  (define (consume-whitespace-and-comments strm)
    (define (consume-whitespace)
      (let ((c (peek-char strm)))
	(or (eof-object? c)
	    (and (char-whitespace? (peek-char strm)) 
		 (read-char strm) 
		 (consume-whitespace)))))
    (define (consume-comments)
      (let ((c (read-char strm)))
	(cond ((and (char=? c #\*) (char=? (peek-char strm) #\/)) 
	       (read-char strm))
	      ((eof-object? c))
	      (else (consume-comments)))))
    (let loop ((c (read-char strm)))
      (cond ((eof-object? c) #f)
	    ((char-whitespace? c) (consume-whitespace) (loop (read-char strm)))
	    ((and (char=? c #\/) (char=? (peek-char strm) #\*))
	     (read-char strm)
	     (consume-comments) 
	     (loop (read-char strm)))
	    (else (unread-char c strm)))))
  
  (define (read-string strm type)
    (let ((c (read-char strm)))
      (cond ((eof-object? c) '())
	    ((char=? c #\\)
	     (let ((pc (peek-char strm)))
	       (cond ((eqv? pc type)
		      (read-char strm) (cons* c type (read-string strm type)))
		     ((eqv? pc #\nl) (read-char strm) (read-string strm type))
		     (else (cons c (read-string strm type))))))
	    ((char=? c type) (list type))
	    (else (cons c (read-string strm type))))))

  (define (read-hex strm)
    (define (read-hex-inner count)
      (let ((c (peek-char strm)))
	(cond ((= count 6) '())
	      ((and (> count 0) (char-whitespace? c)) 
	       (read-char strm) (list c))
	      ((hex? c) (read-char strm) (cons c (read-hex-inner (+ count 1))))
	      (else '()))))
    (let ((h (read-hex-inner 0))) (consume-whitespace-and-comments strm) h))
  
  (define (hex? chr)
    (or (char-numeric? chr) (and (char-ci>=? chr #\A) (char-ci<=? chr #\F))))
  (define (read-escape strm)
    (let ((c (peek-char strm)))
      (if (hex? c) (read-hex strm) (begin (read-char strm) (list c)))))

  (define (read-url strm)
    (define (read-url-inner)
      (let ((c (read-char strm)))
	(cond ((eqv? c #\)) (unread-char c strm) '())
	      ((or (memv c '(#\! #\# #\$ #\% #\&))
		   (and (char-ci>=? c #\*) (char-ci<=? #\~)))
	       (cons c (read-url-inner)))
	      ((eqv? c #\\) (append (cons c (read-escape strm))
				    (read-url-inner)))
	      (else (unread-char c strm) '()))))
    (read-url-inner))

  (define (start-char? chr)
    (or (char-alphabetic? chr) (char=? chr #\_) (> (char->integer chr) 127)))
  (define (read-ident strm)
    (define (read-start)
      (let ((c (read-char strm)))
	(cond ((start-char? c) (cons c (read-start)))
	      ((char=? c #\\) (append (cons c (read-escape strm)) 
				      (read-start)))
	      (else (unread-char c strm) '()))))
    (define (read-rest) 
      (let ((c (read-char strm)))
	(cond ((or (start-char? c) (char-numeric? c) (char=? c #\-)) 
	       (cons c (read-rest)))
	      ((char=? c #\\) (append (cons c (read-escape strm)) 
				      (read-rest)))
	      (else (unread-char c strm) '()))))
    (let ((s (read-start)))
      (if (null? s) s (append s (read-rest)))))
  
  (lambda ()
    (consume-whitespace-and-comments strm) 
    (let ((c (read-char strm)))
      (cond ((eof-object? c) '(*eoi*))
	    ((and (eqv? c #\<) (next? strm "!--")) '(CDO))
	    ((and (eqv? c #\-) (next? strm "->")) '(CDC))
	    ((and (eqv? c #\~) (next? strm "=")) '(INCLUDES))
	    ((and (eqv? c #\|) (next? strm "=")) '(DASHMATCH))
	    ((eqv? c #\*) '(STAR))
	    
	    ((or (and (eqv? c #\.) (char-numeric? (peek-char strm)))
		 (and (char-numeric? c)))
	     (let* ((n (read-number strm (eqv? c #\.)))
		    (o (list->string (cons c n))))
	       (cond 
		((next-ci? strm "em") (cons 'EMS (string-append o "em")))
		((next-ci? strm "ex") (cons 'EXS (string-append o "ex")))
		((next-ci? strm "px") (cons 'LENGTH (string-append o "px")))
		((next-ci? strm "cm") (cons 'LENGTH (string-append o "cm")))
		((next-ci? strm "mm") (cons 'LENGTH (string-append o "mm"))) 
		((next-ci? strm "in") (cons 'LENGTH (string-append o "in")))
		((next-ci? strm "pt") (cons 'LENGTH (string-append o "pt")))
		((next-ci? strm "pc") (cons 'LENGTH (string-append o "pc")))
		((next-ci? strm "deg") (cons 'ANGLE (string-append o "deg")))
		((next-ci? strm "rad") (cons 'ANGLE (string-append o "rad"))) 
		((next-ci? strm "grad") (cons 'ANGLE (string-append o "grad")))
		((next-ci? strm "ms") (cons 'TIME (string-append o "ms")))
		((next-ci? strm "s") (cons 'TIME (string-append o "s")))
		((next-ci? strm "hz") (cons 'FREQUENCY (string-append o "hz")))
		((next-ci? strm "khz") 
		 (cons 'FREQUENCY (string-append o "khz")))
		((next-ci? strm "%") (cons 'PERCENTAGE (string-append o "%")))
		(else (let ((i (read-ident strm)))
			(if (null? i) 
			    (cons 'NUMBER o)
			    (cons 'DIMENSION 
				  (string-append o (list->string i)))))))))

	    ((eqv? c #\.) '(DOT))
	    
	    ((eqv? c #\;) '(SEMICOLON))
	    ((eqv? c #\:) '(COLON))
	    ((eqv? c #\=) '(EQUALS))
	    ((eqv? c #\[) '(LBRACK))
	    ((eqv? c #\]) '(RBRACK))
	    ((eqv? c #\)) '(RPAREN))
	    ((eqv? c #\}) '(RBRACE))
	    ((eqv? c #\{) '(LBRACE))
	    ((eqv? c #\+) '(PLUS))
	    ((eqv? c #\>) '(GREATER))
	    ((eqv? c #\,) '(COMMA))
	    ((eqv? c #\-) '(MINUS))
	    ((or (eqv? c #\') (eqv? c #\")) 
	     (let ((s (read-string strm c)))
	       (if (or (null? s) (not (char=? (car (last-pair s)) c)))
		   (cons 'INVALID (list->string (cons c s)))
		   (cons 'STRING (list->string (cons c s))))))
	    ((or (start-char? c) (eqv? c #\\))
	     (unread-char c strm) 
	     (let ((i (list->string (read-ident strm))))
	       (if (char=? (peek-char strm) #\()
		   (begin (read-char strm) 
			  (if (equal? i "url")
			      (begin 
				(consume-whitespace-and-comments strm)
				(let ((r (let ((c (read-char strm)))
					   (if (or (eqv? c #\') (eqv? c #\"))
					       (list->string 
						(cons c (read-string strm c)))
					       (begin
						 (unread-char c strm)
						 (list->string 
						  (read-url strm)))))))
				  (consume-whitespace-and-comments strm)
				  (read-char strm)
				  (cons 'URI r)))
			      (cons 'FUNCTION (string-append i "("))))
		   (cons 'IDENT i))))
	    ((eqv? c #\#)
	     (let ((i (read-ident strm)))
	       (if (null? i) 
		   (cons 'DELIM c) 
		   (cons 'HASH (list->string (cons c i))))))
	    ((eqv? c #\@)
	     (cond ((next? strm "import") '(IMPORT_SYM))
		   ((next? strm "page") '(PAGE_SYM))
		   ((next? strm "media") '(MEDIA_SYM))
		   ((next? strm "charset") '(CHARSET_SYM))
		   (else '(BAD_AT_KEYWORD))))
	    
	    ((eqv? c #\!) 
	     (consume-whitespace-and-comments strm) 
	     (if (next? strm "important") '(IMPORTANT_SYM) (cons 'DELIM c)))
	    (else (cons 'DELIM c))))))
