;; scss.scm: main module exports and implementations for SCSS
;; Copyright (C) 2005 Julian Graham

;; SCSS is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2 of the License, or
;; (at your option) any later version.
;;
;; SCSS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with SCSS; if not, write to the Free Software
;; Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA

(define-module (scss scss)
  #:export (scss:css->scss
	    scss:scss->css

	    scss:set-user-stylesheet!
	    scss:set-author-stylesheet!
	    scss:set-agent-stylesheet!

	    scss:create-cascade
	    scss:create-ruleset

	    scss:color->hex

	    scss:set-property!
	    scss:get-property
	    scss:add-selector!
	    scss:remove-selector!

	    scss:set-sxml-parent-function!
	    scss:set-dot-handler!
	    scss:set-id-handler!
	    scss:set-pseudo-class-handler!
	    scss:set-pseudo-element-handler!

	    scss:stylesheet?
	    scss:cascade?

	    scss:select-value))

(load "lexer.scm")
(load "parser.scm")

(use-modules (ice-9 readline) (ice-9 regex) (srfi srfi-1))
(activate-readline)

(define unicode-regex "(\\\\[0-9a-f]{1,6}(\\r\\n|[ \\n\\r\\t\\f])?)")
(define unicode-regexc (make-regexp unicode-regex))
(define escape-regex (string-append "(" unicode-regex "|(\\\\[^nrf]))"))
(define escape-regexc (make-regexp escape-regex))
(define nonascii-regex "([^[:alnum:][:punct:]])")
;; (define nonascii-regex "")
(define nonascii-regexc (make-regexp nonascii-regex))
(define nmstart-regex 
  (string-append "([_a-zA-Z]|" nonascii-regex "|" escape-regex ")"))
(define nmstart-regexc (make-regexp nmstart-regex))
(define nmchar-regex
  (string-append "([_a-zA-Z0-9-]|" nonascii-regex "|" escape-regex ")"))
(define nmchar-regexc (make-regexp nmchar-regex))
(define ident-regex (string-append "([-]?" nmstart-regex nmchar-regex "*)"))
(define ident-regexc (make-regexp ident-regex))
(define name-regex (string-append nmchar-regex "+"))
(define nl-regex "(\\n|(\\r\\n)|\\r|\\f)")
(define string1-regex "(\\\"((\\\\[nrf\\\"])|([^\\\\\\\"][^\\\"]?))*\\\")")
(define string2-regex "(\\'((\\\\[nrf\\'])|([^\\\\\\'][^\\']?))*\\')")
(define string-regex (string-append "(" string1-regex "|" string2-regex ")"))
(define string-regexc (make-regexp string-regex))
(define w-regex "([ \\t\\r\\n\\f]*)")
(define integer-regex "([\\-|+]?[0-9]+)")
(define number-regex (string-append "(" integer-regex "(\\.[0-9]+)?)"))
(define angle-regex (string-append "(" number-regex "((deg)|(grad)|(rad)))"))
(define percentage-regex (string-append "(" number-regex "%)"))
(define length-regex 
  (string-append "(" number-regex 
		 "((em)|(ex)|(px)|(in)|(cm)|(mm)|(pt)|(pc)))"))
(define color-regex 
  (string-append "((maroon)|(red)|(orange)|(yellow)|(olive)|(purple)|"
		 "(fuchsia)|(lime)|(green)|(navy)|(blue)|(aqua)|(teal)|"
		 "(black)|(silver)|(gray)|(#[a-fA-F0-9]{6}))"))
(define border-style-regex 
  (string-append "((none)|(hidden)|(dotted)|(dashed)|(solid)|(double)|"
		 "(groove)|(ridge)|(inset)|(outset))"))
(define border-width-regex 
  (string-append "((thin)|(medium)|(thick)|" length-regex ")"))
(define svoice-regex (string-append "(" ident-regex "|" string-regex ")"))
(define gvoice-regex "((male)|(female)|(child))")

(define gfont-regex "((serif)|(sans-serif)|(cursive)|(fantasy)|(monospace))")
(define sfont-regex string-regex)

(define ffamily-regex 
  (string-append "(((" sfont-regex "|" gfont-regex ")(\\w+" sfont-regex "|" 
		 gfont-regex ")*)|(inherit))"))
(define fsize-regex 
  (string-append "(((xx-small)|(x-small)|(small)|(medium)|(large)|(x-large)|"
		 "(xx-large))|((larger)|(smaller))|" length-regex "|"
		 percentage-regex "|(inherit))"))
(define fstyle-regex "((normal)|(italic)|(oblique)|(inherit))")
(define fvariant-regex "((normal)|(small-caps)|(inherit))")
(define fweight-regex 
  (string-append "((normal)|(bold)|(bolder)|(lighter)|(100)|(200)|(300)|(400)|"
		 "(500)|(600)|(700)|(800)|(900)|(inherit))"))

(define lheight-regex (string-append "((normal)|" number-regex "|" length-regex
				     "|" percentage-regex "|(inherit))"))

;; This isn't 100% accurate, but...
(define uri-regex (string-append "(url\\(" string-regex "\\))"))

(define lsimage-regex (string-append "(" uri-regex "|(none))"))
(define lsposition-regex "((inside)|(outside))")
(define lstype-regex 
  (string-append "((disc)|(circle)|(square)|(decimal)|(decimal-leading-zero)|"
		 "(lower-roman)|(upper-roman)|(lower-greek)|(lower-latin)|"
		 "(upper-latin)|(armenian)|(georgian)|(none))"))

(define margin-regex 
  (string-append "(" length-regex "|" percentage-regex "|(auto))"))

(define counter-regex 
  (string-append "(counter\\(" ident-regex "(, " lstype-regex ")?\\))"))


(define ocolor-regex (string-append "(" color-regex "|(invert))"))
(define ostyle-regex border-style-regex)
(define owidth-regex border-width-regex)

(define time-regex (string-append "(" number-regex "m?s)"))

(define padding-regex 
  (string-append "(" length-regex "|" percentage-regex ")"))
(define pcolor-regex 
  (string-append "(" color-regex "|(transparent)|(inherit))"))
(define pimage-regex (string-append "(" uri-regex "|(none)|(inherit))"))
(define prepeat-regex "((repeat)|(repeat-x)|(repeat-y)|(no-repeat)|(inherit))")
(define pattach-regex "((scroll)|(fixed)|(inherit))")

(define lposition
  (lambda (x)
    (or (exact-match (string-append "((" percentage-regex "|" length-regex 
				    "|(left)|(center)|(right))(" 
				    percentage-regex "|" length-regex 
				    "|(top)|(center)|(bottom))?)"))
	(any-order-or-match "(left)|(center)|(right)" 
			    "(top)|(center)|(bottom)")
	(exact-match "inherit"))))

;; NOTE: Changes to this regex will require updates to the selection functions!

(define attr-sel-regexc 
  (make-regexp (string-append "\\[" ident-regex "(([~|]?=)(" ident-regex "|" 
			      string-regex "))?\\]")))

(define color-table
  '(("maroon" "#800000") ("red" "#ff0000") ("orange" "#ffa500") 
    ("yellow" "#ffff00") ("olive" "#808000") ("purple" "#800080")
    ("fuchsia" "#ffffff") ("lime" "#00ff00") ("green" "#008000") 
    ("navy" "#000080") ("blue" "#0000ff") ("aqua" "#00ffff") ("teal" "#008080")
    ("black" "#000000") ("silver" "#c0c0c0") ("gray" "#808080")))

(define exact-match
  (lambda (pattern str)
    (let ((match (string-match pattern str)))
      (if match (equal? str (match:substring match 0)) match))))

(define exact-exec
  (lambda (pattern str)
    (let ((match (regexp-exec pattern str)))
      (if match (equal? str (match:substring match 0)) match))))

(define any-order-or-match
  (lambda lst
    (lambda (input)
      (let ((l (copy-tree lst)))
	(if (not (null? input))
	    (let ((result #t))
	      (for-each (lambda (x)
			  (if (> (string-length x) 0)
			      (if (null? l) 
				  (set! result #f)
				  (let ((m (find (lambda (y)
						   (if (procedure? y)
						       (apply y (list x))
						       (exact-match y x)))
						 l)))
				    (if m (delete! m l) (set! result #f))))))
			(string-split input #\sp))
	      result)
	    #f)))))

(define matchf (lambda (expr) (lambda (input) (exact-match expr input))))

(define property-table
  `(("azimuth" 
     ,(lambda (x)
	(or (exact-match angle-regex x)
	    (apply (any-order-or-match
		    (string-append "(left-side)|(far-left)|(left)|"
				   "(center-left)|(center)|(center-right)|"
				   "(right)|(far-right)|(right-side)")
		    "behind") (list x))
	    (exact-match "(leftwards)|(rightwards)|(inherit)" x)))
     ("center") #t)
    ("background-attachment" ,(matchf pattach-regex) ("scroll") #f)
    ("background-color" ,(matchf pcolor-regex) ("transparent") #f)
    ("background-image" ,(matchf pimage-regex) ("none") #f)
    ("background-position" lposition ("0%" "0%") #f)
    ("background-repeat" ,(matchf prepeat-regex) ("repeat") #f)
    ("background"
     ,(lambda (x)
	(or (any-order-or-match pcolor-regex pimage-regex prepeat-regex
				pattach-regex lposition)
	    (matchf "inherit")))
     ("black" "none" "no-repeat" "fixed" "0% 0%") #f)
    ("border-collapse" ,(matchf "(collapse)|(separate)|(inherit)")
     ("separate") #t)
    ("border-color" ,(matchf "((color)|(transparent)){1,4}|(inherit)") 
     (color color color color) #f)
    ("border-spacing"
     ,(matchf (string-append "(" length-regex "\\w+" length-regex 
			     "?)|(inherit)")) ("0") #t)
    ("border-style"
     ,(matchf (string-append "(" border-style-regex "){1,4}|(inherit)"))
     ("none" "none" "none" "none") #f)
    ("(border-top)|(border-right)|(border-bottom)|(border-left)"
     ,(lambda (x) 
	(or (any-order-or-match border-width-regex 
				border-style-regex 
				pcolor-regex)
	    (exact-match "(inherit)")))
     ("medium" "none" color) #f)
    (,(string-append "(border-top-color)|(border-right-color)|"
		     "(border-left-color)|(border-bottom-color)")
     ,(matchf pcolor-regex) (color) #f)
    (,(string-append "(border-top-style)|(border-right-style)|"
		     "(border-left-style)|(border-bottom-style)")
     ,(matchf (string-append border-style-regex "|(inherit)")) ("none") #f)
    (,(string-append "(border-top-width)|(border-right-width)|"
		     "(border-bottom-width)|(border-left-width)")
     ,(matchf (string-append border-width-regex "|(inherit)")) ("medium") #f)
    ("border-width" 
     ,(matchf (string-append border-width-regex "{1,4}|(inherit)"))
     ("medium" "medium" "medium" "medium") #f)
    ("border"
     ,(lambda (x) 
	(or (any-order-or-match border-width-regex 
				border-style-regex 
				pcolor-regex)
	    (exact-match "(inherit)")))
     ("medium none" color) #f)
    ("bottom" ,(matchf (string-append length-regex "|" percentage-regex 
				      "|(auto)|inherit"))
     ("auto") #f)
    ("caption-side" ,(matchf "(top)|(bottom)|(inherit)") ("top") #t)
    ("clear" ,(matchf "(none)|(left)|(right)|(both)|(inherit)") ("none") #f)
    ("clip" 
     ,(matchf (string-append "(rect\\((" length-regex "|(auto))(,\\w(" 
			     length-regex "|(auto))){3}\\))|(auto)|(inherit)"))
     ("auto") #f)
    ("color" ,(matchf pcolor-regex) ("white") #t)
    ("content" 
     ,(matchf (string-append "(normal)|(" string-regex "|" uri-regex "|" 
			     counter-regex "|(attr\\(" ident-regex "\\))|"
			     "(open-quote)|(close-quote)|(no-open-quote)|"
			     "(no-close-quote))+|(inherit)"))
     ("normal") #f)
    ("(counter-increment)|(counter-reset)"
     ,(matchf (string-append "(" ident-regex "(\\w+" integer-regex ")?)+|"
			     "(none)|(inherit)"))
     ("none") #f)
    ("(cue-after)|(cue-before)" 
     ,(matchf (string-append uri-regex "|(none)|(inherit)")) ("none") #f)
    ("cue"
     ,(matchf (string-append "(" uri-regex "|(none)|(inherit)(\\w+" uri-regex
			     "|(none)|(inherit))?)|(inherit)"))
     ("none" "none") #f)
    ("cursor"
     ,(matchf (string-append "((" uri-regex ",)*((auto)|(crosshair)|(default)|"
			     "(pointer)|(move)|(e-resize)|(ne-resize)|"
			     "(nw-resize)|(n-resize)|(se-resize)|(sw-resize)|"
			     "(s-resize)|(w-resize)|(text)|(wait)|(help)|"
			     "(progress)))|(inherit)"))
     ("auto") #t)
    ("direction" ,(matchf "(ltr)|(rtl)|(inherit)") ("ltr") #t)
    ("display"
     ,(matchf (string-append "(inline)|(block)|(list-item)|(run-in)|"
			     "(inline-block)|(table)|(inline-table)|"
			     "(table-row-group)|(table-header-group)|"
			     "(table-footer-group)|(table-row)|"
			     "(table-column-group)|(table-column)|"
			     "(table-cell)|(table-caption)|(none)|(inherit)"))
     ("inline") #f)
    ("elevation"
     ,(matchf (string-append angle-regex "|(below)|(level)|(above)|(higher)|"
			     "(lower)|(inherit)"))
     ("level") #t)
    ("empty-cells" ,(matchf "(show)|(hide)|(inherit)") ("show") #t)
    ("float" ,(matchf "(left)|(right)|(none)|(inherit)") ("none") #f)
    ("font-family" ,(matchf ffamily-regex) ("monospace") #t)
    ("font-size" ,(matchf fsize-regex) ("medium") #t)
    ("font-style" ,(matchf fstyle-regex) ("normal") #t)
    ("font-variant" ,(matchf fvariant-regex) ("normal") #t)
    ("font-weight" ,(matchf fweight-regex) ("normal") #t)
    ("font" 
     ,(lambda (x)
	(let ((y (string-append fsize-regex "(\\w+\\/" lheight-regex ")?\\w+"
				ffamily-regex)))
	  (or (and (any-order-or-match fstyle-regex fvariant-regex 
				       fweight-regex y)
		   (string-match (string-append y "$") x))
	      (exact-match y)
	      (exact-match (string-append "(caption)|(icon)|(menu)|"
					  "(message-box)|(small-caption)|"
					  "(status-bar)|(inherit)")))))
     ("normal" "normal" "normal" "medium" "monospace") #t)
    ("(height)|(left)|(right)|(width)|(bottom)|(top)"
     ,(matchf (string-append length-regex "|" percentage-regex "|(auto)|"
			     "(inherit)"))
     ("auto") #f)
    ("letter-spacing"
     ,(matchf (string-append "(normal)|" length-regex "|(inherit)"))
     ("normal") #t)
    ("line-height"
     ,(matchf (string-append "(normal)|" number-regex "|" length-regex "|"
			     percentage-regex "|(inherit)"))
     ("normal") #t)
    ("list-style-image"
     ,(matchf (string-append lsimage-regex "|(inherit)")) ("none") #t)
    ("list-style-position" 
     ,(matchf (string-append lsposition-regex "|(inherit)")) ("outside") #t)
    ("list-style-type"
     ,(matchf (string-append lstype-regex "|(inherit)")) ("disc") #t)
    ("list-style"
     ,(lambda (x)
	(or (any-order-or-match lstype-regex lsposition-regex lsimage-regex)
	    (exact-match "inherit")))
     ("disc" "outside" "none") #t)
    ("(margin-right)|(margin-left)|(margin-top)|(margin-bottom)"
     ,(matchf (string-append length-regex "|" percentage-regex 
			     "|(auto)|(inherit)"))
     ("0") #f)
    ("margin" ,(matchf (string-append margin-regex "{1,4}|(inherit)"))
     ("0" "0" "0" "0") #f)
    ("(max-height)|(max-width)"
     ,(matchf (string-append length-regex "|" percentage-regex "|(none)|"
			     "(inherit)"))
     ("none") #f)
    ("(min-height)|(min-width)"
     ,(matchf (string-append length-regex "|" percentage-regex "|(inherit)"))
     ("0") #f)
    ("orphans" ,(matchf (string-append integer-regex "|(inherit)")) ("2") #t)
    ("outline-color" ,(matchf (string-append ocolor-regex "|(inherit)")) 
     ("invert") #f)
    ("outline-style" ,(matchf (string-append ostyle-regex "|(inherit)"))
     ("none") #f)
    ("outline-width" ,(matchf (string-append owidth-regex "|(inherit)"))
     ("medium") #f)
    ("outline" 
     ,(lambda (x) 
	(or (any-order-or-match ocolor-regex ostyle-regex owidth-regex)
	    (exact-match "inherit")))
     ("invert" "none" "medium") #f)
    ("overflow"
     ,(matchf "(visible)|(hidden)|(scroll)|(auto)|(inherit)") ("visible") #f)
    ("(padding-top)|(padding-right)|(padding-bottom)|(padding-left)"
     ,(matchf (string-append length-regex "|" percentage-regex "|(inherit)"))
     ("0") #f)
    ("padding" ,(matchf (string-append padding-regex "{1,4}|(inherit)"))
     ("0" "0" "0" "0") #f)
    ("(page-break-after)|(page-break-before)"
     ,(matchf "(auto)|(always)|(avoid)|(left)|(right)|(inherit)") ("auto") #f)
    ("page-break-inside" ,(matchf "(avoid)|(auto)|(inherit)") ("auto") #t)
    ("(pause-after)|(pause-before)"
     ,(matchf (string-append time-regex "|" percentage-regex "|(inherit)"))
     ("0") #f)
    ("pause"
     ,(matchf (string-append "(" time-regex "|" percentage-regex "){1,2}|"
			     "(inherit)"))
     ("0" "0") #f)
    ("pitch-range" ,(matchf (string-append number-regex "|(inherit)")) 
     ("50") #t)
    ("pitch" ,(matchf (string-append "(" number-regex "k?Hz)|(x-low)|(low)|"
				     "(medium)|(high)|(x-high)|(inherit)"))
     ("medium") #t)
    ("play-during" 
     ,(matchf (string-append uri-regex "|(" uri-regex "\\w+mix)|(" uri-regex
			     "\\w+repeat)|(" uri-regex "\\w+mix\\w+repeat)|("
			     uri-regex "\\w+repeat\\w+mix)|(auto)|(none)|"
			     "(inherit)"))
     ("auto") #f)
    ("position" ,(matchf "(static)|(relative)|(absolute)|(fixed)|(inherit)") 
     ("static") #f)
    ("quotes" 
     ,(matchf (string-append "(" string-regex "\\w+" string-regex ")+|(none)|"
			     "(inherit)"))
     ("none") #t)
    ("richness" ,(matchf (string-append number-regex "|(inherit)")) ("50") #t)
    ("speak-header" ,(matchf "(once)|(always)|(inherit)") ("once") #t)
    ("speak-numeral" ,(matchf "(digits)|(continuous)|(inherit)") 
     ("continuous") #t)
    ("speak-punctuation" ,(matchf "(code)|(none)|(inherit)") ("none") #t)
    ("speak" ,(matchf "(normal)|(none)|(spell-out)|(inherit)") ("normal") #t)
    ("speech-rate"
     ,(matchf (string-append number-regex "|(x-slow)|(slow)|(medium)|(fast)|"
			     "(x-fast)|(faster)|(slower)|(inherit)"))
     ("medium") #t)
    ("stress" ,(matchf (string-append number-regex "|(inherit)")) ("50") #t)
    ("table-layout" ,(matchf "(auto)|(fixed)|(inherit)") ("auto") #f)
    ("text-align" ,(matchf "(left)|(right)|(center)|(justify)|(inherit)")
     ("left") #t)
    ("text-decoration"
     ,(lambda (x) 
	(or (exact-match "none")
	    (any-order-or-match "underline" "overline" "line-through" "blink")
	    (exact-match "inherit")))
     ("none") #f)
    ("text-indent" ,(matchf (string-append length-regex "|" percentage-regex 
					   "|(inherit)"))
     ("0") #t)
    ("text-transform" 
     ,(matchf "(capitalize)|(uppercase)|(lowercase)|(none)|(inherit)") 
     ("none") #t)
    ("unicode-bidi" ,(matchf "(normal)|(embed)|(bidi-override)|(inherit)")
     ("normal") #f)
    ("vertical-align"
     ,(matchf (string-append "(baseline)|(sub)|(super)|(top)|(text-top)|"
			     "(middle)|(bottom)|(text-bottom)|" 
			     percentage-regex "|" length-regex "|(inherit)"))
     ("baseline") #f)
    ("visibility" ,(matchf "(visible)|(hidden)|(collapse)|(inherit)")
     ("visible") #t)
    ("voice-family" 
     ,(matchf (string-append "((" svoice-regex "|" gvoice-regex ",)*(" 
			     svoice-regex "|" gvoice-regex "))|(inherit)"))
     ("female") #t)
    ("volume"
     ,(matchf (string-append number-regex "|" percentage-regex 
			     "|(silent)|(x-soft)|(soft)|(medium)|(loud)|"
			     "(xloud)|(inherit)"))
     ("medium") #t)
    ("white-space" 
     ,(matchf "(normal)|(pre)|(nowrap)|(pre-wrap)|(pre-line)|(inherit)")
     ("normal") #t)
    ("widows" ,(matchf (string-append integer-regex "|(inherit)")) ("2") #t)
    ("word-spacing" 
     ,(matchf (string-append "(normal)|" length-regex "|(inherit)"))
     ("normal") #t)		    
    ("z-index" ,(matchf (string-append "(auto)|" integer-regex "|(inherit)")) 
     ("auto") #f)))

(define sxml-node? 
  (lambda (x) (and (list? x) (> (length x) 1) (symbol? (car x)))))
(define sxml-doc? (lambda (x) (and (sxml-node? x) (eq? (car x) '*TOP*))))
(define get-sxml-parent
  (lambda (doc node)
    (if (memq node doc) doc (let ((c (filter list? (cdr doc))))
			      (if c (find (lambda (x) 
					    (get-sxml-parent x node))
					  c))
			      #f))))
(define sxml-node-name
  (lambda (node)
    (let* ((str (symbol->string (car node)))
	   (ri (string-rindex str #\:)))
      (if (not ri) str (substring str ri)))))      
(define sxml-attr-val
  (lambda (node name)
    (let ((attrs (find (lambda (x) (and (list? x) (eq? (car x) '@)))
		       (cdr node))))
      (if attrs (find (lambda (x) 
			(and (list? x) (equal? (sxml-node-name x) name))) 
		      (cdr attrs)) #f))))			 
(define scss:set-sxml-parent-function!
  (lambda (proc) 
    (verify-arg-types "scss:set-sxml-parent-function!" 
		      (list procedure?) 
		      (list proc)
		      1)
    (set! get-sxml-parent proc)))

(define internal-dot-handler (lambda (sel doc node) #f))
(define scss:set-dot-handler! 
  (lambda (p) 
    (verify-arg-types "scss:set-dot-handler!" (list procedure?) (list p) 1)
    (set! internal-dot-handler p)))

(define internal-id-handler (lambda (str doc node) #f))
(define scss:set-id-handler! 
  (lambda (p) 
    (verify-arg-types "scss:set-id-handler!" (list procedure?) (list p) 1)
    (set! internal-id-handler p)))

(define internal-pseudo-class-handler (lambda (str doc node) #f))
(define scss:set-pseudo-class-handler!
  (lambda (p)
    (verify-arg-types "scss:set-pseudo-class-handler!" 
		      (list procedure?) 
		      (list p)
		      1)
    (set! internal-pseudo-class-handler p)))

(define internal-pseudo-element-handler (lambda (str doc node) #f))
(define scss:set-pseudo-element-handler!
  (lambda (p)
    (verify-arg-types "scss:set-pseudo-element-handler!"
		      (list procedure?) 
		      (list p)
		      1)
    (set! internal-pseudo-element-handler p)))

(define null-merge (lambda (x y) (if (not (null? x)) (cons x y) y)))

(define merge-selectors
  (lambda (selector-list)
    (let ((l (list (list))))
      (for-each (lambda (x) 
		  (if (equal? x ",")
		      (append! l (list (list)))
		      (if (not (null? (car l)))
			  (set-car! (list-cdr-ref l (- (length l) 1))
				    (append (list-ref l (- (length l) 1))
					    (list x)))
			  (set-car! l (list x)))))
		selector-list)
      l)))

(define parse-selector
  (lambda (sel)
    (let ((sel-block (string-append sel " { }")))
      (lexer-init 'string sel-block)
      (caar (scss-parser lexer-wrapper (lambda e (error #t e)))))))

(define pseudo-class? 
  (lambda (str) 
    (member str '(":first-child" ":link" ":visited" ":hover" ":active" ":focus"
		  ":lang"))))

(define pseudo-element? 
  (lambda (str)
    (member str '(":first-line" ":first-letter" ":before" ":after"))))

(define validate-selector 
  (lambda (sel)
    (let* ((id-regexc (make-regexp (string-append "#" ident-regex)))
	   (dot-regexc (make-regexp (string-append "\\." ident-regex))))
      (for-each 
       (lambda (x)
	 (let ((read-ident #f))
	   (for-each 
	    (lambda (y)
	      (cond ((or (equal? y "*")
			 (exact-exec ident-regexc y)
			 (exact-exec attr-sel-regexc y))
		     (set! read-ident #t))
		    ((and read-ident (or (equal? "+" y) (equal? ">" y)))
		     (set! read-ident #f))
		    ((or (and read-ident (or (pseudo-class? y) 
					     (pseudo-element? y)))
			 (exact-exec id-regexc y)
			 (exact-exec dot-regexc y)))
		    (else (error #t (string-append "scss: invalid selector "
						   (apply string-append x))))))
	      x)))
       sel))))

(define calc-precedence
  (lambda (item)
    (cond ((eq? (car item) 'agent) 1)
	  ((eq? (car item) 'user) (if (and (= (length (caddr item)) 3) 
					   (eq? (list-ref item 2) '!))
				      5
				      2))
	  (else (if (and (= (length (caddr item)) 3)
			 (eq? (list-ref item 2) '!))
		    4
		    3)))))

(define calc-specificity
  (lambda (sel) 
    (let ((count (lambda (pred lst) (length (filter pred lst)))))
      (+ (* (count (lambda (x) (equal? (substring x 0 1) "#")) sel) 100)
	 (* (count (lambda (x) (or (pseudo-class? x) 
				   (equal? (substring x 0 1) "[")
				   (equal? (substring x 0 1) ".")))
		   sel) 10)
	 (count (lambda (x) (or (pseudo-element? x) 
				(exact-exec ident-regexc x)))
		sel)))))

(define check-important 
  (lambda (v) 
    (let ((slist (string-split v #\sp)))
      (if (equal? (car (last-pair slist)) "!important")
	  (let ((relist (list-head slist (- (length slist) 1))))
	    (list (fold (lambda (x y) (if (null? y) x (string-append x " " y)))
			(list-head relist (- (length relist) 1))
			(last-pair relist)) 
		  '!))
	  (list v)))))

(define validate-property 
  (lambda (property value)
    (let ((pe (find (lambda (l) (exact-match (car l) property)) 
		    property-table)))
      (if pe 
	  (if (apply (cadr pe) (list value)) 
	      #t 
	      (error #t (string-append 
			 "invalid value " value " for property " property)))
	  (error #t (string-append "invalid property name " property))))))

(define verify-arg-types
  (lambda (name predicates args startnum)
    (if (not (= (length predicates) (length args)))
	(error #t "predicate argument mismatch"))
    (let ((i 0))
      (for-each 
       (lambda (x) 
	 (if (not (apply x (list (list-ref args i))))
	     (error #t (string-append name 
				      ": wrong argument type in position "
				      (number->string (+ startnum i)))))
	 (set! i (+ i 1)))
       predicates))))

(define display-list (lambda lst (display lst)))

(define lexer-wrapper
  (lambda () 
    (let ((tok (lexer))) 
      (if (and (car tok) (not (eqv? (car tok) 0))) tok '(*eoi*)))))

(define scss:css->scss 
  (lambda (port)
    (verify-arg-types "scss:css->scss" (list port?) (list port) 1)
    (lexer-init 'port port)
    (scss-parser lexer-wrapper display-list)))

(define scss:scss->css
  (lambda (stylesheet port)
    (let* ((ser-selector (lambda (selector)
			   (display (car selector) port)))
	   (ser-property (lambda (pair) 
			  (display "\t" port)
			  (display (car pair) port)
			  (display ": " port)
			  (display (cadr pair) port)
			  (display ";" port)
			  (newline port)))
	   (ser-block (lambda (block)
			(ser-selector (car block))
			(display " {" port)
			(newline port)
			(for-each (lambda (x) (ser-property x)) (cadr block))
			(display "}" port)
			(newline port))))			  
      (verify-arg-types "scss:scss->css"
			(list scss:stylesheet? port?) 
			(list stylesheet port)
			1)
      (for-each (lambda (block) (ser-block block)) stylesheet))))

(define scss:create-cascade 
  (lambda arglist
    (if (null? arglist) 
	(list (list) (list) (list))
	(if (= (length arglist) 3)
	    (begin (verify-arg-types "scss:create-cascade"
				     (make-list 3 scss:stylesheet?) 
				     arglist
				     1)
		   arglist)
	    (error #t "wrong number of arguments to scss:create-cascade")))))
    

(define scss:create-ruleset
  (lambda (sel-str . rest)
    (verify-arg-types "scss:create-ruleset"
		      (list string? 
			    (lambda (x) 
			      (and (list? x)
				   (every (lambda (y) 
					    (and (list? y)
						 (>= (length y) 2)
						 (string? (car y))
						 (string? (cadr y))
						 (if (= (length y) 3)
						     (eq? (caddr y) '!)
						     #t)))
					  x))))
		      (list sel-str rest)
		      1)
    (let ((sel (parse-selector sel-str)))
      (list sel rest))))
						     

(define scss:set-author-stylesheet!
  (lambda (cascade authorsheet) 
    (verify-arg-types "scss:set-author-stylesheet!"
		      (list scss:cascade? scss:stylesheet?)
		      (list cascade authorsheet)
		      1)
    (set-car! (list-cdr-ref cascade 2) authorsheet)))

(define scss:set-user-stylesheet!
  (lambda (cascade usersheet) 
    (verify-arg-types "scss:set-user-stylesheet!"
		      (list scss:cascade? scss:stylesheet?)
		      (list cascade usersheet)
		      1)
    (set-car! cascade usersheet)))

(define scss:set-agent-stylesheet!
  (lambda (cascade agentsheet) 
    (verify-arg-types "scss:set-agent-stylesheet!"
		      (list scss:cascade? scss:stylesheet?)
		      (list cascade agentsheet)
		      1)
    (set-car! (list-cdr-ref cascade 1) agentsheet)))

(define scss:cascade? 
  (lambda (cascade)
    (and (list? cascade) 
	 (= (length cascade) 3) 
	 (every (lambda (x) (or (null? x) (scss:stylesheet? x))) cascade))))
(define scss:stylesheet? 
  (lambda (stylesheet)
    (and (list? stylesheet)
	 (every (lambda (x) 
		  (and (list? x)
		       (= (length x) 2)
		       (list? (car x))
		       (every (lambda (y)
				(and (list? y)
				     (false-if-exception validate-selector)))
			      (car x))
		       (list? (cadr x))
		       (every (lambda (y)
				(and (list? y)
				     (= (length y) 2)
				     (every string? y)))
			      (cadr x))))
		stylesheet))))

(define scss:set-property!
  (lambda (stylesheet selector property value)
    (verify-arg-types "scss:set-property!"
		      (list scss:stylesheet? string? string? string?)
		      (list stylesheet selector property value)
		      1)
    (validate-property property value)
    (let ((sel (find (lambda (x) (equal? (caar x) selector)) stylesheet)))
      (if (not sel)
	  (append! stylesheet `((,(list selector)) (,(list property value))))
	  (let ((p (find (lambda (x) (equal? (car x) property)) (cadr sel))))
	    (if p 
		(set-car! (cdr p) value))
		(append! (cadr sel) (list (list property value))))))))

(define scss:add-ruleset!
  (lambda (stylesheet selector)
    (verify-arg-types "scss:add-ruleset!"
		      (list scss:stylesheet? string?) 
		      (list stylesheet selector)
		      1)
    (if (not (find (lambda (x) (equal? (caar x) selector)) stylesheet))
	(append! stylesheet (list (list (list selector)))))))
(define scss:remove-property!
  (lambda (stylesheet selector property) '()))
(define scss:remove-selector!
  (lambda (stylesheet selector) '()))
(define scss:remove-ruleset!
  (lambda (stylesheet selector) '()))

(define scss:color->hex 
  (lambda (color)
    (verify-arg-types "scss:color->hex" (list string?) (list color) 1)
    (find (lambda (x) (equal? (car x) color)) color-table)))

(define get-prop-entry 
  (lambda (p) (find (lambda (x) (exact-match (car x) p)) property-table)))

(define lookup-in-cascade (lambda (cascade selector property) '()))

(define get-default (lambda (property-entry) '()))

(define get-ancestors 
  (lambda (doc node)
    (if (not (null? node))
	(let ((p (get-sxml-parent doc node))) 
	  (if (and p (not (null? p))) (cons p (get-ancestors doc p)) (list)))
	(list))))

(define expand-stylesheet
  (lambda (stylesheet)
    (let ((ret (list)))
      (for-each (lambda (x)
		  (for-each (lambda (y)
			      (if (not (null? (cdr x)))
				  (set! ret (append ret 
						    (list (cons y (cdr x)))))))
			    (car x)))
		stylesheet)
      ret)))

(define get-general-property
  (lambda (prop pvlist)
    (let* ((split-matches 
	    (lambda (pattern str)
	      (let* ((lst (list))
		     (f (lambda (x) 
			  (set! lst (append lst (list (match:substring x)))))))
		(regexp-substitute/global #f pattern str f)
		lst)))
	   (f (lambda (p) (find (lambda (x) (equal? p (car x))) pvlist)))
	   (prop-bits (string-split prop #\-))
	   (g (lambda (pat d x)
		(let ((ml (split-matches pat (cadr x))))
		  (cond ((= (length ml) 1) (car ml)) 
			((= (length ml) 2)
			 (if (member d '("top" "bottom")) (car ml) (cadr ml)))
			((= (length ml) 3)
			 (cond ((equal? d "top") (car ml))
			       ((member d '("left" "right")) (cadr ml))
			       (else (caddr ml))))
			((= (length ml) 4)
			 (list-ref 
			  ml (list-index '("top" "right" "bottom" "left") d)))
			(else (list))))))
	   (h (lambda (str pattern)
		(let ((match (if (string? pattern) 
				 (string-match pattern str)
				 (regexp-exec pattern str))))
		  (if match (match:substring match) (list))))))
      (cond ((= (length prop-bits) 1) (list))
	    ((equal? (car prop-bits) "border")
	     (if (= (length prop-bits) 2)
		 (let ((b (f "border"))) 
		   (if b (g (string-append "(" 
					   border-width-regex "|" 
					   border-style-regex "|" 
					   pcolor-regex 
					   ")")
			    (cadr prop-bits) 
			    b) 
		       (list)))
		 (let ((bd (get-general-property 
			    (string-append (car prop-bits) "-" 
					   (cadr prop-bits))
			    pvlist)))
		   (if (not (null? bd))
		       (cond ((equal? (caddr prop-bits) "color")
			      (h bd color-regex))
			     ((equal? (caddr prop-bits) "width")
			      (h bd border-width-regex))
			     ((equal? (caddr prop-bits) "style")
			      (h bd border-style-regex)))
		       bd))))
	    ((equal? (car prop-bits) "padding")
	     (let ((p (f "padding"))) 
	       (if p (g padding-regex (cadr prop-bits) p) (list))))
	    ((equal? (car prop-bits) "margin")
	     (let ((m (f "margin"))) 
	       (if m (g margin-regex (cadr prop-bits) m) (list))))
	    ((equal? (car prop-bits) "background")
	     (let ((b (f "background")))
	       (cond ((equal? (cadr prop-bits) "color") (h b pcolor-regex))
		     ((equal? (cadr prop-bits) "attach") (h b pattach-regex))
		     ((equal? (cadr prop-bits) "image") (h b pimage-regex))
		     ((equal? (cadr prop-bits) "repeat") (h b prepeat-regex))
;; FIX THIS!!!
		     ((equal? (cadr prop-bits) "position"))
		     (else (list)))))
	    ((and (equal? (car prop-bits) "list")
		  (equal? (cadr prop-bits) "style"))
	     (let ((l (f "list-style")))
	       (cond ((equal? (caddr prop-bits) "position")
		      (h l (string-append lsposition-regex "|(inherit)")))
		     ((equal? (caddr prop-bits) "image")
		      (h l (string-append lsimage-regex "|(inherit)")))
		     ((equal? (caddr prop-bits) "type")
		      (h l (string-append lstype-regex "|(inherit)")))
		     (else (list)))))
	    ((equal? (car prop-bits) "outline")
	     (let ((o (f "outline")))
	       (cond ((equal? (cadr prop-bits) "color")
		      (h o (string-append ocolor-regex "|(inherit)")))
		     ((equal? (cadr prop-bits) "style")
		      (h o (string-append ostyle-regex "|(inherit)")))
		     ((equal? (cadr prop-bits) "width")
		      (h o (string-append owidth-regex "|(inherit)")))
		     (else (list)))))
	    ((equal? (car prop-bits) "font")
	     (let ((font (f "font"))) 
	       (cond ((equal? (cadr prop-bits) "family") 
		      (h font ffamily-regex))
		     ((equal? (cadr prop-bits) "size") (h font fsize-regex))
		     ((equal? (cadr prop-bits) "style") (h font fstyle-regex))
		     ((equal? (cadr prop-bits) "variant")
		      (h font fvariant-regex))
		     ((equal? (cadr prop-bits) "weight") 
		      (h font fweight-regex))
		     (else (list)))))	    
	    (else (list))))))

;; This method is really just searching for selectors that contain the required
;; property...

(define expand-properties
  (lambda (prop expanded-stylesheet s)
    (let ((ret (list)))
      (for-each (lambda (x) 
		  (let ((p (find (lambda (y) (equal? (car y) prop)) (cadr x))))
		    (if (not p) 
			(let ((q (get-general-property prop (cadr x))))
			  (if (not (null? q))
			      (set! p (list prop q)))))
		    (if p (set! ret (append ret (list (list s (car x) p)))))))
		expanded-stylesheet)
      ret)))

(define lookup-value
  (lambda (cascade selector prop)
    (let* ((g (lambda (y) (equal? (car y) prop)))
	   (ge (lambda (y) (equal? (caaddr y) prop)))
	   (h (lambda (x) (and (= (length (car x)) 1)
			       (or (equal? (caar x) selector)
				   (equal? (caar x) "*")))))
	   (agent-matches (if (not (null? (car cascade)))
			      (filter h (expand-stylesheet (car cascade)))
			      (list)))
	   (user-matches (if (not (null? (cadr cascade)))
			     (filter h (expand-stylesheet (cadr cascade)))
			     (list)))
	   (author-matches (if (not (null? (caddr cascade)))
			       (filter h (expand-stylesheet (caddr cascade)))
			       (list)))
	   (results (append (filter ge (expand-properties 
					prop agent-matches 'agent))
			    (filter ge (expand-properties 
					prop user-matches 'user))
			    (filter ge (expand-properties 
					prop author-matches 'author))))
	   (sr (sort (sort results (lambda (x y) (> (calc-precedence x)
						    (calc-precedence y))))
		     (lambda (x y) (> (calc-specificity (cadr x))
				      (calc-specificity (cadr y)))))))
      (if (not (null? sr))
	  (cadr (list-ref (car sr) 2))
	  sr))))

(define select-value
  (lambda (cascade node doc prop)
    (letrec 
	((f (lambda (sel nodes)
	      (let ((ni (letrec 
			    ((next-ident 
			      (lambda (sel-list)
				(if (not (null? sel-list)) 
				    (if (exact-exec ident-regexc
						    (car sel-list)) 
					(car sel-list)
					(next-ident (cdr sel-list)))
				    '()))))
			  (next-ident sel))))
		(cond ((null? sel) #t)
		      ((null? nodes) #f)
		      ((equal? (car sel) ":first-child")
		       (if (>= (length nodes) 2)
			   (let ((cl (find (lambda (x) 
					     (and (list x) 
						  (not (eq? (car x) '@))))
					   (cadr nodes))))
			     (if (eq? (car nodes) (car cl))
				 (f (cdr sel) nodes)
				 #f))
			   #f))
		      ((pseudo-class? (car sel))
		       (if (and (equal? (sxml-node-name (car nodes)) ni)
				(internal-pseudo-class-handler 
				 (substring (car sel) 1) doc (car nodes)))
			   (f (cdr sel) nodes)
			   #f))
		      ((pseudo-element? (car sel))
		       (if (and (equal? (sxml-node-name (car nodes)) ni)
				(internal-pseudo-element-handler 
				 (substring (car sel) 1) doc (car nodes)))
			   (f (cdr sel) nodes)
			   #f))
		      ((equal? (substring (car sel) 0 1) "#")
		       (if (and (equal? (sxml-node-name (car nodes)) ni)
				(internal-id-handler 
				 (substring (car sel) 1) doc (car nodes)))
			   (f (cdr sel) nodes)
			   #f))
		      ((equal? (substring (car sel) 0 1) ".")
		       (letrec ((g (lambda (x) 
				     (if (and (not (null? x))
					      (equal? (substring (car x) 0 1)
						      "."))
					 (cons (substring (car x) 1) 
					       (g (cdr x)))
					 '())))
				(l (cons (substring (car sel) 1) 
					 (g (cdr sel)))))
			 (if (and (equal? (sxml-node-name (car nodes)) ni)
				  (internal-dot-handler l doc (car nodes)))
			     (f (list-cdr-ref sel (length l)) nodes)
			     #f)))
		      ((equal? (substring (car sel) 0 1) "[")
		       (let* ((match (regexp-exec attr-sel-regexc 
						  (car sel)))
			      (l (match:substring match 1))
			      (t (match:substring match 15))
			      (r (match:substring match 16))
			      (v (sxml-attr-val node l)))
			 (if (and (string? r)
				  (or (and (equal? (substring r 0 1) "\"")
					   (equal? (substring 
						    r (- (string-length r) 1))
						   "\""))
				      (and (equal? (substring r 0 1) "'")
					   (equal? (substring 
						    r (- (string-length r) 1))
						   "'"))))
			     (set! r (substring r 1 (- (string-length r) 1))))
			 (if (and v (or (and (equal? t "=")
					     (equal? (cadr v) 
						     (substring 
						      r 1 (- (length r) 2))))
					(and (equal? t "~=")
					     (find (lambda (x) (equal? x r))
						   (string-split (cadr v) 
								 #\sp)))
					(and (equal? t "|=")
					     (> (string-length (cadr v))
						(+ (string-length r) 1))
					     (equal? (substring 
						      (cadr v) 
						      0 
						      (+ (string-length r) 
							 1))
						     (string-append r "-")))
					(not t)))
			     (f (cdr sel) nodes)
			     #f)))
		      ((equal? (car sel) (sxml-node-name (car nodes)))
		       (let ((i (list-index 
				 (lambda (x) 
				   (equal? (sxml-node-name x) ni)) 
				 (cdr nodes))))
			 (cond ((null? (cdr sel)) #t)
			       (i (f (cdr sel) (list-cdr-ref nodes i)))
			       (else #f))))
		      ((equal? (car sel) "*") (f (cdr sel) (cdr nodes)))
		      (else #f))))))
      (let* ((a (cons node (get-ancestors doc node)))
	     (g (lambda (y) (equal? (car y) prop)))
	     (ge (lambda (y) (equal? (caaddr y) prop)))
	     (h (lambda (x) (and (f (reverse (car x)) a) 
				 (not (null? (cdr x)))
				 (find g (cadr x)))))
	     (agent-matches (if (not (null? (car cascade)))
				(filter h (expand-stylesheet (car cascade)))
				(list)))
	     (user-matches (if (not (null? (cadr cascade)))
			       (filter h (expand-stylesheet (cadr cascade)))
			       (list)))
	     (author-matches (if (not (null? (caddr cascade)))
				 (filter h (expand-stylesheet (caddr cascade)))
				 (list)))
	     (results (append (filter ge (expand-properties 
					  prop agent-matches 'agent))
			      (filter ge (expand-properties 
					  prop user-matches 'user))
			      (filter ge (expand-properties 
					  prop author-matches 'author))))
	     (sr (sort (sort results (lambda (x y) (> (calc-precedence x)
						      (calc-precedence y))))
		       (lambda (x y) (> (calc-specificity (cadr x))
					(calc-specificity (cadr y)))))))
	(if (not (null? sr))
	    (cadr (list-ref (car sr) 2))
	    sr)))))

;; This is a hack -- need to fix this for cases where default val depends on
;; other defaults / lookups.

(define get-default-prop-value
  (lambda (pe) 
    (if (or (not pe) (null? pe)) '() (car (list-ref pe 2)))))

(define scss:select-value
  (lambda (source selector . r)
    (let ((cascade #t)
	  (xml #f)
	  (throw-error (lambda () 
			 (error #t (string-append "wrong number of arguments "
						  "to scss:select-value")))))
      (verify-arg-types "scss:select-value"
			(list (lambda (x) (or (scss:cascade? x) 
					      (if (scss:stylesheet? x)
						  (begin (set! cascade #f) #t)
						  #f)))
			      (lambda (x) (or (string? x) 
					      (if (sxml-node? x)
						  (begin (set! xml #t) #t)
						  #f))))
			(list source selector)
			1)
      (if (not xml)
	  (begin
	    (verify-arg-types "scss:select-value" (list string?) r 3)
	    (let ((val (lookup-value (if cascade
					 source 
					 (list (list) (list) source))
				     selector
				     (car r))))
	      (if (null? val)
		  (get-default-prop-value (get-prop-entry (car r)))
		  val)))
	  (begin
	    (verify-arg-types "scss:select-value" (list sxml-doc? string?) r 3)
	    (let ((pe (get-prop-entry (cadr r))))
	      (letrec ((f (lambda (c n d p) 
			    (let ((v (select-value c n d p))
				  (par (get-sxml-parent d n)))
			      (if (or (and (null? v) 
					   par
					   pe
					   (list-ref pe 3))
				      (equal? v "inherit"))
				  (f c par d p) 
				  v)))))
		(let ((val (f (if cascade source (list (list) (list) source))
			      selector
			      (car r)
			      (cadr r))))
		  (if (null? val) 
		      (get-default-prop-value pe)
		      val)))))))))
