#!/usr/bin/python
#     Copyright (C) 2005, 2006 Stewart Adam
#    This file is part of FWBackups.

#    FWBackups is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.

#    FWBackups is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.

#    You should have received a copy of the GNU General Public License
#    along with FWBackups; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

# -- file based on file from Kadischi package -- 
import os
import sys
import shvar
import rpm
import urlgrabber.grabber as grabber

def normalize_path (parts, rootpath = None):
	if rootpath == None:
		rootpath = '/'
	path = rootpath
	for part in parts:
		path = '/'.join([path, part])
	return os.path.normpath (path)

def path_exists (path, quiet = False):
	if os.path.exists (path):
		if not quiet:
			flc_log ('Path %s exists. OK' % path)
		return True
	else:
		if not quiet:
			flc_log ('Error - path %s doesn`t exist.' % path)
		return False

def url_exists (url, quiet = False):
	try:
		u = grabber.urlopen (url, retry = 5)
		if not quiet:
			flc_log ('URL %s contacted. OK' % url)
		u.close ()
		return True
	except grabber.URLGrabError, e:
		if not quiet:
			flc_log ("An error occurred contacting %s." % url)
			flc_log ("URLGrabError:\n %s" % e.strerror)
		return False

def clear_rpm_db_files (rootdir):
	"""Removes rpm database files in the destination if they exist"""
	def rmpath (p):
		try: os.remove (p)
        	except OSError: pass
	rmpath (normalize_path ('var/lib/rpm/__db.001', rootdir))
	rmpath (normalize_path ('var/lib/rpm/__db.002', rootdir))
	rmpath (normalize_path ('var/lib/rpm/__db.003', rootdir))
	
def check_installed_rpms (rootdir, required_rpms, clear_db = True):
        """Sanity check the target system for required RPMs"""
        # By default, clears out the db files before and after doing any operations
        missing_rpms = []
	if clear_db:
		clear_rpm_db_files (rootdir)
        ts = rpm.TransactionSet (rootdir)
        for package in required_rpms:
            mi = ts.dbMatch ('name', package)
            if mi.count () == 0:
                missing_rpms.append (package)
        if clear_db:
		clear_rpm_db_files (rootdir)
	return missing_rpms

def get_kernel_version (rootdir):
        kernel_version = None
        clear_rpm_db_files (rootdir)
        ts = rpm.TransactionSet (rootdir)
        for mi in ts.dbMatch ('name', 'kernel'):
            kernel_version = "%s-%s" % (mi['version'], mi['release'])
        clear_rpm_db_files (rootdir)
        return kernel_version

def clean_directory (dir, remove_when_done=True, keep_child_directories=False):
	"""Removes all files in a directory, optionally removing the directory afterwards"""
	assert os.path.isdir and not os.path.islink (dir)
	file_list = os.listdir (dir)

	for f in file_list:
        	path = os.path.join (dir, f)
        	if os.path.isdir (path) and not os.path.islink (path):
			clean_directory (path,
					not keep_child_directories,
					keep_child_directories)
        	else:
			os.remove (path)

	if remove_when_done:
		os.rmdir (dir)

def sysconfig_file (rootdir, name):
	path = normalize_path(['/etc/sysconfig/', name], rootdir)
	return shvar.shvarFile (path)

def flc_log (msg):
	sys.stdout.write (msg + '\n')

def cleanup (builddir):
	os.system ("umount %s 2>/dev/null" % normalize_path(['/system/proc'],builddir))
	if os.path.exists (normalize_path([builddir,'product'])):
		clean_directory ('/tmp/product')
		shutil.move (normalize_path([builddir,'product']), '/tmp/product')
	elif os.path.exists (normalize_path([builddir,'.buildstamp'])):
		os.unlink ('/tmp/product/.buildstamp')
		shutil.move (normalize_path([builddir,'.buildstamp']), '/tmp/product/.buildstamp')
	else:
		os.unlink ('/tmp/product/.buildstamp')
	clean_directory (builddir)
	
def run (command, builddir = None):
	rc = os.system (command)
	if rc != 0:
		flc_log ('\n  ***  Fatal error: %s returned non zero (%s) exit code. Aborting execution.\n' % (command.rsplit() [0], rc))
		if builddir != None:
			flc_log ('Cleaning up temporary files...')
			cleanup (builddir)
			flc_log ('Done.')
		else:
			flc_log ('You should clean temporary files.')
		sys.exit (1)
