/* Schedwi
   Copyright (C) 2014 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* @file xmem.c
 * Memory and string functions with memory allocation error detection.
 */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#include <lwc_log.h>
#include <xmem.h>


/**
 * malloc with abort on error
 *
 * @param[in] size The size of the requested memory
 * @return Pointer to the allocated memory
 */
void *
xmalloc (size_t size)
{
	void *value = malloc (size);

	if (value == 0) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		exit (EXIT_FAILURE);
	}
	return value;
}


/**
 * realloc with abort on error
 *
 * @param[in] ptr  Pointer to the buffer to resize
 * @param[in] size The size of the requested memory
 * @return Pointer to the allocated memory
 */
void *
xrealloc (void *ptr, size_t size)
{
	void *value = realloc (ptr, size);

	if (value == 0) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		exit (EXIT_FAILURE);
	}
	return value;
}


/**
 * calloc with abort on error
 *
 * @param[in] nmemb Number of elements to allocate
 * @param[in] size  The size of each element
 * @return Pointer to the allocated memory
 */
void *
xcalloc (size_t nmemb, size_t size)
{
	void *value = calloc (nmemb, size);

	if (value == 0) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		exit (EXIT_FAILURE);
	}
	return value;
}


/**
 * strdup with abort on error
 *
 * @param[in] s The string to copy
 * @return Pointer to the copied string (to be freed by the caller)
 */
char *
xstrdup (const char *s)
{
	char *value = strdup (s);

	if (value == 0) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		exit (EXIT_FAILURE);
	}
	return value;
}


/**
 * strndup with abort on error
 *
 * @param[in] s The string to copy
 * @param[in] n Number of characters to copy
 * @return Pointer to the copied string (to be freed by the caller)
 */
char *
xstrndup(const char *s, size_t n)
{
	char *value = strndup (s, n);

	if (value == 0) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		exit (EXIT_FAILURE);
	}
	return value;
}

/*------------------------======= End Of File =======------------------------*/
