/* Schedwi
   Copyright (C) 2007-2014 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* utils.c -- Several functions */

#include <schedwi.h>

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_SIGNAL_H
#include <signal.h>
#endif

#if HAVE_FCNTL_H
#include <fcntl.h>
#endif

#if HAVE_CTYPE_H
#include <ctype.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#include <utils.h>
#include <xmem.h>

#if defined(_GNU_SOURCE) && !HAVE_CLEARENV
extern char **environ;
#endif


/*
 * Convert a unsigned long long int to a string
 *
 * Return:
 *   The number of characters copied in str
 */
unsigned int
copy_ulltostr (unsigned long long int v, char *str)
{
	char c;
	unsigned int len, i, j;

	if (str == NULL) {
	    return 0;
	}

	i = 0;
	do {
		str[i++] = '0' + v % 10;
		v /= 10;
	} while (v > 0);
	str[i] = '\0';
	len = i;

	/* Reverse the string */
	j = 0;
	i--;
	while (j < i) {
	    c = str[i];
	    str[i--] = str[j];
	    str[j++] = c;
	}

	return len;
}


/*
 * Return the filename part of a path
 *
 * Return:
 *   A pointer to the filename part in the provided path
 */
const char *
base_name (const char *path)
{
	const char * s, * prev;
	size_t sep_length;

#ifdef HAVE_ASSERT_H
	assert (path != 0);
#endif
	prev = 0;
	s    = path;
	sep_length = strlen (DIR_SEP);
	while ((s = strstr (s, DIR_SEP)) != 0) {
		s += sep_length;
		prev = s;
	}
	if (prev == 0) {
		return path;
	}
	return prev;
}


/*
 * Print copyright and version information to stdout
 */
void
version ()
{
	printf ("%s (Schedwi) %s\n", program_invocation_name, PACKAGE_VERSION);
	(void)puts("Copyright (C) 2007-2015 Herve Quatremain");
	(void)puts("Schedwi is free software, covered by the GNU");
	(void)puts("General Public License version 3 or later.");
	(void)puts("You may redistribute copies of Schedwi under");
	(void)puts("the terms of this license.");
}


/*
 * Remove a file. If the remove fails, then try to truncate (size=0) the
 * file
 *
 * Return:
 *   0 --> File removed or truncated
 *  -1 --> Error (errno is set)
 */
int
my_unlink (const char *file_name)
{
	FILE *f;
	int save_errno;

	if (file_name == NULL || file_name[0] == '\0') {
		return 0;
	}

	/* Remove the file */
	if (unlink (file_name) != 0) {

		/* File does not exist */
		if (errno == ENOENT) {
			return 0;
		}

		/* Can't remove the file.  Try to truncate it */
		save_errno = errno;
		f = fopen (file_name, "wb");
		if (f == NULL) {
			/* Fail to truncate the file.  Return an error */
			errno = save_errno;
			return -2;
		}
		(void)fclose (f);
	}
	return 0;
}


/*
 * Close all the opened file with a file descriptor above 2
 */
void
close_opened_files ()
{
	static char done = 0;
	int i;


	if (done == 0) {
		for (i = getdtablesize () - 1; i >= 3; i--) {
			close (i);
		}
		done = 1;
	}
}


/*
 * Set the standard file descriptors (0, 1 and 2) to /dev/null
 */
void
reset_standard_fd ()
{
	int fd;


	close (0);
	close (1);
	close (2);

	/*
	 * Redirect 0 (stdin), 1 (stdout) and 2 (stderr) to /dev/null.  This
	 * way, if sub-proccesses/modules try to inadvertently read/write to
	 * this file descriptors it will not mess other ones (pid file,
	 * socket, database connection, ...)
	 */
	fd = open ("/dev/null", O_RDWR); /* `fd' should be 0 */
	dup2 (fd, 1);
	dup2 (fd, 2);
}


/*
 * RAZ the process structures:
 *    - Close all open files and set the default file descriptors to /dev/null
 *    - Empty the environment
 *    - Set the default action for all the signals
 *    - Remove the signal mask
 *
 * Failure of one of this step is not reported
 *
 * Note:
 *    This function is called in two places by the Schedwi agent (schedwiclnt):
 *       - At the start of the agent, when it goes into background
 *       - Every time a job is started
 *
 *    In the Schedwi server (schedwisrv) it is just called once at the
 *    beginning, when the program goes into background.
 *
 *    Clear the environment may cause issues.  In particular, some variables
 *    may be required by the job subprocesses.  For instance, using Cygwin, the
 *    PATH environment variable contains the path to the Cygwin DLL
 *    (cygwin1.dll) which is required to fork/exec the job command.  This is
 *    why the environment is not cleared for Cygwin only.
 *    On other systems, other variables may be required (LD_LIBRARY_PATH for
 *    instance for SunOS and Linux) but no special care is taken and they
 *    will be erased by this function.  If they are required they can be
 *    defined as an environment for the job in the Schedwi server.
 */
void
sanitise ()
{
	int i;
	struct sigaction sa;
	sigset_t mask;


	/* Close all files */
	close_opened_files ();
	reset_standard_fd ();

	/* Clear the environment */
#ifndef CYGWIN_IN_USE
#if HAVE_CLEARENV
	clearenv ();
#elif defined(_GNU_SOURCE)
	environ = NULL;
#endif
#endif

	/* Clear the signal actions and mask */
	memset (&sa, 0, sizeof (struct sigaction));
	sa.sa_handler = SIG_DFL;
	sigemptyset (&sa.sa_mask);
	for (i = 0; i < 128; i++) {
		sigaction (i, &sa, NULL);
	}
	sigfillset (&mask);
	sigprocmask (SIG_UNBLOCK, &mask, NULL);
}


/*
 * Clean the process
 */
void
clean_process ()
{
	/* Change default directory to `/' */
#if HAVE_CHDIR
	(void)chdir ("/");
#endif /* HAVE_CHDIR */

	/*
	 * Clean the process.  Protect the file descriptor 3 as it may
	 * be used by GnuTLS.
	 */
	sanitise ();

#if HAVE_UMASK
	umask (0027);
#endif
}


/*
 * Start the daemon
 *
 * Return:
 *   0 --> No error
 *  -1 --> fork error (errno is set)
 */
int
go_daemon()
{
#if HAVE_PID_T
	pid_t p;
#else
	int p;
#endif

#if HAVE_WORKING_FORK
	p = fork ();
	if (p < 0) {
		return -1;
	}
	if (p > 0) {
		exit (0);
	}
#endif

#if HAVE_SETSID
	setsid();
#endif

#if HAVE_WORKING_FORK
	p = fork ();
	if (p < 0) {
		return -1;
	}
	if (p > 0) {
		exit (0);
	}
#endif

	clean_process ();
	return 0;
}


/*
 * Create the intermediate directories leading to `file_path'.  The last
 * component is not created.
 *
 * @param[in] file_path Full file name.
 * @param[in] user      Owner of the created directories (if NULL or empty, the
 *                      owner is not changed)
 * @param[in] group     Group owning of the created directories (if NULL or
 *                      empty, the owner is not changed)
 * @return 0 on success,
 *        -1 if one component already exists and is not a directory or
 *        -2 on failure to create a directory or change its owner
 *        (errno is set)
 */
int
mkpath (const char *file_path, const char *user, const char *group)
{
	char *path, *p;
	size_t sep_length;
	struct stat stat_str;
	int save_errno;
#if HAVE_CHOWN
	uid_t uid;
	gid_t gid;

	uid = -1;
	get_userid (user, &uid);
	gid = -1;
	get_groupid (group, &gid);
#endif

#ifdef HAVE_ASSERT_H
	assert (file_path != NULL && file_path[0] != '\0');
#endif

	path = xstrdup (file_path);

	sep_length = strlen (DIR_SEP);
	for (p = strstr (path, DIR_SEP);
	     p != NULL;
	     p = strstr (p + sep_length, DIR_SEP))
	{
		if (p == path) {
			continue;
		}
		*p = '\0';
		if (stat (path, &stat_str) == 0) {
			if (S_ISDIR (stat_str.st_mode) == 0) {
				/* Not a directory */
				free (path);
				return -1;
			}
			*p = DIR_SEP[0];
			continue;
		}
		if (mkdir (path, 0755) != 0) {
			save_errno = errno;
			free (path);
			errno = save_errno;
			return -2;
		}
#if HAVE_CHOWN
		if ((uid != -1 || gid != -1) && chown (path, uid, gid) != 0) {
			save_errno = errno;
			free (path);
			errno = save_errno;
			return -2;
		}
#endif
		*p = DIR_SEP[0];
	}
	free (path);
	return 0;
}


static int fd_pid_file = -1;
static char *pid_file_name = NULL;

/*
 * Write the PID of the current process to the provided file.
 *
 * Return:
 *   0 --> No error
 *  -1 --> Fail to open the file (errno is set)
 *   1 --> An other daemon is already running
 */
int
write_pid_file (const char *file_name)
{
	char pid_str[26];

	if (file_name == NULL || file_name[0] == '\0') {
		return 0;
	}

	if (pid_file_name != NULL) {
		close_pid_file ();
	}
	pid_file_name= xstrdup (file_name);

	/* If need be, create the intermediate directories */
	mkpath (file_name, NULL, NULL);

	/* Open the PID file */
	fd_pid_file = open (file_name, O_RDWR|O_CREAT, 00644);
	if (fd_pid_file < 0) {
		return -1;
	}

	/*
	 * Try to lock it.  If it fails it's probably because an other
	 * daemon is already running
	 */
#if HAVE_LOCKF && F_TLOCK
	if (lockf (fd_pid_file, F_TLOCK, 0) < 0) {
		return 1;
	}
#elif HAVE_FLOCK && LOCK_EX && LOCK_NB
	if (flock (fd_pid_file, LOCK_EX | LOCK_NB) < 0) {
		return 1;
	}
#endif

	/* Write the PID to the file */
	(void)snprintf (pid_str, 26, "%ld\n", (long int)getpid ());
	pid_str [26 - 1] = '\0';
	(void)write (fd_pid_file, pid_str, strlen (pid_str));
	return 0;
}


/*
 * Close and remove the file containing the PID of the daemon
 */
void
close_pid_file ()
{
	if (pid_file_name != NULL) {
		my_unlink (pid_file_name);
		free (pid_file_name);
		pid_file_name = NULL;
	}

	if (fd_pid_file >= 0) {
		close (fd_pid_file);
	}
	fd_pid_file = -1;
}


/*
 * Change the user id of the current process
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (errno is set)
 */
static int
change_user_id (uid_t user)
{
#if HAVE_SETREUID
	if (setreuid (user, user) != 0) {
		return -1;
	}
#else

#if HAVE_SETUID
	if (setuid (user) != 0) {
		return -1;
	}
#endif /* HAVE_SETUID */

#if HAVE_SETEUID
	if (seteuid (user) != 0) {
		return -1;
	}
#endif /* HAVE_SETEUID */

#endif /* HAVE_SETREUID */

	return 0;
}


/*
 * Change the group id of the current process
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (errno is set)
 */
static int
change_group_id (gid_t group)
{
#if HAVE_SETREGID
	if (setregid (group, group) != 0) {
		return -1;
	}
#else

#if HAVE_SETGID
	if (setgid (group) != 0) {
		return -1;
	}
#endif /* HAVE_SETGID */

#if HAVE_SETEGID
	if (setegid (group) != 0) {
		return -1;
	}
#endif /* HAVE_SETEGID */

#endif /* HAVE_SETREGID */

	return 0;
}

/*
 * Convert a user name or ID in a user ID.
 *
 * @param name:
 *         User name or ID.
 * @param uid:
 *         Output UID.
 * @return:
 *         0 on success (uid is set).  uid is set to -1 if name is NULL or
 *         empty.  -1 if the user cannot be find.
 */
int
get_userid (const char *name, uid_t *uid)
{
	unsigned int i;
#if HAVE_GETPWNAM || HAVE_GETPWENT
	struct passwd *pass;
#endif

	if (name == NULL) {
		*uid = -1;
		return 0;
	}

	for (i = 0; name[i] != '\0' && isspace (name[i]) != 0; i++);
	if (name[i] == '\0') {
		*uid = -1;
		return 0;
	}
	if (isdigit (name[i]) != 0) {
		*uid = (uid_t) atoi (name + i);
		return 0;
	}

#if HAVE_GETPWNAM

	pass = getpwnam (name + i);
	if (pass == NULL) {
		return -1;
	}
	*uid = pass->pw_uid;
	return 0;

#elif HAVE_GETPWENT && HAVE_ENDPWENT

	while ((pass = getpwent()) != NULL) {
		if (strcmp (pass->pw_name, name + i) == 0) {
			*uid = pass->pw_uid;
			endpwent ();
			return 0;
		}
	}
	endpwent ();
	return -1;

#else
	*uid = -1;
	return 0;
#endif
}


/*
 * Change the user id of the current process.  name may contain a number or
 * a name in the password file
 *
 * Return:
 *   0 --> No error
 *  -1 --> System error (errno is set)
 *  -2 --> User name not found in the password file
 */
int
change_user (const char *name)
{
	uid_t uid;

	if (get_userid (name, &uid) < 0) {
		return -2;
	}

	if (uid == -1) {
		return 0;
	}

	return change_user_id (uid);
}


/*
 * Convert a group name or ID in a group ID.
 *
 * @param name:
 *         Group name or ID.
 * @param gid:
 *         Output GID.
 * @return:
 *         0 on success (gid is set).  gid is set to -1 if name is NULL or
 *         empty.  -1 if the group cannot be find.
 */
int
get_groupid (const char *name, gid_t *gid)
{
	unsigned int i;
#if HAVE_GETGRNAM || HAVE_GETGRENT
	struct group *group_str;
#endif

#ifdef HAVE_ASSERT_H
	assert (gid != NULL);
#endif

	if (name == NULL) {
		*gid = -1;
		return 0;
	}

	for (i = 0; name[i] != '\0' && isspace (name[i]) != 0; i++);
	if (name[i] == '\0') {
		*gid = -1;
		return 0;
	}
	if (isdigit (name[i]) != 0) {
		*gid = (gid_t) atoi (name + i);
		return 0;
	}

#if HAVE_GETGRNAM

	group_str = getgrnam (name + i);
	if (group_str == NULL) {
		return -1;
	}
	*gid = group_str->gr_gid;
	return 0;

#elif HAVE_GETGRENT && HAVE_ENDGRENT

	while ((group_str = getgrent()) != NULL) {
		if (strcmp (group_str->gr_name, name + i) == 0) {
			*gid = group_str->gr_gid;
			endgrent ();
			return 0;
		}
	}
	endgrent ();
	return -1;

#else
	*gid = -1;
	return 0;
#endif
}


/*
 * Change the group id of the current process.  name may contain a number or
 * a name in the group file
 *
 * Return:
 *   0 --> No error
 *  -1 --> System error (errno is set)
 *  -2 --> Group name not found in the group file
 */
int
change_group (const char *name)
{
	gid_t gid;

	if (get_groupid (name, &gid) < 0) {
		return -2;
	}

	if (gid == -1) {
		return 0;
	}

	return change_group_id (gid);
}


/*
 * Escape the provided string.  In this string, `"' are added
 * around the argument (if add_quotes equal 1) and `"' and `\' in the
 * argument are escaped by \.
 *
 * Return:
 *    The argument string (to be freed by the caller)
 */
char *
escape_string (const char *str, unsigned int str_len, char add_quotes)
{
	int i, j;
	char *s;

#ifdef HAVE_ASSERT_H
	assert (str != NULL);
#endif

	/* Count the number of `"' */
	i = 0;
	s = strchr (str, '"');
	while (s != NULL) {
		i++;
		s++;
		s = strchr (s, '"');
	}

	/* Count the number of `\' */
	s = strchr (str, '\\');
	while (s != NULL) {
		i++;
		s++;
		s = strchr (s, '\\');
	}

	s = (char *) xmalloc (str_len + i + 3);

	i = 0;
	if (add_quotes != 0) {
		s[i++] = '"';
	}

	for (j = 0; j < str_len; j++) {
		if (str[j] == '"' || str[j] == '\\') {
			s[i++] = '\\';
		}
		s[i++] = str[j];
	}
	if (add_quotes != 0) {
		s[i++] = '"';
	}
	s[i] = '\0';

	return s;
}


/*
 * Read a file and return a newly allocated string with the content,
 * and set `*length' to the length of the string.  The string is
 * zero-terminated, but the terminating zero byte is not counted in
 * `*length'.
 *
 * Return:
 *   The content of the file (to be freed by free()) OR
 *   NULL in case of error (errno is set and `*length' is undefined)
 */
char *
read_file (const char *filename, size_t *length)
{
	struct stat stat_str;
	FILE *f;
	char *file_content;
	size_t nb_read;
	int save_errno;


#if HAVE_ASSERT_H
	assert (filename != NULL);
#endif

	/* Retrieve the file size */
	if (stat (filename, &stat_str) != 0) {
		return NULL;
	}

	/* Create a buffer for reading the whole file */
	file_content = (char *) xmalloc (stat_str.st_size + 1);

	/* Open and read the whole file */
	f = fopen (filename, "rb");
	if (f == NULL) {
		save_errno = errno;
		free (file_content);
		errno = save_errno;
		return NULL;
	}

	clearerr (f);
	nb_read  = fread (file_content, 1, stat_str.st_size, f);
	if (nb_read != stat_str.st_size && ferror (f) != 0) {
		save_errno = errno;
		fclose (f);
		free (file_content);
		errno = save_errno;
		return NULL;
	}
	fclose (f);

	file_content[stat_str.st_size] = '\0';
	if (length != NULL) {
		*length = stat_str.st_size;
	}
	return file_content;
}

/*------------------------======= End Of File =======------------------------*/
