/* Schedwi
   Copyright (C) 2007-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * @file user_commands.c
 * Deal with user commands to stop/start/... jobs
 */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#include <job_status_state.h>
#include <lwc_linkedlist.h>
#include <stopjob.h>
#include <sql_hierarchy.h>
#include <commands.h>
#include <xmem.h>
#include <job_status_set.h>
#include <lwc_log.h>
#include <job_status_node.h>
#include <user_commands.h>


/**
 * Perform the user requested action (like stopping a job, or changing the
 * status of a job/jobset)
 *
 * @param[in] cmd The command_action_ptr object that contains the details of
 *                the action to perform and the job on which it must apply.
 *
 * @return 0 on success or -1 on error.  An error message is logged by
 *         lwc_writeLog()
 */
static int
commands_action (command_action_ptr cmd)
{
	job_status_state status;
	char *s, *fmt, *err_msg, *path;
	size_t len;
	lwc_LL *hierarchy_list;
	int ret;
	job_status_node_ptr job_details;


	if (cmd == NULL) {
		return 0;
	}

	/* Get the job hierarchy */
	err_msg = NULL;
	hierarchy_list = hierarchy_list_new (	cmd->workload, cmd->job_id,
						&err_msg);
	if (hierarchy_list == NULL) {
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			free (err_msg);
		}
		else {
			lwc_writeLog (	LOG_CRIT,
		_("Workload %d: Job/Jobset Id %lld: cannot retrieve details"),
					cmd->workload, cmd->job_id);
		}
		return -1;
	}
	path = hierarchy_list_to_job_full_path (hierarchy_list);

	switch (cmd->cmd) {

	/* Stop the job/jobset */
	case COMMAND_STOP:
		lwc_writeLog (LOG_INFO,
	_("Workload %d: %s (id %lld): stop request by user (%s)"),
			cmd->workload, path, cmd->job_id,
			(cmd->user_details == NULL) ? "-" : cmd->user_details);
		err_msg = NULL;
		if (stopjob (	hierarchy_list, path,
				cmd->workload, cmd->job_id, &err_msg) != 0)
		{
			if (err_msg != NULL) {
				lwc_writeLog (LOG_ERR,
			_("Workload %d: %s (id %lld): failed to stop: %s"),
					cmd->workload, path, cmd->job_id,
					err_msg);
				free (err_msg);
			}
			else {
				lwc_writeLog (LOG_ERR,
				_("Workload %d: %s (id %lld): failed to stop"),
					cmd->workload, path, cmd->job_id);
			}
			free (path);
			hierarchy_list_destroy (hierarchy_list);
			return -1;
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		free (path);
		hierarchy_list_destroy (hierarchy_list);
		return 0;

	/* Change the status of the job/jobset */
	case COMMAND_CHANGE_STATUS:
		status = job_status_state_int2status (cmd->parameter);
		lwc_writeLog (LOG_INFO,
_("Workload %d: %s (id %lld): new status %s requested by user (%s)"),
			cmd->workload, path, cmd->job_id,
			job_status_state2str (status),
			(cmd->user_details == NULL) ? "-" : cmd->user_details);
		job_details = job_status_node_get (	cmd->workload,
							cmd->job_id,
							path);
		free (path);
		if (job_details == NULL) {
			hierarchy_list_destroy (hierarchy_list);
			return -1;
		}
		job_status_node_set_status (job_details, status, 0);
		if (cmd->user_details == NULL) {
			job_status_node_set_message (	job_details,
							_("Forced by user"));
		}
		else {
			fmt = _("Forced by %s");
			len = strlen (fmt) + strlen (cmd->user_details);
			s = (char *) xmalloc (len);
			snprintf (s, len, fmt, cmd->user_details);
			job_status_node_set_message (job_details, s);
			free (s);
		}
		ret = job_status_force (hierarchy_list, job_details);
		job_status_node_destroy (job_details);
		hierarchy_list_destroy (hierarchy_list);
		return ret;

	default:
		break;
	}
	free (path);
	hierarchy_list_destroy (hierarchy_list);
	return 0;
}


/*
 * Check if some commands are pending and if yes, process them.
 */
void
user_command_check ()
{
	lwc_LL *command_list;
	command_action_ptr cmd;


	command_list = command_get_list ();
	lwc_rewindLL (command_list);
	while ((cmd = (command_action_ptr) lwc_nextLL (command_list)) != NULL)
	{
		if (commands_action (cmd) != 0) {
			command_failed (cmd);
		}
		else {
			command_done (cmd);
		}
	}
	command_destroy_list (command_list);
}

/*-----------------============== End Of File ==============-----------------*/
