/* Schedwi
   Copyright (C) 2007-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* testfile.c -- Test the presence of a file on a agent */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lwc_log.h>
#include <sql_common.h>
#include <sql_hosts.h>
#include <sql_get_environment.h>
#include <net_utils.h>
#include <startjob_json_builder.h>
#include <xmem.h>
#include <testfile.h>


/*
 * Error callback function for the sql_get_environment*() functions
 */
static void
sql_get_environment_error_logger (	void *data, const char *msg,
					int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
_("Database error while trying to retrieve a job environment variables"));
	}
}


/*
 * Error callback function for the sql_host_get_by_id() function
 */
static void
sql_host_get_main_row_error_logger (	void *data, const char *msg,
					int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
		_("Database error while retrieving the agent host details"));
	}
}


/*
 * Callback for the sql_get_environment() function which retrieves the command
 * environment variables
 *
 * Return: Always 0
 */
static int
environment_callback (	void *obj, unsigned int pos,
			const char *name, const char *value)
{
	startjob_json_builder_ptr json_obj = (startjob_json_builder_ptr) obj;


#if HAVE_ASSERT_H
	assert (json_obj != NULL && name != NULL && value != NULL);
#endif

	startjob_json_builder_add_environment (json_obj, name, value);
	return 0;
}


/*
 * Check the existance of the provided file on the provided host ID
 *
 * Return:
 *   0 --> No error.  The file exists on the remote host.  If not
 *         NULL, out_message contains a reply message sent by the agent.  It
 *         may be NULL if the agent didn't send back a message.  out_message
 *         must be freed by the caller by free()
 *   1 --> The file does not exists on the remote agent (or error on the
 *         agent side).  If not NULL, out_message contains a reply message
 *         sent by the agent.  It may be NULL if the agent didn't send back
 *         a message.  out_message must be freed by the caller by free()
 *  -1 --> Error (a message is logged using lwc_writeLog()).  If not NULL,
 *         out_message contains an error message.  It may be NULL if there
 *         is not enough memory to allocate a buffer for this message.
 *         out_message must be freed by the caller by free()
 */
int
testfile (	lwc_LL *hierarchy_list,
		int workload_date, unsigned long long int job_id,
		unsigned long long int host_id, const char *filename,
		char **out_message)
{
	startjob_json_builder_ptr json_obj;
	char *err_msg = NULL;
	row_item_t *host_details;
	int ret;
	schedwi_BIO *b;
	unsigned int last_pos;


#if HAVE_ASSERT_H
	assert (filename != NULL);
#endif

	/* Build the request */
	json_obj = startjob_json_builder_new ();
	startjob_json_builder_add_jobid (json_obj, workload_date, job_id);
	startjob_json_builder_add_filename (	json_obj,
						filename, strlen (filename));


	/* Get the host environment variables */
	if (sql_get_environment_host (	workload_date,
					host_id,
					environment_callback,
					json_obj,
					&last_pos,
					sql_get_environment_error_logger,
					NULL) != 0)
	{
		startjob_json_builder_destroy (json_obj);
		if (out_message != NULL) {
			err_msg =
	_("Server error: cannot retrieve the host environment variables");
			*out_message = xstrdup (err_msg);
		}
		return -1;
	}

	/* Get the job environment variables */
	if (sql_get_environment (	workload_date, hierarchy_list,
					last_pos,
					environment_callback,
					json_obj,
					sql_get_environment_error_logger,
					NULL) != 0)
	{
		startjob_json_builder_destroy (json_obj);
		if (out_message != NULL) {
			err_msg =
	_("Server error: cannot retrieve the job environment variables");
			*out_message = xstrdup (err_msg);
		}
		return -1;
	}

	/* Get the details (IP, port, ...) for the agent host */
	if (sql_host_get_by_id (host_id,
				&host_details,
				sql_host_get_main_row_error_logger,
				NULL) != 0)
	{
		startjob_json_builder_destroy (json_obj);
		if (out_message != NULL) {
			err_msg =
		_("Server error: cannot retrieve the remote agent details");
			*out_message = xstrdup (err_msg);
		}
		return -1;
	}

	/* Establish the connection to the agent */
	b = net_client (host_details[2].value_string,  /* TCP port */
			host_details[1].value_string,  /* Agent host name */
			(char)sql_row_item2ll (&(host_details[3])),
			host_details[4].value_string,  /* Certificate */
			(unsigned int)(host_details[4].len));
	sql_free_row (host_details);
	if (b == NULL) {
		startjob_json_builder_destroy (json_obj);
		if (out_message != NULL) {
			err_msg =
_("Server error: cannot establish the connection with the remote agent");
			*out_message = xstrdup (err_msg);
		}
		return -1;
	}

	/* Send the request */
	if (net_write_request (json_obj, b, "testfile") != 0) {
		net_close (b);
		startjob_json_builder_destroy (json_obj);
		if (out_message != NULL) {
			err_msg =
	_("Server error: cannot send the request to the remote agent");
			*out_message = xstrdup (err_msg);
		}
		return -1;
	}
	startjob_json_builder_destroy (json_obj);

	/* Read the result */
	ret = net_read_result (b, out_message);
	net_close (b);
	return ret;
}

/*-----------------============== End Of File ==============-----------------*/
