/* Schedwi
   Copyright (C) 2007-2014 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* stopjob.c -- Ask to stop a job */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <utils.h>
#include <lwc_log.h>
#include <net_utils.h>
#include <sql_common.h>
#include <sql_hierarchy.h>
#include <sql_hosts.h>
#include <xmem.h>
#include <memsend.h>
#include <stopjob.h>

/*
 * Error callback function for the sql_host_get_by_id() function
 */
static void
sql_host_get_main_row_error_logger (	void *data, const char *msg,
					int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
		_("Database error while retrieving the agent host details"));
	}
}


/*
 * Stop a job
 *
 * Return:
 *   0 --> No error.  The TERM signal has been successfully sent to the job
 *         on the remote client.  This does not mean that the job will stop.
 *         If not NULL, out_message contains a reply message sent by the
 *         client.  It may be NULL if the agent didn't send back a message.
 *         out_message must be freed by the caller by free()
 *   1 --> The agent cannot stop the job.  If not NULL, out_message
 *         contains a reply message sent by the agent.  It may be NULL if the
 *         agent didn't send back a message.  out_message must be freed by
 *         the caller by free()
 *  -1 --> Error (a message is logged using lwc_writeLog()).
 */
int
stopjob (	lwc_LL *hierarchy_list, const char *path,
		int workload_date, unsigned long long int job_id,
		char **out_message)
{
	schedwi_BIO *b;
	unsigned int len, len_workload;
	int ret_read, ret;
	char workload_str[25], job_id_str[25];
	row_item_t host_id;
	char *err_msg = NULL, *req;
	row_item_t *host_details;
	memsend_t *membuff;


#if HAVE_ASSERT_H
        assert (hierarchy_list != NULL && path != NULL);
#endif

	/* Get the host ID associated with the job */
	if  (get_job_parameter (workload_date, hierarchy_list,
				"job_host_s", "host_id",
				&host_id, &err_msg) != 0)
	{
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			free (err_msg);
		}
		else {
			lwc_writeLog (	LOG_CRIT,
_("Workload %d: %s (id %lld): error while retrieving the agent host name"),
					workload_date, path, job_id);
		}
		return -1;
	}
	if (host_id.type == RES_END) {
		lwc_writeLog (	LOG_CRIT,
_("Workload %d: %s (id %lld): agent host name missing in the database"),
				workload_date, path, job_id);
		return -1;
	}

	/* Get the host details */
	ret = sql_host_get_by_id (
			(unsigned long long int) sql_row_item2ll (&host_id),
			&host_details,
			sql_host_get_main_row_error_logger,
			NULL);
	if (ret != 0) {
		return -1;
	}

	/* Convert the job ID to a string */
	len = copy_ulltostr (job_id, job_id_str);

	/* Build the full job ID */
	len_workload = copy_ulltostr (workload_date, workload_str);
	len += len_workload + 1;
	req = (char *) xmalloc (len + 1);
	strcpy (req, workload_str);
	req[len_workload] = SCHEDWI_WORKLOAD_SEPARATOR;
	strcpy (req + len_workload + 1, job_id_str);

	/* Establish the connection to the agent */
	b = net_client (host_details[2].value_string,  /* TCP port */
			host_details[1].value_string,  /* Agent host name */
			(char)sql_row_item2ll (&(host_details[3])),
			host_details[4].value_string, /* Certificate */
			(unsigned int)(host_details[4].len));
	sql_free_row (host_details);
	if (b == NULL) {
		free (req);
		return -1;
	}

	/* Send the request */
	membuff = memsend_new ();
	memsend_append (membuff, "[ \"killjob\", { \"jobs\" : [ \"");
	memsend_append_len (membuff, req, len);
	memsend_append (membuff, "\" ] } ]");
	free (req);
	if (net_write (b, memsend_getstring (membuff),
			memsend_getlength (membuff)) != 0)
	{
		memsend_destroy (membuff);
		net_close (b);
		return -1;
	}
	memsend_destroy (membuff);

	/* Read the result */
	ret_read = net_read_result (b, out_message);
	net_close (b);
	return ret_read;
}

/*-----------------============== End Of File ==============-----------------*/
