/* Schedwi
   Copyright (C) 2007-2014 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * @file startjob_json_builder.c
 * Build the string which contains the network request to start a job.
 */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <memsend.h>
#include <utils.h>
#include <xmem.h>
#include <result_mgnt.h>
#include <startjob_json_builder.h>


/**
 * Create a new startjob_json_builder_ptr structure
 *
 * @return The new structure to be freed by the caller with
           startjob_json_builder_destroy()
 */
startjob_json_builder_ptr
startjob_json_builder_new ()
{
	startjob_json_builder_ptr ptr;


	ptr = (startjob_json_builder_ptr) xmalloc(
					sizeof (startjob_json_builder));
	ptr->json_key_values = ptr->env = ptr->args = NULL;
	return ptr;
}


/**
 * Destroy (free) the provided startjob_json_builder_ptr structure
 */
void
startjob_json_builder_destroy (startjob_json_builder_ptr ptr)
{
	if (ptr != NULL) {
		lwc_delLL (ptr->json_key_values, (void (*)(const void *))free);
		lwc_delLL (ptr->env, (void (*)(const void *))free);
		lwc_delLL (ptr->args, (void (*)(const void *))free);
		free (ptr);
	}
}


/*
 * Add an argument to the startjob_json_builder object
 */
void
startjob_json_builder_add_argument (	startjob_json_builder_ptr ptr,
					const char *val, unsigned int len_val)
{
#if HAVE_ASSERT_H
	assert (ptr != NULL && val != NULL);
#endif

	if (ptr->args == NULL) {
		ptr->args = lwc_newLL ();
	}
	lwc_addEndLL (ptr->args, escape_string (val, len_val, 1));
}


/*
 * Add an environment variable to the startjob_json_builder object
 */
void
startjob_json_builder_add_environment (	startjob_json_builder_ptr ptr,
					const char *key, const char *value)
{
	unsigned int len_key, len_value;
	char *s;


#if HAVE_ASSERT_H
	assert (ptr != NULL && key != NULL && value != NULL);
#endif

	if (ptr->env == NULL) {
		ptr->env = lwc_newLL ();
	}

	/* Compute the size of the strings */
	len_key   = strlen (key);
	len_value = strlen (value);

	s = (char *) xmalloc (len_key + len_value + 2);
	strcpy (s, key);
	s[len_key] = '=';
	strcpy (s + len_key + 1, value);
	lwc_addEndLL (ptr->env, escape_string (s, len_key + len_value + 1, 1));
	free (s);
}


/*
 * Add a json key/value string to the startjob_json_builder object
 */
void
startjob_json_builder_add_key_value_string (	startjob_json_builder_ptr ptr,
						const char *key,
						const char *val,
						unsigned int len_val)
{
	unsigned int len_key;
	char *tmp, *s;


#if HAVE_ASSERT_H
	assert (ptr != NULL && key != NULL && val != NULL);
#endif

	if (ptr->json_key_values == NULL) {
		ptr->json_key_values = lwc_newLL ();
	}

	tmp = escape_string (val, len_val, 1);

	len_key = strlen (key);
	s = (char *) xmalloc (len_key + strlen (tmp) + 5);
	s[0] = '"';
	strcpy (s + 1, key);
	s[len_key + 1] = '"';
	s[len_key + 2] = ':';
	strcpy (s + len_key + 3, tmp);
	free (tmp);
	lwc_addEndLL (ptr->json_key_values, s);
}


/*
 * Add a boolean parameter to the startjob_json_builder object
 */
void
startjob_json_builder_add_key_value_bool (	startjob_json_builder_ptr ptr,
						const char *key, char val)
{
	unsigned int len_key;
	char *s;


#if HAVE_ASSERT_H
	assert (ptr != NULL && key != NULL);
#endif

	if (ptr->json_key_values == NULL) {
		ptr->json_key_values = lwc_newLL ();
	}

	len_key = strlen (key);
	s = (char *) xmalloc (len_key + 10);
	s[0] = '"';
	strcpy (s + 1, key);
	s[len_key + 1] = '"';
	s[len_key + 2] = ':';
	strcpy (s + len_key + 3, (val == 0) ? "false" : "true");
	lwc_addEndLL (ptr->json_key_values, s);
}


/*
 * Add the Job ID parameter to the key/value list
 */
void
startjob_json_builder_add_jobid (	startjob_json_builder_ptr ptr,
					int workload_date,
					unsigned long long int job_id)
{
	unsigned int len_key, len_workload, len_job_id;
	char *s;
	char *json_key = "\"job id\":";


#if HAVE_ASSERT_H
	assert (ptr != NULL);
#endif

	if (ptr->json_key_values == NULL) {
		ptr->json_key_values = lwc_newLL ();
	}

	len_key = strlen (json_key);

	s = (char *) xmalloc (len_key + WORKLOAD_LEN + 30);
	strcpy (s, json_key);
	s[len_key] = '"';
	len_workload = copy_ulltostr (workload_date, s + len_key + 1);
	s[len_key + len_workload + 1] = SCHEDWI_WORKLOAD_SEPARATOR;
	len_job_id = copy_ulltostr (job_id, s + len_key + len_workload + 2);
	s[len_key + len_workload + len_job_id + 2] = '"';
	s[len_key + len_workload + len_job_id + 3] = '\0';
	lwc_addEndLL (ptr->json_key_values, s);
}


/**
 * Send the resquest on the network
 *
 * @param[in] ptr The startjob_json_builder object to send.
 * @param[in] b The network connection.
 * @param[in] module_name Module name.
 * @return 0 on success and -1 on error (an error message is logged by
 *         lwc_writeLog())
 */
int
net_write_request (	startjob_json_builder_ptr ptr,
			schedwi_BIO *b,
			const char *module_name)
{
	int ret;
	char *s, first1, first2;
	memsend_t *membuff;


#if HAVE_ASSERT_H
	assert (ptr != NULL && module_name != NULL);
#endif

	membuff = memsend_new ();
	memsend_append (membuff, "[ \"");
	memsend_append (membuff, module_name);
	memsend_append (membuff, "\", { ");

	/* Send the key/values */
	lwc_rewindLL (ptr->json_key_values);
	first1 = 1;
	while ((s = (char *)lwc_nextLL (ptr->json_key_values)) != NULL) {
		if (first1 != 1) {
			memsend_append (membuff, ", ");
		}
		memsend_append (membuff, s);
		first1 = 0;
	}

	/* Send the arguments */
	if (lwc_getNumNode (ptr->args) != 0) {
		if (first1 != 1) {
			memsend_append (membuff, ", ");
		}
		memsend_append (membuff, "\"args\":[");
		lwc_rewindLL (ptr->args);
		first2 = 1;
		while ((s = (char *)lwc_nextLL (ptr->args)) != NULL) {
			if (first2 != 1) {
				memsend_append (membuff, ",");
			}
			memsend_append (membuff, s);
			first2 = 0;
		}
		memsend_append (membuff, "]");
	}

	/* Send the environment */
	if (lwc_getNumNode (ptr->env) != 0) {
		if (first1 != 1) {
			memsend_append (membuff, ", ");
		}
		memsend_append (membuff, "\"env\":[");
		lwc_rewindLL (ptr->env);
		first2 = 1;
		while ((s = (char *)lwc_nextLL (ptr->env)) != NULL) {
			if (first2 != 1) {
				memsend_append (membuff, ",");
			}
			memsend_append (membuff, s);
			first2 = 0;
		}
		memsend_append (membuff, "]");
	}
	memsend_append (membuff, " } ]");

	ret = net_write (b, memsend_getstring (membuff),
			 memsend_getlength (membuff));
	memsend_destroy (membuff);
	if (ret != 0) {
		return ret;
	}

	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
