/* Schedwi
   Copyright (C) 2007-2014 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* @file startjob.c
 * Start a job.
 */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <utils.h>
#include <lwc_log.h>
#include <net_utils.h>
#include <sql_hosts.h>
#include <sql_status.h>
#include <sql_common.h>
#include <sql_hierarchy.h>
#include <sql_get_environment.h>
#include <startjob_json_builder.h>
#include <xmem.h>
#include <startjob.h>

/*
 * Error callback function for the sql_host_get_by_id() function
 */
static void
sql_host_get_main_row_error_logger (	void *data, const char *msg,
					int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
		_("Database error while retrieving the agent host details"));
	}
}


/*
 * Error callback function for the sql_status_update_host_id() function
 */
static void
sql_status_update_host_id_error_logger (void *data, const char *msg,
					int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_WARNING, msg);
	}
	else {
		lwc_writeLog (LOG_WARNING,
		_("Database error while updating the status (host) of a job"));
	}
}


/*
 * Error callback function for the sql_get_environment*() functions
 */
static void
sql_get_environment_error_logger (	void *data, const char *msg,
					int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
_("Database error while trying to retrieve a job environment variables"));
	}
}


/*
 * Callback for the sql_get_environment() function which retrieves the command
 * environment variables
 *
 * Return: Always 0
 */
static int
environment_callback (	void *obj, unsigned int pos,
			const char *name, const char *value)
{
	startjob_json_builder_ptr json_obj = (startjob_json_builder_ptr) obj;


#if HAVE_ASSERT_H
	assert (json_obj != NULL && name != NULL && value != NULL);
#endif

	startjob_json_builder_add_environment (json_obj, name, value);
	return 0;
}


/*
 * Callback for the SQL function which retrieves the command arguments
 *
 * Return:
 *   0 --> No error
 *   1 --> Stop the sql function.  No arguments for this job
 */
static int
argument_callback (void *obj, row_item_t *row)
{
	startjob_json_builder_ptr json_obj = (startjob_json_builder_ptr) obj;


#if HAVE_ASSERT_H
	assert (json_obj != NULL);
#endif

	if (row == NULL) {
		return 0;
	}

	/*
	 * row[0] --> Position
	 * row[1] --> Argument
	 */

	if (sql_row_item2ll (&(row[0])) == -1) {
		/* No argument for the command */
		return 1;
	}
	startjob_json_builder_add_argument (	json_obj,
						row[1].value_string,
						row[1].len);
	return 0;
}


/*
 * Start a job
 *
 * Return:
 *   0 --> No error.  Job successfully started on the remote client.  If not
 *         NULL, out_message contains a reply message sent by the agent.  It
 *         may be NULL if the agent didn't send back a message.  out_message
 *         must be freed by the caller by free()
 *   1 --> The agent cannot start the job.  If not NULL, out_message
 *         contains a reply message sent by the agent.  It may be NULL if the
 *         agent didn't send back a message.  out_message must be freed by
 *         the caller by free()
 *   2 --> The job does not have an associated command or it's a detach job and
 *         no need to wait for a result.  out_message is not changed
 *  -1 --> Error (a message is logged using lwc_writeLog()).  If not NULL,
 *         out_message contains an error message.  It may be NULL if there
 *         is not enough memory to allocate a buffer for this message.
 *         out_message must be freed by the caller by free()
 */
int
startjob (	lwc_LL *hierarchy_list, const char *path,
		int workload_date, unsigned long long int job_id,
		char **out_message)
{
	startjob_json_builder_ptr json_obj;
	row_item_t value;
	unsigned long long int host_id;
	lwc_LL *host_ids;
	char *s, *err_msg = NULL;
	int ret;
	char detach;
	row_item_t *host_details;
	schedwi_BIO *b;
	unsigned int last_pos;


	/*
	 * Find a host to run the job
	 */

	/*
 	 * First, get the list of host IDs (from the clusters and hosts
	 * tables) associated with the job.
	 */
	if (get_job_hosts (	workload_date, hierarchy_list,
				&host_ids, &err_msg) != 0)
	{
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			if (out_message != NULL) {
				*out_message = err_msg;
			}
			else {
				free (err_msg);
			}
		}
		else {
			lwc_writeLog (	LOG_CRIT,
	_("Workload %d: %s (id %lld): cannot retrieve the agent host details"),
					workload_date, path, job_id);
			if (out_message != NULL) {
				err_msg =
		_("Server error: cannot retrieve the agent host details");
				*out_message = xstrdup (err_msg);
			}
		}
		return -1;
	}
	if (host_ids == NULL) {
		lwc_writeLog (	LOG_CRIT,
_("Workload %d: %s (id %lld): agent host name missing in the database"),
				workload_date, path, job_id);
		if (out_message != NULL) {
			err_msg =
		_("Server error: cannot retrieve the agent host details");
			*out_message = xstrdup (err_msg);
		}
		return -1;
	}

	/* Then, select the host which is the least busy */
	if (sql_status_select_host (host_ids, &host_id, &err_msg) != 0) {
		lwc_delLL (host_ids, (void (*)(const void *))free);
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			if (out_message != NULL) {
				*out_message = err_msg;
			}
			else {
				free (err_msg);
			}
		}
		else {
			lwc_writeLog (	LOG_CRIT,
	_("Workload %d: %s (id %lld): cannot retrieve the agent host details"),
					workload_date, path, job_id);
			if (out_message != NULL) {
				err_msg =
		_("Server error: cannot retrieve the agent host details");
				*out_message = xstrdup (err_msg);
			}
		}
		return -1;
	}
	lwc_delLL (host_ids, (void (*)(const void *))free);


	/*
	 * Create and fill the startjob_json_builder object
	 */

	json_obj = startjob_json_builder_new ();
	startjob_json_builder_add_jobid (json_obj, workload_date, job_id);
	startjob_json_builder_add_path (json_obj, path, strlen(path));

	/* Get the host environment variables */
	if (sql_get_environment_host (	workload_date,
					host_id,
					environment_callback,
					json_obj,
					&last_pos,
					sql_get_environment_error_logger,
					NULL) != 0)
	{
		startjob_json_builder_destroy (json_obj);
		if (out_message != NULL) {
			err_msg =
	_("Server error: cannot retrieve the host environment variables");
			*out_message = xstrdup (err_msg);
		}
		return -1;
	}

	/* Get the job environment variables */
	if (sql_get_environment (	workload_date, hierarchy_list,
					last_pos,
					environment_callback,
					json_obj,
					sql_get_environment_error_logger,
					NULL) != 0)
	{
		startjob_json_builder_destroy (json_obj);
		if (out_message != NULL) {
			err_msg =
	_("Server error: cannot retrieve the job environment variables");
			*out_message = xstrdup (err_msg);
		}
		return -1;
	}

	/* Get the command */
	if (get_job_parameter (	workload_date, hierarchy_list,
				"job_command_s", "command",
				&value, &err_msg) != 0)
	{
		startjob_json_builder_destroy (json_obj);
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			if (out_message != NULL) {
				*out_message = err_msg;
			}
			else {
				free (err_msg);
			}
		}
		else {
			lwc_writeLog (	LOG_CRIT,
		_("Workload %d: %s (id %lld): cannot retrieve the command"),
					workload_date, path, job_id);
			if (out_message != NULL) {
				err_msg =
				_("Server error: cannot retrieve job command");
				*out_message = xstrdup (err_msg);
			}
		}
		return -1;
	}

	if (	   value.type != RES_END
		&& value.value_string != NULL
		&& value.value_string[0] != '\0'
		&& (	   value.value_string[0] != '-'
			|| value.value_string[1] != '\0'))
	{
		startjob_json_builder_add_command (	json_obj,
							value.value_string,
							value.len);
		free (value.value_string);
	}
	else {
		/* No command, the job is then considered completed */
		if (value.type != RES_END && value.value_string != NULL) {
			free (value.value_string);
		}
		startjob_json_builder_destroy (json_obj);
		return 2;
	}

	/* Get the username */
	if (get_job_parameter (	workload_date, hierarchy_list,
				"job_username_s", "username",
				&value, &err_msg) != 0)
	{
		startjob_json_builder_destroy (json_obj);
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			if (out_message != NULL) {
				*out_message = err_msg;
			}
			else {
				free (err_msg);
			}
		}
		else {
			lwc_writeLog (	LOG_CRIT,
		_("Workload %d: %s (id %lld): cannot retrieve the username"),
					workload_date, path, job_id);
			if (out_message != NULL) {
				err_msg =
			_("Server error: cannot retrieve the username");
				*out_message = xstrdup (err_msg);
			}
		}
		return -1;
	}

	if (value.type != RES_END) {
		startjob_json_builder_add_username (	json_obj,
							value.value_string,
							value.len);
		if (value.value_string != NULL) {
			free (value.value_string);
		}
	}
	else {
		startjob_json_builder_destroy (json_obj);
		lwc_writeLog (	LOG_CRIT,
		_("Workload %d: %s (id %lld): cannot retrieve the username"),
				workload_date, path, job_id);
		if (out_message != NULL) {
			err_msg =
			_("Server error: cannot retrieve the username");
			*out_message = xstrdup (err_msg);
		}
		return -1;
	}

	/* Get the output file name */
	if (get_job_parameter (	workload_date, hierarchy_list,
				"job_file_out_s", "file_out",
				&value, &err_msg) != 0)
	{
		startjob_json_builder_destroy (json_obj);
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			if (out_message != NULL) {
				*out_message = err_msg;
			}
			else {
				free (err_msg);
			}
		}
		else {
			lwc_writeLog (	LOG_CRIT,
	_("Workload %d: %s (id %lld): cannot retrieve the output file"),
					workload_date, path, job_id);
			if (out_message != NULL) {
				err_msg =
			_("Server error: cannot retrieve the output file");
				*out_message = xstrdup (err_msg);
			}
		}
		return -1;
	}

	if (value.type != RES_END) {
		startjob_json_builder_add_fileout (	json_obj,
							value.value_string,
							value.len);
		if (value.value_string != NULL) {
			free (value.value_string);
		}
	}

	/* Get the error file name */
	if (get_job_parameter (	workload_date, hierarchy_list,
				"job_file_err_s", "file_err",
				&value, &err_msg) != 0)
	{
		startjob_json_builder_destroy (json_obj);
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			if (out_message != NULL) {
				*out_message = err_msg;
			}
			else {
				free (err_msg);
			}
		}
		else {
			lwc_writeLog (	LOG_CRIT,
		_("Workload %d: %s (id %lld): cannot retrieve the error file"),
					workload_date, path, job_id);
			if (out_message != NULL) {
				err_msg =
			_("Server error: cannot retrieve the error file");
				*out_message = xstrdup (err_msg);
			}
		}
		return -1;
	}

	if (value.type != RES_END) {
		startjob_json_builder_add_filerr (	json_obj,
							value.value_string,
							value.len);
		if (value.value_string != NULL) {
			free (value.value_string);
		}
	}

	/* Get the Linux Control Group name */
	if (get_job_parameter (	workload_date, hierarchy_list,
				"job_control_group_s", "control_group",
				&value, &err_msg) != 0)
	{
		startjob_json_builder_destroy (json_obj);
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			if (out_message != NULL) {
				*out_message = err_msg;
			}
			else {
				free (err_msg);
			}
		}
		else {
			lwc_writeLog (	LOG_CRIT,
	_("Workload %d: %s (id %lld): cannot retrieve the control group"),
					workload_date, path, job_id);
			if (out_message != NULL) {
				err_msg =
			_("Server error: cannot retrieve the control group");
				*out_message = xstrdup (err_msg);
			}
		}
		return -1;
	}

	if (value.type != RES_END) {
		startjob_json_builder_add_cgroup (	json_obj,
							value.value_string,
							value.len);
		if (value.value_string != NULL) {
			free (value.value_string);
		}
	}

	/* Get the load environment flag */
	if (get_job_parameter (	workload_date, hierarchy_list,
				"job_loadenv_s", "loadenv",
				&value, &err_msg) != 0)
	{
		startjob_json_builder_destroy (json_obj);
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			if (out_message != NULL) {
				*out_message = err_msg;
			}
			else {
				free (err_msg);
			}
		}
		else {
			lwc_writeLog (	LOG_CRIT,
_("Workload %d: %s (id %lld): cannot retrieve the `load user env' flag"),
					workload_date, path, job_id);
			if (out_message != NULL) {
				err_msg =
	_("Server error: cannot retrieve the `load user env' flag");
				*out_message = xstrdup (err_msg);
			}
		}
		return -1;
	}

	if (value.type != RES_END) {
		s = (sql_row_item2ll(&value) == 0) ? "0" : "1";
		startjob_json_builder_add_loadenv (json_obj, s, 1);
	}

	/* Get the detach mode flag */
	if (get_job_parameter (	workload_date, hierarchy_list,
				"job_detach_s", "detach",
				&value, &err_msg) != 0)
	{
		startjob_json_builder_destroy (json_obj);
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			if (out_message != NULL) {
				*out_message = err_msg;
			}
			else {
				free (err_msg);
			}
		}
		else {
			lwc_writeLog (	LOG_CRIT,
	_("Workload %d: %s (id %lld): cannot retrieve the `detach mode' flag"),
					workload_date, path, job_id);
			if (out_message != NULL) {
				err_msg =
		_("Server error: cannot retrieve the `detach mode' flag");
				*out_message = xstrdup (err_msg);
			}
		}
		return -1;
	}

	detach = 0;
	if (value.type != RES_END) {
		if (sql_row_item2ll(&value) == 0) {
			s = "0";
		}
		else {
			s = "1";
			detach = 1;
		}
		startjob_json_builder_add_detach (json_obj, s, 1);
	}

	/* Get the command arguments */
	ret = get_job_parameters_multi (workload_date, hierarchy_list,
				"job_arguments_s", "position,argument",
				"position",
				argument_callback, json_obj, &err_msg);
	if (ret != 0 && ret != 2) {
		startjob_json_builder_destroy (json_obj);
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			if (out_message != NULL) {
				*out_message = err_msg;
			}
			else {
				free (err_msg);
			}
		}
		else {
			lwc_writeLog (	LOG_CRIT,
	_("Workload %d: %s (id %lld): cannot retrieve the command arguments"),
					workload_date, path, job_id);
			if (out_message != NULL) {
				err_msg =
		_("Server error: cannot retrieve the command arguments");
				*out_message = xstrdup (err_msg);
			}
		}
		return -1;
	}

	/* Get the details (IP, port, ...) for the agent host */
	if (sql_host_get_by_id (host_id,
				&host_details,
				sql_host_get_main_row_error_logger,
				NULL) != 0)
	{
		startjob_json_builder_destroy (json_obj);
		if (out_message != NULL) {
			err_msg =
		_("Server error: cannot retrieve the remote agent details");
			*out_message = xstrdup (err_msg);
		}
		return -1;
	}

	/*
	 * Update the database to specify the host on which the job is
	 * going to run
	 */
	sql_status_update_host_id (	workload_date, job_id, host_id,
					sql_status_update_host_id_error_logger,
					NULL);

	/* Establish the connection to the agent */
	b = net_client (host_details[2].value_string,  /* TCP port */
			host_details[1].value_string,  /* Agent host name */
			(char)sql_row_item2ll (&(host_details[3])),
			host_details[4].value_string,  /* Certificate */
			(unsigned int)(host_details[4].len));
	sql_free_row (host_details);
	if (b == NULL) {
		startjob_json_builder_destroy (json_obj);
		if (out_message != NULL) {
			err_msg =
_("Server error: cannot establish the connection with the remote agent");
			*out_message = xstrdup (err_msg);
		}
		return -1;
	}

	/* Send the request */
	if (net_write_request (json_obj, b, "runjob") != 0) {
		net_close (b);
		startjob_json_builder_destroy (json_obj);
		if (out_message != NULL) {
			err_msg =
	_("Server error: cannot send the request to the remote agent");
			*out_message = xstrdup (err_msg);
		}
		return -1;
	}
	startjob_json_builder_destroy (json_obj);

	/* Read the result */
	ret = net_read_result (b, out_message);
	net_close (b);
	if (ret == 0 && detach == 1) {
		return 2;
	}
	return ret;
}

/*-----------------============== End Of File ==============-----------------*/
