/* Schedwi
   Copyright (C) 2007-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* sql_status.c -- job status management functions */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <ping.h>
#include <xmem.h>
#include <lwc_log.h>
#include <job_status_state.h>
#include <schedwi_time.h>
#include <sql_common.h>
#include <sql_hierarchy.h>
#include <sql_status.h>

#define SQLSTATUS_WORKLOAD_LIST "SELECT DISTINCT workload_date FROM job_status WHERE workload_date=%d OR workload_date=%d OR time_status_set>%ld ORDER BY workload_date"
#define SQLSTATUS_GET "SELECT status,time_status_set,retry_num,wait_reason,error_msg,duration FROM job_status WHERE job_id=%ld AND workload_date=%d"
#define SQLSTATUS_UPDATE "UPDATE job_status SET status=%d,time_status_set=%ld,retry_num=%d,wait_reason=%d,error_msg=%s,duration=%d WHERE job_id=%ld AND workload_date=%d"
#define SQLSTATUS_SET "INSERT INTO job_status (workload_date,job_id,status,time_status_set,retry_num,wait_reason,error_msg,duration) VALUES (%d,%ld,%d,%ld,%d,%d,%s,%d)"
#define SQLSTATUS_UPDATE_WAIT_REASON "UPDATE job_status SET wait_reason=%d WHERE job_id=%ld AND workload_date=%d"
#define SQLSTATUS_UPDATE_HOST "UPDATE job_status SET host_id=%ld WHERE job_id=%ld AND workload_date=%d"
#define SQLSTATUS_DEL "DELETE FROM job_status WHERE workload_date=%d"
#define SQLSTATUS_DEL_TABLES "DELETE FROM %s_s WHERE workload_date=%d"
#define SQLSTATUS_CLEAN_TABLES "DELETE FROM %s WHERE workload_date=%d"
#define SQLSTATUS_COPY "INSERT INTO %s_s SELECT *,%d FROM %s"
#define SQLSTATUS_GET_OLD "SELECT workload_date FROM %s_s WHERE workload_date<%d"
#define SQLSTATUS_HOST_COUNT_RUNNING "SELECT host_id,COUNT(*) FROM job_status WHERE status=2 AND host_id IN (%s) GROUP BY host_id ORDER BY COUNT(*)"
#define SQLSTATUS_IS_JOB_PRESENT "SELECT 1 FROM job_status WHERE job_id=%ld AND workload_date=%d"
#define SQLSTATUS_NUM_WAITING_LINKS "SELECT COUNT(*) FROM job_status WHERE workload_date=%d AND status=%d AND (wait_reason&%d)<>0"


/*
 * List of tables to copy.  These tables will be copied in tables with the
 * same name but suffixed by `_s'
 */
static char * status_tables[] = {	"calendars",
					"job_main",
					"job_start_limit",
					"job_max_duration",
					"job_retries",
					"job_retries_interval",
					"job_username",
					"job_file_out",
					"job_file_err",
					"job_control_group",
					"job_loadenv",
					"job_detach",
					"job_manual_command",
					"job_success_return_code",
					"job_command",
					"job_arguments",
					"environments",
					"environment_var",
					"job_environment",
					"host_environment",
					"job_host",
					"job_cluster",
					"constraint_file",
					"jobset_background",
					"job_icon_default",
					"job_icon_completed",
					"job_icon_failed",
					"job_icon_running",
					"job_icon_waiting",
					"jobset_icon_default",
					"jobset_icon_completed",
					"jobset_icon_failed",
					"jobset_icon_running",
					"jobset_icon_waiting",
					"link",
					NULL
				};
/* Tables to clean at the same time as the job_status table */
static char * to_clean_tables[] = {
					"acknowledge_manual",
					"commands",
					NULL
				};

/**
 * Get all the available workloads from the job_status database table that
 * have been updated after the given time (as in time(2)).  The workload
 * matching the given time and the one before that are always returned.
 *
 * @param[out] rows The workload list to be freed by the caller by
 *                  lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
 *                  This list may be empty if no workload has been updated
 *                  after the given time (in time_status_set)
 * @param[in] time_status_set The time (as in time(2)) after which to get the
 *                            workloads.
 * @param[in] error_func Error function to call in case of error. This function
 *                       is called with user_data_error_func as its first
 *                       parameter and the error message as the second
 *                       parameter.  error_func can be NULL.
 * @param[in] user_data_error_func User provided parameter to pass to the
 *                                 error_func function in case of error.
 * @return
 *     0 --> No error.  rows contains the retrieved workload date and must be
 *           freed by the caller by
 *              lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_status_updated_workload_list (
			lwc_LL **rows,
			time_t time_status_set,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;
	time_t yesterday;


#if HAVE_ASSERT_H
	assert (rows != NULL);
#endif

	yesterday = time_status_set - 24 * 60 * 60;
	ret = sql_select (NULL, NULL, &err_msg, NULL, rows,
		SQLSTATUS_WORKLOAD_LIST,
		SQL_INT, (long int)schedwi_time_to_date_int (time_status_set),
		SQL_INT, (long int)schedwi_time_to_date_int (
					(yesterday >= 0) ? yesterday: 0),
		SQL_INT, (long int)time_status_set,
		SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Get the status of the provided job/jobset
 *
 * Return:
 *     0 --> No error.  status contains the retrieved status and
 *           time_status_set the associated start time.
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_status_get (int workload_date,
		unsigned long long int job_id,
		int *status,
		time_t *time_status_set,
		int *retry_num,
		int *wait_reason,
		char **message,
		long int *duration,
		void (*error_func)(void *, const char *, int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	lwc_LL *rows;
	int ret;
	row_item_t *row;


#if HAVE_ASSERT_H
	assert (status != NULL);
#endif

	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows,
				SQLSTATUS_GET,
				SQL_INT, (long int)job_id,
				SQL_INT, (long int)workload_date,
				SQL_END);

	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	row = (row_item_t *) lwc_delStartLL (rows);

	/* Message */
	if (message != NULL) {
		if (	   row == NULL
			|| row[4].value_string == NULL
			|| row[4].value_string[0] == '\0')
		{
			*message = NULL;
		}
		else {
			*message = xstrdup (row[4].value_string);
		}
	}

	/* Status */
	*status = (row == NULL) ? 0 : (int)sql_row_item2ll (&(row[0]));

	/* Start time */
	if (time_status_set != NULL) {
		*time_status_set = (row == NULL) ? 0
					: (time_t)sql_row_item2ll (&(row[1]));
	}

	/* Current retry number */
	if (retry_num != NULL) {
		*retry_num = (row == NULL) ? 0
				: (int)sql_row_item2ll (&(row[2]));
	}

	/* Why the job/jobset is waiting? */
	if (wait_reason != NULL) {
		*wait_reason = (row == NULL) ? 0
				: (int)sql_row_item2ll (&(row[3]));
	}

	/* Duration */
	if (duration != NULL) {
		*duration = (row == NULL) ? 0
				: (long int)sql_row_item2ll (&(row[5]));
	}

	sql_free_row (row);
	lwc_delLL (rows, (void (*)(const void *))sql_free_row);
	return 0;
}


/**
 * Change the reason why the given job/jobset is in the waiting state.
 *
 * @param[in] workload_date Workload date of the job/jobset (YYYYMMDD).
 * @param[in] job_id The database ID of the job or jobset.
 * @param[in] wait_reason The reason why the job/jobset is waiting.  This is
 * 			  a bit mask of the flags defined in sql_status.h:
 * 			  WAIT_REASON_MASK_PARENT, WAIT_REASON_MASK_TIME, ...
 * @param[in] error_func Error function to call in case of error. This function
 *                       is called with user_data_error_func as its first
 *                       parameter and the error message as the second
 *                       parameter.  error_func can be NULL.
 * @param[in] user_data_error_func User provided parameter to pass to the
 *                                 error_func function in vase of error.
 * @return
 *     0 --> Success
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_status_update_wait_reason (	int workload_date,
				unsigned long long int job_id,
				int wait_reason,
				void (*error_func)(void *, const char *, int),
				void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;


	ret = sql_non_select (  NULL, NULL, &err_msg, NULL, NULL, NULL,
				SQLSTATUS_UPDATE_WAIT_REASON,
				SQL_INT, (long int)wait_reason,
				SQL_INT, (long int)job_id,
				SQL_INT, (long int)workload_date,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Set the status of the provided job/jobset
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_status_set (int workload_date,
		unsigned long long int job_id,
		int status,
		time_t time_status_set,
		int retry_num,
		int wait_reason,
		const char *error_message,
		long int duration,
		void (*error_func)(void *, const char *, int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;
	unsigned long long int nb = 0;


	if (time_status_set <= 0) {
		time_status_set = time (NULL);
	}
	ret = sql_non_select (  NULL, NULL, &err_msg, NULL, &nb, NULL,
				SQLSTATUS_UPDATE,
				SQL_INT, (long int)status,
				SQL_INT, (long int)time_status_set,
				SQL_INT, (long int)retry_num,
				SQL_INT, (long int)wait_reason,
				SQL_STRING, (error_message != NULL)
							? error_message: "",
				SQL_INT, duration,
				SQL_INT, (long int)job_id,
				SQL_INT, (long int)workload_date,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	/* Update successful.  There was already a row for this job */
	if (nb != 0) {
		return 0;
	}

	/* There is not yet a status row for this job.  Add one */
	err_msg = NULL;
	ret = sql_non_select (  NULL, NULL, &err_msg, NULL, NULL, NULL,
				SQLSTATUS_SET,
				SQL_INT, (long int)workload_date,
				SQL_INT, (long int)job_id,
				SQL_INT, (long int)status,
				SQL_INT, (long int)time_status_set,
				SQL_INT, (long int)retry_num,
				SQL_INT, (long int)wait_reason,
				SQL_STRING, (error_message != NULL)
							? error_message: "",
				SQL_INT, duration,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/**
 * Delete a workload
 *
 * @param[in] workload_date Workload date of the job/jobsets (YYYYMMDD) to
 *                          delete.
 * @param[in] error_func Error function to call in case of error. This function
 *                       is called with user_data_error_func as its first
 *                       parameter and the error message as the second
 *                       parameter.  error_func can be NULL.
 * @param[in] user_data_error_func User provided parameter to pass to the
 *                                 error_func function in vase of error.
 * @return
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_status_delete (	int workload_date,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int err, i;
	int ret;
	char *buff;
	unsigned int buff_len;


	buff = NULL;
	buff_len = 0;

	/* First, delete the entry in the job_status table */
	ret = sql_non_select (  &buff, &buff_len, &err_msg, NULL, NULL, NULL,
				SQLSTATUS_DEL,
				SQL_INT, (long int)workload_date,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		if (buff != NULL) {
			free (buff);
		}
		return ret;
	}

	/* Delete from the tables to clean */
	err = 0;
	for (i = 0; to_clean_tables[i] != NULL; i++);
	do {
		i--;
		ret = sql_non_select (  &buff, &buff_len, &err_msg, NULL, NULL,
					NULL,
					SQLSTATUS_CLEAN_TABLES,
					SQL_STRING_NON_ESCAPE,
							to_clean_tables[i],
					SQL_INT, (long int)workload_date,
					SQL_END);
		if (ret != 0) {
			err = ret;
			if (error_func != NULL) {
				error_func (	user_data_error_func,
						err_msg, ret);
			}
			if (err_msg != NULL) {
				free (err_msg);
			}
		}
	} while (i > 0);

	/* And then delete the entries for all the other tables */
	for (i = 0; status_tables[i] != NULL; i++);
	do {
		i--;
		ret = sql_non_select (  &buff, &buff_len, &err_msg, NULL, NULL,
					NULL,
					SQLSTATUS_DEL_TABLES,
					SQL_STRING_NON_ESCAPE,
							status_tables[i],
					SQL_INT, (long int)workload_date,
					SQL_END);
		if (ret != 0) {
			err = ret;
			if (error_func != NULL) {
				error_func (	user_data_error_func,
						err_msg, ret);
			}
			if (err_msg != NULL) {
				free (err_msg);
			}
		}
	} while (i > 0);

	if (buff != NULL) {
		free (buff);
	}

	if (err != 0) {
		return err;
	}
	return 0;
}


/*
 * Check for old workloads
 * For workloads older than the provided workload, the callback() function is
 * called with the following parameters:
 *    - The provided user_data
 *    - Table name (without the `_s')
 *    - The workload date
 *
 * Return:
 *     2 --> Old workloads found
 *     0 --> No error.
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_status_get_old (	int workload_date,
			int (*callback)(void *, const char *, int),
			void *user_data,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	lwc_LL *rows;
	row_item_t *row;
	unsigned int i;
	int ret, global_ret;
	char *buff;
	unsigned int buff_len;

#if HAVE_ASSERT_H
	assert (callback != NULL);
#endif

	buff = NULL;
	buff_len = 0;
	global_ret = 0;
	for (i = 0; status_tables[i] != NULL; i++) {
		ret = sql_select (&buff, &buff_len, &err_msg, NULL, &rows,
				SQLSTATUS_GET_OLD,
				SQL_STRING_NON_ESCAPE, status_tables[i],
				SQL_INT, (long int)workload_date,
				SQL_END);
		if (ret != 0) {
			if (error_func != NULL) {
				error_func (	user_data_error_func, err_msg,
						ret);
			}
			if (err_msg != NULL) {
				free (err_msg);
			}
			if (buff != NULL) {
				free (buff);
			}
			return ret;
		}

		while ((row = (row_item_t *) lwc_delStartLL (rows)) != NULL) {
			/*
			 * row[0] --> workload_date
			 */
			callback (	user_data, status_tables[i],
					(int)sql_row_item2ll (&(row[0])));
			sql_free_row (row);
			global_ret = 2;
		}
		lwc_delLL (rows, (void (*)(const void *))sql_free_row);
	}

	if (buff != NULL) {
		free (buff);
	}

	return global_ret;
}


/*
 * Create a new workload by copying tables
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_status_copy_to_workload (int workload_date,
		void (*error_func)(void *, const char *, int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int i;
	int ret;
	char *buff;
	unsigned int buff_len;


	buff = NULL;
	buff_len = 0;

	/*
	 * First, delete the workload from all the tables if it already
	 * exists. It should not, but to be on the safe side...
	 */
	sql_status_delete (workload_date, NULL, NULL);

	/* Copy all the tables */
	for (i = 0; status_tables[i] != NULL; i++) {
		ret = sql_non_select (  &buff, &buff_len, &err_msg, NULL, NULL,
					NULL,
					SQLSTATUS_COPY,
					SQL_STRING_NON_ESCAPE,
							status_tables[i],
					SQL_INT, (long int)workload_date,
					SQL_STRING_NON_ESCAPE,
							status_tables[i],
					SQL_END);
		if (ret != 0) {
			if (error_func != NULL) {
				error_func (	user_data_error_func,
						err_msg, ret);
			}
			if (err_msg != NULL) {
				free (err_msg);
			}
			if (buff != NULL) {
				free (buff);
			}

			/* Try to remove the successfully copied tables */
			sql_status_delete (workload_date, NULL, NULL);
			return ret;
		}
	}
	if (buff != NULL) {
		free (buff);
	}

	return 0;
}


/*
 * From the given host ID list, return the host ID on which a few number of
 * jobs are running.
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if err_msg is not NULL, it contains the
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message)
 *    -2 --> SQL error (if err_msg is not NULL it contains an error
 *           message which must be freed by the caller.  *err_msg may be NULL
 *           if there is not enough memory to store the error message)
 */
int
sql_status_select_host (lwc_LL *host_ids, unsigned long long int *host_id_out,
			char **err_msg)
{
	char *s;
	lwc_LL *rows;
	row_item_t *row;
	int num_ids, num_rows, ret, i, j;
	unsigned long long int *a, *id;
	char found;


	num_ids = lwc_getNumNode (host_ids);

#if HAVE_ASSERT_H
	assert (host_ids != NULL && num_ids > 0 && host_id_out != NULL);
#endif

	if (num_ids == 1) {
		/* Only one host ID in the list */
		lwc_rewindLL (host_ids);
		*host_id_out = *((unsigned long long int *)lwc_nextLL (
								host_ids));
		/* We don't try to ping this host, just return its ID */
		return 0;
	}

	/* Convert the list in a string suitable for the SQL request */
	s = hierarchy_list_to_str (host_ids);
	ret = sql_select (	NULL, NULL, err_msg, NULL, &rows,
				SQLSTATUS_HOST_COUNT_RUNNING,
				SQL_STRING_NON_ESCAPE, s,
				SQL_END);
	free (s);
	if (ret != 0) {
		return ret;
	}

	/*
	 * Store all the IDs in an array sorted from the least to the most
	 * used host (so later on, these hosts will be ping in the order to
	 * find the least used working host)
	 */
	a = (unsigned long long int *) xmalloc ((num_ids + 1)
					* sizeof (unsigned long long int));
	i = 0;
	/*
	 * First, store all the IDs not in the returned SQL query (hosts
	 * that don't have any running jobs)
	 */
	num_rows = lwc_getNumNode (rows);
	if (num_ids > num_rows) {
		lwc_rewindLL (host_ids);
		while ((id = (unsigned long long int *)lwc_nextLL (host_ids))
						!= NULL)
		{
			found = 0;
			lwc_rewindLL (rows);
			while ((row = (row_item_t *)lwc_nextLL (rows))
						!= NULL)
			{
				if (*id ==
			(unsigned long long int) sql_row_item2ll (&(row[0])))
				{
					found = 1;
					break;
				}
			}
			if (found == 0) {
				a[i++] = *id;
			}
		}
	}
	/* Then store the IDs of hosts with running jobs */
	while ((row = (row_item_t *) lwc_delStartLL (rows)) != NULL) {
		a[i++] = (unsigned long long int) sql_row_item2ll (&(row[0]));
		sql_free_row (row);
	}
	lwc_delLL (rows, (void (*)(const void *)) sql_free_row);

	/* Return the ID of the first host in the array that can be pinged */
	for (j = 0; j < i; j++) {
		if (ping_by_id (a[j]) == 0) {
			*host_id_out = a[j];
			free (a);
			return 0;
		}
	}

	/* All the pings failed. Return the first host anyway */
	*host_id_out = a[0];
	free (a);
	lwc_writeLog (LOG_WARNING, _("None of the hosts replied to ping"));
	return 0;
}


/**
 * Change the host ID associated with the given job/jobset.
 *
 * @param[in] workload_date Workload date of the job/jobset (YYYYMMDD).
 * @param[in] job_id The database ID of the job or jobset.
 * @param[in] host_id The new host ID.
 * @param[in] error_func Error function to call in case of error. This function
 *                       is called with user_data_error_func as its first
 *                       parameter and the error message as the second
 *                       parameter.  error_func can be NULL.
 * @param[in] user_data_error_func User provided parameter to pass to the
 *                                 error_func function in vase of error.
 * @return
 *     0 --> Success
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_status_update_host_id (	int workload_date,
				unsigned long long int job_id,
				unsigned long long int host_id,
				void (*error_func)(void *, const char *, int),
				void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;


	ret = sql_non_select (  NULL, NULL, &err_msg, NULL, NULL, NULL,
				SQLSTATUS_UPDATE_HOST,
				SQL_INT, (long int)host_id,
				SQL_INT, (long int)job_id,
				SQL_INT, (long int)workload_date,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/**
 * Check if the given workload is already in the job_status table.
 *
 * @param[in] workload_date Workload date of the job/jobset (YYYYMMDD).
 * @param[in] error_func Error function to call in case of error. This function
 *                       is called with user_data_error_func as its first
 *                       parameter and the error message as the second
 *                       parameter.  error_func can be NULL.
 * @param[in] user_data_error_func User provided parameter to pass to the
 *                                 error_func function in vase of error.
 * @return
 *     0 --> The given workload is not in the job_status table.
 *     1 --> The workload is in the job_status table.
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_status_check_workload (	int workload_date,
				void (*error_func)(void *, const char *, int),
				void *user_data_error_func)
{
	char *err_msg = NULL;
	lwc_LL *rows;
	int ret;


	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows,
				SQLSTATUS_IS_JOB_PRESENT,
				SQL_INT, (long int)ROOT_JOBSET,
				SQL_INT, (long int)workload_date,
				SQL_END);

	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	if (lwc_getNumNode (rows) == 0) {
		lwc_delLL (rows, (void (*)(const void *))sql_free_row);
		return 0;
	}
	lwc_delLL (rows, (void (*)(const void *))sql_free_row);
	return 1;
}


/**
 * Return the number of jobs/jobsets waiting on links not resolved.
 *
 * @param[in] workload_date The workload date.
 * @param[in] error_func Error function to call in case of error. This function
 *                       is called with user_data_error_func as its first
 *                       parameter and the error message as the second
 *                       parameter.  error_func can be NULL.
 * @param[in] user_data_error_func User provided parameter to pass to the
 *                                 error_func function in case of error.
 * @return
 *     The number of jobs/jobsets waiting on links.
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_status_num_waiting_links (	int workload_date,
				void (*error_func)(void *, const char *, int),
				void *user_data_error_func)
{
	char *err_msg = NULL;
	lwc_LL *rows;
	int ret;
	row_item_t *row;


	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows,
				SQLSTATUS_NUM_WAITING_LINKS,
				SQL_INT, (long int)workload_date,
				SQL_INT, JOB_STATUS_STATE_WAITING,
				SQL_INT, WAIT_REASON_MASK_LINKS,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	row = (row_item_t *) lwc_delStartLL (rows);
	lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
	if (row == NULL) {
		return 0;
	}
	ret = (int) sql_row_item2ll (&(row[0]));
	sql_free_row (row);
	return ret;
}

/*-----------------============== End Of File ==============-----------------*/
