/* Schedwi
   Copyright (C) 2007-2014 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* sql_stat.c -- Update and retrieve statistics (table job_stat) */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <sql_stat.h>


#define SQL_STAT_SUCCESS_UPDATE "UPDATE job_stat SET num_success=num_success+1,total_duration=total_duration+%ld,last_duration=%ld WHERE job_id=%ld"
#define SQL_STAT_FAIL_UPDATE "UPDATE job_stat SET num_failed=num_failed+1,last_duration=%ld WHERE job_id=%ld"
#define SQL_STAT_SUCCESS_INSERT "INSERT INTO job_stat (job_id,num_success,num_failed,total_duration,last_duration) VALUES (%ld,1,0,%ld,%ld)"
#define SQL_STAT_FAIL_INSERT "INSERT INTO job_stat (job_id,num_success,num_failed,total_duration,last_duration) VALUES (%ld,0,1,0,%ld)"


/*
 * Update the statistics for the provided successful job
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_stat_success (	unsigned long long int job_id,
			long int duration,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;
	unsigned long long int nb = 0;


	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, &nb, NULL,
				SQL_STAT_SUCCESS_UPDATE,
				SQL_INT, duration,
				SQL_INT, duration,
				SQL_INT, (long int)job_id,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	/* Update successful.  There was already a row for this job */
	if (nb != 0) {
		return 0;
	}

	/* There is not yet a statistic row for this job.  Add one */
	err_msg = NULL;
	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL, NULL,
				SQL_STAT_SUCCESS_INSERT,
				SQL_INT, (long int)job_id,
				SQL_INT, duration,
				SQL_INT, duration,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	return 0;
}


/*
 * Update the statistics for the provided failed job
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_stat_fail (	unsigned long long int job_id,
		long int duration,
		void (*error_func)(void *, const char *, int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;
	unsigned long long int nb = 0;


	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, &nb, NULL,
				SQL_STAT_FAIL_UPDATE,
				SQL_INT, duration,
				SQL_INT, (long int)job_id,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	/* Update successful.  There was already a row for this job */
	if (nb != 0) {
		return 0;
	}

	/* There is not yet a statistic row for this job.  Add one */
	err_msg = NULL;
	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL, NULL,
				SQL_STAT_FAIL_INSERT,
				SQL_INT, (long int)job_id,
				SQL_INT, duration,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
