/* Schedwi
   Copyright (C) 2007-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* sql_link.c -- functions to retrieve link details from the database */

#include <schedwi.h>

#if HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include <lwc_linkedlist.h>
#include <sql_common.h>
#include <sql_link.h>

#define SQL_LINK_NOT_RESOLVED "SELECT COUNT(*) FROM link_s,job_status WHERE link_s.job_id_source=%ld AND link_s.workload_date=%d AND link_s.workload_date=job_status.workload_date AND link_s.job_id_destination=job_status.job_id AND ((link_s.required_status=5 AND job_status.status<>3 AND job_status.status<>4) OR (link_s.required_status<>5 AND link_s.required_status<>job_status.status))"


/*
 * Check if the links are resolved for the given job/jobset.
 *
 * @param[in] workload_date The workload date.
 * @param[in] job_id The ID of the job or jobset to check.
 * @param[in] error_func Error function to call in case of error. This function
 *                       is called with user_data_error_func as its first
 *                       parameter and the error message as the second
 *                       parameter.  error_func can be NULL.
 * @param[in] user_data_error_func User provided parameter to pass to the
 *                                 error_func function in case of error.
 * @return
 *     The number of waiting links (0 means that all the links are resolved) or
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_link_are_resolved (	int workload_date,
			unsigned long long int job_id,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	lwc_LL *rows;
	int ret;
	row_item_t *row;


	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows,
				SQL_LINK_NOT_RESOLVED,
				SQL_INT, (long int)job_id,
				SQL_INT, (long int)workload_date,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	row = (row_item_t *) lwc_delStartLL (rows);
	lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
	if (row == NULL) {
		return 1;
	}
	ret = (int) sql_row_item2ll (&(row[0]));
	sql_free_row (row);
	return ret;
}

/*-----------------============== End Of File ==============-----------------*/
