/* Schedwi
   Copyright (C) 2007-2014 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* sql_hosts.c -- hosts table management functions */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <utils.h>
#include <xmem.h>
#include <sql_hosts.h>

#define SQL_HOST_DETAIL_FROM_NAMES "SELECT id,hostname,portnum,sslenable,sslcert,description FROM hosts WHERE hostname IN (%s) AND portnum=%s"
#define SQL_HOST_DETAIL_FROM_ID "SELECT id,hostname,portnum,sslenable,sslcert,description FROM hosts WHERE id=%ld"
#define SQL_HOST_NEW "INSERT INTO hosts (hostname,portnum,sslenable,sslcert,description) VALUES (%s,%s,%d,%s,%s)"
#define SQL_HOST_CERT "SELECT sslenable,sslcert FROM hosts WHERE hostname IN (%s) ORDER BY sslenable DESC"
#define SQL_HOST_REPLACE_UPDATE_NODESCR "UPDATE hosts SET sslenable=%d,sslcert=%s WHERE hostname=%s AND portnum=%s"
#define SQL_HOST_REPLACE_UPDATE_DESCR "UPDATE hosts SET sslenable=%d,sslcert=%s,description=%s WHERE hostname=%s AND portnum=%s"


/*
 * Get the host details from the database.  The provided NULL terminated
 * host_names array contains all the known names of the searched host.
 *
 * Return:
 *     0 --> No error.  row_out is set and must be freed by the caller by
 *                   sql_free_row (row_out);
 *           row_out is an array of row_item_t objects as follow:
 *                   row_out[0] --> host ID
 *                   row_out[1] --> Host name
 *                   row_out[2] --> TCP port
 *                   row_out[3] --> SSL?
 *                   row_out[4] --> SSL agent certificate
 *                   row_out[5] --> Host description
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_host_get_by_name (	char **host_names, const char *portnum,
			row_item_t **row_out,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int i, len, j;
	lwc_LL *rows;
	row_item_t *row;
	char *s, *tmp;
	dbi_conn sql_handler;
	int ret;


#if HAVE_ASSERT_H
	assert (host_names != NULL && portnum != NULL && row_out != NULL);
#endif

	/*
	 * Convert the host names array to a string suitable
	 * for the SQL request
	 */
	for (i = len = 0; host_names[i] != NULL; i++) {
		len += 2 * strlen (host_names[i]) + 3;
	}
	s = (char *) xmalloc (len);

	/* Get the SQL handler */
	sql_handler = begin_sql (&err_msg);
	if (sql_handler == NULL) {
		free (s);
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, 0);
		}
		return -2;
	}

	/* Build the string */
	for (i = j = 0; host_names[i] != NULL; i++) {
		tmp = sql_escape (sql_handler, host_names[i]);
		if (tmp == NULL) {
			end_sql ();
			free (s);
			if (error_func != NULL) {
				error_func (	user_data_error_func,
						_("Memory allocation error"),
						0);
			}
			return -1;
		}
		strcpy (s + j, tmp);
		j += strlen (tmp);
		s[j++] = ',';
		free (tmp);
	}
	if (j != 0) {
		s[j - 1] = '\0';
	}
	else {
		s[j] = '\0';
	}

	err_msg = NULL;
	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows,
				SQL_HOST_DETAIL_FROM_NAMES,
				SQL_STRING_NON_ESCAPE, s,
				SQL_STRING, portnum,
				SQL_END);
	free (s);
	end_sql ();

	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	row = (row_item_t *) lwc_delStartLL (rows);
	lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
	if (row == NULL) {
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Host not defined in the database"),
					ret);
		}
		return -2;
	}

	*row_out = row;
	return 0;
}


/*
 * Get the host details from the database.
 *
 * Return:
 *     0 --> No error.  row_out is set and must be freed by the caller by
 *                   sql_free_row (row_out);
 *           row_out is an array of row_item_t objects as follow:
 *                   row_out[0] --> host ID (the same as the host_id parameter)
 *                   row_out[1] --> Host name
 *                   row_out[2] --> TCP port
 *                   row_out[3] --> SSL?
 *                   row_out[4] --> SSL agent certificate
 *                   row_out[5] --> Host description
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_host_get_by_id (	unsigned long long int host_id,
			row_item_t **row_out,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	lwc_LL *rows;
	row_item_t *row;
	dbi_conn sql_handler;
	int ret;


#if HAVE_ASSERT_H
	assert (row_out != NULL);
#endif

	/* Get the SQL handler */
	sql_handler = begin_sql (&err_msg);
	if (sql_handler == NULL) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, 0);
		}
		return -2;
	}

	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows,
				SQL_HOST_DETAIL_FROM_ID,
				SQL_INT, (long int)host_id,
				SQL_END);
	end_sql ();

	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	row = (row_item_t *) lwc_delStartLL (rows);
	lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
	if (row == NULL) {
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Host not defined in the database"),
					ret);
		}
		return -2;
	}

	*row_out = row;

	return 0;
}


/*
 * Retrieve the certificate for the provided host (which is known by a name
 * in the provided array)
 *
 * Return:
 *     2 --> Host not found in the database
 *     0 --> No error.  sslenable, certificate and certificate_len are set (if
 *           not NULL).  certificate must be freed by the caller (by free())
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_host_certificate (	char **names, char *sslenable,
			char **certificate, unsigned long int *certificate_len,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	lwc_LL *rows;
	row_item_t *row;
	unsigned int ret, i, j, len_names;
	char *s, *t;


#if HAVE_ASSERT_H
	assert (names != NULL);
#endif

	/* Total size of the names */
	for (len_names = i = 0; names[i] != NULL; i++) {
		len_names += strlen (names[i]);
	}
	len_names += 3 * i; /* For the two `'' and the `,' */

	/* names is empty */
	if (i == 0) {
		error_func (	user_data_error_func,
				_("Host not found in the database"), 0);
		return -2;
	}
	s = (char *) xmalloc (len_names);

	/*
	 * Copy all the names in a new string.  Each name are inserted
	 * between `''.  Names are separated by `,'
	 */
	for (i = j = 0; names[i] != NULL; i++) {
		/*
		 * Basic check against SQL injections. Host names containing
		 * the character `'' are  ignored.
		 */
		if (strchr (names[i], '\'') == NULL) {
			s[j++] = '\'';
			strcpy (s + j, names[i]);
			j += strlen (names[i]);
			s[j++] = '\'';
			s[j++] = ',';
		}
	}
	if (j != 0) {
		s[j - 1] = '\0';
	}
	else {
		s[0] = '\0';
	}


	/* SQL request */
	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows,
				SQL_HOST_CERT,
				SQL_STRING_NON_ESCAPE, s,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		free (s);
		return ret;
	}

	/* Only keep the first row and delete the others */
	row = (row_item_t *) lwc_delStartLL (rows);
	lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
	if (row == NULL) {
		if (error_func != NULL) {
			t = (char *) xmalloc (
			strlen (
			_("Host not found in the database: looking for "))
				+ j + 2);
			strcpy (t,
			_("Host not found in the database: looking for "));
			strcat (t, s);
			error_func (user_data_error_func, t, 0);
			free (t);
		}
		free (s);
		return -2;
	}
	free (s);

	/*
	 * row[0] --> SSL enabled
	 * row[1] --> SSL certificate
	 */
	if (certificate != NULL) {
		/* Copy the certificate in a new string */
		s = (char *) xmalloc (row[1].len + 1);
		memcpy (s, row[1].value_string, row[1].len);
		s[row[1].len] = '\0';
		*certificate = s;
	}
	if (certificate_len != NULL) {
		*certificate_len = row[1].len;
	}
	if (sslenable != NULL) {
		*sslenable = (char) sql_row_item2ll (&(row[0]));
	}
	sql_free_row (row);
	return 0;
}


/*
 * Insert/update a host in the database
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_host_replace (const char *hostname, const char *portnum,
		char sslenable, const char *sslcert,
		const char *description,
		void (*error_func)(void *, const char *, int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;
	unsigned long long int nb = 0;

#if HAVE_ASSERT_H
	assert (hostname != NULL && portnum != NULL);
#endif

	/* Try to update the existing host in the database */
	if (description == NULL) {
		ret = sql_non_select (	NULL, NULL, &err_msg, NULL, &nb, NULL,
					SQL_HOST_REPLACE_UPDATE_NODESCR,
					SQL_BOOL, sslenable,
					SQL_STRING, sslcert,
					SQL_STRING, hostname,
					SQL_STRING, portnum,
					SQL_END);
	}
	else {
		ret = sql_non_select (	NULL, NULL, &err_msg, NULL, &nb, NULL,
					SQL_HOST_REPLACE_UPDATE_DESCR,
					SQL_BOOL, sslenable,
					SQL_STRING, sslcert,
					SQL_STRING, description,
					SQL_STRING, hostname,
					SQL_STRING, portnum,
					SQL_END);
	}
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	/* Update successful.  There was already a row for this job */
	if (nb != 0) {
		return 0;
	}

	/* This host is not yet in the database */
	err_msg = NULL;
	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL, NULL,
				SQL_HOST_NEW,
				SQL_STRING, hostname,
				SQL_STRING, portnum,
				SQL_BOOL, sslenable,
				SQL_STRING, sslcert,
				SQL_STRING, (description == NULL)
						? "": description,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
