/* Schedwi
   Copyright (C) 2007-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* sql_hierarchy.c -- Get the job hierarchy and parameters */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <utils.h>
#include <xmem.h>
#include <sql_hierarchy.h>

#define SQL_GET_JOB "SELECT job_main_s.name,job_main_s.parent,job_main_s.start_time,job_main_s.manual,job_main_s.type,job_start_limit_s.start_limit FROM job_main_s LEFT JOIN job_start_limit_s ON job_main_s.id=job_start_limit_s.job_id AND job_main_s.workload_date=job_start_limit_s.workload_date WHERE job_main_s.id=%ld AND job_main_s.workload_date=%d"
#define SQL_GET_JOB_PARAMS "SELECT %s FROM %s WHERE job_id=%ld AND workload_date=%d"
#define SQL_GET_JOB_PARAMS_TIME "SELECT %s FROM %s WHERE id=%ld AND workload_date=%d"
#define SQL_GET_JOB_PARAMS_SORTED "SELECT %s FROM %s WHERE job_id=%ld AND workload_date=%d ORDER BY %s"
#define SQL_GET_JOB_ID "SELECT id,parent,name,type FROM job_main WHERE name IN (%s)"
#define SQL_GET_JOB_CLUSTER "SELECT host_clusters.host_id FROM host_clusters,job_cluster_s WHERE  job_cluster_s.job_id=%ld AND job_cluster_s.workload_date=%d AND job_cluster_s.cluster_id = host_clusters.cluster_id"
#define SQL_GET_JOB_HOST "SELECT host_id FROM job_host_s WHERE job_id=%ld AND workload_date=%d"


/* Object to store a job details */
struct hierarchy_item {
	unsigned long long int id;
	char *name;
	char manual;
	time_t start_time;
	char node_type;
	short int start_limit;
};


/**
 * Free the provided hierarchy_item object.
 */
static void
free_hierarchy_item (struct hierarchy_item *ptr)
{
	if (ptr != NULL) {
		if (ptr->name != NULL) {
			free (ptr->name);
		}
		free (ptr);
	}
}


/*
 * Free the provided hierarchy list (built by hierarchy_list_new())
 */
void
hierarchy_list_destroy (lwc_LL *lst)
{
	lwc_delLL (lst, (void (*)(const void *))free_hierarchy_item);
}


/*
 * Get the hierarchy list of a job.  This list can then be passed as a
 * parameter to the get_job_parameters() and get_job_parameter() functions and
 * must be freed by the caller.
 * The first item of the list is the ID of the provided job (`job_id') and
 * the last one is the root jobset
 *
 * Return:
 *     The list to be freed by the caller using hierarchy_list_destroy()
 *     or NULL in case of error (if err_msg is not NULL it contains an
 *     error message which must be freed by the caller.  *err_msg may be
 *     NULL if there is not enough memory to store the error message)
 */
lwc_LL *
hierarchy_list_new (	int workload_date, unsigned long long int job_id,
			char **err_msg)
{
	dbi_conn sql;
	int ret;
	lwc_LL *lst, *rows;
	row_item_t *row;
	char *buf;
	unsigned int buf_len;
	struct hierarchy_item *ptr;


	/* Connect to the database */
	sql = begin_sql (err_msg);
	if (sql == NULL) {
		return NULL;
	}

	/* Create the result list */
	lst = lwc_newLL ();

	/* Initialize the first id to look in the database */
	ptr = (struct hierarchy_item *)xmalloc (sizeof(struct hierarchy_item));
	ptr->id = job_id;
	ptr->name = NULL;
	ptr->manual = 0;
	ptr->start_time = 0;
	ptr->node_type = 0;
	ptr->start_limit = 0;
	lwc_addEndLL (lst, ptr);

	buf = NULL;
	buf_len = 0;
	while (1) {
		/* Get the parent of the job */
		ret = sql_select (&buf, &buf_len, err_msg, NULL, &rows,
				SQL_GET_JOB,
				SQL_INT, (long int)job_id,
				SQL_INT, (long int)workload_date,
				SQL_END);
		if (ret != 0) {
			end_sql ();
			lwc_delLL (lst,
				(void (*)(const void *))free_hierarchy_item);
			if (buf != NULL) {
				free (buf);
			}
			return NULL;
		}

		/* Retrieve the result (only one row) */
		row = (row_item_t *) lwc_delStartLL (rows);
		if (row == NULL) {
			/*
			 * Nothing found - The root of the hierarchy
			 * has been reached
			 */
			lwc_delLL (rows, (void (*)(const void *))sql_free_row);
			break;
		}
		lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
		/*
		 * row[0] --> Name
		 * row[1] --> parent jobset ID
		 * row[2] --> Start time
		 * row[3] --> Automatic (0) or manual (1)
		 * row[4] --> Type (0 for jobset, 1 for job)
	 	 * row[5] --> Start limit in minutes
		 */

		/* Copy the name */
		if (job_id == ROOT_JOBSET) {   /* Root jobset */
			ptr->name = xstrdup ("/");
		}
		else {
			ptr->name = xstrdup (row[0].value_string);
		}

		/* Retrieve the start time */
		ptr->start_time = (time_t) sql_row_item2ll (&(row[2]));

		/* Copy the manual flag */
		ptr->manual = (char) sql_row_item2ll (&(row[3]));

		/* Copy the type */
		ptr->node_type = (char)sql_row_item2ll (&(row[4]));

		/* Copy the start limit */
		ptr->start_limit = (char)sql_row_item2ll (&(row[5]));

		/* Copy and add the parent id to the list */
		job_id = (unsigned long long int) sql_row_item2ll (&(row[1]));
		sql_free_row (row);
		ptr = (struct hierarchy_item *) xmalloc (
					sizeof (struct hierarchy_item));
		ptr->id = job_id;
		ptr->name = NULL;
		lwc_addEndLL (lst, ptr);
	}

	/* Close the database and free the working buffer */
	end_sql ();
	if (buf != NULL) {
		free (buf);
	}

	/* Return the list */
	return lst;
}


/**
 * Add a job/jobset as a leaf to the given jobset hierarchy list.
 *
 * @param[in] lst The jobset hierarchy list in which the job or jobset should
 *                be added. This list must have been build by
 *                hierarchy_list_new()
 * @param[in] workload_date The workload date.
 * @param[in] job_id The ID of the job or jobset to add to the list.
 * @param[out] err_msg The error message set (if not NULL) in case of error.
 *                     To be freed by the caller with free().
 *                     This error message can be NULL if there is not enough
 *                     memory to store the error message.
 * @return 0 on success or -1 on error (if err_msg is not NULL, it contains the
 *         error message)
 */
int
hierarchy_list_push_job (lwc_LL *lst, int workload_date,
			unsigned long long int job_id,
			char **err_msg)
{
	dbi_conn sql;
	struct hierarchy_item *ptr, *ptr2;
	lwc_LL *rows;
	row_item_t *row;


	if (lst == NULL) {
		return 0;
	}

	/* Connect to the database */
	sql = begin_sql (err_msg);
	if (sql == NULL) {
		return -1;
	}

	/* Get the job/jobset details from the database */
	if (sql_select (NULL, NULL, err_msg, NULL, &rows,
			SQL_GET_JOB,
			SQL_INT, (long int)job_id,
			SQL_INT, (long int)workload_date,
			SQL_END) != 0)
	{
		end_sql ();
		return -1;
	}

	/* Retrieve the result (only one row is expected) */
	row = (row_item_t *) lwc_delStartLL (rows);
	if (row == NULL) {
		/* Not found */
		lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
		end_sql ();
		return 0;
	}
	lwc_delLL (rows, (void (*)(const void *)) sql_free_row);

	/*
	 * row[0] --> Name
	 * row[1] --> parent jobset ID
	 * row[2] --> Start time
	 * row[3] --> Automatic (0) or manual (1)
	 * row[4] --> Type (0 for jobset, 1 for job)
	 * row[5] --> Start limit in minutes
	 */

	/* Create and initialize the linked list node */
	ptr = (struct hierarchy_item *)xmalloc (sizeof(struct hierarchy_item));
	ptr->id = job_id;
	ptr->manual = 0;
	ptr->start_time = 0;
	ptr->node_type = 0;
	ptr->start_limit = 0;

	/* Copy the name */
	if (job_id == ROOT_JOBSET) {   /* Root jobset */
		ptr->name = xstrdup ("/");
	}
	else {
		ptr->name = xstrdup (row[0].value_string);
	}

	/* Retrieve the start time */
	ptr->start_time = (time_t) sql_row_item2ll (&(row[2]));

	/* Copy the manual flag */
	ptr->manual = (char) sql_row_item2ll (&(row[3]));

	/* Copy the type */
	ptr->node_type = (char)sql_row_item2ll (&(row[4]));

	/* Copy the start limit */
	ptr->start_limit = (char)sql_row_item2ll (&(row[5]));

	sql_free_row (row);
	end_sql ();

	/* Copy the start limit from the parent jobset */
	if (ptr->start_limit <= 0) {
		lwc_rewindLL (lst);
		ptr2 = (struct hierarchy_item *) lwc_nextLL (lst);
		if (ptr2 != NULL) {
			ptr->start_limit = ptr2->start_limit;
		}
	}
	lwc_addStartLL (lst, ptr);
	return 0;
}


/**
 * Remove the job/jobset leaf (deepest) from the given hierarchy list.
 *
 * @param[in] lst The jobset hierarchy list from which the leaf job or jobset
 *                should be removed.
 */
void
hierarchy_list_pop_job (lwc_LL *lst)
{
	struct hierarchy_item *ptr;


	if (lst != NULL) {
		ptr = (struct hierarchy_item *) lwc_delStartLL (lst);
		free_hierarchy_item (ptr);
	}
}


/**
 * Get the type (job or jobset) of the leaf from the given hierarchy.
 *
 * @param[in] lst The jobset hierarchy list from which the type of the leaf
 *                should be returned.
 * @return The type of the object (0 for jobset or 1 for job)
 */
char
hierarchy_list_get_type (lwc_LL *lst)
{
	struct hierarchy_item *ptr;


#if HAVE_ASSERT_H
	assert (lst != NULL);
#endif

	lwc_rewindLL (lst);
	ptr = (struct hierarchy_item *) lwc_nextLL (lst);

#if HAVE_ASSERT_H
	assert (ptr != NULL);
#endif

	return ptr->node_type;
}


/**
 * Get the manual flag of the leaf from the given hierarchy.
 *
 * @param[in] lst The jobset hierarchy list from which the manual state of the
 *                leaf should be returned.
 * @return The manual flag of the object (0 not manual or 1 manual)
 */
char
hierarchy_list_get_manual (lwc_LL *lst)
{
	struct hierarchy_item *ptr;


#if HAVE_ASSERT_H
	assert (lst != NULL);
#endif

	lwc_rewindLL (lst);
	ptr = (struct hierarchy_item *) lwc_nextLL (lst);

#if HAVE_ASSERT_H
	assert (ptr != NULL);
#endif

	return ptr->manual;
}


/**
 * Get the start time of the leaf from the given hierarchy.
 *
 * @param[in] lst The jobset hierarchy list from which the start time state of
 *                the leaf should be returned.
 * @return The start time (time(2)) of the object.  -1 means that the
 *         job/jobset should start when the parent jobset starts.
 */
time_t
hierarchy_list_get_start_time (lwc_LL *lst)
{
	struct hierarchy_item *ptr;


#if HAVE_ASSERT_H
	assert (lst != NULL);
#endif

	lwc_rewindLL (lst);
	ptr = (struct hierarchy_item *) lwc_nextLL (lst);

#if HAVE_ASSERT_H
	assert (ptr != NULL);
#endif

	return ptr->start_time;
}


/**
 * Get the real start time of the given job/jobset.
 *
 * @param[in] lst The jobset hierarchy list from which the start time 
 *                should be retrieved.
 * @return The start time (time(2)) of the object.
 */
time_t
hierarchy_list_get_real_start_time (lwc_LL *lst)
{
	struct hierarchy_item *ptr;


#if HAVE_ASSERT_H
	assert (lst != NULL);
#endif

	lwc_rewindLL (lst);
	while (	   (ptr = (struct hierarchy_item *) lwc_nextLL (lst)) != NULL
		&& ptr->start_time <= 0);
	return (ptr == NULL) ? 0 : ptr->start_time;
}


/**
 * Get the start limit (in minutes) of the given job/jobset.
 *
 * @param[in] lst The jobset hierarchy list from which the start limit 
 *                should be retrieved.
 * @return The start limit in minutes.
 */
short int
hierarchy_list_get_start_limit (lwc_LL *lst)
{
	struct hierarchy_item *ptr;


#if HAVE_ASSERT_H
	assert (lst != NULL);
#endif

	lwc_rewindLL (lst);
	while (	   (ptr = (struct hierarchy_item *) lwc_nextLL (lst)) != NULL
		&& ptr->start_limit <= 0);
	return (ptr == NULL) ? 0 : ptr->start_limit;
}


/*
 * Convert the given hierarchy list in a path name (job names separated by a
 * `/')
 *
 * @return The path to be freed by the caller with free()
 */
char *
hierarchy_list_to_job_full_path (lwc_LL *lst)
{
	char prec_was_root; 
	char **a, *p;
	struct hierarchy_item *ptr;
	int i;
	size_t len;


	if (lwc_getNumNode (lst) <= 0) {
		p = (char *) xmalloc (1);
		p[0] = '\0';
		return p;
	}

	a = (char **) xmalloc (sizeof (char *) * lwc_getNumNode (lst));
	i = lwc_getNumNode (lst) - 1;
	len = 0;
	lwc_rewindLL (lst);
	while ((ptr = (struct hierarchy_item *) lwc_nextLL (lst)) != NULL) {
		a[i--] = ptr->name;
		if (ptr->name != NULL) {
			len += strlen (ptr->name);
		}
	}
	p = (char *) xmalloc (len + lwc_getNumNode (lst) + 2);
	p[0] = '\0';
	prec_was_root = 0;
	for (i = 0; i < lwc_getNumNode (lst); i++) {
		if (a[i] != NULL) {
			if (prec_was_root == 0) {
				strcat (p, "/");
			}
			if (a[i][0] != '/') {
				strcat (p, a[i]);
				prec_was_root = 0;
			}
			else {
				prec_was_root = 1;
			}
		}
	}
	free (a);
	return p;
}


/*
 * Convert a hierarchy array to a string of job IDs separated by commas
 *
 * @return The string (to be freed by the caller with free()) or NULL if
 *         a in NULL (or l is 0)
 */
char *
hierarchy_array_to_str (unsigned long long int *a, unsigned int l)
{
	char *s;
	unsigned int i, j, n;


	if (a == NULL) {
		return NULL;
	}

	s = (char *) xmalloc (l * 26);
	for (i = j = 0; j < l; j++) {
		n = copy_ulltostr (a[j], s + i);
		s[i + n] = ',';
		i += n + 1;
	}
	if (i > 0) {
		s[i - 1] = '\0';
	}
	return s;
}


/*
 * Convert a hierarchy list to a string of job IDs separated by commas
 *
 * @return The string (to be freed by the caller with free()) or NULL if
 *         the given list is empty.
 */
char *
hierarchy_list_to_str (lwc_LL *lst)
{
	char *s;
	unsigned int i, n;
	struct hierarchy_item *ptr;


	if (lwc_getNumNode (lst) <= 0) {
		return NULL;
	}

	s = (char *) xmalloc (lwc_getNumNode (lst) * 26);
	i = 0;
	lwc_rewindLL (lst);
	while ((ptr = (struct hierarchy_item *) lwc_nextLL (lst)) != NULL) {
		n = copy_ulltostr (ptr->id, s + i);
		s[i + n] = ',';
		i += n + 1;
	}
	if (i > 0) {
		s[i - 1] = '\0';
	}
	return s;
}


/*
 * Convert a hierarchy list to an array of job IDs
 *
 * @return The job ID array (to be freed by the caller with free()) or NULL if
 *         the given list is empty.
 */
unsigned long long int *
hierarchy_list_to_array (lwc_LL *lst)
{
	unsigned int i;
	struct hierarchy_item *ptr;
	unsigned long long int *ids_out;


	if (lwc_getNumNode (lst) <= 0) {
		return NULL;
	}

	ids_out = (unsigned long long int *) xmalloc (lwc_getNumNode (lst)
					* sizeof (unsigned long long int));
	i = 0;
	lwc_rewindLL (lst);
	while ((ptr = (struct hierarchy_item *) lwc_nextLL (lst)) != NULL) {
		ids_out[i++] = ptr->id;
	}
	return ids_out;
}


/*
 * Retrieve parameters for a job hierarchy.  The provided hierarchy list
 * (in lst) must have been built by hierarchy_list_new() prior to calling
 * this function.
 *
 * Return:
 *     0 --> No error.  values contains the retrieved parameters and must be
 *           freed by the caller by sql_free_row().
 *           If the parameters has not been found in the hierarchy (ie. no
 *           ancestor defines these parameters), values is set to NULL.
 *    -1 --> Memory allocation error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message).
 *    -2 --> Database error (if err_msg is not NULL it contains an error
 *           message which must be freed by the caller.  *err_msg may be NULL
 *           if there is not enough memory to store the error message).
 */
int
get_job_parameters (	int workload_date, lwc_LL *lst,
			const char *tablename, const char *colnames,
			row_item_t **values, char **err_msg)
{
	dbi_conn sql;
	int ret;
	lwc_LL *rows;
	row_item_t *row;
	char *buf;
	unsigned int buf_len;
	struct hierarchy_item *ptr;


#if HAVE_ASSERT_H
	assert (   lst != NULL && tablename != NULL && colnames != NULL
		&& values != NULL);
#endif

	/* Connect to the database */
	sql = begin_sql (err_msg);
	if (sql == NULL) {
		return -2;
	}

	buf = NULL;
	buf_len = 0;
	lwc_rewindLL (lst);
	while ((ptr = (struct hierarchy_item *) lwc_nextLL (lst)) != NULL) {
		/* Try to get the required parameter */
		ret = sql_select (&buf, &buf_len, err_msg, NULL, &rows,
				SQL_GET_JOB_PARAMS,
				SQL_STRING_NON_ESCAPE, colnames,
				SQL_STRING_NON_ESCAPE, tablename,
				SQL_INT, (long int)(ptr->id),
				SQL_INT, (long int)workload_date,
				SQL_END);
		if (ret != 0) {
			end_sql ();
			if (buf != NULL) {
				free (buf);
			}
			return ret;
		}

		/* Retrieve the result (only one row) */
		row = (row_item_t *) lwc_delStartLL (rows);
		if (row == NULL) {
			/*
			 * Nothing found - The same request will then be
			 * issued for the parent
			 */
			lwc_delLL (rows, (void (*)(const void *))sql_free_row);
			continue;
		}
		lwc_delLL (rows, (void (*)(const void *)) sql_free_row);

		/* Copy the returned values */
		*values = row;

		/* Close the database and free the working buffer */
		end_sql ();
		if (buf != NULL) {
			free (buf);
		}
		return 0;
	}

	/*
	 * Parameters not found in the database
	 */

	/* Close the database and free the working buffer */
	end_sql ();
	if (buf != NULL) {
		free (buf);
	}
	*values = NULL;

	return 0;
}


/*
 * Retrieve a parameter for a job hierarchy.  The provided hierarchy list
 * (in lst) must have been built by hierarchy_list_new() prior to calling
 * this function.
 *
 * Return:
 *     0 --> No error.  value contains the retrieved parameter.
 *           If the parameter has not been found in the hierarchy (ie. no
 *           ancestor defines this parameter), value is set to the type
 *           RES_END (value->type == RES_END)
 *           If the parameter is set and is a string, the caller must free the
 *           value_string member.
 *    -1 --> Memory allocation error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message).
 *    -2 --> Database error (if err_msg is not NULL it contains an error
 *           message which must be freed by the caller.  *err_msg may be NULL
 *           if there is not enough memory to store the error message).
 */
int
get_job_parameter (	int workload_date, lwc_LL *lst,
			const char *tablename, const char *colname,
			row_item_t *value, char **err_msg)
{
	int ret;
	row_item_t *row;
	char *s;


#if HAVE_ASSERT_H
	assert (   lst != NULL && tablename != NULL && colname != NULL
		&& value != NULL);
#endif


	ret = get_job_parameters (	workload_date, lst, tablename, colname,
					&row, err_msg);
	if (ret != 0) {
		return ret;
	}

	/* Parameter not found */
	if (row == NULL) {
		value->type = RES_END;
		return 0;
	}

	/* Copy the returned value */
	if (row[0].type == RES_STRING || row[0].type == RES_BIN) {
		s = (char *) xmalloc (row[0].len + 1);
		memcpy (s, row[0].value_string, row[0].len);
		s[row[0].len]       = '\0';
		value->type         = row[0].type;
		value->len          = row[0].len;
		value->value_string = s;
	}
	else {
		memcpy (value, &(row[0]), sizeof (row_item_t));
	}

	sql_free_row (row);

	return 0;
}


/*
 * Retrieve parameters for a job hierarchy.  The provided hierarchy list
 * (in lst) must have been built by hierarchy_list_new() prior to calling
 * this function.  The provided rows parameter is filled with
 * the retrieved parameters.  rows must be freed by the caller by
 *     lwc_delLL (rows, (void (*)(const void *))sql_free_row)
 * If there is no row retrieved from the database, rows is set to NULL.
 *
 * Return:
 *     0 --> No error.  rows is set and must be freed by the caller by
 *               lwc_delLL (rows, (void (*)(const void *))sql_free_row)
 *    -1 --> Memory allocation error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message).
 *    -2 --> Database error (if err_msg is not NULL it contains an error
 *           message which must be freed by the caller.  *err_msg may be NULL
 *           if there is not enough memory to store the error message).
 */
int
get_job_parameters_multi_row (
		int workload_date, lwc_LL *lst,
		const char *tablename, const char *colnames,
		const char *sort_colname,
		lwc_LL **rows, char **err_msg)
{
	dbi_conn sql;
	int ret;
	lwc_LL *r;
	char *buf;
	unsigned int buf_len;
	struct hierarchy_item *ptr;


#if HAVE_ASSERT_H
	assert (   lst != NULL && tablename != NULL && colnames != NULL
		&& rows != NULL);
#endif

	/* Connect to the database */
	sql = begin_sql (err_msg);
	if (sql == NULL) {
		return -2;
	}

	buf = NULL;
	buf_len = 0;
	lwc_rewindLL (lst);
	while ((ptr = (struct hierarchy_item *) lwc_nextLL (lst)) != NULL) {
		/* Try to get the required parameter */
		if (sort_colname == NULL) {
			ret = sql_select (&buf, &buf_len, err_msg, NULL, &r,
				SQL_GET_JOB_PARAMS,
				SQL_STRING_NON_ESCAPE, colnames,
				SQL_STRING_NON_ESCAPE, tablename,
				SQL_INT, (long int)(ptr->id),
				SQL_INT, (long int)workload_date,
				SQL_END);
		}
		else {
			ret = sql_select (&buf, &buf_len, err_msg, NULL, &r,
				SQL_GET_JOB_PARAMS_SORTED,
				SQL_STRING_NON_ESCAPE, colnames,
				SQL_STRING_NON_ESCAPE, tablename,
				SQL_INT, (long int)(ptr->id),
				SQL_INT, (long int)workload_date,
				SQL_STRING_NON_ESCAPE, sort_colname,
				SQL_END);
		}

		if (ret != 0) {
			end_sql ();
			if (buf != NULL) {
				free (buf);
			}
			return ret;
		}

		/*
		 * If nothing has been found, the same request will be
		 * issued for the parent
		 */
		if (lwc_getNumNode (r) <= 0) {
			lwc_delLL (r, (void (*)(const void *))sql_free_row);
			continue;
		}

		*rows = r;

		/* Close the database and free the working buffer */
		end_sql ();
		if (buf != NULL) {
			free (buf);
		}
		return 0;
	}

	/*
	 * Parameters not found in the database
	 */
	*rows = NULL;

	/* Close the database and free the working buffer */
	end_sql ();
	if (buf != NULL) {
		free (buf);
	}

	return 0;
}


/*
 * Retrieve parameters for a job hierarchy.  The provided hierarchy list
 * (in lst) must have been built by hierarchy_list_new() prior to calling
 * this function.  The provided callback function is called for each retrieved
 * row.  Its first parameter is the provided user_data object, the second
 * is the row array.
 * If callback returns a value other than 0, get_job_parameters_multi()
 * returns a value of 2.
 *
 * Return:
 *     2 --> The callback function didn't return 0
 *     0 --> No error.
 *    -1 --> Memory allocation error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message).
 *    -2 --> Database error (if err_msg is not NULL it contains an error
 *           message which must be freed by the caller.  *err_msg may be NULL
 *           if there is not enough memory to store the error message).
 */
int
get_job_parameters_multi (
		int workload_date, lwc_LL *lst,
		const char *tablename, const char *colnames,
		const char *sort_colname,
		int (*callback)(void *, row_item_t *), void *user_data,
		char **err_msg)
{
	int ret;
	lwc_LL *rows;
	row_item_t *row;


#if HAVE_ASSERT_H
	assert (   lst != NULL && tablename != NULL && colnames != NULL
		&& callback != NULL);
#endif

	/* Retrieve the rows */
	ret = get_job_parameters_multi_row (	workload_date,
						lst, tablename, colnames,
						sort_colname, &rows, err_msg);
	if (ret != 0) {
		return ret;
	}

	/* Call the callback function for each row */
	while ((row = (row_item_t *) lwc_delStartLL (rows)) != NULL) {
		ret = callback (user_data, row);
		sql_free_row (row);
		if (ret != 0) {
			lwc_delLL (rows, (void (*)(const void *))sql_free_row);
			return 2;
		}
	}
	lwc_delLL (rows, (void (*)(const void *))sql_free_row);

	return 0;
}


/*
 * Retrieve the host IDs for a job hierarchy.  The provided hierarchy list
 * (in lst) must have been built by hierarchy_list_new() prior to calling
 * this function.
 *
 * Return:
 *     0 --> No error.  host_ids contains the host ID list and must be
 *           freed by the caller by
 *               lwc_delLL (host_ids, (void (*)(const void *))free);
 *           If no host ID  has been found in the hierarchy, host_ids is
 *           set to NULL.
 *    -1 --> Memory allocation error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message).
 *    -2 --> Database error (if err_msg is not NULL it contains an error
 *           message which must be freed by the caller.  *err_msg may be NULL
 *           if there is not enough memory to store the error message).
 */
int
get_job_hosts (	int workload_date, lwc_LL *lst, lwc_LL **host_ids,
		char **err_msg)
{
	dbi_conn sql;
	int ret;
	lwc_LL *rows, *ids;
	row_item_t *row;
	char *buf;
	unsigned int buf_len;
	struct hierarchy_item *ptr;
	unsigned long long int *id;


#if HAVE_ASSERT_H
	assert (lst != NULL && host_ids != NULL);
#endif

	/* Connect to the database */
	sql = begin_sql (err_msg);
	if (sql == NULL) {
		return -2;
	}

	buf = NULL;
	buf_len = 0;
	lwc_rewindLL (lst);
	while ((ptr = (struct hierarchy_item *) lwc_nextLL (lst)) != NULL) {
		/* Try to get the host IDs from a cluster */
		ret = sql_select (&buf, &buf_len, err_msg, NULL, &rows,
				SQL_GET_JOB_CLUSTER,
				SQL_INT, (long int)(ptr->id),
				SQL_INT, (long int)workload_date,
				SQL_END);
		if (ret != 0) {
			end_sql ();
			if (buf != NULL) {
				free (buf);
			}
			return ret;
		}

		/* Not in a cluster */
		if (lwc_getNumNode (rows) <= 0) {
			/*
			 * Check if the job is directly associated with a host
			 */
			lwc_delLL (rows, (void (*)(const void *))sql_free_row);
			ret = sql_select (&buf, &buf_len, err_msg, NULL, &rows,
					SQL_GET_JOB_HOST,
					SQL_INT, (long int)(ptr->id),
					SQL_INT, (long int)workload_date,
					SQL_END);
			if (ret != 0) {
				end_sql ();
				if (buf != NULL) {
					free (buf);
				}
				return ret;
			}
			if (lwc_getNumNode (rows) <= 0) {
				/*
				 * Nothing found - The same request will then
				 * be issued for the parent
				 */
				lwc_delLL (rows,
					(void (*)(const void *))sql_free_row);
				continue;
			}
		}

		ids = lwc_newLL ();

		/*
		 * Retrieve the host IDs and stored them in the list to
		 * be returned
		 */
		while ((row = (row_item_t *) lwc_delStartLL (rows)) != NULL) {
			id = (unsigned long long int *)
				xmalloc (sizeof (unsigned long long int));
			*id = (unsigned long long int) sql_row_item2ll (
								&(row[0]));
			lwc_addEndLL (ids, id);
			sql_free_row (row);
		}
		lwc_delLL (rows, (void (*)(const void *)) sql_free_row);

		/* Copy the returned values */
		*host_ids = ids;

		/* Close the database and free the working buffer */
		end_sql ();
		if (buf != NULL) {
			free (buf);
		}
		return 0;
	}

	/*
	 * Host ID not found in the database
	 */

	/* Close the database and free the working buffer */
	end_sql ();
	if (buf != NULL) {
		free (buf);
	}
	*host_ids = NULL;

	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
