/* Schedwi
   Copyright (C) 2007-2014 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* @file sql_get_environment.c
 * Get the environment for a job searching its ancestors (the job inherit the
 * environments of its ancestors).
 */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <sql_common.h>
#include <sql_hierarchy.h>
#include <xmem.h>
#include <sql_get_environment.h>

#define SQL_GET_JOB_NOENV "SELECT job_main_s.id FROM job_main_s WHERE job_main_s.id IN (%s) AND job_main_s.has_env=0 AND job_main_s.workload_date=%d"
#define SQL_GET_ENVIRONMENT_REQ "SELECT job_environment_s.job_id,job_environment_s.position,environment_var_s.position,environment_var_s.env_key,environment_var_s.env_value FROM environment_var_s,job_environment_s WHERE job_environment_s.job_id IN (%s) AND job_environment_s.workload_date=%d AND environment_var_s.workload_date=%d AND job_environment_s.env_id=environment_var_s.env_id ORDER BY job_environment_s.position,environment_var_s.position"
#define SQL_GET_ENVIRONMENT_HOST_REQ "SELECT host_environment_s.position,environment_var_s.position,environment_var_s.env_key,environment_var_s.env_value FROM environment_var_s,host_environment_s WHERE host_environment_s.host_id=%ld AND host_environment_s.workload_date=%d AND environment_var_s.workload_date=%d AND host_environment_s.env_id=environment_var_s.env_id ORDER BY host_environment_s.position,environment_var_s.position"


/*
 * Get the environment from the database for a job specified by the
 * provided hierarchy list (in lst).  For each environment variable,
 * the callback() function is called with the following parameters:
 *    - The position
 *    - The variable name
 *    - The variable value
 * If the callback function return a value other than 0, the
 * sql_get_environment() exits with a return code of 2.
 *
 * Return:
 *     2 --> The callback function returned a value other than 0
 *     0 --> No error.
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_get_environment (int workload_date, lwc_LL *lst, unsigned int next_pos,
	int (*callback)(void *, unsigned int, const char *, const char *),
	void *user_data,
	void (*error_func)(void *, const char *, int),
	void *user_data_error_func)
{
	char *err_msg = NULL;
	char *hierarchy_str;
	unsigned int i, j, ids_len;
	int ret;
	lwc_LL *rows, **lists;
	row_item_t *row;
	unsigned long long int jobid, *ids;


#if HAVE_ASSERT_H
	assert (lwc_getNumNode (lst) > 0 && callback != NULL);
#endif


	/* Build the job ID list to be used by the SQL request */
	hierarchy_str = hierarchy_list_to_str (lst);

	/* From the jobset hierarchy, get the jobsets where has_env is 0 */
	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows,
				SQL_GET_JOB_NOENV,
				SQL_STRING_NON_ESCAPE, hierarchy_str,
				SQL_INT, (long int)workload_date,
				SQL_END);
	free (hierarchy_str);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	/* Recreate the new jobset hierarchy */
	ids_len = lwc_getNumNode (lst);
	ids = hierarchy_list_to_array (lst);
	while ((row = (row_item_t *) lwc_delStartLL (rows)) != NULL) {
		/* row[0] --> Job ID */
		jobid = (unsigned long long int)sql_row_item2ll (&(row[0]));
		sql_free_row (row);
		for (j = 0; j < ids_len; j++) {
			if (ids[j] == jobid) {
				ids_len = j;
				break;
			}
		}
	}
	lwc_delLL (rows, (void (*)(const void *))sql_free_row);
	/* No environment variable to set */
	if (ids_len == 0) {
		free (ids);
		return 0;
	}

	/* Build the job ID list to be used by the SQL request */
	hierarchy_str = hierarchy_array_to_str (ids, ids_len);

	/* Create the array of environment variable lists for each job ID */
	lists = (lwc_LL **) xmalloc (ids_len * sizeof (lwc_LL *));

	for (i = 0; i < ids_len; i++) {
		lists[i] = lwc_newLL ();
	}

	/* Get the environment variables from the database */
	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows,
				SQL_GET_ENVIRONMENT_REQ,
				SQL_STRING_NON_ESCAPE, hierarchy_str,
				SQL_INT, (long int)workload_date,
				SQL_INT, (long int)workload_date,
				SQL_END);
	free (hierarchy_str);
	if (ret != 0) {
		for (i = 0; i < ids_len; i++) {
			lwc_delLL (lists[i], NULL);
		}
		free (lists);
		free (ids);
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	while ((row = (row_item_t *) lwc_delStartLL (rows)) != NULL) {
		/*
		 * row[0] --> Job ID
		 * row[1] --> Position 1
		 * row[2] --> Position 2
		 * row[3] --> Environment variable name
		 * row[4] --> Environment variable value
		 */
		jobid = (unsigned long long int)sql_row_item2ll (&(row[0]));
		for (i = 0; i < ids_len && ids[i] != jobid; i++);
		if (i < ids_len) {
			lwc_addEndLL (lists[i], row);
		}
		else {
			sql_free_row (row);
		}
	}
	lwc_delLL (rows, (void (*)(const void *))sql_free_row);
	free (ids);

	i = ids_len;
	do {
		i--;
		while ((row = (row_item_t *)lwc_delStartLL (lists[i])) != NULL)
		{
			/*
			 * row[0] --> Job ID
			 * row[1] --> Position 1
			 * row[2] --> Position 2
			 * row[3] --> Environment variable name
			 * row[4] --> Environment variable value
			 */
			ret = callback (user_data, next_pos++,
					row[3].value_string,
					row[4].value_string);
			sql_free_row (row);
			if (ret != 0) {
				for (i = 0; i < ids_len; i++) {
					lwc_delLL (lists[i],
					(void (*)(const void *))sql_free_row);
				}
				free (lists);
				return 2;
			}
		}
	} while (i > 0);

	for (i = 0; i < ids_len; i++) {
		lwc_delLL (lists[i], (void (*)(const void *))sql_free_row);
	}
	free (lists);
	return 0;
}


/*
 * Get the environment from the database for a host specified by the
 * provided host_id string.  For each environment variable,
 * the callback() function is called with the following parameters:
 *    - The position
 *    - The variable name
 *    - The variable value
 * If the callback function return a value other than 0, the
 * sql_get_environment_host() function exits with a return code of 2.
 * If not NULL, next_pos is set with the position of the last environment
 * variable.  This value may be used in the call to the sql_get_environment()
 * function.
 *
 * Return:
 *     2 --> The callback function returned a value other than 0
 *     0 --> No error.
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_get_environment_host (int workload_date, unsigned long long int host_id,
	int (*callback)(void *, unsigned int, const char *, const char *),
	void *user_data,
	unsigned int *next_pos,
	void (*error_func)(void *, const char *, int),
	void *user_data_error_func)
{
	char *err_msg = NULL;
	lwc_LL *rows;
	row_item_t *row;
	unsigned int i;
	int ret;


#if HAVE_ASSERT_H
	assert (callback != NULL);
#endif

	/* SQL request */
	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows,
				SQL_GET_ENVIRONMENT_HOST_REQ,
				SQL_INT, (long int)host_id,
				SQL_INT, (long int)workload_date,
				SQL_INT, (long int)workload_date,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	i = 1;
	while ((row = (row_item_t *) lwc_delStartLL (rows)) != NULL) {
		/*
		 * row[0] --> Position 1
		 * row[1] --> Position 2
		 * row[2] --> Environment variable name
		 * row[3] --> Environment variable value
		 */
		ret = callback (user_data, i++,
				row[2].value_string, row[3].value_string);
		sql_free_row (row);
		if (ret != 0) {
			lwc_delLL (rows, (void (*)(const void *))sql_free_row);
			return 2;
		}
	}
	lwc_delLL (rows, (void (*)(const void *))sql_free_row);

	if (next_pos != NULL) {
		*next_pos = i;
	}
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
