/* Schedwi
   Copyright (C) 2007-2014 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* sql_common.c -- Useful SQL functions */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_TIME_H
#include <time.h>
#endif

#if TM_IN_SYS_TIME
#include <sys/time.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <pthread.h>

#include <xmem.h>
#include <conf.h>
#include <sql_common.h>

static dbi_conn sql = NULL;
static int nb_handlers = 0;


/*
 * Mutex to protect libdbi requests (libdbi is not yet thread-safe)
 */
static pthread_mutex_t lock = PTHREAD_MUTEX_INITIALIZER;


/*
 * Get a libdbi SQL handler
 *
 * Return:
 *    The handler or
 *    NULL in case of error.  If err_msg is not NULL, it is set with the error
 *    message (it must be freed by the caller)
 */
dbi_conn
begin_sql (char **err_msg)
{
	pthread_mutex_lock (&lock);
	if (sql != NULL && nb_handlers > 0) {
		/*
		 * As there is a memory leak bug in the SQLite/SQLite3 drivers
		 * in the dbi_conn_ping() function (in dbi_conn_ping()),  it
		 * cannot be used and we suppose that the DB is still alive
		 */
		if (is_sqlite () || dbi_conn_ping (sql) != 0) {
			/* Connection still alive */
			nb_handlers++;
			pthread_mutex_unlock (&lock);
			return sql;
		}
	}
	sql = schedwi_dbi_init (err_msg);
	if (sql == NULL) {
		pthread_mutex_unlock (&lock);
		return NULL;
	}
	if (nb_handlers < 0) {
		nb_handlers = 1;
	}
	else {
		nb_handlers++;
	}
	pthread_mutex_unlock (&lock);
	return sql;
}


/*
 * Free a libdbi SQL handler
 */
void
end_sql ()
{
	nb_handlers--;
	if (sql != NULL && nb_handlers <= 0) {
		pthread_mutex_lock (&lock);
		schedwi_dbi_close (sql);
		sql = NULL;
		nb_handlers = 0;
		pthread_mutex_unlock (&lock);
	}
}


/*
 * Built an SQL error message
 * Warning - a mutex must be set before calling this function.
 *
 * If not NULL, msg_out contains the error message (to be freed by the caller)
 */
void
compose_error_message (const char *msg, char **msg_out)
{
	const char *msg_sql;
	char *separator;
	unsigned int msg_len;

	if (msg_out == NULL) {
		return;
	}

	dbi_conn_error(sql, &msg_sql);

	if (msg == NULL) {
		msg_len = 0;
	}
	else {
		msg_len = strlen (msg);
	}

	separator = _(": ");
	*msg_out = (char *) xmalloc (	  msg_len
					+ strlen (separator)
					+ strlen (msg_sql)
					+ 1);
	(*msg_out)[0] = '\0';
	if (msg != NULL) {
		strcpy (*msg_out, msg);
	}
	if (msg != NULL && msg_sql[0] != '\0') {
		strcat (*msg_out, separator);
	}
	strcat (*msg_out, msg_sql);
}


/*
 * Escape the provided string (if the string is NULL, an empty string is
 * returned - ie. the first char is 0 and the string still need to be
 * freed by the caller). The returned string is itself in quotes.
 * Warning - a mutex must be set before calling this function.
 *
 * Return:
 *   The escaped string (to be freed by the caller) or
 *   NULL in case of error
 */
char *
sql_escape (dbi_conn sql, const char *s)
{
	char *res;

#if HAVE_ASSERT_H
	assert (sql != NULL);
#endif

	if (dbi_conn_quote_string_copy (sql, (s == NULL) ? "": s, &res) == 0) {
		return NULL;
	}
	return res;
}


/*
 * Escape the provided binary string (if the string is NULL, an empty string is
 * returned - ie. the first char is 0 and the string still need to be
 * freed by the caller). The returned string is itself in quotes.
 * Warning - a mutex must be set before calling this function.
 *
 * Return:
 *   The escaped string (to be freed by the caller) or
 *   NULL in case of error
 */
static unsigned char *
sql_escape_bin (dbi_conn sql, const unsigned char *s, size_t len)
{
	unsigned char *res;

#if HAVE_ASSERT_H
	assert (sql != NULL);
#endif

	if (dbi_conn_quote_binary_copy (sql,
					(s == NULL) ? (unsigned char*)"" : s,
					len,
					&res) == 0)
	{
		return NULL;
	}
	return res;
}


#define MAX_NUM_ARGS 10

struct sql_args {
	sql_type type;
	union {
		long int value_int;
		char *value_string;
		char value_bool;
	};
};
typedef struct sql_args sql_args_t;


/*
 * Free the content of a sql_args array
 */
static void
free_args (sql_args_t *args, int len)
{
	if (args != NULL) {
		while (--len >= 0) {
			if (args[len].type == SQL_STRING) {
				if (args[len].value_string != NULL) {
					free (args[len].value_string);
				}
			}
		}
	}
}


/*
 * Run a database request
 *
 * The result is stored in the provided result parameter and must be freed
 * by the caller by dbi_result_free().
 * buf (and buf_len) is used as a working buffer.  If buf is NULL, an internal
 * buffer is used. If *buf is NULL, sql_request() will allocate a buffer for
 * internal use (*buf must be freed by the caller) and *buf (and *buf_len) will
 * be set with this buffer.  Alternatively, before calling sql_request() *buf
 * can contain a pointer to a malloc()-allocated buffer *buf_len bytes in size.
 * If the buffer is not large enough, it will be resized, updating *buf and
 * *buf_len to reflect the buffer address and size.
 * Warning - a mutex must be set before calling this function.
 *
 * The variable argument list should contain a sql_type followed by the value.
 * For the sql_type SQL_BIN, the value must be followed by the lenght of the
 * string value.
 * The list is terminated by SQL_END.  For examples:
 *
 *     dbi_result result;
 *     sql_request (	sql, &result, NULL, NULL, &err,
 *   			"DELETE FROM hosts WHERE id=%ld AND hostname=%s",
 *   			SQL_INT, 12,
 *   			SQL_STRING, "foo.company.org",
 *   			SQL_END);
 *     dbi_result_free(result);
 *
 *     img_data = (char *)gdk_pixdata_serialize (pixdata, &len_img_data);
 *     sql_request (	sql, &result, NULL, NULL, &err,
 *			"INSERT INTO imgs (id,data) VALUES (%ld,%s)",
 *			SQL_INT, 144,
 *			SQL_BIN, img_data, len_img_data,
 *                      SQL_END);
 *     dbi_result_free(result);
 *     g_free (img_data);
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message)
 *    -2 --> Database error (if err_msg is not NULL it contains an error
 *           message which must be freed by the caller.  *err_msg may be NULL
 *           if there is not enough memory to store the error message).
 */
static int
sql_request (dbi_conn sql, dbi_result *result,
		char **buf, unsigned int *buf_len,
		char **err_msg,
		const char *fmt, va_list ap)
{
	sql_args_t args[MAX_NUM_ARGS];
	sql_type t;
	int i, ret, try;
	char c;
	char *s;
	unsigned char *us;
	unsigned int req_len;
	char *req;
	char req_to_free;
	size_t bin_len;

#if HAVE_ASSERT_H
	assert (sql != NULL && result != NULL && fmt != NULL);
#endif

	req_len = 0;
	for (	i = 0;
		i < MAX_NUM_ARGS && (t = va_arg (ap, sql_type)) != SQL_END;
		i++)
	{
		args[i].type = t;

		switch (t) {

		case SQL_INT:
			args[i].value_int = va_arg (ap, long int);
			req_len += 22;
			break;

		case SQL_BOOL:
			c = va_arg (ap, int);
			args[i].value_bool = (c == 0 || c == '0') ? 0 : 1;
			req_len += 1;
			break;

		case SQL_STRING:
			/* Escape the provided string */
			s = sql_escape (sql, va_arg (ap, char *));
			if (s == NULL) {
				free_args (args, i);
				if (err_msg != NULL) {
					s =_("Memory allocation error");
					*err_msg = xstrdup (s);
				}
				return -1;
			}
			args[i].value_string = s;
			req_len += strlen (s);
			break;

		case SQL_STRING_NON_ESCAPE:
			s = va_arg (ap, char *);
			args[i].value_string = s;
			req_len += strlen (s);
			break;

		case SQL_BIN:
			/* Escape the binary string */
			us = va_arg (ap, unsigned char *);
			bin_len = va_arg (ap, size_t);
			us = sql_escape_bin (sql, us, bin_len);
			if (us == NULL) {
				free_args (args, i);
				if (err_msg != NULL) {
					s =_("Memory allocation error");
					*err_msg = xstrdup (s);
				}
				return -1;
			}
			args[i].value_string = (char *)us;
			req_len += strlen ((char *)us);
			break;

		default:
			break;
		}
	}

	/* Fill the end of the args array with NULL strings */
	while (i < MAX_NUM_ARGS) {
		args[i].type = SQL_STRING;
		args[i].value_string = NULL;
		i++;
	}

	/* Allocate memory for the SQL request */
	req_len += strlen (fmt) + 1;
	if (buf == NULL || buf_len == NULL) {
		/* Create a new buffer */
		req = (char *) xmalloc (req_len);
		req_to_free = 1;
	}
	else {
		/* Try to use the provided buffer */
		if (*buf_len < req_len || *buf == NULL) {
			if (*buf != NULL) {
				free (*buf);
			}
			*buf = (char *) xmalloc (req_len);
			*buf_len = req_len;
		}
		req = *buf;
		req_to_free = 0;
	}

	/* Build the request */
	ret = snprintf (req, req_len, fmt,
		(   args[0].type == SQL_STRING || args[0].type == SQL_BIN
		 || args[0].type == SQL_STRING_NON_ESCAPE)
			? args[0].value_string
			: ((args[0].type == SQL_INT) 	? args[0].value_int
							: args[0].value_bool)
#if MAX_NUM_ARGS > 1
		,(  args[1].type == SQL_STRING || args[1].type == SQL_BIN
		 || args[1].type == SQL_STRING_NON_ESCAPE)
			? args[1].value_string
			: ((args[1].type == SQL_INT)	? args[1].value_int
							: args[1].value_bool)
#if MAX_NUM_ARGS > 2
		,(  args[2].type == SQL_STRING || args[2].type == SQL_BIN
		 || args[2].type == SQL_STRING_NON_ESCAPE)
			? args[2].value_string
			: ((args[2].type == SQL_INT)	? args[2].value_int
							: args[2].value_bool)
#if MAX_NUM_ARGS > 3
		,(  args[3].type == SQL_STRING || args[3].type == SQL_BIN
		 || args[3].type == SQL_STRING_NON_ESCAPE)
			? args[3].value_string
			: ((args[3].type == SQL_INT)	? args[3].value_int
							: args[3].value_bool)
#if MAX_NUM_ARGS > 4
		,(  args[4].type == SQL_STRING || args[4].type == SQL_BIN
		 || args[4].type == SQL_STRING_NON_ESCAPE)
			? args[4].value_string
			: ((args[4].type == SQL_INT)	? args[4].value_int
							: args[4].value_bool)
#if MAX_NUM_ARGS > 5
		,(  args[5].type == SQL_STRING || args[5].type == SQL_BIN
		 || args[5].type == SQL_STRING_NON_ESCAPE)
			? args[5].value_string
			: ((args[5].type == SQL_INT)	? args[5].value_int
							: args[5].value_bool)
#if MAX_NUM_ARGS > 6
		,(  args[6].type == SQL_STRING || args[6].type == SQL_BIN
		 || args[6].type == SQL_STRING_NON_ESCAPE)
			? args[6].value_string
			: ((args[6].type == SQL_INT)	? args[6].value_int
							: args[6].value_bool)
#if MAX_NUM_ARGS > 7
		,(  args[7].type == SQL_STRING || args[7].type == SQL_BIN
		 || args[7].type == SQL_STRING_NON_ESCAPE)
			? args[7].value_string
			: ((args[7].type == SQL_INT)	? args[7].value_int
							: args[7].value_bool)
#if MAX_NUM_ARGS > 8
		,(  args[8].type == SQL_STRING || args[8].type == SQL_BIN
		 || args[8].type == SQL_STRING_NON_ESCAPE)
			? args[8].value_string
			: ((args[8].type == SQL_INT)	? args[8].value_int
							: args[8].value_bool)
#if MAX_NUM_ARGS > 9
		,(  args[9].type == SQL_STRING || args[9].type == SQL_BIN
		 || args[9].type == SQL_STRING_NON_ESCAPE)
			? args[9].value_string
			: ((args[9].type == SQL_INT)	? args[9].value_int
							: args[9].value_bool)
#if MAX_NUM_ARGS > 10
		,(  args[10].type == SQL_STRING || args[10].type == SQL_BIN
		 || args[10].type == SQL_STRING_NON_ESCAPE)
			? args[10].value_string
			: ((args[10].type == SQL_INT)	? args[10].value_int
							: args[10].value_bool)
#if MAX_NUM_ARGS > 11
		,(  args[11].type == SQL_STRING || args[11].type == SQL_BIN
		 || args[11].type == SQL_STRING_NON_ESCAPE)
			? args[11].value_string
			: ((args[11].type == SQL_INT)	? args[11].value_int
							: args[11].value_bool)
#if MAX_NUM_ARGS > 12
		,(  args[12].type == SQL_STRING || args[12].type == SQL_BIN
		 || args[12].type == SQL_STRING_NON_ESCAPE)
			? args[12].value_string
			: ((args[12].type == SQL_INT)	? args[12].value_int
							: args[12].value_bool)
#if MAX_NUM_ARGS > 13
		,(  args[13].type == SQL_STRING || args[13].type == SQL_BIN
		 || args[13].type == SQL_STRING_NON_ESCAPE)
			? args[13].value_string
			: ((args[13].type == SQL_INT)	? args[13].value_int
							: args[13].value_bool)
#if MAX_NUM_ARGS > 14
		,(  args[14].type == SQL_STRING || args[14].type == SQL_BIN
		 || args[14].type == SQL_STRING_NON_ESCAPE)
			? args[14].value_string
			: ((args[14].type == SQL_INT)	? args[14].value_int
							: args[14].value_bool)
#if MAX_NUM_ARGS > 15
		,(  args[15].type == SQL_STRING || args[15].type == SQL_BIN
		 || args[15].type == SQL_STRING_NON_ESCAPE)
			? args[15].value_string
			: ((args[15].type == SQL_INT)	? args[15].value_int
							: args[15].value_bool)
#if MAX_NUM_ARGS > 16
		,(  args[16].type == SQL_STRING || args[16].type == SQL_BIN
		 || args[16].type == SQL_STRING_NON_ESCAPE)
			? args[16].value_string
			: ((args[16].type == SQL_INT)	? args[16].value_int
							: args[16].value_bool)
#if MAX_NUM_ARGS > 17
		,(  args[17].type == SQL_STRING || args[17].type == SQL_BIN
		 || args[17].type == SQL_STRING_NON_ESCAPE)
			? args[17].value_string
			: ((args[17].type == SQL_INT)	? args[17].value_int
							: args[17].value_bool)
#if MAX_NUM_ARGS > 18
		,(  args[18].type == SQL_STRING || args[18].type == SQL_BIN
		 || args[18].type == SQL_STRING_NON_ESCAPE)
			? args[18].value_string
			: ((args[18].type == SQL_INT)	? args[18].value_int
							: args[18].value_bool)
#if MAX_NUM_ARGS > 19
		,(  args[19].type == SQL_STRING || args[19].type == SQL_BIN
		 || args[19].type == SQL_STRING_NON_ESCAPE)
			? args[19].value_string
			: ((args[19].type == SQL_INT)	? args[19].value_int
							: args[19].value_bool)
#if MAX_NUM_ARGS > 20
		,(  args[20].type == SQL_STRING || args[20].type == SQL_BIN
		 || args[20].type == SQL_STRING_NON_ESCAPE)
			? args[20].value_string
			: ((args[20].type == SQL_INT)	? args[20].value_int
							: args[20].value_bool)
#endif
#endif
#endif
#endif
#endif
#endif
#endif
#endif
#endif
#endif
#endif
#endif
#endif
#endif
#endif
#endif
#endif
#endif
#endif
#endif
	);

#if HAVE_ASSERT_H
	assert (ret < req_len && ret >=0);
#endif

	free_args (args, MAX_NUM_ARGS);

	/*
	 * Run the request.
	 * In case of failure, retry several times because in the case of
	 * SQLite, the database may be locked temporarily by an other
	 * process (schedwi or schedwigui).
	 */
	try = 0;
	do {
		if (++try > 1) {
#if HAVE_NANOSLEEP
			struct timespec time_req;

			time_req.tv_sec = 1;
			time_req.tv_nsec = 0;
			nanosleep (&time_req, NULL);
#elif HAVE_SLEEP
			sleep (1);
#else
			usleep (1000000);
#endif
		}
		*result = dbi_conn_query (sql, req);
	} while (*result == NULL && dbi_conn_error (sql, NULL) && try < 5);
	if (req_to_free != 0) {
		free (req);
	}
	if (*result == NULL && dbi_conn_error (sql, NULL)) {
		compose_error_message (_("Database error"), err_msg);
		return -2;
	}
	return 0;
}


/*
 * Run a non-select database request (insert, delete, update, ...)
 *
 * buf (and buf_len) is used as a working buffer.  If buf is NULL, an internal
 * buffer is used. If *buf is NULL, sql_non_select() will allocate a buffer for
 * internal use (*buf must be freed by the caller) and *buf (and *buf_len) will
 * be set with this buffer.  Alternatively,  before calling sql_non_select()
 * *buf can contain a pointer to a malloc()-allocated buffer *buf_len bytes in
 * size.  If the buffer is not large enough, it will be resized, updating *buf
 * and *buf_len to reflect the buffer address and size.
 *              id (out) --> ID of the last inserted item
 *   affected_rows (out) --> Number of affected rows
 *        tablename (in) --> Name of updated table. This is used only if `id'
 *                           is not NULL and it is required with some databases
 *                           (postgresql) to get the last inserted item from
 *                           a sequence.
 *
 * The variable argument list should contain a sql_type followed by the value.
 * The list is terminated by SQL_END.  For example:
 *
 *     sql_non_select (	NULL, NULL, &err, NULL, NULL, "hosts",
 *   			"DELETE FROM hosts WHERE id=%ld AND hostname=%s",
 *   			SQL_INT, 12,
 *   			SQL_STRING, "foo.company.org",
 *   			SQL_END);
 *
 * Return:
 *     0 --> No error (if id is not NULL it contains the value of the last ID
 *           inserted or updated in the database.  This is only relevant for
 *           INSERT and UPDATE requests)
 *    -1 --> Memory allocation error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message)
 *    -2 --> Database error (if err_msg is not NULL it contains an error
 *           message which must be freed by the caller.  *err_msg may be NULL
 *           if there is not enough memory to store the error message).
 */
int
sql_non_select (	char **buf, unsigned int *buf_len, char **err_msg,
			unsigned long long int *id,
			unsigned long long int *affected_rows,
			const char *tablename,
			const char *fmt, ...)
{
	va_list ap;
	int ret;
	dbi_conn sql;
	dbi_result result;
	char *sequence, *s;

#if HAVE_ASSERT_H
	assert (fmt != NULL);
#endif

	/* Connect to the database */
	sql = begin_sql (err_msg);
	if (sql == NULL) {
		return -2;
	}

	/* Run the request */
	pthread_mutex_lock (&lock);
	va_start (ap, fmt);
	ret = sql_request (sql, &result, buf, buf_len, err_msg, fmt, ap);
	va_end (ap);

	if (ret != 0) {
		pthread_mutex_unlock (&lock);
		end_sql ();
		return ret;
	}

	/* Get the number of affected rows */
	if (affected_rows != NULL) {
		*affected_rows = dbi_result_get_numrows_affected (result);
	}

	/* Get the last id inserted in the database */
	if (id != NULL) {
		*id = dbi_conn_sequence_last (sql, NULL);
		/*
		 * If the dbi_conn_sequence_last() failed, try to use the
		 * sequence which is called <tablename>_<column>_seq
		 */
		if (*id == 0 && tablename != NULL && tablename[0] != '\0') {
			s = "_id_seq";
			sequence = (char *) xmalloc (	  strlen (tablename)
							+ strlen (s) + 1);
			strcpy (sequence, tablename);
			strcat (sequence, s);
			*id = dbi_conn_sequence_last (sql, sequence);
			free (sequence);
		}
	}

	if (result != NULL) {
		dbi_result_free (result);
	}
	pthread_mutex_unlock (&lock);
	end_sql ();

	return 0;
}


/*
 * Free a result row
 */
void
sql_free_row (row_item_t *row)
{
	int i;

	if (row != NULL) {
		for (i = 0; row[i].type != RES_END; i++) {
			if (	   row[i].type == RES_STRING
				|| row[i].type == RES_BIN)
			{
				free (row[i].value_string);
			}
		}
		free (row);
	}
}


/*
 * Return the int value in the provided row item (whatever the type is - signed
 * or unsigned)
 */
long long int
sql_row_item2ll (const row_item_t *item)
{
	if (item != NULL) {
		if (item->type == RES_INT || item->type == RES_UINT) {
			return (long long int)((item->type == RES_INT)
							? item->value_int
							: item->value_uint);
		}
		if (item->type == RES_STRING) {
			return (long long int) strtoull (item->value_string,
							 NULL, 0);
		}
	}
	return 0;
}


/*
 * Run a select database request
 *
 * buf (and buf_len) is used as a working buffer.  If buf is NULL, an internal
 * buffer is used. If *buf is NULL, sql_select() will allocate a buffer for
 * internal use (*buf must be freed by the caller) and *buf (and *buf_len) will
 * be set with this buffer.  Alternatively,  before calling sql_select() *buf
 * can contain a pointer to a malloc()-allocated buffer *buf_len bytes in size.
 * If the buffer is not large enough, it will be resized, updating *buf and
 * *buf_len to reflect the buffer address and size.
 * The returned `row' list is a list of the rows. Each row is an array of
 * row_item_t. The last item of the array has the `type' field set
 * to `RES_END'.
 *
 * The variable argument list should contain a sql_type followed by the value.
 * For the sql_type SQL_BIN, the value must be followed by the lenght of the
 * string value.
 * The list is terminated by SQL_END.  For example:
 *
 *         sql_select (	NULL, NULL, &err, NULL, &list_row,
 *   			"SELECT FROM hosts WHERE id=%ld AND hostname=%s",
 *   			SQL_INT, 12,
 *   			SQL_STRING, "foo.company.org",
 *   			SQL_END);
 *
 * Return:
 *     0 --> No error.  rows contains the list of rows and must be freed by the
 *           caller by
 *              lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
 *           If not NULL, num_columns contains the number of columns.
 *    -1 --> Memory allocation error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message)
 *    -2 --> Database error (if err_msg is not NULL it contains an error
 *           message which must be freed by the caller.  *err_msg may be NULL
 *           if there is not enough memory to store the error message).
 */
int
sql_select (	char **buf, unsigned int *buf_len, char **err_msg,
		unsigned int *num_columns, lwc_LL **rows,
		const char *fmt, ...)
{
    va_list ap;
    dbi_conn sql;
    dbi_result result;
    lwc_LL *list;
    char *s;
    unsigned int cols, i, attrs;
    int ret;
    row_item_t *row_array;

#if HAVE_ASSERT_H
    assert (rows != NULL && fmt != NULL);
#endif

    /* Allocate memory for the linked list */
    list = lwc_newLL ();

    /* Connect to the database */
    sql = begin_sql (err_msg);
    if (sql == NULL) {
        lwc_delLL (list, NULL);
        return -2;
    }

    /* Run the request */
    pthread_mutex_lock (&lock);
    va_start (ap, fmt);
    ret = sql_request (sql, &result, buf, buf_len, err_msg, fmt, ap);
    va_end (ap);

    if (ret != 0) {
        pthread_mutex_unlock (&lock);
        end_sql ();
        lwc_delLL (list, NULL);
        return ret;
    }

    /* Number of columns */
    cols = dbi_result_get_numfields (result);
    if (cols == DBI_FIELD_ERROR) {
        compose_error_message (_("Database error"), err_msg);
        dbi_result_free (result);
        pthread_mutex_unlock (&lock);
        end_sql ();
        lwc_delLL (list, NULL);
        return -2;
    }

    /* Copy each returned row */
    while (dbi_result_next_row (result)) {

        /* Allocate a new row array */
        row_array = (row_item_t *) xmalloc (sizeof (row_item_t) * (cols + 1));
        row_array[cols].type = RES_END;

        /* Copy the returned row */
        for (i = 0; i < cols; i++) {

            attrs = dbi_result_get_field_attribs_idx (result, i + 1);
            switch (dbi_result_get_field_type_idx (result, i + 1)) {

                case DBI_TYPE_INTEGER:
                    if (attrs & DBI_INTEGER_UNSIGNED) {
                        row_array[i].type = RES_UINT;
                        if (attrs & DBI_INTEGER_SIZE1) {
                            row_array[i].value_uint =
                                dbi_result_get_uchar_idx (result, i + 1);
                        }
                        else
                        if (attrs & DBI_INTEGER_SIZE2) {
                            row_array[i].value_uint =
                                dbi_result_get_ushort_idx (result, i + 1);
                        }
                        else
                        if (attrs & (DBI_INTEGER_SIZE3|DBI_INTEGER_SIZE4)) {
                            row_array[i].value_uint =
                                dbi_result_get_uint_idx (result, i + 1);
                        }
                        else {
                            row_array[i].value_uint =
                                dbi_result_get_ulonglong_idx (result, i + 1);
                        }
                    }
                    else {
                        row_array[i].type = RES_INT;
                        if (attrs & DBI_INTEGER_SIZE1) {
                            row_array[i].value_int =
                                dbi_result_get_char_idx (result, i + 1);
                        }
                        else
                        if (attrs & DBI_INTEGER_SIZE2) {
                            row_array[i].value_int =
                                dbi_result_get_short_idx (result, i + 1);
                        }
                        else
                        if (attrs & (DBI_INTEGER_SIZE3|DBI_INTEGER_SIZE4)) {
                            row_array[i].value_int =
                                dbi_result_get_int_idx (result, i + 1);
                        }
                        else {
                            row_array[i].value_int =
                                dbi_result_get_longlong_idx (result, i + 1);
                        }
                    }
                    break;

                case DBI_TYPE_DECIMAL:
                    if (attrs & DBI_DECIMAL_UNSIGNED) {
                        row_array[i].type = RES_UINT;
                        if (attrs & DBI_DECIMAL_SIZE4) {
                            row_array[i].value_uint =
                                dbi_result_get_uint_idx (result, i + 1);
                        }
                        else {
                            row_array[i].value_uint =
                                dbi_result_get_ulonglong_idx (result, i + 1);
                        }
                    }
                    else {
                        row_array[i].type = RES_INT;
                        if (attrs & DBI_DECIMAL_SIZE4) {
                            row_array[i].value_int =
                                dbi_result_get_int_idx (result, i + 1);
                        }
                        else {
                            row_array[i].value_int =
                                dbi_result_get_longlong_idx (result, i + 1);
                        }
                    }
                    break;

                case DBI_TYPE_STRING:
                    row_array[i].type = RES_STRING;
                    row_array[i].len  =
                              dbi_result_get_field_length_idx (result, i + 1);
                    row_array[i].value_string =
                              dbi_result_get_string_copy_idx (result, i + 1);

                    if (   (   row_array[i].value_string == NULL
                            && row_array[i].len > 0)
                         || strcmp (row_array[i].value_string, "ERROR") == 0)
                    {
                        if (row_array[i].value_string != NULL) {
                            free (row_array[i].value_string);
                            compose_error_message (_("Database error"),
                                                   err_msg);
                        }
                        else {
                            if (err_msg != NULL) {
                                s = _("Memory allocation error");
                                *err_msg = xstrdup (s);
                            }
                        }
                        dbi_result_free (result);
                        pthread_mutex_unlock (&lock);
                        end_sql ();
                        lwc_delLL (list, (void (*)(const void *))sql_free_row);
                        row_array[i].type = RES_END;
                        sql_free_row (row_array);
                        return -1;
                    }
                    break;

                case DBI_TYPE_BINARY:
                    row_array[i].type = RES_BIN;
                    row_array[i].len  =
                              dbi_result_get_field_length_idx (result, i + 1);
                    row_array[i].value_string = (char *)
                              dbi_result_get_binary_copy_idx (result, i + 1);
                    if (   (   row_array[i].value_string == NULL
                            && row_array[i].len > 0)
                         || strcmp (row_array[i].value_string, "ERROR") == 0)
                    {
                        if (row_array[i].value_string != NULL) {
                            free (row_array[i].value_string);
                            compose_error_message (_("Database error"),
                                                   err_msg);
                        }
                        else {
                            if (err_msg != NULL) {
                                s = _("Memory allocation error");
                                *err_msg = xstrdup (s);
                            }
                        }
                        dbi_result_free (result);
                        pthread_mutex_unlock (&lock);
                        end_sql ();
                        lwc_delLL (list, (void (*)(const void *))sql_free_row);
                        row_array[i].type = RES_END;
                        sql_free_row (row_array);
                        return -1;
                    }
                    break;

                case DBI_TYPE_DATETIME:
                    row_array[i].type = RES_TIME;
                    row_array[i].value_time =
                              dbi_result_get_datetime_idx (result, i + 1);
                    break;
            }
        }

        /* Add the new row to the linked list */
        lwc_addEndLL (list, row_array);
    }

    dbi_result_free (result);
    pthread_mutex_unlock (&lock);
    end_sql ();
    if (num_columns != NULL) {
        *num_columns = cols;
    }
    *rows = list;
    return 0;
}


/*
 * Return a SQL standard date/time which represents now + sec (sec can
 * be negative to compute a date/time in the past)
 *
 * Return:
 *   The date/string to be freed by the caller by free()
 */
char *
sql_date_interval (int sec)
{
	char *d;
	time_t t;
	struct tm stm;

	time (&t);
	t += sec;
	localtime_r (&t, &stm);
	d = (char *) xmalloc (25);
	strftime (d, 24, "%Y-%m-%d %H:%M:%S", &stm);
	return d;
}

/*-----------------============== End Of File ==============-----------------*/
