/* Schedwi
   Copyright (C) 2007-2014 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* sql_commands.c -- commands management functions */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <sql_common.h>
#include <sql_commands.h>


#define SQLCOMMAND_GET "SELECT DISTINCT workload_date,job_id,username,command,parameter,insert_time FROM commands WHERE command_status=0 AND insert_time >= %s ORDER BY workload_date,job_id,insert_time"
#define SQLCOMMAND_ADD "INSERT INTO commands (workload_date,job_id,username,command,parameter) VALUES (%d,%ld,%s,%d,%d)"
#define SQLCOMMAND_DONE "UPDATE commands SET command_status=1 WHERE job_id=%ld AND workload_date=%d AND command=%d AND parameter=%d"
#define SQLCOMMAND_FAILED "UPDATE commands SET command_status=2 WHERE job_id=%ld AND workload_date=%d AND command=%d AND parameter=%d"


/*
 * Retrieve the commands to run.  For each command, the provided callback()
 * is called with the provided parameters:
 *   - the provided user_data parameter
 *   - the workload
 *   - the job ID
 *   - the username
 *   - the command
 *   - the parameter
 * If the callback function returns a value other than 0, the
 * sql_command_get_commands() function exits with a return code of 2.
 *
 * Return:
 *     2 --> The callback function returned a value other than 0
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_command_get_commands (
		int (*callback)(void *, int, unsigned long long int,
				const char *, int, int),
		void *user_data,
		void (*error_func)(void *, const char *, int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;
	lwc_LL *rows;
	row_item_t *row;
	char *d;

#if HAVE_ASSERT_H
	assert (callback != NULL);
#endif

	d = sql_date_interval (-10 * 60); /* now - 10 minutes */
	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows,
				SQLCOMMAND_GET,
				SQL_STRING, d,
				SQL_END);
	free (d);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	while ((row = (row_item_t *) lwc_delStartLL (rows)) != NULL) {
		/*
		 * row[0] --> Workload
		 * row[1] --> Job ID
		 * row[2] --> User name
		 * row[3] --> Command
		 * row[4] --> Parameter
		 * row[5] --> Insert time
		 */
		if (callback (	user_data,
			(int)sql_row_item2ll (&(row[0])),
			(unsigned long long int)sql_row_item2ll (&(row[1])),
			row[2].value_string,
			(int)sql_row_item2ll (&(row[3])),
			(int)sql_row_item2ll (&(row[4]))) != 0)
		{
			sql_free_row (row);
			lwc_delLL (rows, (void (*)(const void *))sql_free_row);
			return 2;
		}
		sql_free_row (row);
	}
	lwc_delLL (rows, (void (*)(const void *))sql_free_row);

	return 0;
}


/*
 * Add the provided command for the Schedwi server to run
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error.  If err_msg is not NULL it contains
 *           an error message to be freed by the caller by free()
 *    -2 --> SQL error.  If err_msg is not NULL it contains
 *           an error message to be freed by the caller by free()
 */
int
sql_command_add (	int workload_date, unsigned long long job_id,
			const char *user, int command, int parameter,
			char **err_msg)
{
	return sql_non_select (	NULL, NULL, err_msg, NULL, NULL, NULL,
				SQLCOMMAND_ADD,
				SQL_INT, (long int)workload_date,
				SQL_INT, (long int)job_id,
				SQL_STRING, (user == NULL) ? "": user,
				SQL_INT, (long int)command,
				SQL_INT, (long int)parameter,
				SQL_END);
}


/*
 * Update the status of the command in the database
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_command_done (	int workload_date, unsigned long long int job_id,
			int command, int parameter,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;


	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL, NULL,
				SQLCOMMAND_DONE,
				SQL_INT, (long int)job_id,
				SQL_INT, (long int)workload_date,
				SQL_INT, (long int)command,
				SQL_INT, (long int)parameter,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Update the status of the command in the database
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_command_failed (	int workload_date, unsigned long long int job_id,
			int command, int parameter,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;


	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL, NULL,
				SQLCOMMAND_FAILED,
				SQL_INT, (long int)job_id,
				SQL_INT, (long int)workload_date,
				SQL_INT, (long int)command,
				SQL_INT, (long int)parameter,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
