/* Schedwi
   Copyright (C) 2007-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* sql_calendar.c -- calendars table management functions */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <sql_common.h>
#include <sql_calendar.h>

#define SQL_CAL_ONLY_LIST "SELECT calendars_s.id,calendars_s.name,calendars_s.formula FROM calendars_s,job_main_s WHERE calendars_s.workload_date=%d AND job_main_s.workload_date=%d AND calendars_s.entry_type='0' AND calendars_s.id=job_main_s.cal_id"


/**
 * Retrieve all the calendars (and not the folders) from the database.  Only
 * used calendars (associated with at least a job or a jobsets) are retrieved.
 *
 * @param[in] workload_date Workload date of the calendars, jobs and jobsets
 *                          (YYYYMMDD).
 * @param[out] rows List of calendars to be freed by the caller by
 *                    lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
 *                  Each item in the list is of type row_item_t:  it's an array
 *                  with element 0 set to the ID of the calendar, element 1
 *                  to its name and element 2 to its formula.
 * @param[in] error_func Error function to call in case of error. This function
 *                       is called with user_data_error_func as its first
 *                       parameter and the error message as the second
 *                       parameter.  error_func can be NULL.
 * @param[in] user_data_error_func User provided parameter to pass to the
 *                                 error_func function in vase of error.
 * @return
 *     0 --> No error.  rows contains the retrieved calendars and must be freed
 *           by the caller by
 *              lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_cal_only_list (	int workload_date,
			lwc_LL **rows,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;


#if HAVE_ASSERT_H
	assert (rows != NULL);
#endif

	ret = sql_select (	NULL, NULL, &err_msg, NULL, rows,
				SQL_CAL_ONLY_LIST,
				SQL_INT, (long int)workload_date,
				SQL_INT, (long int)workload_date,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
