/* Schedwi
   Copyright (C) 2013-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* sql_acknowledge_manual.c -- acknowledge_manual database table functions */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <sql_common.h>
#include <xmem.h>
#include <sql_acknowledge_manual.h>

#define SQLACK_ADD "INSERT INTO acknowledge_manual (workload_date,job_id,status,urn,password) VALUES (%d,%ld,0,%s,%s)"
#define SQLACK_GET "SELECT username,ack_time,status FROM acknowledge_manual WHERE job_id=%ld AND workload_date=%d"


/*
 * Add an acknowledge request
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error.  If err_msg is not NULL it contains
 *           an error message to be freed by the caller by free()
 *    -2 --> SQL error.  If err_msg is not NULL it contains
 *           an error message to be freed by the caller by free()
 */
int
sql_acknowledge_add (	int workload_date, unsigned long long int job_id,
			const char *urn, const char *password, char **err_msg)
{
	return sql_non_select (	NULL, NULL, err_msg, NULL, NULL, NULL,
				SQLACK_ADD,
				SQL_INT, (long int)workload_date,
				SQL_INT, (long int)job_id,
				SQL_STRING, (urn == NULL) ? "": urn,
				SQL_STRING, (password == NULL) ? "": password,
				SQL_END);
}


/*
 * Get the details of the provided job/jobset acknowledge state
 *
 * Return:
 *     0 --> No error.  status contains the retrieved status and start_time
 *           the associated start time. username is set with the name of the
 *           user who did the acknowledgement and must be freed by the caller
 *           with free(). status is set to -1 if the requested entry does not
 *           exists in the database.
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_acknowledge_get (	int workload_date,
			unsigned long long int job_id,
			int *status,
			time_t *ack_time,
			char **username,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	lwc_LL *rows;
	int ret;
	row_item_t *row;


#if HAVE_ASSERT_H
	assert (status != NULL);
#endif

	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows,
				SQLACK_GET,
				SQL_INT, (long int)job_id,
				SQL_INT, (long int)workload_date,
				SQL_END);

	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	row = (row_item_t *) lwc_delStartLL (rows);

	/*
	 * row[0] --> username
	 * row[1] --> ack_time
	 * row[2] --> status
 	 */ 	
	if (username != NULL) {
		if (	   row == NULL
			|| row[0].value_string == NULL
			|| row[0].value_string[0] == '\0')
		{
			*username = NULL;
		}
		else {
			*username = xstrdup (row[0].value_string);
		}
	}

	/* Status */
	*status = (row == NULL) ? -1 : (int)sql_row_item2ll (&(row[2]));

	/* Start time */
	if (ack_time != NULL) {
		*ack_time = (row == NULL) ? 0 : row[1].value_time;
	}

	sql_free_row (row);
	lwc_delLL (rows, (void (*)(const void *))sql_free_row);
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
