/* Schedwi
   Copyright (C) 2007-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* schedwisrv.c -- Schedwi server */

#include <schedwi.h>

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifndef _GNU_SOURCE
extern char **environ;
#endif

#if HAVE_TIME_H
#include <time.h>
#endif

#if HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif

#if HAVE_SIGNAL_H
#include <signal.h>
#endif

#if HAVE_GETOPT_H
#include <getopt.h>
#endif

#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <error.h>

#include <sql_common.h>
#include <utils.h>
#include <conf.h>
#include <conf_srv.h>
#include <lwc_log.h>
#include <net_utils.h>
#include <signal_utils.h>
#include <check_new_workload.h>
#include <update_status_jobset_tree.h>
#include <module.h>
#include <cert_ca.h>
#include <xmem.h>
#include <get_job_status_from_agents.h>
#include <user_commands.h>


static const char *configuration_file = NULL;
static char **save_argv, **save_env;
#if HAVE_CHDIR && HAVE_GETCWD
#define MAX_SIZE_CWD 500
static char save_cwd[MAX_SIZE_CWD];
#endif

static fd_set sock_set;
static int sock_max;


/*
 * Print a help message to stderr
 */
void
help ()
{
	int i;
	char * msg[] = {
N_("The Schedwi server."),
"",
#if HAVE_GETOPT_LONG
N_("  -c, --config=FILE  use the configuration file FILE rather than"),
N_("                     the default one"),
N_("  -t, --test         Run syntax tests for the configuration file. The"),
N_("                     program immediately exits after these syntax"),
N_("                     parsing tests with either a return code of 0 (syntax"),
N_("                     OK) or return code not equal to 0 (syntax error)"),
#if HAVE_WORKING_FORK
N_("  -f, --foreground   keep the program in foreground"),
#endif
N_("  -h, --help         display this help and exit"),
N_("  -V, --version      print version information and exit"),
#else /* HAVE_GETOPT_LONG */
N_("  -c FILE            use the configuration file FILE rather than"),
N_("                     the default one"),
N_("  -t                 Run syntax tests for the configuration file. The"),
N_("                     program immediately exits after these syntax"),
N_("                     parsing tests with either a return code of 0 (syntax"),
N_("                     OK) or return code not equal to 0 (syntax error)"),
#if HAVE_WORKING_FORK
N_("  -f                 keep the program in foreground"),
#endif
N_("  -h                 display this help and exit"),
N_("  -V                 print version information and exit"),
#endif /* ! HAVE_GETOPT_LONG */
"",
NULL
	};

#if HAVE_ASSERT_H
	assert (configuration_file != NULL);
#endif

	fprintf (stderr, _("Usage: %s [OPTION]...\n"),
		 program_invocation_name);

	for (i = 0; msg[i] != NULL; i++) {
		if (msg[i][0] != '\0') {
			fputs (_(msg[i]), stderr);
		}
		fputc ('\n', stderr);
	}

	fputs (	_("The default configuration file is "), stderr);
	fputs (configuration_file, stderr);
	fputc ('\n', stderr);
	fprintf (stderr,
		 _("Exit status is %d on error (%d means no error).\n"),
		 EXIT_FAILURE, EXIT_SUCCESS);
	fputc ('\n', stderr);
	fputs (_("Report bugs to "), stderr);
	fputs (PACKAGE_BUGREPORT, stderr);
	fputc ('\n', stderr);
}


/*
 * Signal handler for SIGHUP
 */
static void
sighup_action ()
{
	/*
	 * Before anything else be sure that the configuration file
	 * can be read
	 */
	if (access (configuration_file, R_OK) != 0) {
		lwc_writeLog (	LOG_ERR, _("%s: %s"),
				configuration_file, strerror (errno));
		return;
	}

	lwc_closeLog ();
	net_close_sock_set (&sock_set, &sock_max);
	close_pid_file ();
	end_sql ();
	net_destroy ();
	conf_destroy_srv ();
	module_exit ();
	sanitise ();
#if HAVE_CHDIR && HAVE_GETCWD
	if (save_cwd[0] != '\0') {
		chdir (save_cwd);
	}
#endif
	execve (save_argv[0], save_argv, save_env);
	error (EXIT_FAILURE, errno, "execve");
}



/*
 * Signal handler for SIGTERM
 */
static void
sigterm_action (int sig)
{
	int i;

	lwc_writeLog (LOG_INFO, _("schedwisrv exiting"));
	net_close_sock_set (&sock_set, &sock_max);
	for (i = 0; save_env[i] != NULL; i++) {
		free (save_env[i]);
	}
	free (save_env);
	free (save_argv);
	exit (EXIT_SUCCESS);
}


/*
 * Main function
 *
 * The exit code is 1 in case of error or 0 if killed (SIGTERM)
 */
int
main (int argc, char **argv)
{
	int i, ret;
	const char *group_id, *user_id;
	const char *pid_file, *log_file, *date_format;
	const char *iface, *server_port, *address_family, *allow_from;
	const char *server_key, *server_crt, *ca_crt, *ca_key, *ca_crl;
	char quick_random;
	long int facility, today_workload_skip;
	char test, foreground;
	dbi_conn sql_handler;
	char *err_msg;
	struct sigaction sa;
#if HAVE_GETOPT_LONG
	int option_index;
	struct option long_options[] =
	{
		{"help",       0, 0, 'h'},
		{"version",    0, 0, 'V'},
		{"test",       0, 0, 't'},
		{"foreground", 0, 0, 'f'},
		{"config",     1, 0, 'c'},
		{0, 0, 0, 0}
	};
#endif


#if HAVE_SETLOCALE
	setlocale (LC_ALL, "");
#endif
#if HAVE_BINDTEXTDOMAIN
	bindtextdomain (PACKAGE, LOCALEDIR);
#endif
#if HAVE_TEXTDOMAIN
	textdomain (PACKAGE);
#endif

	/* Set default values for options */
	test = foreground = 0;
	configuration_file = SCHEDWI_DEFAULT_CONFFILE_SRV;

	/* Parse options */
	while (1) {
#if HAVE_GETOPT_LONG
		option_index = 0;
		ret = getopt_long (argc, argv, "hVtfc:",
					long_options, &option_index);
#else
		ret = getopt (argc, argv, "hVtfc:");
#endif
		if (ret == -1) {
			break;
		}

		switch (ret) {
			case 'h':
				help ();
				return EXIT_SUCCESS;
			case 'V':
				version ();
				return EXIT_SUCCESS;
			case 't':
				test = 1;
				break;
			case 'f':
				foreground = 1;
				break;
			case 'c':
				configuration_file = optarg;
				break;
			default:
				help ();
				return EXIT_FAILURE;
		}
	}

	if (test == 1) {
		printf (_("Configuration file: %s\n"), configuration_file);
	}

	/*
	 * seed for the rand() function that is used through the program
	 */
	srand ((unsigned int) time (NULL));

	/*
	 * Initialize the module engine
	 */
	if (module_init () != 0) {
		return EXIT_FAILURE;
	}
	atexit (module_exit);

	/*
	 * Read the configuration file
	 */
	ret = conf_init_srv (configuration_file);
	if (ret == -2) {
		error (EXIT_FAILURE, errno, "%s", configuration_file);
	}
	if (ret != 0) {
		return EXIT_FAILURE;
	}
	atexit (conf_destroy_srv);

	ret =  conf_get_param_string ("SSLCACertificateFile", &ca_crt);
	ret += conf_get_param_string ("SSLCACertificateKeyFile", &ca_key);
	ret += conf_get_param_string ("SSLCACRLFile", &ca_crl);
	ret += conf_get_param_string ("SSLCertificateFile", &server_crt);
	ret += conf_get_param_string ("SSLCertificateKeyFile", &server_key);
	ret += conf_get_param_bool ("SSLQuickRandom", &quick_random);
	ret += conf_get_param_string ("USER", &user_id);
	ret += conf_get_param_string ("GROUP", &group_id);
	ret += conf_get_param_string ("PID_FILE", &pid_file);
	ret += conf_get_param_int (	"TODAY_WORKLOAD_SKIP",
					&today_workload_skip);
#if HAVE_ASSERT_H
	assert (ret == 0);
#endif

	/*
	 * Close opened files (but keep the standard fd 0, 1 and 2 because
	 * some error messages may be printed on stderr)
	 * This has to be done before the net_init_gnutls() function below
	 * because the gnutls_global_init() library call will open a file
	 * descriptor on /dev/urandom.
	 */
	close_opened_files ();

	/*
	 * Initialize the GnuTLS layer
	 */
	if (net_init_gnutls (quick_random) != 0) {
		return EXIT_FAILURE;
	}
	atexit (net_destroy);

	/*
	 * Create the keys and certificates if needed
	 */
	if (	   test == 0
		&& cert_check_and_create (	ca_crt, ca_key,
						server_crt, server_key,
						user_id, group_id,
						quick_random) != 0)
	{
		return EXIT_FAILURE;
	}

	/*
	 * Initialize the network (before detaching the program from its
	 * terminal as a password may be asked for the SSL private key)
	 */
	if ((	   test == 0
		|| (	   access (ca_crt, F_OK) == 0
			&& access (server_crt, F_OK) == 0
			&& access (server_key, F_OK) == 0
		   )
	    ) && net_init (	server_crt, server_key, ca_crt,
				ca_crl, quick_random) != 0)
	{
		return EXIT_FAILURE;
	}

	/*
	 * For a daemon that is going to run in background, create a PID file
	 * and change owner.  This is for sysV daemons.
	 * Systemd daemons are started in the foreground (systemd will set
	 * the owner and manage the PID file - see the daemon man page for
	 * systemd)
	 */
	if (foreground == 0) {

		/*
		 * Create the directory that will hold the PID file
		 */
		mkpath (pid_file, user_id, group_id);
	}
	
	/*
	 * Change the group id and the user id of the process
	 */
	ret = change_group (group_id);
	if (ret != 0) {
		if (ret == -2) {
			fprintf (stderr,
				_("Unknown group name: %s\n"),
				group_id);
		}
		else {
			fprintf (stderr,
			_("Failed to change the group ID to %s: %s\n"),
				group_id, strerror (errno));
		}
		return EXIT_FAILURE;
	}

	ret = change_user (user_id);
	if (ret != 0) {
		if (ret == -2) {
			fprintf (stderr,
				_("Unknown user name: %s\n"),
				user_id);
		}
		else {
			fprintf (stderr,
			_("Failed to change the user ID to %s: %s\n"),
				user_id, strerror (errno));
		}
		return EXIT_FAILURE;
	}

	/* It was only a test of the configuration file */
	if (test == 1) {
		/* Just try a connection to the database */
		err_msg = NULL;
		sql_handler = begin_sql (&err_msg);
		if (sql_handler == NULL) {
			if (err_msg != NULL) {
				fputs (err_msg, stderr);
				free (err_msg);
			}
			else {
				fputs (_("Failed to connect to database"),
					stderr);
			}
			fputc ('\n', stderr);
			return EXIT_FAILURE;
		}
		end_sql ();

		fputs (_("Fine."), stdout);
		fputc ('\n', stdout);
		return EXIT_SUCCESS;
	}

	/* Copy argv and environ to use them to exec the process on SIGHUP */
	save_argv = (char **) xmalloc (sizeof (char *) * (argc + 1));
	for (i = 0; i < argc; i++) {
		save_argv[i] = argv[i];
	}
	save_argv[i] = NULL;

	for (i = 0; environ[i] != NULL; i++);
	save_env = (char **) xmalloc (sizeof (char *) * (i + 1));
	for (i = 0; environ[i] != NULL; i++) {
		save_env[i] = xstrdup (environ[i]);
	}
	save_env[i] = NULL;

#if HAVE_CHDIR && HAVE_GETCWD
	if (getcwd (save_cwd, MAX_SIZE_CWD) == NULL) {
		save_cwd[0] = '\0';
	}
#endif


	/*
	 * Go background
	 */
	if (foreground == 0 && go_daemon () != 0) {
		error (EXIT_FAILURE, errno, "fork");
	}


	/*
	 * For sysV daemons, open the log file and write the PID file.
	 * For systemd service (foreground = 1), systemd will already have done
	 * all that (log messages are sent to stderr)
	 */
	if (foreground == 0) {

		/*
		 * Open the log file
		 */
		log_file = date_format = NULL;
		facility = -1;
		ret = conf_get_param_string ("LOG_FILE", &log_file);
		ret += conf_get_param_syslog_facility ("SYSLOG", &facility);
		ret += conf_get_param_string ("DATE_FORMAT", &date_format);
#if HAVE_ASSERT_H
		assert (ret == 0);
#endif
	
		lwc_newLog (    PACKAGE_NAME,
				(facility != -1) ? 1 : 0,
				(facility != -1) ? (int)facility : 0,
				log_file,
				date_format);
		atexit (lwc_closeLog);
		lwc_writeLog (LOG_INFO, _("schedwisrv starting"));
	
	
		/*
		 * Create the PID file and check that an other process is not
		 * already running
		 */
		ret = write_pid_file (pid_file);
		switch (ret) {
			case -1:
				lwc_writeLog (	LOG_CRIT,
					_("Cannot create the PID file %s: %s"),
						pid_file,
						strerror (errno));
				break;
	
			case 1:
				lwc_writeLog (LOG_CRIT,
				_("An other daemon is already running"));
				break;
		}
		if (ret != 0) {
			for (i = 0; save_env[i] != NULL; i++) {
				free (save_env[i]);
			}
			free (save_env);
			free (save_argv);
			return EXIT_FAILURE;
		}
		atexit (close_pid_file);
	}


	/*
	 * Database connection
	 */
	err_msg = NULL;
	sql_handler = begin_sql (&err_msg);
	if (sql_handler == NULL) {
		if (err_msg != NULL) {
			lwc_writeLog (	LOG_CRIT,
					_("Failed to connect to database: %s"),
					err_msg);
			free (err_msg);
		}
		else {
			lwc_writeLog (	LOG_CRIT,
					_("Failed to connect to database"));
		}
		for (i = 0; save_env[i] != NULL; i++) {
			free (save_env[i]);
		}
		free (save_env);
		free (save_argv);
		return EXIT_FAILURE;
	}
	atexit (end_sql);


	/*
	 * SSL setup
	 */
	if (quick_random == 0) {
		lwc_writeLog (LOG_INFO,
		_("Initializing the SSL layer (may take several minutes)..."));
	}
	if (net_init_server () != 0) {
		for (i = 0; save_env[i] != NULL; i++) {
			free (save_env[i]);
		}
		free (save_env);
		free (save_argv);
		return EXIT_FAILURE;
	}
	if (quick_random == 0) {
		lwc_writeLog (LOG_INFO, _("Done"));
	}


	/*
	 * Network setup
	 */
	ret = conf_get_param_string ("IFACE_LISTEN", &iface);
	ret += conf_get_param_string ("SERVER_PORT", &server_port);
	ret += conf_get_param_string ("ADDRESS_FAMILY", &address_family);
	ret += conf_get_param_string ("ALLOW_FROM", &allow_from);
#if HAVE_ASSERT_H
	assert (ret == 0);
#endif

	FD_ZERO (&sock_set);
	sock_max = -1;
	if (net_server (server_port, iface, address_family,
			&sock_set, &sock_max) != 0)
	{
		for (i = 0; save_env[i] != NULL; i++) {
			free (save_env[i]);
		}
		free (save_env);
		free (save_argv);
		return EXIT_FAILURE;
	}


	/*
	 * Signal handlers
	 */
	memset (&sa, 0, sizeof (struct sigaction));
	sa.sa_handler = sigterm_action;
	sigemptyset (&(sa.sa_mask));
	sa.sa_flags = 0;
	if (sigaction (SIGTERM, &sa, NULL) != 0) {
		lwc_writeLog (
			LOG_CRIT,
			_("Cannot install signal handler for SIGTERM: %s"),
			strerror (errno));
		net_close_sock_set (&sock_set, &sock_max);
		for (i = 0; save_env[i] != NULL; i++) {
			free (save_env[i]);
		}
		free (save_env);
		free (save_argv);
		return EXIT_FAILURE;
	}

	memset (&sa, 0, sizeof (struct sigaction));
	sa.sa_handler = sighup_action;
	sigemptyset (&(sa.sa_mask));
	sa.sa_flags = 0;
	if (sigaction (SIGHUP, &sa, NULL) != 0) {
		lwc_writeLog (
			LOG_CRIT,
			_("Cannot install signal handler for SIGHUP: %s"),
			strerror (errno));
		net_close_sock_set (&sock_set, &sock_max);
		for (i = 0; save_env[i] != NULL; i++) {
			free (save_env[i]);
		}
		free (save_env);
		free (save_argv);
		return EXIT_FAILURE;
	}


	/*
	 * Main loop
	 */
	while (1) {

		signal_mask_all ();
		check_new_workload (today_workload_skip);
		signal_unmask ();

		get_job_status_from_agents (&sock_set, sock_max, allow_from);

		signal_mask_all ();
		user_command_check ();
		signal_unmask ();

		signal_mask_all ();
		update_status_jobset_tree ();
		signal_unmask ();
	}
	return EXIT_SUCCESS;
}

/*------------------------======= End Of File =======------------------------*/
