/* Schedwi
   Copyright (C) 2007-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* schedwireg.c -- Schedwi registrar daemon */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_SYS_SELECT_H
#include <sys/select.h>
#else
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifndef _GNU_SOURCE
extern char **environ;
#endif

#if HAVE_TIME_H
#include <time.h>
#endif

#if HAVE_SIGNAL_H
#include <signal.h>
#endif

#if HAVE_GETOPT_H
#include <getopt.h>
#endif

#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <error.h>

#include <utils.h>
#include <conf.h>
#include <conf_srv.h>
#include <module.h>
#include <signal_utils.h>
#include <lwc_log.h>
#include <net_utils.h>
#include <net_utils_ssl.h>
#include <net_utils_sock.h>
#include <net_parse_reg.h>
#include <xmem.h>


static const char *configuration_file = NULL;
static char **save_argv, **save_env;
#if HAVE_CHDIR && HAVE_GETCWD
#define MAX_SIZE_CWD 500
static char save_cwd[MAX_SIZE_CWD];
#endif


/*
 * Print a help message to stderr
 */
void
help ()
{
	int i;
	char * msg[] = {
N_("The Schedwi registrar daemon."),
"",
#if HAVE_GETOPT_LONG
N_("  -c, --config=FILE  use the configuration file FILE rather than"),
N_("                     the default one"),
N_("  -t, --test         Run syntax tests for the configuration file. The"),
N_("                     program immediately exits after these syntax"),
N_("                     parsing tests with either a return code of 0 (syntax"),
N_("                     OK) or return code not equal to 0 (syntax error)"),
#if HAVE_WORKING_FORK
N_("  -f, --foreground   keep the program in foreground"),
#endif
N_("  -h, --help         display this help and exit"),
N_("  -V, --version      print version information and exit"),
#else
N_("  -c FILE            use the configuration file FILE rather than"),
N_("                     the default one"),
N_("  -t                 Run syntax tests for the configuration file. The"),
N_("                     program immediately exits after these syntax"),
N_("                     parsing tests with either a return code of 0 (syntax"),
N_("                     OK) or return code not equal to 0 (syntax error)"),
#if HAVE_WORKING_FORK
N_("  -f                 keep the program in foreground"),
#endif
N_("  -h                 display this help and exit"),
N_("  -V                 print version information and exit"),
#endif
"",
NULL
	};

#if HAVE_ASSERT_H
	assert (configuration_file != NULL);
#endif

	fprintf (stderr, _("Usage: %s [OPTION]...\n"),
		 program_invocation_name);

	for (i = 0; msg[i] != NULL; i++) {
		if (msg[i][0] != '\0') {
			fputs (_(msg[i]), stderr);
		}
		fputc ('\n', stderr);
	}

	fputs (	_("The default configuration file is "), stderr);
	fputs (configuration_file, stderr);
	fputc ('\n', stderr);
	fprintf (stderr,
		 _("Exit status is %d on error (%d means no error).\n"),
		 EXIT_FAILURE, EXIT_SUCCESS);
	fputc ('\n', stderr);
	fputs (_("Report bugs to "), stderr);
	fputs (PACKAGE_BUGREPORT, stderr);
	fputc ('\n', stderr);
}


/*
 * Signal handler for SIGHUP
 */
static void
sighup_action (int sig)
{
	/*
	 * Before anything else be sure that the configuration file
	 * can be read
	 */
	if (access (configuration_file, R_OK) != 0) {
		lwc_writeLog (	LOG_ERR, _("%s: %s"),
				configuration_file, strerror (errno));
		return;
	}

	lwc_closeLog ();
	close_pid_file ();
	net_destroy_ssl ();
	conf_destroy_srv ();
	module_exit ();
	sanitise ();
#if HAVE_CHDIR && HAVE_GETCWD
	if (save_cwd[0] != '\0') {
		chdir (save_cwd);
	}
#endif
	execve (save_argv[0], save_argv, save_env);
	error (EXIT_FAILURE, errno, "execve");
}


/*
 * Signal handler for SIGTERM
 */
static void
sigterm_action (int sig)
{
	int i;

	lwc_writeLog (LOG_INFO, _("schedwireg exiting"));
	for (i = 0; save_env[i] != NULL; i++) {
		free (save_env[i]);
	}
	free (save_env);
	free (save_argv);
	exit (EXIT_SUCCESS);
}


/*
 * Wait until the CA certificate file is available (it is generated by
 * schedwisrv at first start)
 *
 * Return:
 *   0 --> OK
 *  -1 --> the certificate file is there but not readable
 */
static int
wait_for_ca_cert (const char *ca_crt)
{
	int count;
#if HAVE_NANOSLEEP
	struct timespec req;


	req.tv_sec = 10;
	req.tv_nsec = 0;
#endif
	count = 0;
	while (1) {
		if (access (ca_crt, F_OK) == 0) {
			if (access (ca_crt, R_OK) == 0) {
				return 0;
			}
			lwc_writeLog (LOG_ERR,
			_("%s (SSLCACertificateFile) is not readable: %s\n"),
				ca_crt, strerror (errno));
			return -1;
		}
		if (count == 0) {
			lwc_writeLog (LOG_INFO,
_("The CA certficate file %s (SSLCACertificateFile) is missing"),
				ca_crt);
			lwc_writeLog (LOG_INFO,
_("This file should automatically be generated by schedwisrv at first start"));
			lwc_writeLog (LOG_INFO,
_("This is a long process that may take several minutes"));
			lwc_writeLog (LOG_INFO,
_("I'll recheck a little bit latter"));
		}
		else
		if (count % 60 == 0) {
			lwc_writeLog (LOG_INFO,
			_("CA certficate file still not there.  Waiting..."));
		}
		count++;
#if HAVE_NANOSLEEP
		nanosleep (&req, NULL);
#elif HAVE_SLEEP
		sleep (10);
#elif HAVE_USLEEP
		usleep (10 * 1000000);
#endif
	}
}


/*
 * Main function
 *
 * The exit code is 1 in case of error or 0 if killed (SIGTERM)
 */
int
main (int argc, char **argv)
{
	fd_set sock_set;
	int sock_max, in_sock;
	char *buff;
	size_t buff_len;
	int i, ret;
	const char *group_id, *user_id;
	const char *client_cert_dir, *client_csr_dir, *client_savecsr_dir;
	const char *ca_crt;
	struct sigaction sa;
	const char *pid_file, *log_file, *date_format;
	const char *iface, *registrar_port, *address_family, *allow_from;
	long int facility;
	char test, foreground;
	char **client_names;
#if HAVE_GETOPT_LONG
	int option_index;
	struct option long_options[] =
	{
		{"help",       0, 0, 'h'},
		{"version",    0, 0, 'V'},
		{"test",       0, 0, 't'},
		{"foreground", 0, 0, 'f'},
		{"config",     1, 0, 'c'},
		{0, 0, 0, 0}
	};
#endif


#if HAVE_SETLOCALE
	setlocale (LC_ALL, "");
#endif
#if HAVE_BINDTEXTDOMAIN
	bindtextdomain (PACKAGE, LOCALEDIR);
#endif
#if HAVE_TEXTDOMAIN
	textdomain (PACKAGE);
#endif

	/* Set default values for options */
	test = foreground = 0;
	configuration_file = SCHEDWI_DEFAULT_CONFFILE_SRV;


	/* Parse options */
	while (1) {
#if HAVE_GETOPT_LONG
		option_index = 0;
		ret = getopt_long (argc, argv, "hVtfc:",
					long_options, &option_index);
#else
		ret = getopt (argc, argv, "hVtfc:");
#endif

		if (ret == -1) {
			break;
		}

		switch (ret) {
			case 'h':
				help ();
				return EXIT_SUCCESS;
			case 'V':
				version ();
				return EXIT_SUCCESS;
			case 't':
				test = 1;
				break;
			case 'f':
				foreground = 1;
				break;
			case 'c':
				configuration_file = optarg;
				break;
			default:
				help ();
				return EXIT_FAILURE;
		}
	}

	if (test == 1) {
		printf (_("Configuration file: %s\n"), configuration_file);
	}

	/*
	 * Initialize the module engine
	 */
	if (module_init () != 0) {
		return EXIT_FAILURE;
	}
	atexit (module_exit);

	/*
	 * Read the configuration file
	 */
	ret = conf_init_srv (configuration_file);
	if (ret == -2) {
		error (EXIT_FAILURE, errno, "%s", configuration_file);
	}
	if (ret != 0) {
		return EXIT_FAILURE;
	}
	atexit (conf_destroy_srv);

	/*
	 * Close opened files (but keep the standard fd 0, 1 and 2 because
	 * some error messages may be printed on stderr)
	 * This has to be done before the net_init_gnutls() function below
	 * because the gnutls_global_init() library call will open a file
	 * descriptor on /dev/urandom.
	 */
	close_opened_files ();

	/*
	 * Initialize the GnuTLS layer
	 */
	if (net_init_gnutls (1) != 0) {
		return EXIT_FAILURE;
	}
	atexit (net_destroy_ssl);

	/*
	 * For a daemon that is going to run in background, create a PID file
	 * and change owner.  This is for sysV daemons.
	 * Systemd daemons are started in the foreground (systemd will set
	 * the owner and manage the PID file - see the daemon man page for
	 * systemd)
	 */
	if (foreground == 0) {

		/*
		 * Create the directory that will hold the PID file
		 */
		ret =  conf_get_param_string ("USER", &user_id);
		ret += conf_get_param_string ("GROUP", &group_id);
		ret += conf_get_param_string ("PID_FILE_REGISTRAR", &pid_file);
#if HAVE_ASSERT_H
		assert (ret == 0);
#endif

		/*
		 * Create the directory that will hold the PID file
		 */
		mkpath (pid_file, user_id, group_id);
	
		/*
		 * Change the group id and the user id of the process
		 */
		ret = change_group (group_id);
		if (ret != 0) {
			if (ret == -2) {
				fprintf (stderr,
					_("Unknown group name: %s\n"),
					group_id);
			}
			else {
				fprintf (stderr,
				_("Failed to change the group ID to %s: %s\n"),
					group_id, strerror (errno));
			}
			return EXIT_FAILURE;
		}
	
		ret = change_user (user_id);
		if (ret != 0) {
			if (ret == -2) {
				fprintf (stderr,
					_("Unknown user name: %s\n"),
					user_id);
			}
			else {
				fprintf (stderr,
				_("Failed to change the user ID to %s: %s\n"),
					user_id, strerror (errno));
			}
			return EXIT_FAILURE;
		}
	}

	/*
	 * Check if we can write in the working directories and read the
	 * certificate authority files
	 */
	ret =  conf_get_param_string (	"SSLAgentCertificateDir",
					&client_cert_dir);
	ret += conf_get_param_string (	"SSLAgentRequestDir",
					&client_csr_dir);
	ret += conf_get_param_string (	"SSLAgentSaveRequestDir",
					&client_savecsr_dir);
	ret += conf_get_param_string ("SSLCACertificateFile", &ca_crt);

#if HAVE_ASSERT_H
	assert (ret == 0);
#endif
	if (access (client_cert_dir, R_OK|W_OK|X_OK) != 0) {
		fprintf (stderr,
		_("Cannot write in %s (SSLAgentCertificateDir): %s\n"),
			client_cert_dir, strerror (errno));
		ret = 1;
	}

	if (access (client_csr_dir, R_OK|W_OK|X_OK) != 0) {
		fprintf (stderr,
			_("Cannot write in %s (SSLAgentRequestDir): %s\n"),
			client_csr_dir, strerror (errno));
		ret = 1;
	}

	if (access (client_savecsr_dir, R_OK|W_OK|X_OK) != 0) {
		fprintf (stderr,
		_("Cannot write in %s (SSLAgentSaveRequestDir): %s\n"),
			client_savecsr_dir, strerror (errno));
		ret = 1;
	}

	if (test == 1) {
		if (access (ca_crt, R_OK) != 0) {
			fprintf (stderr,
			_("Cannot read %s (SSLCACertificateFile): %s\n"),
				ca_crt, strerror (errno));
			ret = 1;
		}
	}

	if (ret != 0) {
		return EXIT_FAILURE;
	}

	/* It was only a test of the configuration file */
	if (test == 1) {
		fputs (_("Fine."), stdout);
		fputc ('\n', stdout);
		return EXIT_SUCCESS;
	}

	/* Copy argv and environ to use them to exec the process on SIGHUP */
	save_argv = (char **) xmalloc (sizeof (char *) * (argc + 1));
	for (i = 0; i < argc; i++) {
		save_argv[i] = argv[i];
	}
	save_argv[i] = NULL;

	for (i = 0; environ[i] != NULL; i++);
	save_env = (char **) xmalloc (sizeof (char *) * (i + 1));
	for (i = 0; environ[i] != NULL; i++) {
		save_env[i] = xstrdup (environ[i]);
	}
	save_env[i] = NULL;

#if HAVE_CHDIR && HAVE_GETCWD
	if (getcwd (save_cwd, MAX_SIZE_CWD) == NULL) {
		save_cwd[0] = '\0';
	}
#endif


	/*
	 * Go background
	 */
	if (foreground == 0 && go_daemon () != 0) {
		error (EXIT_FAILURE, errno, "fork");
	}


	/*
	 * For sysV daemons, open the log file and write the PID file.
	 * For systemd service (foreground = 1), systemd will already have done
	 * all that.
	 */
	if (foreground == 0) {

		/*
		 * Open the log file
		 */
		log_file = date_format = NULL;
		facility = -1;
		ret = conf_get_param_string ("LOG_FILE", &log_file);
		ret += conf_get_param_syslog_facility ("SYSLOG", &facility);
		ret += conf_get_param_string ("DATE_FORMAT", &date_format);
#if HAVE_ASSERT_H
		assert (ret == 0);
#endif
	
		lwc_newLog (    PACKAGE_NAME,
				(facility != -1) ? 1 : 0,
				(facility != -1) ? (int)facility : 0,
				log_file,
				date_format);
		atexit (lwc_closeLog);
		lwc_writeLog (LOG_INFO, _("schedwireg starting"));
	
	
		/*
		 * Create the PID file and check that an other process is not
		 * already running
		 */
		ret = write_pid_file (pid_file);
		switch (ret) {
			case -1:
				lwc_writeLog (	LOG_CRIT,
					_("Cannot create the PID file %s: %s"),
						pid_file,
						strerror (errno));
				break;
	
			case 1:
				lwc_writeLog (LOG_CRIT,
				_("An other daemon is already running"));
				break;
		}
		if (ret != 0) {
			for (i = 0; save_env[i] != NULL; i++) {
				free (save_env[i]);
			}
			free (save_env);
			free (save_argv);
			return EXIT_FAILURE;
		}
		atexit (close_pid_file);
	}


	/*
	 * Signal handlers
	 */
	memset (&sa, 0, sizeof (struct sigaction));
	sa.sa_handler = sigterm_action;
	sigemptyset (&(sa.sa_mask));
	sa.sa_flags = 0;
	if (sigaction (SIGTERM, &sa, NULL) != 0) {
		lwc_writeLog (
			LOG_CRIT,
			_("Cannot install signal handler for SIGTERM: %s"),
			strerror (errno));
		for (i = 0; save_env[i] != NULL; i++) {
			free (save_env[i]);
		}
		free (save_env);
		free (save_argv);
		return EXIT_FAILURE;
	}

	memset (&sa, 0, sizeof (struct sigaction));
	sa.sa_handler = sighup_action;
	sigemptyset (&(sa.sa_mask));
	sa.sa_flags = 0;
	if (sigaction (SIGHUP, &sa, NULL) != 0) {
		lwc_writeLog (
			LOG_CRIT,
			_("Cannot install signal handler for SIGHUP: %s"),
			strerror (errno));
		for (i = 0; save_env[i] != NULL; i++) {
			free (save_env[i]);
		}
		free (save_env);
		free (save_argv);
		return EXIT_FAILURE;
	}

	memset (&sa, 0, sizeof (struct sigaction));
	sa.sa_handler = SIG_IGN;
	sigemptyset (&(sa.sa_mask));
	sa.sa_flags = 0;
	if (sigaction (SIGPIPE, &sa, NULL) != 0) {
		lwc_writeLog (
			LOG_CRIT,
			_("Cannot install signal handler for SIGPIPE: %s"),
			strerror (errno));
		for (i = 0; save_env[i] != NULL; i++) {
			free (save_env[i]);
		}
		free (save_env);
		free (save_argv);
		return EXIT_FAILURE;
	}


	/*
	 * Wait until the CA certificate file is available
	 */
	if (wait_for_ca_cert (ca_crt) != 0) {
		for (i = 0; save_env[i] != NULL; i++) {
			free (save_env[i]);
		}
		free (save_env);
		free (save_argv);
		return EXIT_FAILURE;
	}


	/*
	 * Network setup
	 */
	ret = conf_get_param_string ("IFACE_LISTEN", &iface);
	ret += conf_get_param_string ("REGISTRAR_PORT", &registrar_port);
	ret += conf_get_param_string ("ADDRESS_FAMILY", &address_family);
#if HAVE_ASSERT_H
	assert (ret == 0);
#endif

	FD_ZERO (&sock_set);
	sock_max = -1;
	if (net_server (registrar_port, iface, address_family,
			&sock_set, &sock_max) != 0)
	{
		for (i = 0; save_env[i] != NULL; i++) {
			free (save_env[i]);
		}
		free (save_env);
		free (save_argv);
		return EXIT_FAILURE;
	}


	/*
	 * Main loop
	 */
	buff = NULL;
	buff_len = 0;
	do {
		ret = conf_get_param_string ("ALLOW_FROM", &allow_from);
#if HAVE_ASSERT_H
		assert (ret == 0);
#endif
		/* Wait for connections */
		client_names = NULL;
		in_sock = net_accept_sock (	&sock_set, sock_max,
						allow_from, &client_names);
		if (in_sock < 0) {
			if (buff != NULL) {
				free (buff);
			}
			for (i = 0; save_env[i] != NULL; i++) {
				free (save_env[i]);
			}
			free (save_env);
			free (save_argv);
			net_close_sock_set (&sock_set, &sock_max);
			return EXIT_FAILURE;
		}

		/* Parse the request */
		signal_mask_all ();
		net_parse_reg (in_sock, &buff, &buff_len,
				(const char *const *)client_names);
		destroy_client_names (client_names);
		net_close_sock (in_sock);
		signal_unmask ();
	} while (1);
	return EXIT_SUCCESS;
}

/*------------------------======= End Of File =======------------------------*/
