/* Schedwi
   Copyright (C) 2007-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* schedwiping.c -- Test the connection with agents (schedwiclnt) */

#include <schedwi.h>

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif

#if HAVE_SIGNAL_H
#include <signal.h>
#endif

#if HAVE_GETOPT_H
#include <getopt.h>
#endif

#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <error.h>

#include <sql_common.h>
#include <utils.h>
#include <conf.h>
#include <conf_srv.h>
#include <module.h>
#include <net_utils.h>
#include <net_utils_ssl.h>
#include <ping.h>

static const char *configuration_file = NULL;


/*
 * Print a help message to stderr
 */
void
help ()
{
	int i;
	char * msg[] = {
N_("Test the connection to a Schedwi agent (schedwiclnt)."),
N_("This program must be run on the same host as the Schedwi server."),
"",
#if HAVE_GETOPT_LONG
N_("  -c, --config=FILE  use the configuration file FILE rather than"),
N_("                     the default one"),
N_("  -h, --help         display this help and exit"),
N_("  -V, --version      print version information and exit"),
#else /* HAVE_GETOPT_LONG */
N_("  -c FILE            use the configuration file FILE rather than"),
N_("                     the default one"),
N_("  -h                 display this help and exit"),
N_("  -V                 print version information and exit"),
#endif /* ! HAVE_GETOPT_LONG */
"",
NULL
	};

#if HAVE_ASSERT_H
	assert (configuration_file != NULL);
#endif

	fprintf (stderr, _("Usage: %s [OPTION]... HOST[:PORT]\n"),
			program_invocation_name);

	for (i = 0; msg[i] != NULL; i++) {
		if (msg[i][0] != '\0') {
			fputs (_(msg[i]), stderr);
		}
		fputc ('\n', stderr);
	}

	fputs (	_("The default configuration file is "), stderr);
	fputs (configuration_file, stderr);
	fputc ('\n', stderr);
	fprintf (stderr,
		 _("Exit status is %d on error (%d means no error).\n"),
		 EXIT_FAILURE, EXIT_SUCCESS);
	fputc ('\n', stderr);
	fputs (_("Report bugs to "), stderr);
	fputs (PACKAGE_BUGREPORT, stderr);
	fputc ('\n', stderr);
}


/*
 * Main function
 *
 * The exit code is 1 in case of error or 0 on success
 */
int
main (int argc, char **argv)
{
	const char *server_key, *server_crt, *ca_crt, *ca_crl;
	char quick_random;
	char *agent_name;
	int ret;
	dbi_conn sql_handler;
	char *err_msg;
#if HAVE_GETOPT_LONG
	int option_index;
	struct option long_options[] =
	{
		{"help",       0, 0, 'h'},
		{"version",    0, 0, 'V'},
		{"config",     1, 0, 'c'},
		{0, 0, 0, 0}
	};
#endif


#if HAVE_SETLOCALE
	setlocale (LC_ALL, "");
#endif
#if HAVE_BINDTEXTDOMAIN
	bindtextdomain (PACKAGE, LOCALEDIR);
#endif
#if HAVE_TEXTDOMAIN
	textdomain (PACKAGE);
#endif

	/* Set default values for options */
	configuration_file = SCHEDWI_DEFAULT_CONFFILE_SRV;

	/* Parse options */
	while (1) {
#if HAVE_GETOPT_LONG
		option_index = 0;
		ret = getopt_long (argc, argv, "hVc:",
					long_options, &option_index);
#else
		ret = getopt (argc, argv, "hVc:");
#endif

		if (ret == -1) {
			break;
		}

		switch (ret) {
			case 'h':
				help ();
				return EXIT_SUCCESS;
			case 'V':
				version ();
				return EXIT_SUCCESS;
			case 'c':
				configuration_file = optarg;
				break;
			default:
				help ();
				return EXIT_FAILURE;
		}
	}

	/* Agent host name missing */
	if (optind >= argc) {
		fputs (_("Agent host name required\n"), stderr);
		help ();
		return EXIT_FAILURE;
	}

	/* Too many parameters */
	if (optind + 1 != argc) {
		fputs (_("Too many parameters\n"), stderr);
		help ();
		return EXIT_FAILURE;
	}

	agent_name = argv[optind];

	/*
	 * Initialize the module engine
	 */
	if (module_init () != 0) {
		return EXIT_FAILURE;
	}

	/*
	 * Read the configuration file
	 */
	ret = conf_init_srv (configuration_file);
	if (ret == -2) {
		error (EXIT_FAILURE, errno, "%s", configuration_file);
	}
	if (ret != 0) {
		module_exit ();
		return EXIT_FAILURE;
	}

	ret =  conf_get_param_string ("SSLCACertificateFile", &ca_crt);
	ret += conf_get_param_string ("SSLCertificateFile", &server_crt);
	ret += conf_get_param_string ("SSLCertificateKeyFile", &server_key);
	ret += conf_get_param_string ("SSLCACRLFile", &ca_crl);
	ret += conf_get_param_bool ("SSLQuickRandom", &quick_random);
#if HAVE_ASSERT_H
	assert (ret == 0);
#endif

	/*
	 * Initialize the GnuTLS layer
	 */
	if (net_init_gnutls (quick_random) != 0) {
		conf_destroy_srv ();
		module_exit ();
		return EXIT_FAILURE;
	}

	/*
	 * Initialize the network
	 */
	if (net_init (	server_crt, server_key, ca_crt, ca_crl,
			 quick_random) != 0)
	{
		net_destroy ();
		conf_destroy_srv ();
		module_exit ();
		return EXIT_FAILURE;
	}

	/*
	 * Database connection
	 */
	err_msg = NULL;
	sql_handler = begin_sql (&err_msg);
	if (sql_handler == NULL) {
		if (err_msg != NULL) {
			fprintf (stderr,
				_("Failed to connect to database: %s\n"),
				err_msg);
			free (err_msg);
		}
		else {
			fputs (_("Failed to connect to database\n"), stderr);
		}
		net_destroy ();
		conf_destroy_srv ();
		module_exit ();
		return EXIT_FAILURE;
	}

	ret = ping_by_names (agent_name);
	end_sql ();
	net_destroy ();
	conf_destroy_srv ();
	module_exit ();
	if (ret == 0) {
		fputs (_("The agent is running.\n"), stdout);
		return EXIT_SUCCESS;
	}

	return EXIT_FAILURE;
}

/*------------------------======= End Of File =======------------------------*/
