/* Schedwi
   Copyright (C) 2007-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* schedwidbchecks.c -- Check and repair database consistency */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_TIME_H
#include <time.h>
#endif
#if TM_IN_SYS_TIME
#include <sys/time.h>
#endif

#if HAVE_GETOPT_H
#include <getopt.h>
#endif

#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <error.h>

#include <sql_common.h>
#include <utils.h>
#include <conf.h>
#include <conf_srv.h>
#include <module.h>
#include <xmem.h>
#include <schedwi_time.h>
#include <sql_status.h>
#include <sql_db_checks.h>


#define LOST_FOUND_STRFTIME "schedwidbchecks - %c"

typedef enum
{
	NORMAL,
	QUIET,
	VERBOSE
} levels;

static levels verbose_level = NORMAL;
static const char *configuration_file = NULL;
static char autofix = 0;
static char *lost_found_name;
static unsigned long long int lost_found_jobset_id = 0;
static unsigned long long int lost_found_calendar_folder_id = 0;
static int x, y;


/*
 * Print a help message to stderr
 */
void
help ()
{
	int i;
	char * msg[] = {
N_("Check the Schedwi database consistency."),
N_("This program must be run on the same host as the Schedwi server."),
"",
#if HAVE_GETOPT_LONG
N_("  -c, --config=FILE  use the configuration file FILE rather than"),
N_("                     the default one"),
N_("  -h, --help         display this help and exit"),
N_("  -V, --version      print version information and exit"),
N_("  -q, --quiet        suppress all normal output"),
N_("  -v, --verbose      verbose database checks"),
N_("  -f, --fix          try to repair the database"),
#else /* HAVE_GETOPT_LONG */
N_("  -c FILE            use the configuration file FILE rather than"),
N_("                     the default one"),
N_("  -h                 display this help and exit"),
N_("  -V                 print version information and exit"),
N_("  -q                 suppress all normal output"),
N_("  -v                 verbose database checks"),
N_("  -f                 try to repair the database"),
#endif /* ! HAVE_GETOPT_LONG */
"",
NULL
	};

#if HAVE_ASSERT_H
	assert (configuration_file != NULL);
#endif

	fprintf (stderr, _("Usage: %s [OPTION]...\n"),
			program_invocation_name);

	for (i = 0; msg[i] != NULL; i++) {
		if (msg[i][0] != '\0') {
			fputs (_(msg[i]), stderr);
		}
		fputc ('\n', stderr);
	}

	fputs (	_("The default configuration file is "), stderr);
	fputs (configuration_file, stderr);
	fputc ('\n', stderr);
	fprintf (stderr,
 _("Exit status is %d on error and 2 in case of database inconsistency\n"),
 		EXIT_FAILURE);
	fprintf (stderr, _("(%d means no error).\n"), EXIT_SUCCESS);
	fputc ('\n', stderr);
	fputs (_("Report bugs to "), stderr);
	fputs (PACKAGE_BUGREPORT, stderr);
	fputc ('\n', stderr);
}


/*
 * Error callback function for the sql functions
 */
static void
sql_error_logger (void *data, const char *msg, int err_code)
{
	if (msg != NULL) {
		fputs (msg, stderr);
		fputc ('\n', stderr);
	}
	else {
		fputs (_("Database error\n"), stderr);
	}
}


/*
 * Callback for the sql_job_main_check_parent() function when an error
 * is detected
 */
static int
check_job_main_cb_parent (	void *data, unsigned long long int id,
				const char *name,
				char job_type, const char *description)
{
	switch (verbose_level) {
		case NORMAL:
			printf (_("%s: %s (%lld): error: orphan\n"),
				(job_type == 0) ? _("Jobset") : _("Job"),
				name, id);
			break;
		case VERBOSE:
			printf (_("%s: %s (%lld): error: orphan\n"),
				(job_type == 0) ? _("Jobset") : _("Job"),
				name, id);
			if (description != NULL && description[0] != '\0') {
				fputc ('\t', stdout);
				fputs (description, stdout);
				fputc ('\n', stdout);
			}
			break;
		default:
			break;
	}
	if (autofix == 1) {
		switch (verbose_level) {
			case NORMAL:
			case VERBOSE:
				printf (
				_("\tRe-attaching this orphan to `/%s'.\n"),
					lost_found_name);
				break;
			default:
				break;
		}
		if (	   lost_found_jobset_id != 0
			|| sql_check_create_lost_found_jobset (
				lost_found_name,
	_("Created by schedwidbchecks to receive orphan jobs and jobsets"),
				x, y, &lost_found_jobset_id,
				sql_error_logger, NULL) == 0)
		{
			sql_check_attach_orphan_job (id,
						lost_found_jobset_id,
						sql_error_logger, NULL);
		}
	}
	return 0;
}


/*
 * Callback for the sql_job_main_check_calendar() function when an error
 * is detected
 */
static int
check_job_main_cb_calendar (	void *data, unsigned long long int id,
				const char *name,
				char job_type, const char *description)
{
	switch (verbose_level) {
		case NORMAL:
			printf (_("%s: %s (%lld): error: unknown calendar\n"),
				(job_type == 0) ? _("Jobset") : _("Job"),
				name, id);
			break;
		case VERBOSE:
			printf (_("%s: %s (%lld): error: unknown calendar\n"),
				(job_type == 0) ? _("Jobset") : _("Job"),
				name, id);
			if (description != NULL && description[0] != '\0') {
				fputc ('\t', stdout);
				fputs (description, stdout);
				fputc ('\n', stdout);
			}
			break;
		default:
			break;
	}
	if (autofix == 1) {
		switch (verbose_level) {
			case NORMAL:
			case VERBOSE:
				fputs (_("\tFixing (using parent calendar)\n"),
					stdout);
				break;
			default:
				break;
		}
		sql_check_fix_cal_id (id, sql_error_logger, NULL);
	}
	return 0;
}


/*
 * Callback for the sql_job_main_check_time() function when an error
 * is detected
 */
static int
check_job_main_cb_time (void *data, unsigned long long int id,
			const char *name, char job_type,
			const char *description, int start_time)
{
	switch (verbose_level) {
		case NORMAL:
			printf (
_("%s: %s (%lld): error: wrong start time `%d' (must be between -1 and 2359)\n"),
				(job_type == 0) ? _("Jobset") : _("Job"),
				name, id, start_time);
			break;
		case VERBOSE:
			printf (
_("%s: %s (%lld): error: wrong start time `%d' (must be between -1 and 2359)\n"),
				(job_type == 0) ? _("Jobset") : _("Job"),
				name, id, start_time);
			if (description != NULL && description[0] != '\0') {
				fputc ('\t', stdout);
				fputs (description, stdout);
				fputc ('\n', stdout);
			}
			break;
		default:
			break;
	}
	if (autofix == 1) {
		switch (verbose_level) {
			case NORMAL:
			case VERBOSE:
				fputs (
				_("\tFixing (using parent start time)\n"),
					stdout);
				break;
			default:
				break;
		}
		sql_check_fix_start_time (id, sql_error_logger, NULL);
	}
	return 0;
}


/*
 * Check the job main parameters (parent, calendar and start time value)
 */
static int
check_job_main ()
{
	int ret;

	if (verbose_level == VERBOSE) {
		fputs (_("Checking jobs/jobsets main parameters:\n"), stdout);
	}

	ret = 0;
	switch (sql_job_main_check_parent (	check_job_main_cb_parent,
						NULL, sql_error_logger, NULL))
	{
		case 0: break;
		case 2: ret = 2; break;
		default: return -1;
	}

	switch (sql_job_main_check_calendar (	check_job_main_cb_calendar,
						NULL, sql_error_logger, NULL))
	{
		case 0: break;
		case 2: ret = 2; break;
		default: return -1;
	}

	switch (sql_job_main_check_time (	check_job_main_cb_time,
						NULL, sql_error_logger, NULL))
	{
		case 0: break;
		case 2: ret = 2; break;
		default: return -1;
	}

	if (ret == 0 && verbose_level == VERBOSE) {
		fputs (_("\tOK\n"), stdout);
	}

	return ret;
}



/*
 * Database tables to check.  The job_id field is checked to see if the
 * associated job still exists
 */
static char *job_tables[][3] = {
  { "job_start_limit",	"start_limit",		N_("start limits") },
  { "job_max_duration",	"max_duration",		N_("max durations") },
  { "job_retries",	"retries",		N_("number of retries") },
  { "job_retries_interval", "retries_interval",	N_("retry intervals") },
  { "job_username",	"username",		N_("execution user names") },
  { "job_file_out",	"file_out",		N_("output file names") },
  { "job_file_err",	"file_err",		N_("error file names") },
  { "job_control_group", "control_group",	N_("Linux Control Group") },
  { "job_loadenv",	"loadenv",	N_("load user environment variables") },
  { "job_detach",	"detach",	N_("start job in detach mode") },
  { "job_manual_command", "manual_command", N_("manual job alert command") },
  { "job_success_return_code",	"success_ret",	N_("return codes") },
  { "job_command",	"command",		N_("remote commands") },
  { "job_arguments",	"argument",		N_("command parameters") },
  { "jobset_background",	"0",		N_("background images") },
  { "job_icon_default",		"0",	N_("default icons for jobs") },
  { "job_icon_completed",	"0",	N_("completed icons for jobs") },
  { "job_icon_failed",		"0",	N_("failed icons for jobs") },
  { "job_icon_running",		"0",	N_("running icons for jobs") },
  { "job_icon_waiting",		"0",	N_("waiting icons for jobs") },
  { "jobset_icon_default",	"0",	N_("default icons for jobsets") },
  { "jobset_icon_completed",	"0",	N_("completed icons for jobsets") },
  { "jobset_icon_failed",	"0",	N_("failed icons for jobsets") },
  { "jobset_icon_running",	"0",	N_("running icons for jobsets") },
  { "jobset_icon_waiting",	"0",	N_("waiting icons for jobsets") },
  { "job_stat",		"num_success",	N_("job statistics") },
  { NULL, NULL, NULL }
};


/*
 * Callback for the sql_check_id() function when an error is detected
 */
static int
check_job_parameters_cb (void *data, row_item_t *id, row_item_t *value)
{
	char *table_name = (char *)data;
	unsigned long long int jobid;
	long long int v;

	jobid = (unsigned long long int) sql_row_item2ll (id);
	switch (verbose_level) {
		case NORMAL:
		case VERBOSE:
			if (value->type == RES_INT || value->type == RES_UINT)
			{
				v = sql_row_item2ll (value);
				if (v != 0) {
					printf (
		_("table %s: error: unknown job Id `%lld' for value `%lld'\n"),
						table_name, jobid, v);
				}
				else {
					printf (
				_("table %s: error: unknown job Id `%lld'\n"),
						table_name, jobid);
				}
			}
			else
			if (value->type == RES_STRING) {
				if (	   value->value_string != NULL
					&& (value->value_string)[0] != '\0')
				{
					printf (
		_("table %s: error: unknown job Id `%lld' for value `%s'\n"),
					table_name, jobid, value->value_string);
				}
				else {
					printf (
				_("table %s: error: unknown job Id `%lld'\n"),
						table_name, jobid);
				}
			}
			else {
				printf (
				_("table %s: error: unknown job Id `%lld'\n"),
					table_name, jobid);
			}
			break;

		default:
			break;
	}

	if (autofix == 1) {
		switch (verbose_level) {
			case NORMAL:
			case VERBOSE:
				fputs (_("\tDeleting.\n"), stdout);
				break;
			default:
				break;
		}
		sql_check_delete1 (	table_name, "job_id", jobid,
					sql_error_logger, NULL);
	}
	return 0;
}


/*
 * Check that the job parameters are associated with an existing job
 */
static int
check_job_parameters ()
{
	int ret, global_ret;
	unsigned int i;

	global_ret = 0;
	for (i = 0; job_tables[i][0] != NULL; i++) {
		ret = 0;
		if (verbose_level == VERBOSE) {
			printf (_("Checking %s:\n"), job_tables[i][2]);
		}

		switch (sql_check_id (	job_tables[i][0], "job_id",
					"job_main", "id", job_tables[i][1],
					check_job_parameters_cb,
					job_tables[i][0],
					sql_error_logger, NULL))
		{
			case 0: break;
			case 2: ret = global_ret = 2; break;
			default: return -1;
		}
		if (ret == 0 && verbose_level == VERBOSE) {
			fputs (_("\tOK\n"), stdout);
		}
	}
	return global_ret;
}

#ifdef NO_NEED_TO_CHECK
/*
 * Callback for the sql_hosts_check_cert() function when an error
 * is detected
 */
static int
check_hosts_cb_cert (	void *data, unsigned long long int id, const char *name,
			const char *tcp_port, const char *description)
{
	switch (verbose_level) {
		case NORMAL:
			printf (
_("host: %s (port %s): error: SSL enabled but no SSL certificate provided\n"),
				name, tcp_port);
			break;
		case VERBOSE:
			printf (
_("host: %s (port %s): error: SSL enabled but no SSL certificate provided\n"),
				name, tcp_port);
			if (description != NULL && description[0] != '\0') {
				fputc ('\t', stdout);
				fputs (description, stdout);
				fputc ('\n', stdout);
			}
			break;
		default:
			break;
	}
	if (autofix == 1) {
		switch (verbose_level) {
			case NORMAL:
			case VERBOSE:
				fputs (_("\tDisabling SSL.\n"), stdout);
				break;
			default:
				break;
		}
		sql_check_disable_ssl (id, sql_error_logger, NULL);
	}
	return 0;
}


/*
 * Check the hosts SSL parameters
 */
static int
check_hosts ()
{
	int ret;

	if (verbose_level == VERBOSE) {
		fputs (_("Checking hosts parameters:\n"), stdout);
	}

	ret = 0;
	switch (sql_hosts_check_cert (	check_hosts_cb_cert,
					NULL, sql_error_logger, NULL))
	{
		case 0: break;
		case 2: ret = 2; break;
		default: return -1;
	}

	if (ret == 0 && verbose_level == VERBOSE) {
		fputs (_("\tOK\n"), stdout);
	}

	return ret;
}
#endif


/*
 * Callback for the sql_calendars_check_parent() function when an error
 * is detected
 */
static int
check_calendars_cb_parent (	void *data, unsigned long long int id,
				const char *name,
				char cal_type, const char *description)
{
	switch (verbose_level) {
		case NORMAL:
			printf (_("%s: %s (%lld): error: orphan\n"),
				(cal_type == 0) ?	_("Calendar") :
							_("Calendar folder"),
				name, id);
			break;
		case VERBOSE:
			printf (_("%s: %s (%lld): error: orphan\n"),
				(cal_type == 0) ?	_("Calendar") :
							_("Calendar folder"),
				name, id);
			if (description != NULL && description[0] != '\0') {
				fputc ('\t', stdout);
				fputs (description, stdout);
				fputc ('\n', stdout);
			}
			break;
		default:
			break;
	}
	if (autofix == 1) {
		switch (verbose_level) {
			case NORMAL:
			case VERBOSE:
				printf (
				_("\tRe-attaching this orphan to `/%s'.\n"),
					lost_found_name);
				break;
			default:
				break;
		}
		if (	   lost_found_calendar_folder_id != 0
			|| sql_check_create_lost_found_cal_folder (
				lost_found_name,
		_("Created by schedwidbchecks to receive orphan calendars"),
				&lost_found_calendar_folder_id,
				sql_error_logger, NULL) == 0)
		{
			sql_check_attach_orphan_cal (id,
						lost_found_calendar_folder_id,
						sql_error_logger, NULL);
		}
	}
	return 0;
}


/*
 * Comment out (add a `#') all the lines of the provided string.
 * Return:
 *     A new allocated string (to be freed by the caller by free()) which
 *     contains the commented out input string
 */
static char *
comment_lines (const char *src)
{
	char *dst;
	const char *s;
	char *d, *found;
	unsigned int len;

	dst = (char *) xmalloc (strlen (src) * 2 + 2);
	dst[0] = '#';

	d = dst + 1;
	s = src;
	while (*s != '\0') {

		found = strchr (s, '\n');
		if (found == NULL) {
			strcpy (d, s);
			return dst;
		}
		len = found - s + 1;
		strncpy (d, s, len);
		d[len] = '#';
		d += len + 1;
		s = found + 1;
	}
	*d = '\0';
	return dst;
}


/*
 * Callback for the sql_calendars_check_formula() function when an error
 * is detected
 */
static int
check_calendars_cb_formula (	void *data, unsigned long long int id,
				const char *name,
				const char *description, const char *formula)
{
	char *new_formula;

	switch (verbose_level) {
		case NORMAL:
			printf (
		_("Calendar: %s (%lld): error: formula systax error `%s'\n"),
				name, id, formula);
			break;
		case VERBOSE:
			printf (
		_("Calendar: %s (%lld): error: formula systax error `%s'\n"),
				name, id, formula);
			if (description != NULL && description[0] != '\0') {
				fputc ('\t', stdout);
				fputs (description, stdout);
				fputc ('\n', stdout);
			}
			break;
		default:
			break;
	}
	if (autofix == 1) {
		new_formula = comment_lines (formula);
		if (new_formula == NULL) {
			fputs (_("Memory allocation error\n"), stderr);
			return -1;
		}
		switch (verbose_level) {
			case NORMAL:
			case VERBOSE:
				fputs (	_("\tCommenting out the formula.\n"),
					stdout);
				break;
			default:
				break;
		}
		sql_check_comment_out_formula (	id, new_formula,
						sql_error_logger, NULL);
		free (new_formula);
	}
	return 0;
}


/*
 * Check the calendars
 */
static int
check_calendars ()
{
	int ret;

	if (verbose_level == VERBOSE) {
		fputs (_("Checking calendars:\n"), stdout);
	}

	ret = 0;
	switch (sql_calendars_check_parent (	check_calendars_cb_parent,
						NULL, sql_error_logger, NULL))
	{
		case 0: break;
		case 2: ret = 2; break;
		default: return -1;
	}

	switch (sql_calendars_check_formula (	check_calendars_cb_formula,
						NULL, sql_error_logger, NULL))
	{
		case 0: break;
		case 2: ret = 2; break;
		default: return -1;
	}

	if (ret == 0 && verbose_level == VERBOSE) {
		fputs (_("\tOK\n"), stdout);
	}

	return ret;
}


/*
 * Callback functions for sql_check_id() when an error is detected
 */
static int
check_constraint_files_cb1 (void *data, row_item_t *id, row_item_t *filename)
{
	unsigned long long int jobid;

	jobid = (unsigned long long int) sql_row_item2ll (id);
	switch (verbose_level) {
		case NORMAL:
		case VERBOSE:
			if (	   filename->value_string != NULL
				&& (filename->value_string)[0] != '\0')
			{
				printf (
_("Constraint file: error: unknown job/jobset Id `%lld' for file name `%s'\n"),
					jobid, filename->value_string);
			}
			else {
				printf (
		_("Constraint file: error: unknown job/jobset Id `%lld'\n"),
					jobid);
			}
			break;
		default:
			break;
	}
	if (autofix == 1) {
		switch (verbose_level) {
			case NORMAL:
			case VERBOSE:
				fputs (_("\tDeleting.\n"), stdout);
				break;
			default:
				break;
		}
		sql_check_delete1 (	"constraint_file", "job_id", jobid,
					sql_error_logger, NULL);
	}
	return 0;
}

static int
check_constraint_files_cb2 (void *data, row_item_t *id, row_item_t *filename)
{
	unsigned long long int hostid;

	hostid = (unsigned long long int) sql_row_item2ll (id);
	switch (verbose_level) {
		case NORMAL:
		case VERBOSE:
			if (	   filename->value_string != NULL
				&& (filename->value_string)[0] != '\0')
			{
				printf (
_("Constraint file: error: unknown host Id `%lld' for file name `%s'\n"),
					hostid, filename->value_string);
			}
			else {
				printf (
			_("Constraint file: error: unknown host Id `%lld'\n"),
					hostid);
			}
			break;
		default:
			break;
	}
	if (autofix == 1) {
		switch (verbose_level) {
			case NORMAL:
			case VERBOSE:
				fputs (_("\tDeleting.\n"), stdout);
				break;
			default:
				break;
		}
		sql_check_delete1 (	"constraint_file", "host_id", hostid,
					sql_error_logger, NULL);
	}
	return 0;
}


/*
 * Check the constraint files
 */
static int
check_constraint_files ()
{
	int ret;

	if (verbose_level == VERBOSE) {
		fputs (_("Checking constraint files:\n"), stdout);
	}

	ret = 0;
	switch (sql_check_id (	"constraint_file", "job_id",
				"job_main", "id", "filename",
				check_constraint_files_cb1, NULL,
				sql_error_logger, NULL))
	{
		case 0: break;
		case 2: ret = 2; break;
		default: return -1;
	}

	switch (sql_check_id (	"constraint_file", "host_id",
				"hosts", "id", "filename",
				check_constraint_files_cb2, NULL,
				sql_error_logger, NULL))
	{
		case 0: break;
		case 2: ret = 2; break;
		default: return -1;
	}

	if (ret == 0 && verbose_level == VERBOSE) {
		fputs (_("\tOK\n"), stdout);
	}

	return ret;
}


/*
 * Callback functions for sql_check_id() when an error is detected
 */
static int
check_links_cb1 (void *data, row_item_t *id, row_item_t *job_id_dest)
{
	unsigned long long int jobid, jobid_d;

	jobid = (unsigned long long int) sql_row_item2ll (id);
	jobid_d = (unsigned long long int) sql_row_item2ll (job_id_dest);
	switch (verbose_level) {
		case NORMAL:
		case VERBOSE:
			printf (
_("Link: error: unknown source job/jobset Id `%lld' for the link between Ids `%lld' and `%lld'\n"),
				jobid, jobid, jobid_d);
			break;
		default:
			break;
	}
	if (autofix == 1) {
		switch (verbose_level) {
			case NORMAL:
			case VERBOSE:
				fputs (_("\tDeleting.\n"), stdout);
				break;
			default:
				break;
		}
		sql_check_delete1 (	"link", "job_id_source", jobid,
					sql_error_logger, NULL);
	}
	return 0;
}

static int
check_links_cb2 (void *data, row_item_t *id, row_item_t *job_id_source)
{
	unsigned long long int jobid, jobid_s;

	jobid = (unsigned long long int) sql_row_item2ll (id);
	jobid_s = (unsigned long long int) sql_row_item2ll (job_id_source);
	switch (verbose_level) {
		case NORMAL:
		case VERBOSE:
			printf (
_("Link: error: unknown destination job/jobset Id `%lld' for the link between Ids `%lld' and `%lld'\n"),
				jobid, jobid_s, jobid);
			break;
		default:
			break;
	}
	if (autofix == 1) {
		switch (verbose_level) {
			case NORMAL:
			case VERBOSE:
				fputs (_("\tDeleting.\n"), stdout);
				break;
			default:
				break;
		}
		sql_check_delete1 (	"link", "job_id_destination", jobid,
					sql_error_logger, NULL);
	}
	return 0;
}


/*
 * Check the links
 */
static int
check_links ()
{
	int ret;

	if (verbose_level == VERBOSE) {
		fputs (_("Checking links:\n"), stdout);
	}

	ret = 0;
	switch (sql_check_id (	"link", "job_id_source",
				"job_main", "id", "job_id_destination",
				check_links_cb1, NULL,
				sql_error_logger, NULL))
	{
		case 0: break;
		case 2: ret = 2; break;
		default: return -1;
	}

	switch (sql_check_id (	"link", "job_id_destination",
				"job_main", "id", "job_id_source",
				check_links_cb2, NULL,
				sql_error_logger, NULL))
	{
		case 0: break;
		case 2: ret = 2; break;
		default: return -1;
	}

	if (ret == 0 && verbose_level == VERBOSE) {
		fputs (_("\tOK\n"), stdout);
	}

	return ret;
}


/*
 * Callback functions for sql_check_id() when an error is detected
 */
static int
check_job_host_cb1 (void *data, row_item_t *id, row_item_t *host_id)
{
	unsigned long long int jobid;

	jobid = (unsigned long long int) sql_row_item2ll (id);
	switch (verbose_level) {
		case NORMAL:
		case VERBOSE:
			printf (
			_("job/host: error: unknown job/jobset Id `%lld'\n"),
				jobid);
			break;
		default:
			break;
	}
	if (autofix == 1) {
		switch (verbose_level) {
			case NORMAL:
			case VERBOSE:
				fputs (_("\tDeleting.\n"), stdout);
				break;
			default:
				break;
		}
		sql_check_delete1 (	"job_host", "job_id", jobid,
					sql_error_logger, NULL);
	}
	return 0;
}

static int
check_job_host_cb2 (void *data, row_item_t *id, row_item_t *job_id)
{
	unsigned long long int hostid;

	hostid = (unsigned long long int) sql_row_item2ll (id);
	switch (verbose_level) {
		case NORMAL:
		case VERBOSE:
			printf (_("job/host: error: unknown host Id `%lld'\n"),
				hostid);
			break;
		default:
			break;
	}
	if (autofix == 1) {
		switch (verbose_level) {
			case NORMAL:
			case VERBOSE:
				fputs (_("\tDeleting.\n"), stdout);
				break;
			default:
				break;
		}
		sql_check_delete1 (	"job_host", "host_id", hostid,
					sql_error_logger, NULL);
	}
	return 0;
}


/*
 * Check the association hosts/jobs
 */
static int
check_job_host ()
{
	int ret;

	if (verbose_level == VERBOSE) {
		fputs (_("Checking the association between jobs and hosts:\n"),
			stdout);
	}

	ret = 0;
	switch (sql_check_id (	"job_host", "job_id",
				"job_main", "id", "host_id",
				check_job_host_cb1, NULL,
				sql_error_logger, NULL))
	{
		case 0: break;
		case 2: ret = 2; break;
		default: return -1;
	}

	switch (sql_check_id (	"job_host", "host_id",
				"hosts", "id", "job_id",
				check_job_host_cb2, NULL,
				sql_error_logger, NULL))
	{
		case 0: break;
		case 2: ret = 2; break;
		default: return -1;
	}

	if (ret == 0 && verbose_level == VERBOSE) {
		fputs (_("\tOK\n"), stdout);
	}

	return ret;
}


/*
 * Callback functions for sql_check_id() when an error is detected
 */
static int
check_job_cluster_cb1 (void *data, row_item_t *id, row_item_t *host_id)
{
	unsigned long long int jobid;

	jobid = (unsigned long long int) sql_row_item2ll (id);
	switch (verbose_level) {
		case NORMAL:
		case VERBOSE:
			printf (
		_("job/cluster: error: unknown job/jobset Id `%lld'\n"),
				jobid);
			break;
		default:
			break;
	}
	if (autofix == 1) {
		switch (verbose_level) {
			case NORMAL:
			case VERBOSE:
				fputs (_("\tDeleting.\n"), stdout);
				break;
			default:
				break;
		}
		sql_check_delete1 (	"job_cluster", "job_id", jobid,
					sql_error_logger, NULL);
	}
	return 0;
}

static int
check_job_cluster_cb2 (void *data, row_item_t *id, row_item_t *job_id)
{
	unsigned long long int clusterid;

	clusterid = (unsigned long long int) sql_row_item2ll (id);
	switch (verbose_level) {
		case NORMAL:
		case VERBOSE:
			printf (
			_("job/cluster: error: unknown cluster Id `%lld'\n"),
				clusterid);
			break;
		default:
			break;
	}
	if (autofix == 1) {
		switch (verbose_level) {
			case NORMAL:
			case VERBOSE:
				fputs (_("\tDeleting.\n"), stdout);
				break;
			default:
				break;
		}
		sql_check_delete1 (	"job_cluster", "cluster_id", clusterid,
					sql_error_logger, NULL);
	}
	return 0;
}


/*
 * Check the association clusters/jobs
 */
static int
check_job_cluster ()
{
	int ret;

	if (verbose_level == VERBOSE) {
		fputs (
		    _("Checking the association between jobs and clusters:\n"),
			stdout);
	}

	ret = 0;
	switch (sql_check_id (	"job_cluster", "job_id",
				"job_main", "id", "cluster_id",
				check_job_cluster_cb1, NULL,
				sql_error_logger, NULL))
	{
		case 0: break;
		case 2: ret = 2; break;
		default: return -1;
	}

	switch (sql_check_id (	"job_cluster", "cluster_id",
				"clusters", "id", "job_id",
				check_job_cluster_cb2, NULL,
				sql_error_logger, NULL))
	{
		case 0: break;
		case 2: ret = 2; break;
		default: return -1;
	}

	if (ret == 0 && verbose_level == VERBOSE) {
		fputs (_("\tOK\n"), stdout);
	}

	return ret;
}


/*
 * Callback functions for sql_check_id() when an error is detected
 */
static int
check_job_environment_cb1 (void *data, row_item_t *id, row_item_t *env_id)
{
	unsigned long long int jobid;

	jobid = (unsigned long long int) sql_row_item2ll (id);
	switch (verbose_level) {
		case NORMAL:
		case VERBOSE:
			printf (
		_("job/environment: error: unknown job/jobset Id `%lld'\n"),
				jobid);
			break;
		default:
			break;
	}
	if (autofix == 1) {
		switch (verbose_level) {
			case NORMAL:
			case VERBOSE:
				fputs (_("\tDeleting.\n"), stdout);
				break;
			default:
				break;
		}
		sql_check_delete1 (	"job_environment", "job_id", jobid,
					sql_error_logger, NULL);
	}
	return 0;
}

static int
check_job_environment_cb2 (void *data, row_item_t *id, row_item_t *job_id)
{
	unsigned long long int envid;

	envid = (unsigned long long int) sql_row_item2ll (id);
	/*
	 * Ignore the error if the environment ID is 0.  It simply means that
	 * the job does not inherit its parent environment
	 */
	if (envid == 0) {
		return 0;
	}

	switch (verbose_level) {
		case NORMAL:
		case VERBOSE:
			printf (
		_("job/environment: error: unknown environment Id `%lld'\n"),
				envid);
			break;
		default:
			break;
	}
	if (autofix == 1) {
		switch (verbose_level) {
			case NORMAL:
			case VERBOSE:
				fputs (_("\tDeleting.\n"), stdout);
				break;
			default:
				break;
		}
		sql_check_delete1 (	"job_environment", "env_id", envid,
					sql_error_logger, NULL);
	}
	return 0;
}


/*
 * Check the association jobs/environments
 */
static int
check_job_environment ()
{
	int ret;

	if (verbose_level == VERBOSE) {
		fputs (
		_("Checking the association between jobs and environments:\n"),
			stdout);
	}

	ret = 0;
	switch (sql_check_id (	"job_environment", "job_id",
				"job_main", "id", "env_id",
				check_job_environment_cb1, NULL,
				sql_error_logger, NULL))
	{
		case 0: break;
		case 2: ret = 2; break;
		default: return -1;
	}

	switch (sql_check_id (	"job_environment", "env_id",
				"environments", "id", "job_id",
				check_job_environment_cb2, NULL,
				sql_error_logger, NULL))
	{
		case 0: break;
		case 2: ret = 2; break;
		default: return -1;
	}

	if (ret == 0 && verbose_level == VERBOSE) {
		fputs (_("\tOK\n"), stdout);
	}

	return ret;
}


/*
 * Callback functions for sql_check_id() when an error is detected
 */
static int
check_host_environment_cb1 (void *data, row_item_t *id, row_item_t *env_id)
{
	unsigned long long int hostid;

	hostid = (unsigned long long int) sql_row_item2ll (id);
	switch (verbose_level) {
		case NORMAL:
		case VERBOSE:
			printf (
			_("host/environment: error: unknown host Id `%lld'\n"),
				hostid);
			break;
		default:
			break;
	}
	if (autofix == 1) {
		switch (verbose_level) {
			case NORMAL:
			case VERBOSE:
				fputs (_("\tDeleting.\n"), stdout);
				break;
			default:
				break;
		}
		sql_check_delete1 (	"host_environment", "host_id", hostid,
					sql_error_logger, NULL);
	}
	return 0;
}

static int
check_host_environment_cb2 (void *data, row_item_t *id, row_item_t *host_id)
{
	unsigned long long int envid;

	envid = (unsigned long long int) sql_row_item2ll (id);
	switch (verbose_level) {
		case NORMAL:
		case VERBOSE:
			printf (
		_("host/environment: error: unknown environment Id `%lld'\n"),
				envid);
			break;
		default:
			break;
	}
	if (autofix == 1) {
		switch (verbose_level) {
			case NORMAL:
			case VERBOSE:
				fputs (_("\tDeleting.\n"), stdout);
				break;
			default:
				break;
		}
		sql_check_delete1 (	"host_environment", "env_id", envid,
					sql_error_logger, NULL);
	}
	return 0;
}


/*
 * Check the association hosts/environments
 */
static int
check_host_environment ()
{
	int ret;

	if (verbose_level == VERBOSE) {
		fputs (
		_("Checking the association between hosts and environments:\n"),
			stdout);
	}

	ret = 0;
	switch (sql_check_id (	"host_environment", "host_id",
				"hosts", "id", "env_id",
				check_host_environment_cb1, NULL,
				sql_error_logger, NULL))
	{
		case 0: break;
		case 2: ret = 2; break;
		default: return -1;
	}

	switch (sql_check_id (	"host_environment", "env_id",
				"environments", "id", "host_id",
				check_host_environment_cb2, NULL,
				sql_error_logger, NULL))
	{
		case 0: break;
		case 2: ret = 2; break;
		default: return -1;
	}

	if (ret == 0 && verbose_level == VERBOSE) {
		fputs (_("\tOK\n"), stdout);
	}

	return ret;
}


/*
 * Callback functions for sql_check_id() when an error is detected
 */
static int
check_host_cluster_cb1 (void *data, row_item_t *id, row_item_t *env_id)
{
	unsigned long long int hostid;

	hostid = (unsigned long long int) sql_row_item2ll (id);
	switch (verbose_level) {
		case NORMAL:
		case VERBOSE:
			printf (
			_("host/cluster: error: unknown host Id `%lld'\n"),
				hostid);
			break;
		default:
			break;
	}
	if (autofix == 1) {
		switch (verbose_level) {
			case NORMAL:
			case VERBOSE:
				fputs (_("\tDeleting.\n"), stdout);
				break;
			default:
				break;
		}
		sql_check_delete1 (	"host_clusters", "host_id", hostid,
					sql_error_logger, NULL);
	}
	return 0;
}

static int
check_host_cluster_cb2 (void *data, row_item_t *id, row_item_t *host_id)
{
	unsigned long long int clusterid;

	clusterid = (unsigned long long int) sql_row_item2ll (id);
	switch (verbose_level) {
		case NORMAL:
		case VERBOSE:
			printf (
		_("host/cluster: error: unknown cluster Id `%lld'\n"),
				clusterid);
			break;
		default:
			break;
	}
	if (autofix == 1) {
		switch (verbose_level) {
			case NORMAL:
			case VERBOSE:
				fputs (_("\tDeleting.\n"), stdout);
				break;
			default:
				break;
		}
		sql_check_delete1 ("host_clusters", "cluster_id", clusterid,
					sql_error_logger, NULL);
	}
	return 0;
}


/*
 * Check the association hosts/clusters
 */
static int
check_host_cluster ()
{
	int ret;

	if (verbose_level == VERBOSE) {
		fputs (
		_("Checking the association between hosts and clusters:\n"),
			stdout);
	}

	ret = 0;
	switch (sql_check_id (	"host_clusters", "host_id",
				"hosts", "id", "cluster_id",
				check_host_cluster_cb1, NULL,
				sql_error_logger, NULL))
	{
		case 0: break;
		case 2: ret = 2; break;
		default: return -1;
	}

	switch (sql_check_id (	"host_clusters", "cluster_id",
				"clusters", "id", "host_id",
				check_host_cluster_cb2, NULL,
				sql_error_logger, NULL))
	{
		case 0: break;
		case 2: ret = 2; break;
		default: return -1;
	}

	if (ret == 0 && verbose_level == VERBOSE) {
		fputs (_("\tOK\n"), stdout);
	}

	return ret;
}


/*
 * Callback function for sql_check_id() when an error is detected
 */
static int
check_environment_var_cb_env (void *data, row_item_t *id, row_item_t *key)
{
	unsigned long long int envid;

	envid = (unsigned long long int) sql_row_item2ll (id);
	switch (verbose_level) {
		case NORMAL:
		case VERBOSE:
			printf (
_("Environment variable: error: unknown environment Id `%lld' for variable `%s'\n"),
				envid, key->value_string);
			break;
		default:
			break;
	}
	if (autofix == 1) {
		switch (verbose_level) {
			case NORMAL:
			case VERBOSE:
				fputs (_("\tDeleting.\n"), stdout);
				break;
			default:
				break;
		}
		sql_check_delete1 (	"environment_var", "env_id", envid,
					sql_error_logger, NULL);
	}
	return 0;
}


/*
 * Callback function for sql_environment_var_check_value() when an error
 * is detected
 */
static int
check_environment_var_cb_value (void *data, unsigned long long int id,
				const char *name, const char *value)
{
	switch (verbose_level) {
		case NORMAL:
		case VERBOSE:
			printf (
_("Environment variable: error: environment Id `%lld': syntax error in value `%s' for variable `%s'\n"),
				id, value, name);
			break;
		default:
			break;
	}
	if (autofix == 1) {
		switch (verbose_level) {
			case NORMAL:
			case VERBOSE:
				fputs (_("\tDeleting.\n"), stdout);
				break;
			default:
				break;
		}
		sql_check_delete2 (	"environment_var",
					"env_id", "env_value",
					id, value,
					sql_error_logger, NULL);
	}
	return 0;
}


/*
 * Check the environment variables
 */
static int
check_environment_var ()
{
	int ret;

	if (verbose_level == VERBOSE) {
		fputs (
		_("Checking the environment variables:\n"),
			stdout);
	}

	ret = 0;
	switch (sql_check_id (	"environment_var", "env_id",
				"environments", "id", "env_key",
				check_environment_var_cb_env, NULL,
				sql_error_logger, NULL))
	{
		case 0: break;
		case 2: ret = 2; break;
		default: return -1;
	}

	switch (sql_environment_var_check_value (
				check_environment_var_cb_value, NULL,
				sql_error_logger, NULL))
	{
		case 0: break;
		case 2: ret = 2; break;
		default: return -1;
	}

	if (ret == 0 && verbose_level == VERBOSE) {
		fputs (_("\tOK\n"), stdout);
	}

	return ret;
}


/*
 * Callback function for sql_status_get_old() when an error is detected
 */
static int
check_workload_retention_cb (	void *data, const char *table_name,
				int workload)
{
	switch (verbose_level) {
		case NORMAL:
		case VERBOSE:
			printf (
		_("Workload: error: table `%s_s': obsolete workload `%d'\n"),
				table_name, workload);
			break;
		default:
			break;
	}
	if (autofix == 1) {
		switch (verbose_level) {
			case NORMAL:
			case VERBOSE:
				fputs (_("\tDeleting.\n"), stdout);
				break;
			default:
				break;
		}
		sql_status_delete (workload,
				sql_error_logger, NULL);
	}
	return 0;
}


/*
 * Look for too old workloads
 */
static int
check_workload_retention (time_t now, long int max_retention)
{
	time_t old;
	int ret;

	if (verbose_level == VERBOSE) {
		fputs (
		_("Checking for obsolete workloads:\n"),
			stdout);
	}

	old = now - ((max_retention + 1) * 24 * 60 * 60);
	ret = 0;
	switch (sql_status_get_old (schedwi_time_to_date_int (old),
				check_workload_retention_cb, NULL,
				sql_error_logger, NULL))
	{
		case 0: break;
		case 2: ret = 2; break;
		default: return -1;
	}

	if (ret == 0 && verbose_level == VERBOSE) {
		fputs (_("\tOK\n"), stdout);
	}

	return ret;
}


/*
 * Main function
 *
 * The exit code is 1 in case of error
 */
int
main (int argc, char **argv)
{
	int ret, global_ret;
	dbi_conn sql_handler;
	char *err_msg;
	time_t now;
	long int retention_completed, retention_failed;
#if HAVE_GETOPT_LONG
	int option_index;
	struct option long_options[] =
	{
		{"help",       0, 0, 'h'},
		{"version",    0, 0, 'V'},
		{"config",     1, 0, 'c'},
		{"quiet",      0, 0, 'q'},
		{"verbose",    0, 0, 'v'},
		{"fix",        0, 0, 'f'},
		{0, 0, 0, 0}
	};
#endif


#if HAVE_SETLOCALE
	setlocale (LC_ALL, "");
#endif
#if HAVE_BINDTEXTDOMAIN
	bindtextdomain (PACKAGE, LOCALEDIR);
#endif
#if HAVE_TEXTDOMAIN
	textdomain (PACKAGE);
#endif

	/* Set default values for options */
	configuration_file = SCHEDWI_DEFAULT_CONFFILE_SRV;

	/* Parse options */
	while (1) {
#if HAVE_GETOPT_LONG
		option_index = 0;
		ret = getopt_long (argc, argv, "hVc:qvf",
					long_options, &option_index);
#else
		ret = getopt (argc, argv, "hVc:qvf");
#endif
		if (ret == -1) {
			break;
		}

		switch (ret) {
			case 'h':
				help ();
				return EXIT_SUCCESS;
			case 'V':
				version ();
				return EXIT_SUCCESS;
			case 'c':
				configuration_file = optarg;
				break;
			case 'q':
				verbose_level = QUIET;
				break;
			case 'v':
				verbose_level = VERBOSE;
				break;
			case 'f':
				autofix = 1;
				break;
			default:
				help ();
				return EXIT_FAILURE;
		}
	}

	/*
	 * Build the lost+found jobset name.  When orphans are detected,
	 * this jobset is created and the orphans are attached to this new
	 * jobset.  The same applies for orphan calendars (attached to a
	 * calendar forlder with the same name)
	 */
	now = time (NULL);
	lost_found_name = schedwi_time_strftime (LOST_FOUND_STRFTIME, &now);
	if (lost_found_name == NULL) {
		fputs (_("Internal error: buffer too small\n"), stderr);
		return EXIT_FAILURE;
	}

	/*
	 * Compute the coordinates (x and y) of the lost+found jobset on the
	 * canvas of the root jobset
	 */
	srand ((unsigned int) now);
	x = 50 + (int) (150.0 * (rand() / (RAND_MAX + 1.0)));
	y = 50 + (int) (150.0 * (rand() / (RAND_MAX + 1.0)));

	/*
	 * Initialize the module engine
	 */
	if (module_init () != 0) {
		free (lost_found_name);
		return EXIT_FAILURE;
	}

	/*
	 * Read the configuration file
	 */
	ret = conf_init_srv (configuration_file);
	if (ret == -2) {
		error (EXIT_FAILURE, errno, "%s", configuration_file);
	}
	if (ret != 0) {
		free (lost_found_name);
		return EXIT_FAILURE;
	}

	/*
	 * Retrieve the retention duration from the configuration file
	 */
	ret = conf_get_param_int (	"WORKLOAD_PURGE_COMPLETED",
					&retention_completed);
	ret += conf_get_param_int (	"WORKLOAD_PURGE_FAILED",
					&retention_failed);
#if HAVE_ASSERT_H
	assert (ret == 0);
#endif


	/*
	 * Database connection
	 */
	err_msg = NULL;
	sql_handler = begin_sql (&err_msg);
	if (sql_handler == NULL) {
		if (err_msg != NULL) {
			fprintf (stderr,
				_("Failed to connect to database: %s\n"),
				err_msg);
			free (err_msg);
		}
		else {
			fputs (_("Failed to connect to database\n"), stderr);
		}
		conf_destroy_srv ();
		module_exit ();
		free (lost_found_name);
		return EXIT_FAILURE;
	}


	/*
	 * Checks
	 */
	global_ret = EXIT_SUCCESS;

	ret = check_job_main ();
	if (ret < 0) {
		end_sql ();
		conf_destroy_srv ();
		module_exit ();
		free (lost_found_name);
		return EXIT_FAILURE;
	}
	if (ret != 0) {
		global_ret = 2;
	}

	ret = check_job_parameters ();
	if (ret < 0) {
		end_sql ();
		conf_destroy_srv ();
		module_exit ();
		free (lost_found_name);
		return EXIT_FAILURE;
	}
	if (ret != 0) {
		global_ret = 2;
	}

/* Hosts are allowed to have SSL enable without a cert in the DB
	ret = check_hosts ();
	if (ret < 0) {
		end_sql ();
		conf_destroy_srv ();
		module_exit ();
		free (lost_found_name);
		return EXIT_FAILURE;
	}
	if (ret != 0) {
		global_ret = 2;
	}
*/

	ret = check_job_host ();
	if (ret < 0) {
		end_sql ();
		conf_destroy_srv ();
		module_exit ();
		free (lost_found_name);
		return EXIT_FAILURE;
	}
	if (ret != 0) {
		global_ret = 2;
	}

	ret = check_job_cluster ();
	if (ret < 0) {
		end_sql ();
		conf_destroy_srv ();
		module_exit ();
		free (lost_found_name);
		return EXIT_FAILURE;
	}
	if (ret != 0) {
		global_ret = 2;
	}

	ret = check_host_cluster ();
	if (ret < 0) {
		end_sql ();
		conf_destroy_srv ();
		module_exit ();
		free (lost_found_name);
		return EXIT_FAILURE;
	}
	if (ret != 0) {
		global_ret = 2;
	}

	ret = check_calendars ();
	if (ret < 0) {
		end_sql ();
		conf_destroy_srv ();
		module_exit ();
		free (lost_found_name);
		return EXIT_FAILURE;
	}
	if (ret != 0) {
		global_ret = 2;
	}

	ret = check_constraint_files ();
	if (ret < 0) {
		end_sql ();
		conf_destroy_srv ();
		module_exit ();
		free (lost_found_name);
		return EXIT_FAILURE;
	}
	if (ret != 0) {
		global_ret = 2;
	}

	ret = check_links ();
	if (ret < 0) {
		end_sql ();
		conf_destroy_srv ();
		module_exit ();
		free (lost_found_name);
		return EXIT_FAILURE;
	}
	if (ret != 0) {
		global_ret = 2;
	}

	ret = check_environment_var ();
	if (ret < 0) {
		end_sql ();
		conf_destroy_srv ();
		module_exit ();
		free (lost_found_name);
		return EXIT_FAILURE;
	}
	if (ret != 0) {
		global_ret = 2;
	}

	ret = check_job_environment ();
	if (ret < 0) {
		end_sql ();
		conf_destroy_srv ();
		module_exit ();
		free (lost_found_name);
		return EXIT_FAILURE;
	}
	if (ret != 0) {
		global_ret = 2;
	}

	ret = check_host_environment ();
	if (ret < 0) {
		end_sql ();
		conf_destroy_srv ();
		module_exit ();
		free (lost_found_name);
		return EXIT_FAILURE;
	}
	if (ret != 0) {
		global_ret = 2;
	}

	ret = check_workload_retention (now,
		(retention_completed > retention_failed)
			? retention_completed :retention_failed);
	if (ret < 0) {
		end_sql ();
		conf_destroy_srv ();
		module_exit ();
		free (lost_found_name);
		return EXIT_FAILURE;
	}
	if (ret != 0) {
		global_ret = 2;
	}

	/* Destroy the database handler */
	end_sql ();


	/* Destroy the configuration file object */
	conf_destroy_srv ();
	module_exit ();
	free (lost_found_name);

	return global_ret;
}

/*------------------------======= End Of File =======------------------------*/
