/* Schedwi
   Copyright (C) 2007-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* schedwi_time.c -- Time and date management functions */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#include <xmem.h>
#include <schedwi_time.h>


/*
 * Add the provided number of days to the provided schedwi_date object. The
 * result is set in the provided object out. nb, the number of days to add,
 * may be negative to remove days.
 */
void
schedwi_date_add_days (const schedwi_date *in, schedwi_date *out, int nb)
{
	struct tm t;
	time_t ts;


	if (in != NULL && out != NULL) {
		t.tm_sec = 0;
		t.tm_min = 0;
		t.tm_hour = 12;
		t.tm_mday = in->day;
		t.tm_mon = in->month - 1;
		t.tm_year = in->year - 1900;
		t.tm_isdst = -1;

		ts = mktime (&t);
		ts += nb * 86400;

		localtime_r (&ts, &t);

		out->day = t.tm_mday;
		out->month = t.tm_mon + 1;
		out->year = t.tm_year + 1900;
	}
}


/*
 * Add the provided number of minutes to the provided schedwi_time object. The
 * result is returned. nb, the number of minutes to add, may be negative to
 * remove minutes.
 */
schedwi_time
schedwi_time_add_minutes (schedwi_time in, int nb)
{
	return in + (nb * 60);
}


/*
 * Retrieve the number of seconds remaining to midnight
 */
schedwi_time
schedwi_time_to_midnight ()
{
	struct tm t;
	time_t t_now, t_tomorrow;


	time (&t_now);
	t_tomorrow = t_now + 86400;
	localtime_r (&t_tomorrow, &t);
	t.tm_sec = 0;
	t.tm_min = 0;
	t.tm_hour = 0;
	t_tomorrow = mktime (&t);
	return t_tomorrow - t_now;
}


/*
 * Format the provided schedwi_time.  See strftime for the format syntax
 *
 * Return:
 *   The string (to be freed by the caller by free()) or
 *   NULL if the converted time does not fit in the internal buffer.
 */
char *
schedwi_time_strftime (const char *format, const time_t *in)
{
	const char *fmt;
	struct tm t;
	char *s;
	unsigned int len;
	size_t ret;


	fmt = (format != NULL)? format: "%c";

	len = strlen (fmt) + 200;
	s = (char *) xmalloc (len + 1);

	localtime_r (in, &t);
	ret = strftime (s, len, fmt, &t);
	if (ret == 0 || ret == len) {
		free (s);
		return NULL;
	}
	return s;
}


/*
 * Build the schedwi_date object with the current date
 */
void
schedwi_date_now (schedwi_date *out)
{
	struct tm t;
	time_t ts;

	if (out != NULL) {
		time (&ts);
		localtime_r (&ts, &t);

		out->day = t.tm_mday;
		out->month = t.tm_mon + 1;
		out->year = t.tm_year + 1900;
	}
}


/*
 * Convert the provided value to a schedwi_date object. The format of the
 * value must be `YYYYMMDD'
 *
 * Return:
 *    0 --> No error. out is set (if not NULL)
 *   -1 --> The string format is wrong (out is left unchanged)
 */
int
schedwi_date_from_int (int year_month_day, schedwi_date *out)
{
	unsigned short int year;
	unsigned char month;
	unsigned char day;


	year  = year_month_day / 10000;
	month = (year_month_day % 10000) / 100;
	day   = year_month_day % 100;
	if (	   year < 1900 || year > 3000
		|| month < 1 || month > 12
		|| day < 1 || day > 31)
	{
		return -1;
	}

	if (out != NULL) {
		out->year  = year;
		out->month = month;
		out->day   = day;
	}
	return 0;
}


/*
 * Convert the provided string to a schedwi_date object. The format of
 * the string must be `YYYYMMDD'
 *
 * Return:
 *    0 --> No error. out is set (if not NULL)
 *   -1 --> The string format is wrong (out is left unchanged)
 */
int
schedwi_date_from_string (const char *data_str, schedwi_date *out)
{
	if (data_str == NULL) {
		return -1;
	}
	return schedwi_date_from_int ((unsigned int)atoi (data_str), out);
}


/*
 * Convert the provided schedwi_date object to a string
 *
 * Return:
 *   The string (to be freed by the caller by free()) or
 *   NULL in case of error
 */
char *
schedwi_date_to_string (schedwi_date in)
{
	char *s;
	int ret;

	s = (char *) xmalloc (9);
	ret = snprintf (s, 9, "%4.4d%2.2d%2.2d", in.year, in.month, in.day);
	if (ret >= 9 || ret < 0) {
		free (s);
		return NULL;
	}
	return s;
}


/*
 * Compare two dates and return an integer less than, equal to, or greater
 * than zero if the first date is considered to be respectively less than,
 * equal to, or greater than the second.
 */
int
schedwi_date_compar (const schedwi_date *a, const schedwi_date *b)
{
	if (a == NULL) {
		if (b == NULL) {
			return 0;
		}
		return -1;
	}
	if (b == NULL) {
		return 1;
	}
	if (a->year != b->year) {
		return a->year - b->year;
	}
	if (a->month != b->month) {
		return a->month - b->month;
	}
	return a->day - b->day;
}


/*
 * Format the provided number of seconds to a human readable string
 *
 * Return:
 *   The string (to be freed by the caller by free())
 */
char *
schedwi_time_duration_to_str (unsigned long int nb_sec)
{
	char *fmt, *s;
	int sec, minutes, hours;

	if (nb_sec < 60) {
		fmt = _("%ds");
		s = (char *) xmalloc (strlen (fmt) + 5);
		sprintf (s, fmt, (int)nb_sec);
		return s;
	}

	sec = nb_sec % 60;
	nb_sec /= 60;

	if (nb_sec < 60) {
		fmt = _("%dm %ds");
		s = (char *) xmalloc (strlen (fmt) + 10);
		sprintf (s, fmt, (int)nb_sec, sec);
		return s;
	}

	minutes = nb_sec % 60;
	nb_sec /= 60;

	if (nb_sec < 24) {
		fmt = _("%dh %dm %ds");
		s = (char *) xmalloc (strlen (fmt) + 15);
		sprintf (s, fmt, (int)nb_sec, minutes, sec);
		return s;
	}

	hours = nb_sec % 24;
	nb_sec /= 24;

	fmt = _("%ldd %dh %dm %ds");
	s = (char *) xmalloc (strlen (fmt) + 40);
	sprintf (s, fmt, (long int)nb_sec, hours, minutes, sec);
	return s;
}


/**
 * Convert a time_t value (see time(2)) to a schedwi_date object.
 *
 * @param[in] epoch The time to convert.
 * @param[out] out The user-supplied schedwi_date object to fill.  If NULL,
 *                 this function does nothing and returns NULL.
 * @return A pointer to the given out object or NULL if out is NULL.
 */
schedwi_date *
schedwi_date_from_time_t (time_t epoch, schedwi_date *out)
{
	struct tm t;


	if (out == NULL) {
		return NULL;
	}

	localtime_r (&epoch, &t);

	out->day = t.tm_mday;
	out->month = t.tm_mon + 1;
	out->year = t.tm_year + 1900;
	return out;
}


/**
 * Convert the provided schedwi_date object to an integer (YYYYMMDD).
 *
 * @param[in] in A pointer to the schedwi_date object to convert.  If NULL,
 *               the current date is used instead.
 * @return The converted date (YYYYMMDD).
 */
int
schedwi_date_to_int (const schedwi_date *in)
{
	schedwi_date d;


	if (in == NULL) {
		schedwi_date_from_time_t (time (NULL), &d);
		return d.day + 100 * d.month + 10000 * d.year;
	}
	return in->day + 100 * in->month + 10000 * in->year;
}


/**
 * Convert a time_t value (see time(2)) to a date integer (YYYYMMDD).
 *
 * @param[in] epoch The time to convert.
 * @return The converted date (YYYYMMDD).
 */
int
schedwi_time_to_date_int (time_t epoch)
{
	schedwi_date d;


	return schedwi_date_to_int (schedwi_date_from_time_t (epoch, &d));
}


/**
 * Return the time_t (epoch - see time(2)) of midnight.
 *
 * @param[in] epoch A time during the day for which midnight - 0h00m01s -
 *                  should be retrieved.
 * @return The epoch time of midnight.
 */
time_t
schedwi_last_midnight (time_t epoch)
{
	struct tm t;
	schedwi_date today_date;


	schedwi_date_from_time_t (epoch, &today_date);
	t.tm_sec = 1;
	t.tm_min = 0;
	t.tm_hour = 0;
	t.tm_mday = today_date.day;
	t.tm_mon = today_date.month - 1;
	t.tm_year = today_date.year - 1900;
	t.tm_isdst = -1;
	return mktime (&t);
}


/**
 * Compute and return the time_t value from the provided
 * date (in) and time (hour_min).
 *
 * @param[in] in A pointer to the schedwi_date object representing the date.
 *               If NULL, the current date is used instead.
 * @param[in] hour_min Number of hours and minutes from midnight. It is a
 *                     value between 0 and 2359.
 * @return The time_t value
 */
time_t
schedwi_time_convert (const schedwi_date *in, int hour_min)
{
	struct tm t;
	schedwi_date d;


	if (hour_min < 0) {
		return 0;
	}

	t.tm_sec = 0;
	t.tm_min = hour_min % 100 ;
	t.tm_hour = hour_min / 100;
	t.tm_isdst = -1;
	if (in == NULL) {
		schedwi_date_from_time_t (time (NULL), &d);
		t.tm_mday = d.day;
		t.tm_mon = d.month - 1;
		t.tm_year = d.year - 1900;
	}
	else {
		t.tm_mday = in->day;
		t.tm_mon = in->month - 1;
		t.tm_year = in->year - 1900;
	}

	return mktime (&t);
}


/**
 * Compute and return the time_t value from the provided date (epoch) and time
 * (hour_min).
 *
 * @param[in] epoch The time (only the date part of the time is used).
 * @param[in] hour_min Number of hours and minutes from midnight. It is a
 *                     value between 0 and 2359.
 * @return The time_t value
 */
time_t
schedwi_time_convert_from_epoch (time_t epoch, int hour_min)
{
	schedwi_date d;


	return  schedwi_time_convert (	schedwi_date_from_time_t (epoch, &d),
					hour_min);
}


/**
 * Compute and return the time_t value from the provided
 * date (in) and time (str).
 *
 * @param[in] in A pointer to the schedwi_date object representing the date.
 *               If NULL, the current date is used instead.
 * @param[in] str Time representation according to the format HHMM.
 * @return The time_t value
 */
time_t
schedwi_time_convert_from_string (const schedwi_date *in, const char *str)
{
	return schedwi_time_convert (in, (str == NULL) ? 0: atoi (str));
}


/**
 * Compute and return the time_t value from the provided
 * date (in the YYYYMMDD format) and time.
 *
 * @param[in] year_month_day The date (in the YYYYMMDD format).
 * @param[in] hour_min Number of hours and minutes from midnight. It is a
 *                     value between 0 and 2359.
 * @return The time_t value
 */
time_t
schedwi_time_convert_from_int (int year_month_day, int hour_min)
{
	schedwi_date d;


	if (schedwi_date_from_int (year_month_day, &d) == 0) {
		return schedwi_time_convert (&d, hour_min);
	}
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
