/* Schedwi
   Copyright (C) 2010-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* schedwi_dbi.c -- DBI functions */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <conf.h>
#include <conf_srv.h>
#include <xmem.h>
#include <schedwi_dbi.h>

#ifdef HAVE_DBI_R
static dbi_inst *dbiInst_ptr = NULL;
#endif

static char drivername_is_sqlite;

struct CONNINFO {
	const char *driverdir;
	const char *drivername;
	const char *dbname;
	const char *dbdir;
	const char *username;
	const char *password;
	const char *hostname;
	const char *version;
	const char *encoding;
};


/*
 * Is the driver SQLite or SQLite3?
 */
char
is_sqlite ()
{
	return drivername_is_sqlite;
}


/*
 * Fill the provided CONNINFO structure with the parameters retrieved
 * from the configuration file
 */
static void
fill_conninfo (struct CONNINFO* pcinfo)
{
	int ret;

	ret  = conf_get_param_string ("DBI_DRIVERDIR", &(pcinfo->driverdir));
	ret += conf_get_param_string ("DBI_DRIVERNAME", &(pcinfo->drivername));
	ret += conf_get_param_string ("DBI_USER", &(pcinfo->username));
	ret += conf_get_param_string ("DBI_PASSWORD", &(pcinfo->password));
	ret += conf_get_param_string ("DBI_DBDIR", &(pcinfo->dbdir));
	ret += conf_get_param_string ("DBI_DBHOSTNAME", &(pcinfo->hostname));
	ret += conf_get_param_string ("DBI_DBNAME", &(pcinfo->dbname));
	ret += conf_get_param_string ("DBI_FREETDS_VERSION",
					&(pcinfo->version));
	ret += conf_get_param_string ("DBI_ENCODING", &(pcinfo->encoding));
#if HAVE_ASSERT_H
	assert (ret == 0);
#endif

	if (pcinfo->driverdir != NULL && (pcinfo->driverdir)[0] == '\0') {
		pcinfo->driverdir = NULL;
	}
	if (pcinfo->encoding != NULL && (pcinfo->encoding)[0] == '\0') {
		pcinfo->encoding = NULL;
	}

	if (pcinfo->hostname == NULL || (pcinfo->hostname)[0] == '\0') {
		if (	   strcmp(pcinfo->drivername, "firebird") != 0
			&& strcmp(pcinfo->drivername, "msql") != 0
			&& strcmp(pcinfo->drivername, "pgsql") != 0)
		{
			pcinfo->hostname = "localhost";
		}
	}
	else
	if (	   strcmp (pcinfo->hostname, "localhost") == 0
		&& strcmp (pcinfo->drivername, "pgsql") == 0)
	{
		pcinfo->hostname = "";
	}
	if (strncmp (pcinfo->drivername, "sqlite", 6) == 0) {
		drivername_is_sqlite = 1;
	}
	else {
		drivername_is_sqlite = 0;
	}
}


/*
 * Establish a connection to the database
 *
 * Return:
 *   The new connection handler OR
 *   NULL in case of error. If err_msg is not NULL, it is set with the error
 *        message (it must be freed by the caller)
 */
dbi_conn
schedwi_dbi_init (char **err_msg)
{
	struct CONNINFO cinfo;
	int numdrivers;
	char *msg;
	dbi_conn conn;
	int ret;
	const char *err;


#ifdef HAVE_DBI_R
	if (dbiInst_ptr != NULL) {
		dbi_shutdown_r (*dbiInst_ptr);
	}
	else {
		dbiInst_ptr = (dbi_inst *) xmalloc (sizeof (dbi_inst));
	}
#endif
	fill_conninfo (&cinfo);


	/*
	 * Initialize DBI and specify the drivers directory
	 */

#ifdef HAVE_DBI_R
	numdrivers = dbi_initialize_r (cinfo.driverdir, dbiInst_ptr);
#else
	numdrivers = dbi_initialize (cinfo.driverdir);
#endif
	if (numdrivers < 0) {
#ifdef HAVE_DBI_R
		dbi_shutdown_r (*dbiInst_ptr);
		free (dbiInst_ptr);
		dbiInst_ptr = NULL;
#else
		dbi_shutdown ();
#endif
		if (err_msg != NULL) {
			msg = _("Unable to initialize libdbi. Make sure you specified a valid driver directory (DBI_DRIVERDIR configuration variable)");
			*err_msg = xstrdup (msg);
		}
		return NULL;
	}

	if (numdrivers == 0) {
#ifdef HAVE_DBI_R
		dbi_shutdown_r (*dbiInst_ptr);
		free (dbiInst_ptr);
		dbiInst_ptr = NULL;
#else
		dbi_shutdown ();
#endif
		if (err_msg != NULL) {
			msg = _("No database driver found");
			*err_msg = xstrdup (msg);
		}
		return NULL;
	}


	/*
	 * Connexion to the driver
	 */

#ifdef HAVE_DBI_R
	conn = dbi_conn_new_r (cinfo.drivername, *dbiInst_ptr);
#else
	conn = dbi_conn_new (cinfo.drivername);
#endif
	if (conn == NULL) {
#ifdef HAVE_DBI_R
		dbi_shutdown_r (*dbiInst_ptr);
		free (dbiInst_ptr);
		dbiInst_ptr = NULL;
#else
		dbi_shutdown ();
#endif
		if (err_msg != NULL) {
			msg = _("Cannot access the database driver specified in the DBI_DRIVERNAME configuration variable");
			*err_msg = xstrdup (msg);
		}
		return NULL;
	}


	/*
	 * Set options (user name, password, database name, ...)
	 */

	ret = 0;
	
	if (	   strcmp (cinfo.drivername, "mysql") == 0
		|| strcmp (cinfo.drivername, "pgsql") == 0
		|| strcmp (cinfo.drivername, "freetds") == 0)
	{
		ret += dbi_conn_set_option (conn, "host", cinfo.hostname);
		ret += dbi_conn_set_option (conn, "username", cinfo.username);
		ret += dbi_conn_set_option (conn, "password", cinfo.password);
		if (cinfo.encoding != NULL) {
			ret += dbi_conn_set_option (conn,"encoding",
							cinfo.encoding);
		}
		if (strcmp (cinfo.drivername, "freetds") == 0) {
			ret += dbi_conn_set_option (conn, "freetds_version",
							cinfo.version);
		}
	}
	else
	if (strcmp (cinfo.drivername, "msql") == 0) {
		if (cinfo.hostname != NULL) {
			ret += dbi_conn_set_option (conn, "host",
							cinfo.hostname);
		}
	}
	else
	if (strcmp (cinfo.drivername, "sqlite3") == 0) {
		ret += dbi_conn_set_option (conn, "sqlite3_dbdir",
							cinfo.dbdir);
		if (cinfo.encoding != NULL) {
			ret += dbi_conn_set_option (conn,"encoding",
							cinfo.encoding);
		}
	}
	else
	if (strcmp (cinfo.drivername, "sqlite") == 0) {
		ret += dbi_conn_set_option (conn, "sqlite_dbdir", cinfo.dbdir);
		if (cinfo.encoding != NULL) {
			ret += dbi_conn_set_option (conn,"encoding",
							cinfo.encoding);
		}
	}
	if (strcmp (cinfo.drivername, "firebird") == 0) {
		ret += dbi_conn_set_option(conn, "firebird_dbdir",
							cinfo.dbdir);
		ret += dbi_conn_set_option(conn, "host", cinfo.hostname);
		ret += dbi_conn_set_option(conn, "username", cinfo.username);
		ret += dbi_conn_set_option(conn, "password", cinfo.password);
	}

	ret += dbi_conn_set_option(conn, "dbname", cinfo.dbname);
			
	if (ret != 0) {
		dbi_conn_close (conn);
#ifdef HAVE_DBI_R
		dbi_shutdown_r (*dbiInst_ptr);
		free (dbiInst_ptr);
		dbiInst_ptr = NULL;
#else
		dbi_shutdown ();
#endif
		if (err_msg != NULL) {
			msg = _("Cannot set database parameters");
			*err_msg = xstrdup (msg);
		}
		return NULL;
	}


	/*
	 * Database connection
	 */

	if (dbi_conn_connect (conn) < 0) {
		if (err_msg != NULL) {
			dbi_conn_error (conn, &err);
			*err_msg = xstrdup (err);
		}
		dbi_conn_close (conn);
#ifdef HAVE_DBI_R
		dbi_shutdown_r (*dbiInst_ptr);
		free (dbiInst_ptr);
		dbiInst_ptr = NULL;
#else
		dbi_shutdown ();
#endif
		return NULL;
	}

	return conn;
}


/*
 * Close the provided database connection and clean the internal DBI structures
 */
void
schedwi_dbi_close (dbi_conn conn)
{
	dbi_conn_close (conn);
#ifdef HAVE_DBI_R
	if (dbiInst_ptr != NULL) {
		dbi_shutdown_r (*dbiInst_ptr);
		free (dbiInst_ptr);
		dbiInst_ptr = NULL;
	}
#else
	dbi_shutdown ();
#endif
}

/*-----------------============== End Of File ==============-----------------*/

