/* Schedwi
   Copyright (C) 2007-2014 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* result_mgnt.c -- Read/write job result status from/to file */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_CTYPE_H
#include <ctype.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <xmem.h>
#include <utils.h>
#include <result_mgnt.h>
#include <JSON_parser.h>

enum steps {
	S_START,
	S_OUTER_ARRAY,
	S_OUTER_ARRAY_MODULE,
	S_DATA,
	S_KEY_JOBID,
	S_KEY_DURATION,
	S_KEY_KILLED,
	S_KEY_RETURN_CODE,
	S_END
};

struct json_cb {
	enum steps step;
	char error;
	result_t *ptr;
};


/*
 * JSON callback function
 */
static int
json_char_callback (void* ctx, int type, const JSON_value* value)
{
	struct json_cb *s = (struct json_cb *)ctx;
	char *v;


	switch (type) {

	case JSON_T_ARRAY_BEGIN:
		if (s->step == S_START) {
			s->step = S_OUTER_ARRAY;
		}
		break;
	case JSON_T_STRING:
		if (s->step == S_OUTER_ARRAY) {
			s->step = S_OUTER_ARRAY_MODULE;
		}
		else if (s->step == S_KEY_JOBID) {
			v = (char *) xmalloc (value->vu.str.length + 1);
			strncpy (v, value->vu.str.value, value->vu.str.length);
			v[value->vu.str.length] = '\0';
			if (s->ptr->job_id != NULL) {
				free (s->ptr->job_id);
			}
			s->ptr->job_id = v;
			s->ptr->job_id_len = value->vu.str.length;
			s->ptr->workload_int = atoi (v);
			if (value->vu.str.length > WORKLOAD_LEN + 1) {
				s->ptr->job_id_int = strtoull (
							v + WORKLOAD_LEN + 1,
							NULL, 0);
			}
			else {
				s->ptr->job_id_int = 0;
			}
			s->step = S_DATA;
		}
		break;
	case JSON_T_OBJECT_BEGIN:
		if (s->step == S_OUTER_ARRAY_MODULE) {
			s->step = S_DATA;
		}
		break;
	case JSON_T_KEY:
		if (s->step == S_DATA) {
			if (strcasecmp (value->vu.str.value, "job id") == 0) {
				s->step = S_KEY_JOBID;
			}
			else
			if (strcasecmp (value->vu.str.value, "duration") == 0)
			{
				s->step = S_KEY_DURATION;
			}
			else
			if (strcasecmp (value->vu.str.value, "killed") == 0) {
				s->step = S_KEY_KILLED;
			}
			else
			if (strcasecmp (value->vu.str.value,
						"return code") == 0)
			{
				s->step = S_KEY_RETURN_CODE;
			}
		}
		break;
	case JSON_T_INTEGER:
		if (s->step == S_KEY_DURATION) {
			s->ptr->duration = (time_t) value->vu.integer_value;
			s->step = S_DATA;
		}
		else
		if (s->step == S_KEY_RETURN_CODE) {
			s->ptr->exit_code = value->vu.integer_value;
			s->step = S_DATA;
		}
		break;
	case JSON_T_TRUE:
		if (s->step == S_KEY_KILLED) {
			s->ptr->was_killed = 1;
			s->step = S_DATA;
		}
		break;
	case JSON_T_FALSE:
		if (s->step == S_KEY_KILLED) {
			s->ptr->was_killed = 0;
			s->step = S_DATA;
		}
		break;
	case JSON_T_OBJECT_END:
		if (s->step == S_DATA) {
			s->step = S_END;
		}
		break;
	}

	return 1;
}


/*
 * Create a new result structure
 *
 * Return:
 *   The new structure (to free by result_destroy)
 */
result_t *
result_new ()
{
	return (result_t *) xcalloc (1, sizeof (result_t));
}


/*
 * Free the provided structure
 */
void
result_destroy (result_t *r)
{
	if (r != NULL) {
		if (r->job_id != NULL) {
			free (r->job_id);
		}
		free (r);
	}
}


/*
 * Set the job ID and its workload in the result_t object.
 */
void
result_set_id (result_t *r, int workload_date, unsigned long long int job_id)
{
	unsigned int len_workload, len_job_id;


	if (r == NULL) {
		return;
	}

	r->workload_int = workload_date;
	r->job_id_int = job_id;

	/* Build the full job ID */
	r->job_id = (char *) xrealloc (r->job_id, WORKLOAD_LEN + 30);
	len_workload = copy_ulltostr (workload_date, r->job_id);
	(r->job_id)[len_workload] = SCHEDWI_WORKLOAD_SEPARATOR;
	len_job_id = copy_ulltostr (job_id, (r->job_id) + len_workload + 1);
	r->job_id_len = len_workload + len_job_id + 1;
}


/**
 * Convert the string to a result_t structure.
 *
 * @param[in] buffer The json string to parse.
 * @param[in] buffer_size The length of the string to parse.
 * @return The result_t object to be freed by the caller with result_destroy()
 */
result_t * 
str_to_result (const char *buffer, int buffer_size)
{
	int i;
	JSON_config config;
	struct JSON_parser_struct *jc;
	struct json_cb cb_data;


#if HAVE_ASSERT_H
	assert (buffer != NULL);
#endif

	/* JSON parser initialization */
	init_JSON_config (&config);

	cb_data.ptr                   = result_new ();
	cb_data.step                  = S_START;
	cb_data.error                 = 0;
	config.depth                  = 20;
	config.callback               = &json_char_callback;
	config.allow_comments         = 0;
	config.handle_floats_manually = 0;
	config.callback_ctx           = &cb_data;
	config.malloc                 = xmalloc;
	config.free                   = free;

	jc = new_JSON_parser (&config);

	/* Parse the JSON string */
	for (i = 0; i < buffer_size && cb_data.step != S_END; i++) {
		/*
		 * No need to check the return code as the JSON string has
		 * already been parsed once in net_parse.c
		 */
		JSON_parser_char (jc, (unsigned char)(buffer[i]));
	}
	delete_JSON_parser (jc);
	return cb_data.ptr;
}

/*------------------------======= End Of File =======------------------------*/
