/* Schedwi
   Copyright (C) 2007-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* reg_register.c -- Parse the `register' request */

/*
 * Non-SSL agents will try to register every time they are started as they
 * don't have a way to know if they are already registered.  If they are
 * registered they will ignore the return message from the server.
 */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_CTYPE_H
#include <ctype.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <utils.h>
#include <xmem.h>
#include <memsend.h>
#include <net_utils_sock.h>
#include <net_utils_ssl.h>
#include <cert_utils.h>
#include <conf.h>
#include <lwc_log.h>
#include <JSON_parser.h>
#include <reg_error_code.h>
#include <reg_register.h>

enum steps {
	S_START,
	S_OUTER_ARRAY,
	S_OUTER_ARRAY_MODULE,
	S_DATA,
	S_KEY_SSL,
	S_KEY_CSR,
	S_KEY_PORT,
	S_END
};

struct json_cb {
	enum steps step;
	char error;
	char ssl, *csr;
	int port;
};


/*
 * JSON callback function
 */
static int
json_char_callback (void* ctx, int type, const JSON_value* value)
{
	struct json_cb *s = (struct json_cb *)ctx;
	char *v;


	switch (type) {
		case JSON_T_ARRAY_BEGIN:
			if (s->step == S_START) {
				s->step = S_OUTER_ARRAY;
			}
			break;
		case JSON_T_STRING:
			if (s->step == S_OUTER_ARRAY) {
				s->step = S_OUTER_ARRAY_MODULE;
			}
			else
			if (s->step == S_KEY_CSR) {
				v = (char *)xmalloc (value->vu.str.length + 1);
				strncpy (	v, value->vu.str.value,
						value->vu.str.length);
				v[value->vu.str.length] = '\0';
				if (s->csr != NULL) {
					free (s->csr);
				}
				s->csr = v;
				s->step = S_DATA;
			}
			break;
		case JSON_T_OBJECT_BEGIN:
			if (s->step == S_OUTER_ARRAY_MODULE) {
				s->step = S_DATA;
			}
			break;
		case JSON_T_KEY:
			if (s->step == S_DATA) {
				if (strcasecmp (value->vu.str.value,
							"ssl") == 0)
				{
					s->step = S_KEY_SSL;
				}
				else
				if (strcasecmp (value->vu.str.value,
							"csr") == 0)
				{
					s->step = S_KEY_CSR;
				}
				else
				if (strcasecmp (value->vu.str.value,
							"port") == 0)
				{
					s->step = S_KEY_PORT;
				}
			}
			break;
		case JSON_T_TRUE:
			if (s->step == S_KEY_SSL) {
				s->ssl = 1;
				s->step = S_DATA;
			}
			break;
		case JSON_T_FALSE:
			if (s->step == S_KEY_SSL) {
				s->ssl = 0;
				s->step = S_DATA;
			}
			break;
		case JSON_T_INTEGER:
			if (s->step == S_KEY_PORT) {
				s->port = value->vu.integer_value;
				s->step = S_DATA;
			}
			break;
		case JSON_T_OBJECT_END:
			if (s->step == S_DATA) {
				s->step = S_END;
			}
			break;
	}

	return 1;
}


static void
send_error (int sock, const char *error_message, error_reason_t reason)
{
	memsend_t *membuff;


	membuff = memsend_new ();
	memsend_append (membuff, "{ \"success\" : false, \"reason\" : ");
	memsend_append (membuff, reg_error_to_string (reason));
	memsend_append (membuff, ", \"data\" : \"");
	if (error_message != NULL) {
		memsend_append (membuff, error_message);
	}
	else {
		memsend_append (membuff,
				_("Cannot save the agent request"));
	}
	memsend_append (membuff, "\" }");
	net_write_sock (sock, memsend_getstring (membuff),
			memsend_getlength (membuff));
	memsend_destroy (membuff);
}


/*
 * Check if at least one file from `file_names' exists in `path' with
 * the extension `extension'
 *
 * Return:
 *   0  --> file does not exist
 *   >0 --> file exists (the value is the index+1 in `file_names')
 */
static int
check_file (	const char *path, const char *const *file_names,
		const char *port, const char *extension)
{
	size_t l_base, l_extension, l;
	char *base, *s;
	int i;


#if HAVE_ASSERT_H
	assert (   path != 0 && file_names != NULL
		&& port != NULL && extension != NULL);
#endif

	l_base = strlen (path) + strlen (DIR_SEP) + strlen (port) + 1;
	base = (char *) xmalloc (l_base + 1);
	strcpy (base, path);
	strcat (base, DIR_SEP);
	strcat (base, port);
	strcat (base, "_");
	l_extension = strlen (extension);
	for (i = 0; file_names[i] != NULL; i++) {
		l = l_base + strlen (file_names[i]) + l_extension + 1;
		s = (char *) xmalloc (l);
		strcpy (s, base);
		strcpy (s + l_base, file_names[i]);
		strcat (s, extension);
		if (access (s, F_OK) == 0) {
			free (s);
			free (base);
			return (i + 1);
		}
		free (s);
	}
	free (base);
	return 0;
}


int
reg_register (	int sock, const char *buff, ssize_t buff_len,
		const char *const *client_names)
{
	ssize_t i;
	size_t l;
	int ret;
	char *s, *deserialized;
	char sport[7];
	unsigned int l_sport;
	const char *client_cert_dir, *client_csr_dir;
	JSON_config config;
	struct JSON_parser_struct *jc;
	struct json_cb cb_data;
	FILE *f;


	ret =  conf_get_param_string (	"SSLAgentCertificateDir",
					&client_cert_dir);
	ret += conf_get_param_string (	"SSLAgentRequestDir",
					&client_csr_dir);
#if HAVE_ASSERT_H
	assert (ret == 0);
#endif

	if (client_names == NULL || client_names[0] == NULL) {
		send_error (	sock, _("Cannot get agent IP/name details"),
				REG_SYSTEM_ERROR);
		return -1;
	}

	/* JSON parser initialization */
	init_JSON_config (&config);
	cb_data.step                  = S_START;
	cb_data.error                 = 0;
	cb_data.ssl                   = 0;
	cb_data.csr                   = NULL;
	cb_data.port                  = atoi (SCHEDWI_DEFAULT_AGTPORT);
	config.depth                  = 20;
	config.callback               = &json_char_callback;
	config.allow_comments         = 0;
	config.handle_floats_manually = 0;
	config.callback_ctx           = &cb_data;
	config.malloc                 = xmalloc;
	config.free                   = free;

	jc = new_JSON_parser (&config);

	/* Parse the JSON string */
	for (i = 0; i < buff_len && cb_data.step != S_END; i++) {
		/*
		 * No need to check the return code as the JSON string has
		 * already been parsed once in net_parse_reg.c
		 */
		JSON_parser_char (jc, (unsigned char)(buff[i]));
	}
	delete_JSON_parser (jc);

	l_sport = copy_ulltostr (cb_data.port, sport);

	/*
	 * Check if a certificate or request already exists for the agent
	 */
	ret = check_file (client_cert_dir, client_names, sport, CRT_EXTENSION);
	if (ret > 0) {
		lwc_writeLog (LOG_INFO,
_("SSL request received from `%s' (port %s) which already has a certificate in %s.  Use `schedwica --clean %s:%s' if you really want to remove the local certificate"),
			client_names[ret - 1], sport, client_cert_dir,
			client_names[ret - 1], sport);
		send_error (sock,
	_("A certificate for this agent already exists on the server"),
			REG_CRT_PRESENT);
		if (cb_data.csr != NULL) {
			free (cb_data.csr);
		}
		return -1;
	}

	ret = check_file (	client_cert_dir, client_names,
				sport, NOSSL_EXTENSION);
	if (ret > 0) {
		if (cb_data.ssl != 0) {
			lwc_writeLog (LOG_INFO,
_("SSL request received from `%s' (port %s) which already has a record in %s.  Use `schedwica --clean %s:%s' if you really want to remove the local record"),
			client_names[ret - 1], sport, client_cert_dir,
			client_names[ret - 1], sport);
		}
		send_error (sock,
	_("A certificate for this agent already exists on the server"),
				REG_CRT_PRESENT);
		if (cb_data.csr != NULL) {
			free (cb_data.csr);
		}
		return -1;
	}

	ret = check_file (client_csr_dir, client_names, sport, CSR_EXTENSION);
	if (ret > 0) {
		lwc_writeLog (LOG_INFO,
_("SSL request received from `%s' (port %s) which already has a pending request in %s.  Use `schedwica --clean %s:%s' if you really want to remove the local request"),
			client_names[ret - 1], sport, client_csr_dir,
			client_names[ret - 1], sport);
		send_error (sock,
	_("A request for this agent already exists on the server"),
			REG_CSR_PRESENT);
		if (cb_data.csr != NULL) {
			free (cb_data.csr);
		}
		return -1;
	}

	ret = check_file (	client_csr_dir, client_names,
				sport, NOSSL_EXTENSION);
	if (ret > 0) {
		if (cb_data.ssl != 0) {
			lwc_writeLog (LOG_INFO,
_("SSL request received from `%s' (port %s) which already has a pending record in %s.  Use `schedwica --clean %s:%s' if you really want to remove the local record"),
			client_names[ret - 1], sport, client_csr_dir,
			client_names[ret - 1], sport);
		}
		send_error (sock,
	_("A request for this agent already exists on the server"),
			REG_CSR_PRESENT);
		if (cb_data.csr != NULL) {
			free (cb_data.csr);
		}
		return -1;
	}

	/* The agent does not want/support SSL encryption */
	if (cb_data.ssl == 0) {
		if (cb_data.csr != NULL) {
			free (cb_data.csr);
		}
		l =       strlen (client_csr_dir)
			+ strlen (DIR_SEP)
			+ l_sport + 1 /* +1 for "_" */
			+ strlen (NOSSL_EXTENSION)
			+ 1;
		/* Try to find a hostname rather than an IP */
		for (i = 0; 	   client_names[i] != NULL
				&& isdigit (client_names[i][0]) != 0; i++);
		if (client_names[i] != NULL) {
			l += strlen (client_names[i]);
		}
		else {
			l += strlen (client_names[0]);
			i = 0;
		}
		s = (char *) xmalloc (l);
		strcpy (s, client_csr_dir);
		strcat (s, DIR_SEP);
		strcat (s, sport);
		strcat (s, "_");
		strcat (s, client_names[i]);
		strcat (s, NOSSL_EXTENSION);
		f = fopen (s, "wb");
		if (f == NULL) {
			lwc_writeLog (	LOG_ERR, _("%s: %s"),
					s, strerror (errno));
			free (s);
			send_error (sock, NULL, REG_SYSTEM_ERROR);
			return -1;
		}
		free (s);
		for (i = 0; client_names[i] != NULL; i++) {
			fwrite (client_names[i], 1,
				strlen (client_names[i]), f);
			fwrite ("\n", 1, 1, f);
		}
		fclose (f);
	}
	else { /* The agent wants SSL encryption */
		/* Check that the request has been provided */
		if (cb_data.csr == NULL) {
			send_error (sock, _("missing request"), REG_ERROR_CSR);
			return -1;
		}

		/* De-serialize the request */
		deserialized = string2PEM (cb_data.csr);
		free (cb_data.csr);

		/*
		 * Check that at least one hostnames/IP in the certificate
		 * matches the agent name/IP.
		 */
		ret = check_string_request_names (deserialized, client_names);
		switch (ret) {
			case 0: /* No match */
				free (deserialized);
				send_error (sock,
			_("Names in request do not match the agent IP/name"),
					REG_ERROR_CSR);
				return -1;
			case -1: /* Error */
				free (deserialized);
				send_error (sock, NULL, REG_SYSTEM_ERROR);
				return -1;
		}

		/* Create the request file */
		l =       strlen (client_csr_dir)
			+ strlen (DIR_SEP)
			+ l_sport + 1 /* +1 for "_" */
			+ strlen (client_names[ret - 1])
			+ strlen (CSR_EXTENSION)
			+ 1;
		s = (char *) xmalloc (l);
		strcpy (s, client_csr_dir);
		strcat (s, DIR_SEP);
		strcat (s, sport);
		strcat (s, "_");
		strcat (s, client_names[ret - 1]);
		strcat (s, CSR_EXTENSION);
		f = fopen (s, "wb");
		if (f == NULL) {
			lwc_writeLog (	LOG_ERR, _("%s: %s"),
					s, strerror (errno));
			free (s);
			free (deserialized);
			send_error (sock, NULL, REG_SYSTEM_ERROR);
			return -1;
		}
		l = strlen (deserialized);
		if (fwrite (deserialized, 1, l, f) != l) {
			lwc_writeLog (	LOG_ERR, _("%s: write error: %s"),
					s, strerror (errno));
			free (deserialized);
			send_error (sock, NULL, REG_SYSTEM_ERROR);
			fclose (f);
			my_unlink (s);
			free (s);
			return -1;
		}
		free (deserialized);
		if (fclose (f) != 0) {
			lwc_writeLog (	LOG_ERR, _("%s: %s"),
					s, strerror (errno));
			send_error (sock, NULL, REG_SYSTEM_ERROR);
			my_unlink (s);
			free (s);
			return -1;
		}
		free (s);
	}

	s = "{ \"success\" : true, \"data\" : \"ok\" }";
	net_write_sock (sock, s, strlen (s));
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
