/* Schedwi
   Copyright (C) 2013, 2014 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* @file ping.c
 * Check if a remote agent is up/listening or not.
 */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <sql_common.h>
#include <sql_hosts.h>
#include <net_utils.h>
#include <net_utils_sock.h>
#include <lwc_log.h>
#include <ping.h>


/*
 * Error callback function for the sql_host_get_by_name() and
 * sql_host_get_by_id() functions
 */
static void
sql_error_logger (void *data, const char *msg, int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
		_("Database error while retrieving the agent host details"));
	}
}


/**
 * Ping a remote agent.
 *
 * @param[in] host_details
 *            The array with the details of the agent to ping:
 *                   host_details[0] --> host ID
 *                   host_details[1] --> Host name
 *                   host_details[2] --> TCP port
 *                   host_details[3] --> SSL?
 *                   host_details[4] --> SSL agent certificate
 *                   host_details[5] --> Host description
 * @return
 *          0 on success
 *          1 if the remote agent did not give a proper reply (a message is
 *            logged with lwc_writeLog())
 *         -1 on system error (a message is logged with lwc_writeLog())
 */
static int
ping_agent (row_item_t *host_details)
{
	schedwi_BIO *b;
	char *s = "[ \"version\" ]";
	char *result_msg;
	int ret;


#if HAVE_ASSERT_H
	assert (host_details != NULL);
#endif

	/* Establish the connection to the agent */
	b = net_client (host_details[2].value_string,  /* TCP port */
			host_details[1].value_string,  /* Agent host name */
			(char)sql_row_item2ll (&(host_details[3])),
			host_details[4].value_string,  /* Certificate */
			(unsigned int)(host_details[4].len));
	if (b == NULL) {
		return -1;
	}

	/* Send the ping request */
	if (net_write (b, s, strlen (s)) != 0) {
		net_close (b);
		return -1;
	}

	/* Read the result */
	result_msg = NULL;
	ret = net_read_result (b, &result_msg);
	net_close (b);

	/* System error */
	if (ret < 0) {
		if (result_msg != NULL) {
			free (result_msg);
		}
		return -1;
	}

	/* Success */
	if (ret == 0 && result_msg != NULL) {
		free (result_msg);
		return 0;
	}

	/* Failure */
	if (result_msg != NULL) {
		lwc_writeLog (	LOG_ERR,
				_("schedwiping: %s/%s: failed (reply: %s)"),
				host_details[1].value_string,
				host_details[2].value_string,
				result_msg);
		free (result_msg);
	}
	else {
		lwc_writeLog (	LOG_ERR,
				_("schedwiping: %s/%s: failed"),
				host_details[1].value_string,
				host_details[2].value_string);
	}
	return 1;
}


/**
 * Ping a remote agent (given its name)
 *
 * @param[in] agent_name
 *          Name of the agent (as defined in the database) to ping.
 * @return
 *          0 on success
 *          1 if the remote agent did not give a proper reply (a message is
 *            logged with lwc_writeLog())
 *         -1 on system error (a message is logged with lwc_writeLog())
 */
int
ping_by_names (char *agent_name)
{
	char **names;
	char *port;
	row_item_t *host_details;
	int ret;


	/* Retrieve all the names of the agent from the DNS */
	if (build_client_names_from_name (agent_name, &names, &port) != 0) {
		return -1;
	}

	/* Retrieve the agent details from the database */
	if (sql_host_get_by_name (names, port, &host_details,
					sql_error_logger, NULL) != 0)
	{
		free (port);
		destroy_client_names (names);
		return -1;
	}
	free (port);
	destroy_client_names (names);

	/* Ping */
	ret = ping_agent (host_details);
	sql_free_row (host_details);
	return ret;
}


/*
 * Ping a remote agent (given its database ID)
 *
 * @param host_id
 *          The database host ID.
 * @return:
 *          0 on success
 *          1 if the remote agent did not give a proper reply (a message is
 *            logged with lwc_writeLog())
 *         -1 on system error (a message is logged with lwc_writeLog())
 */
int
ping_by_id (unsigned long long int host_id)
{
	row_item_t *host_details;
	int ret;


	/* Retrieve the agent details from the database */
	if (sql_host_get_by_id (	host_id, &host_details,
					sql_error_logger, NULL) != 0)
	{
		return -1;
	}

	/* Ping */
	ret = ping_agent (host_details);
	sql_free_row (host_details);
	return ret;
}

/*-----------------============== End Of File ==============-----------------*/
