/* Schedwi
   Copyright (C) 2007-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* net_parse_reg.c -- Parse and run a network request for the registrar */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <net_utils_sock.h>
#include <lwc_log.h>
#include <JSON_parser.h>
#include <reg_getmycrt.h>
#include <reg_getca.h>
#include <reg_register.h>
#include <reg_whoami.h>
#include <xmem.h>
#include <net_parse_reg.h>


/*
 * JSON callback function
 */
static int
json_char_callback (void* ctx, int type, const JSON_value* value)
{
	char *s = (char *)ctx;

	if (type == JSON_T_STRING) {
		strncpy (s, value->vu.str.value, SCHEDWI_NET_FUNCTION_LENGHT);
		s[SCHEDWI_NET_FUNCTION_LENGHT] = '\0';
	}
	return 1;
}


/*
 * Parse and run the network request
 * buff (and its lenght len) is a working buffer.  It is allocated by this
 * function if NULL and must be freed by the caller.
 *
 * Return:
 *    0 --> No error
 *   -1 --> Error (an error message is displayed by using lwc_writeLog())
 */
int
net_parse_reg (	int sock, char **buff, size_t *len,
		const char *const *client_names)
{
	ssize_t nb_read;
	char module[SCHEDWI_NET_FUNCTION_LENGHT + 1];
	int i;
	JSON_config config;
	struct JSON_parser_struct *jc;


#if HAVE_ASSERT_H
	assert (buff != NULL && len != NULL);
#endif

	nb_read = net_read_sock (sock, buff, len);
	if (nb_read < 0) {
		return -1;
	}

	/* JSON parser initialization */
	init_JSON_config (&config);

	module[0]                     = '\0';
	config.depth                  = 20;
	config.callback               = &json_char_callback;
	config.allow_comments         = 0;
	config.handle_floats_manually = 0;
	config.callback_ctx           = module;
	config.malloc                 = xmalloc;
	config.free                   = free;

	jc = new_JSON_parser (&config);

	/* Look for the first JSON string which is the module name */
	for (i = 0; i < nb_read && module[0]  == '\0'; i++) {
		if (! JSON_parser_char (jc, (unsigned char)((*buff)[i]))) {
			lwc_writeLog (LOG_ERR,
			_("Wrong network request: JSON syntax error"));
			delete_JSON_parser (jc);
			return -1;
		}
	}
	delete_JSON_parser (jc);

	if (module[0] == '\0') {
		lwc_writeLog (LOG_ERR,
			_("Wrong network request: module name not specified"));
		return -1;
	}

	if (strcasecmp (module, "getmycrt") == 0) {
		return reg_getmycrt (sock, *buff, nb_read, client_names);
	}
	if (strcasecmp (module, "whoami") == 0) {
		return reg_whoami (sock, *buff, nb_read, client_names);
	}
	if (strcasecmp (module, "getca") == 0) {
		return reg_getca (sock, *buff, nb_read, client_names);
	}
	if (strcasecmp (module, "register") == 0) {
		return reg_register (sock, *buff, nb_read, client_names);
	}
	lwc_writeLog (LOG_ERR, _("Wrong network request: unknown module `%s'"),
			module);
	return -1;
}

/*-----------------============== End Of File ==============-----------------*/
