/* Schedwi
   Copyright (C) 2007-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* module.c -- Modules management functions */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif


#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <ltdl.h>

#include <lwc_linkedlist.h>
#include <lwc_log.h>
#include <utils.h>
#include <xmem.h>
#include <module.h>


static lwc_LL *module_list = NULL;
static char initialized = 0;

struct module_node {
	char *name;
	char *filename;
	lt_dlhandle handle;
	int (*mod_init)();
	int (*mod_conf)(const char *, const char *);
	int (*mod_status)(	const result_t * const, const char *,
				job_status_state, job_status_state,
				long int, int, const char *);
	int (*mod_check)(const job_status_node_ptr const, time_t, short int);
	void (*mod_exit)();
};


/*
 * Free the provided module node
 */
static void
free_module_node (struct module_node *ptr)
{
	if (ptr != NULL) {
		if (ptr->name != NULL) {
			free (ptr->name);
		}
		if (ptr->filename != NULL) {
			free (ptr->filename);
		}
		if (ptr->mod_exit != NULL) {
			(*(ptr->mod_exit)) ();
		}
		if (ptr->handle != NULL) {
			lt_dlclose (ptr->handle);
		}
		free (ptr);
	}
}


/*
 * Compare a module name with a module object
 *
 * Return:
 *    0 --> the name and the module object name are the same
 *   -1 --> the name is lower than the module object name
 *    1 --> the name is greater than the module object name
 */
static int
compar_module_node1 (const char *a, const struct module_node *b)
{
	if (a == NULL) {
		if (b == NULL) {
			return 0;
		}
		return -1;
	}
	if (b == NULL || b->name == NULL) {
		return 1;
	}

	return strcmp (a, b->name);
}


/*
 * Compare two module objects
 *
 * Return:
 *    0 --> the two module objects have the same name
 *   -1 --> the first module object name is lower than the second one
 *    1 --> the first module object name is greater than the second one
 */
static int
compar_module_node2 (const struct module_node *a, const struct module_node *b)
{
	if (a == NULL || a->name == NULL) {
		if (b == NULL || b->name == NULL) {
			return 0;
		}
		return -1;
	}
	if (b == NULL || b->name == NULL) {
		return 1;
	}

	return strcmp (a->name, b->name);
}


/*
 * Initialize the module engine
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (an error message has been printed by fprintf())
 */
int
module_init ()
{
	if (initialized != 0) {
		return 0;
	}

	LTDL_SET_PRELOADED_SYMBOLS ();

	if (lt_dlinit () != 0) {
		fprintf (	stderr, _("Module initialization: %s\n"),
				lt_dlerror ());
		return -1;
	}

	if (lt_dladdsearchdir (PKGLIBDIR) != 0) {
		fprintf (	stderr,
				_("Module path initialization: %s: %s\n"),
				PKGLIBDIR, lt_dlerror ());
		lt_dlexit ();
		return -1;
	}

	initialized = 1;
	return 0;
}


/*
 * Destroy the module engine
 */
void
module_exit ()
{
	lwc_delLL (module_list, (void (*)(const void *))free_module_node);
	module_list = NULL;
	initialized = 0;
	lt_dlexit ();
}


/*
 * Add a path to the search directory list
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (an error message has been printed by fprintf())
 */
int
module_path (const char *path)
{
	if (path != NULL && lt_dladdsearchdir (path) != 0) {
		fprintf (	stderr,
				_("Module path initialization: %s: %s\n"),
				path, lt_dlerror ());
		return -1;
	}
	return 0;
}


/*
 * Load a module
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (a message has been printed by fprintf())
 *   1 --> The module init function returned a non-zero code
 */
int
module_load (const char *file)
{
	const char *name;
	char *s, *n;
	struct module_node *ptr;


	if (file == NULL) {
		return 0;
	}

	/*
	 * Build the module name (file without the path and
	 * without the extension)
	 */
	name = base_name (file);
	n = xstrdup (name);
	s = strchr (n, '.');
	if (s != NULL && s != n) {
		*s = '\0';
	}

	/* Check if the module is already loaded */
	ptr = (struct module_node *) lwc_searchSortLL (module_list, n,
		(int (*)(const void *, const void *)) compar_module_node1);
	if (ptr != NULL) {
		free (n);
		if (ptr->filename == NULL) {
			fprintf (stderr,
				_("Module %s already loaded\n"),
				file);
		}
		else {
			fprintf (stderr,
				_("Module %s already loaded from %s\n"),
				file, ptr->filename);
		}
		return 0;
	}

	/* Create the list if need be */
	if (module_list == NULL) {
		module_list = lwc_newLL ();
	}

	ptr = (struct module_node *) xmalloc (sizeof (struct module_node));
	ptr->filename = xstrdup (file);

	/* Open the module */
	ptr->handle = lt_dlopenext (file);
	if (ptr->handle == NULL) {
		fprintf (	stderr, _("Module load: error: %s: %s\n"),
				file, lt_dlerror ());
		free (ptr->filename);
		free (ptr);
		free (n);
		return -1;
	}

	/* Load the functions */
	ptr->mod_init = (int (*)()) lt_dlsym (ptr->handle, "schedwi_init");
	ptr->mod_conf = (int (*)(const char *, const char *))
				lt_dlsym (ptr->handle, "schedwi_conf");
	ptr->mod_status = (int (*)(const result_t * const, const char *,
				job_status_state, job_status_state,
				long int, int, const char *)) lt_dlsym (
						ptr->handle, "schedwi_status");
	ptr->mod_check = (int (*)(	const job_status_node_ptr const,
					time_t, short int))
				lt_dlsym (ptr->handle, "schedwi_check");
	ptr->mod_exit = (void (*)()) lt_dlsym (ptr->handle, "schedwi_exit");

	/* Call the module init function */
	if (ptr->mod_init && (*(ptr->mod_init)) () != 0) {
		fprintf (stderr, _("Module %s: init function error\n"), n);
		lt_dlclose (ptr->handle);
		free (ptr->filename);
		free (ptr);
		free (n);
		return 1;
	}

	/* Link the new module object to the list of modules */
	ptr->name = n;
	lwc_addLL (module_list,
		   ptr,
		   (int (*)(const void *, const void *))compar_module_node2);
	return 0;
}


/*
 * Forward a configuration parameter to the corresponding module
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error: unknown module or the module configuration function
 *         return a non-zero value.  A message has been printed by fprintf()
 */
int
module_conf (const char *name, const char *key, const char *value)
{
	struct module_node *ptr;


	if (name == NULL || key == NULL || value == NULL) {
		return 0;
	}

	/* Retrieve the module object */
	ptr = (struct module_node *) lwc_searchSortLL (module_list, name,
		(int (*)(const void *, const void *)) compar_module_node1);
	if (ptr == NULL) {
		fprintf (stderr,
			_("Unknown module `%s' for parameter `%s=%s'\n"),
			name, key, value);
		return -1;
	}

	if (ptr->mod_conf != NULL && (*(ptr->mod_conf)) (key, value) != 0) {
		fprintf (	stderr,
				_("Module %s: error: parameter `%s=%s'\n"),
				name, key, value);
		return -1;
	}
	return 0;
}


/**
 * Call all the module `schedwi_status' functions for the provided job with
 * the provided status
 *
 * @param[in] result A result_t object which contains the job workload, id, ...
 * @param[in] job_name_with_path Full name of the job (with its path)
 * @param[in] new_state New state of the job/jobset
 * @param[in] previous_state Previous state of the job/jobset
 * @param[in] duration How long (in secondes) the job was in the previous_state
 * @param[in] retry_num Retry number if the job has failed and is going to be
 *                      retried
 * @param[in] status_message Description.  For a fail job it may be the reason
 *                           why it failed. May be empty or NULL
 *
 * @return 0 on sucess or -1 if at least one `schedwi_status' function returned
 *         a non-zero code.  An error message is logged by lwc_writeLog().
 */
int
module_status (	const result_t * const result, const char *job_name_with_path,
		job_status_state new_state, job_status_state previous_state,
		long int duration, int retry_num, const char *status_message)
{
	int ret;
	struct module_node *ptr;


#if HAVE_ASSERT_H
	assert (result != NULL && job_name_with_path != NULL);
#endif

	ret = 0;
	lwc_rewindLL (module_list);
	while ((ptr = (struct module_node *)lwc_nextLL (module_list)) != NULL)
	{
		if (	   ptr->mod_status != NULL
			&& (*(ptr->mod_status)) (	result,
							job_name_with_path,
							new_state,
							previous_state,
							duration,
							retry_num,
							status_message) != 0)
		{
			lwc_writeLog (	LOG_ERR,
_("Module %s: schedwi_status: Workload %d: %s (id %lld): from %s to %s: error"),
					ptr->name,
					result->workload_int,
					job_name_with_path,
					result->job_id_int,
					job_status_state2str (previous_state),
					job_status_state2str (new_state));
			ret = -1;
		}
	}
	return ret;
}


/**
 * Call all the module `schedwi_check' functions for the provided job
 *
 * @param[in] job_details A job_status_node_ptr object which contains the
 *                        job/jobset details.
 * @param[in] start_time Time (time(2) format) at which the job should start.
 * @param[in] start_limit Number of minutes for the job/jobset to start past
 *                        its start time. 0 means no limit.
 * @return 0 if no error (the job/jobset can be started: all the
 *         `schedwi_check' functions returned 0). 1 if the job/jobset cannot be
 *         started yet (a `schedwi_check' function returned a code greater than
 *         0). -1 in case of error (A `schedwi_check' function returned a code
 *         lower than 0).  A message has been logged by lwc_writeLog().
 */
int
module_check (	const job_status_node_ptr const job_details, time_t start_time,
		short int start_limit)
{
	int ret;
	struct module_node *ptr;


#if HAVE_ASSERT_H
	assert (job_details != NULL);
#endif

	lwc_rewindLL (module_list);
	while ((ptr = (struct module_node *)lwc_nextLL (module_list)) != NULL)
	{
		if (ptr->mod_check != NULL) {
			ret = (*(ptr->mod_check)) (	job_details,
							start_time,
							start_limit);
			if (ret > 0) {
				return 1;
			}
			if (ret < 0) {
				lwc_writeLog (	LOG_ERR,
	_("Module %s: schedwi_check: Workload %d: %s (id %lld): error"),
					ptr->name,
					job_details->workload_date,
					job_details->job_name_with_path,
					job_details->job_id);
				return -1;
			}
		}
	}
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
