/* Schedwi
   Copyright (C) 2013-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* manual_trigger.c -- Alert someone that a manual job is waiting */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <utils.h>
#include <schedwi_system.h>
#include <conf.h>
#include <schedwi_time.h>
#include <sql_acknowledge_manual.h>
#include <sql_hierarchy.h>
#include <sql_job.h>
#include <lwc_log.h>
#include <xmem.h>
#include <manual_trigger.h>


/*
 * Replace all occurences of keywords in a file.
 *
 * @param file_in:
 *        The source file in which keywords will be looked for.  This file
 *        is not modified.
 * @param fd_out:
 *        Output file descriptor of the output file.
 * @param keys:
 *        List of keywords to look for.  In `file_in' this keywords must
 *        be surrounded by %'s. For instance, for a SCHEDWI_JOBID keyword, all
 *        the occurences of %SCHEDWI_JOBID% in the source file will be replaced
 *        in `fd_out' by the associated value.
 * @param values:
 *        List of values.  `keys' and `values' work together.  The value
 *        at position 1 in associated with the keyword at the same position.
 * @param len:
 *        Number of keywords/values in the `keys' and `values' arrays.
 * @return:
 *        0 in case of success or -1 in case of failure (errno is set)
 */
static int
my_sed(	const char *file_in, int fd_out,
	const char * const *keys, const char * const *values, size_t len)
{
	char *file_content, *s, *d, *k, *save_s;
	size_t length;
	size_t *values_len;
	char **match_keys;
	size_t l, i;


#ifdef HAVE_ASSERT_H
	assert (file_in != NULL && file_in[0] != '\0' && fd_out >= 0);
#endif

	/* Store the whole content of the input file in a string */
	length = 0;
	file_content = read_file (file_in, &length);
	if (file_content == NULL) {
		return -1;
	}

	if (len == 0) {
		/* No keys/values. Just copy the file */
		(void)write (fd_out, file_content, length);
		free (file_content);
		return 0;
	}

	/* Array to store the length of each value */
	values_len = (size_t *) xmalloc (sizeof (size_t) * len);

	/* Array to store the keywords to look for (with surrounding %'s) */
	match_keys = (char **) xmalloc (sizeof (char *) * len);

	for (i = 0; i < len; i++) {
		values_len[i] = strlen (values[i]);
		l = strlen (keys[i]) + 2;
		match_keys[i] = (char *) xmalloc (l + 1);
		/* The key to search for is surrounded by `%' (ie. %JOBID%) */
		match_keys[i][0] = '%';
		strcpy (match_keys[i] + 1, keys[i]);
		match_keys[i][l - 1] = '%';
		match_keys[i][l] = '\0';

		/*
		 * Compute the size of the final array (the output string)
		 * by finding the number of occurences of the keyword to
		 * substitute.
		 */
		s = file_content;
		while ((s = strstr(s, match_keys[i])) != NULL) {
			length += values_len[i];
			s += l;
		}
	}

	/* Do the substitutions of each keyword */
	s = file_content;
	for (i = 0; i < len; i++) {
		save_s = s;
		d = (char *) xmalloc (length);
		l = 0;
		while ((k = strstr (s, match_keys[i])) != NULL) {
			strncpy (d + l, s, (size_t)(k - s));
			l += k - s;
			strcpy (d + l, values[i]);
			l += values_len[i];
			s = k + strlen (match_keys[i]);
		}
		strcpy (d + l, s);
		s = d;
		free (save_s);
		free (match_keys[i]);
	}
	free (match_keys);
	free (values_len);

	/* Write the result to the output file */
	(void)write (fd_out, s, strlen (s));
	free (s);
	return 0;
}


/*
 * Generate a random string.
 *
 * @param num_chars:
 *	Number of characters in the returned string.
 * @return:
 *	The null-terminated random string to be freed by the caller by free()
 */
static char *
rand_string (unsigned int num_chars)
{
	const char *set[3] =	{	"bcdfghjkmnpqrstvwxz",
					"aeiouy",
					"23456789-"
				};
	char *s;
	unsigned int i;
	unsigned int l[3];


	l[0] = strlen (set[0]);
	l[1] = strlen (set[1]);
	l[2] = strlen (set[2]);

	s = (char *) xmalloc (num_chars + 1);
	for (i = 0; i < num_chars - 1; i++) {
		s[i] = set[i % 2][rand () % l[i % 2]];
	}
	s[num_chars - 1 ] = set[2][rand () % l[2]];
	s[num_chars] = '\0';
	return s;
}


/*
 * Generate a template string for the mkstemp() function.
 *
 * @return:
 *       the string (to be freed by the caller by free()) that contains the
 *       template.
 */
static
char *get_template_for_mkstemp()
{
	char *tmp, *s;


#if HAVE_GETENV
	tmp = getenv ("TMPDIR");
	if (tmp == NULL) {
		tmp = "/tmp";
	}
#else
	tmp = "/tmp";
#endif

	s = (char *) xmalloc (strlen (tmp) + strlen (DIR_SEP) + 15);
	strcpy (s, tmp);
	strcat (s, DIR_SEP);
	strcat (s, "schedwiXXXXXX");
	return s;
}


/*
 * Error callback function for sql_acknowledge_get()
 */
static void
sql_get_error_logger (void *data, const char *msg, int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
_("Database error while trying to retrieve manual status of a job/jobset"));
	}
}


/*
 * Error callback function for sql_job_get_manual_command()
 */
static void
sql_job_get_command_error_logger (void *data, const char *msg, int err_code)

{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
_("Database error while trying to retrieve the command to alert for a manual job/jobset"));
	}
}


/**
 * Check if the provided job needs a manual validation to start.  If yes,
 * check that it has been validated.  If no, add an entry in the
 * acknowledge_manual database table (with the http urn and password to use
 * to acknowledge the message from the Web interface) and execute the
 * command to alert operators (job->manual_command)
 *
 * @param[in] ptr Job/jobset details.
 * @param[in] hierarchy_list Hierarchy list (see sql_hierarchy.c) for this
 *                           job/jobset.
 * @param[in] start_limit Number of minutes past the expected start time for
 *                        the job/jobset to start.
 * @return
 *	0 if the job can be started (it's not a manual job or it has been
 *	validated)
 *      1 if the job is waiting for validation and should not start yet.
 *	-1 in case of error (an error message has already been logged with
 *      lwc_writeLog())
 */
int
check_manual_trigger (	job_status_node_ptr ptr, lwc_LL *hierarchy_list,
			time_t start_time, short int start_limit)
{
	const char *template_file;
	int status, i, ret;
	time_t ack_time;
	char *username;
	char *password, *urn, *s, *err_msg, *tmp_file, *manual_command;
	const char *keys[6] = {	"SCHEDWI_JOBPATH", "SCHEDWI_JOBID",
				"SCHEDWI_START_TIME", "SCHEDWI_TIME_LIMIT",
				"SCHEDWI_URN", "SCHEDWI_PASSWORD" };
        char *values[6];
	size_t num_keys = 6;  /* Keep in sync with the num of items in keys! */
	char *variable_system[20];
	time_t t;


#ifdef HAVE_ASSERT_H
	assert (ptr != NULL && hierarchy_list != NULL);
#endif

	/* Is it a manual job/jobset ? */
	if (hierarchy_list_get_manual (hierarchy_list) == 0) {
		return 0;
	}

	/* Retrieve the acknowledge status from the database */
	username = NULL;
	if (sql_acknowledge_get (	ptr->workload_date, ptr->job_id,
					&status, &ack_time, &username,
					sql_get_error_logger, NULL) != 0)
	{
		return -1;
	}

	/* Not yet acknowledged */
	if (status == 0) {
		if (username != NULL) {
			free (username);
		}
		return 1;
	}

	/* Acknowledged */
	if (status == 1) {
		s = schedwi_time_strftime (	SCHEDWI_DEFAULT_DATEFORMAT,
						&ack_time);
		if (s == NULL) {
			/* Convertion failed */
			return 0;
		}
		lwc_writeLog (	LOG_INFO,
	_("Workload %d: %s (id %lld): Start acknowledged by %s at %s"),
				ptr->workload_date, ptr->job_name_with_path,
				ptr->job_id,
				(username != NULL && username[0] != '\0') ?
					username : _("unkwnown"),
				s);
		free (s);
		if (username != NULL) {
			free (username);
		}
		return 0;
	}

	/* No entry in the database.  Create one */
	lwc_writeLog (	LOG_INFO,
_("Workload %d: %s (id %lld): Manual %s ready: requesting acknowledgement"),
			ptr->workload_date, ptr->job_name_with_path,
			ptr->job_id,
			(hierarchy_list_get_type (hierarchy_list) == JOBSET) ?
				_("jobset") : _("job"));

	/*
	 * Add a new entry in the database (URL is the web page that can be
	 * used to acknowledge the start)
	 */
	password = rand_string (6);
	s = rand_string (6);
	urn = (char *) xmalloc (20);   /* schedwi-<s> (s is 6 chars long) */
	strcpy (urn, "schedwi-");
	strcat (urn, s);
	free (s);

	err_msg = NULL;
	ret = sql_acknowledge_add (	ptr->workload_date, ptr->job_id,
					urn, password, &err_msg);
	if (ret != 0) {
		free (urn);
		free (password);
		if (err_msg != NULL) {
			lwc_writeLog (LOG_CRIT, err_msg);
			free (err_msg);
		}
		else {
			lwc_writeLog (	LOG_CRIT,
_("Workload %d: %s (id %lld): Cannot add the acknowledgement req in database"),
					ptr->workload_date,
					ptr->job_name_with_path, ptr->job_id);
		}
		return -1;
	}

	manual_command = NULL;
	if (sql_job_get_manual_command (ptr->workload_date,
					hierarchy_list,
					&manual_command,
					sql_job_get_command_error_logger,
					NULL) != 0)
	{
		free (urn);
		free (password);
		return -1;
	}
	if (manual_command == NULL || manual_command[0] == '\0') {
		if (manual_command != NULL) {
			free (manual_command);
		}
		free (urn);
		free (password);
		return 1;
	}

	/* Convert the template */
	ret = conf_get_param_string (	"MANUAL_TRIGGER_TEMPLATE_FILE",
					&template_file);
#if HAVE_ASSERT_H
	assert (ret == 0);
#endif
	if (access (template_file, R_OK) != 0) {
		lwc_writeLog (	LOG_INFO, _("%s: %s"),
				template_file, strerror (errno));
		tmp_file = NULL;
		variable_system[0] = xstrdup ("SCHEDWI_TEMPLATE=");
	}
	else {
		/* SCHEDWI_JOBPATH */
		values[0] = xstrdup (ptr->job_name_with_path);
			
		/* SCHEDWI_JOBID */
		values[1] = xstrdup (ptr->workload_jobid);

		/* SCHEDWI_START_TIME */
		values[2] = schedwi_time_strftime ("%H:%M", &start_time);

		/* SCHEDWI_TIME_LIMIT */
		if (start_limit > 0) {
			/*
			 * start_limit - 1 because if start_limit is 24 hours
			 * the displayed HH:MM will show the current hour/min
			 * and this may confuse the user.
			 */
			t = start_time + (start_limit - 1) * 60;
			values[3] = schedwi_time_strftime ("%H:%M", &t);
		}
		else {
			values[3] = xstrndup (_("no limit"), 100);
		}

		/* SCHEDWI_URN */
		values[4] = xstrdup (urn);
	
		/* SCHEDWI_PASSWORD */
		values[5] = xstrdup (password);

		/* Get a temporary file name to convert the template */
		tmp_file = get_template_for_mkstemp();
		i = mkstemp (tmp_file);
		if (i < 0) {
			lwc_writeLog (	LOG_ERR, _("%s: %s"),
					tmp_file, strerror (errno));
			free (tmp_file);
			for (i = 0; i < num_keys; i++) {
				free (values[i]);
			}
			free (manual_command);
			free (urn);
			free (password);
			return -1;
		}

		if (my_sed (	template_file, i, keys,
				(const char * const *)values, num_keys) != 0)
		{
			lwc_writeLog (	LOG_ERR, _("%s: %s"),
					template_file, strerror (errno));
			close (i);
			my_unlink (tmp_file);
			free (tmp_file);
			for (i = 0; i < num_keys; i++) {
				free (values[i]);
			}
			free (manual_command);
			free (urn);
			free (password);
			return -1;
		}
		close (i);
		for (i = 0; i < num_keys; i++) {
			free (values[i]);
		}

		variable_system[0] = (char *) xmalloc (
					  strlen ("SCHEDWI_TEMPLATE")
					+ strlen (tmp_file) + 2);
		strcpy (variable_system[0], "SCHEDWI_TEMPLATE=");
		strcat (variable_system[0], tmp_file);
	}

	variable_system[1] = schedwi_time_strftime (
				"SCHEDWI_START_TIME=%H:%M", &start_time);
	variable_system[2] = schedwi_time_strftime (
				"SCHEDWI_START_TIME_EPOCH=%s", &start_time);
	if (start_limit > 0) {
		/*
		 * start_limit - 1 because if start_limit is 24 hours
		 * the displayed HH:MM will show the current hour/min
		 * and this may confuse the user.
		 */
		t = start_time + (start_limit - 1) * 60;
		variable_system[3] = schedwi_time_strftime (
					"SCHEDWI_TIME_LIMIT=%H:%M", &t);
		variable_system[4] = schedwi_time_strftime (
					"SCHEDWI_TIME_LIMIT_EPOCH=%s", &t);
	}
	else {
		variable_system[3] = xstrdup ("SCHEDWI_TIME_LIMIT=");
		variable_system[4] = xstrdup ("SCHEDWI_TIME_LIMIT_EPOCH=0");
	}
	variable_system[5] = (char *) xmalloc (   strlen ("SCHEDWI_PASSWORD")
						+ strlen (password) + 2);
	strcpy (variable_system[5], "SCHEDWI_PASSWORD=");
	strcat (variable_system[5], password);
	variable_system[6] = (char *) xmalloc (   strlen ("SCHEDWI_URN")
						+ strlen (urn) + 2);
	strcpy (variable_system[6], "SCHEDWI_URN=");
	strcat (variable_system[6], urn);

	variable_system[7] = NULL;
	free (urn);
	free (password);

	/* Execute the command to alert of a waiting manual job */
	ret = schedwi_system (	manual_command, ptr->workload_jobid,
				ptr->job_name_with_path,
				(const char * const *)variable_system, 7);
	for (i = 0; i < 7; i++) {
		free (variable_system[i]);
	}
	if (tmp_file != NULL) {
		my_unlink (tmp_file);
		free (tmp_file);
	}
	switch (ret) {
		case 0: break;
		case 127:
			lwc_writeLog (	LOG_ERR,
			_("Workload %d: %s (id %lld): %s: Failed to start"),
					ptr->workload_date,
					ptr->job_name_with_path, ptr->job_id,
					manual_command);
			break;
		case 300:
			lwc_writeLog (	LOG_ERR,
			_("Workload %d: %s (id %lld): %s: Killed"),
					ptr->workload_date,
					ptr->job_name_with_path, ptr->job_id,
					manual_command);
			break;
		case -1:
			lwc_writeLog (	LOG_CRIT,
			_("Workload %d: %s (id %lld): %s: System error"),
					ptr->workload_date,
					ptr->job_name_with_path, ptr->job_id,
					manual_command);
			break;
		default:
			lwc_writeLog (	LOG_ERR,
			_("Workload %d: %s (id %lld): %s: Exit code: %d"),
					ptr->workload_date,
					ptr->job_name_with_path, ptr->job_id,
					manual_command, ret);
			break;
	}
	free (manual_command);
	return 1;
}

/*-----------------============== End Of File ==============-----------------*/
