/* Schedwi
   Copyright (C) 2014, 2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * @file job_status_node.c
 * Manage the job_status_node object (status parameters of a job/jobset)
 */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#include <sql_status.h>
#include <job_status_set.h>
#include <lwc_log.h>
#include <utils.h>
#include <xmem.h>
#include <job_status_node.h>


/*
 * Error callback function for the sql_status_get() and
 * job_status_node_set_wait_reason() functions.
 */
static void
sql_get_error_logger (void *data, const char *msg, int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
		_("Database error while trying to retrieve job status"));
	}
}


/**
 * Return a new job_status_node object.
 * The returned object has to be freed by the caller with
 * job_status_node_destroy()
 *
 * @return The new job_status_node object (to be freed by the caller by
 *         job_status_node_destroy())
 */
job_status_node_ptr
job_status_node_new ()
{
	return (job_status_node_ptr) xcalloc (1, sizeof (job_status_node));
}


/**
 * Destroy (free) the provided job_status_node object.
 *
 * @param[in] ptr The pointer to the object to free.
 */
void
job_status_node_destroy (job_status_node_ptr ptr)
{
	if (ptr != NULL) {
		if (ptr->message != NULL) {
			free (ptr->message);
		}
		if (ptr->job_name_with_path != NULL) {
			free (ptr->job_name_with_path);
		}
		if (ptr->workload_jobid != NULL) {
			free (ptr->workload_jobid);
		}
		free (ptr);
	}
}


/**
 * Return a new populated job_status_node object.
 * The returned object has to be freed by the caller with
 * job_status_node_destroy()
 *
 * @param[in] workload_date Workload date of the job/jobset (YYYYMMDD).
 * @param[in] job_id Job/jobset ID
 * @param[in] job_name_with_path Full job/jobset name.
 * @return The new job_status_node object (to be freed by the caller by
 *         job_status_node_destroy()) or NULL in case of error (a message has
 *         been logged by lwc_writeLog())
 */
job_status_node_ptr
job_status_node_get (	int workload_date,
			unsigned long long int job_id,
			const char *job_name_with_path)
{
	job_status_node_ptr ptr;
	int ret, status;
	unsigned int len_workload;


	ptr = job_status_node_new ();
	ret = sql_status_get (	workload_date, job_id,
				&status,
				&(ptr->time_status_set),
				&(ptr->retry_num),
				&(ptr->wait_reason),
				&(ptr->message),
				&(ptr->duration),
				sql_get_error_logger, NULL);
	if (ret != 0) {
		job_status_node_destroy (ptr);
		return NULL;
	}
	ptr->workload_date = workload_date;
	ptr->job_id = job_id;
	ptr->workload_jobid = (char *) xmalloc (100);
	len_workload = copy_ulltostr (workload_date, ptr->workload_jobid);
	(ptr->workload_jobid)[len_workload] = SCHEDWI_WORKLOAD_SEPARATOR;
	copy_ulltostr (job_id, (ptr->workload_jobid) + len_workload + 1);
	ptr->job_name_with_path = xstrdup ((job_name_with_path != NULL)
							? job_name_with_path
							: _("Unknown"));
	ptr->status = (status == 0)	? JOB_STATUS_STATE_WAITING
		  			: job_status_state_int2status (status);
	return ptr;
}


/**
 * Change the status message in the provided job_status_node object.
 * The new message is only set in the given object; it is not writen in the
 * database (see job_status_node_set())
 *
 * @param[in] ptr The job_status_node object.
 * @param[in] new_message The new status message to set.
 */
void
job_status_node_set_message (job_status_node_ptr ptr, const char *new_message)
{
	if (ptr != NULL) {
		if (ptr->message != NULL) {
			free (ptr->message);
		}
		if (new_message != NULL) {
			ptr->message = xstrdup (new_message);
		}
		else {
			ptr->message = NULL;
		}
	}
}


/**
 * Change the status in the provided job_status_node object.
 * The new status is only set in the given object; it is not writen in the
 * database (see job_status_node_set())
 *
 * @param[in] ptr The job_status_node object.
 * @param[in] new_status The new status to set.
 * @param[in] status_time Time (as in time(2)) at which the status has changed.
 *                        If 0, the current time is going to be used.
 */
void
job_status_node_set_status (	job_status_node_ptr ptr,
				job_status_state new_status,
				time_t status_time)
{
	if (ptr != NULL) {
		if (ptr->previous_status == JOB_STATUS_STATE_UNDEFINED) {
			ptr->previous_status = ptr->status;
		}
		if (ptr->previous_time_status_set == 0) {
			ptr->previous_time_status_set = ptr->time_status_set;
		}
		ptr->status = new_status;
		if (status_time <= 0) {
			status_time = time (NULL);
		}
		ptr->duration = status_time - ptr->previous_time_status_set;
		ptr->time_status_set = status_time;
	}
}


/**
 * Change the waiting reason in the provided job_status_node object and also
 * in the database.
 *
 * @param[in] ptr The job_status_node object.
 * @param[in] wait_reason The waiting reason (see sql_status.h for the mask)
 * @return 0 on success and -1 in case of error (an error message has been
 *         logged by lwc_writeLog()
 */
int
job_status_node_set_wait_reason (job_status_node_ptr ptr, int wait_reason)
{
	if (ptr == NULL || ptr->wait_reason == wait_reason) {
		return 0;
	}

	ptr->wait_reason = wait_reason;
	return sql_status_update_wait_reason (	ptr->workload_date,
						ptr->job_id,
						wait_reason,
						sql_get_error_logger, NULL);
}


/**
 * Update the provided job status into the database and start the job if need
 * be.
 *
 * @param[in] ptr The job_status_node object to update in the database.
 * @return 0 on success and -1 in case of error (an error message has been
 *         logged by lwc_writeLog()
 */
int
job_status_node_set (lwc_LL *hierarchy_list, job_status_node_ptr ptr)
{
	if (ptr == NULL) {
		return 0;
	}

	if (job_status_force (hierarchy_list, ptr) != 0) {
		return -1;
	}
	ptr->previous_status = JOB_STATUS_STATE_UNDEFINED;
	ptr->previous_time_status_set = 0;
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
