/* Schedwi
   Copyright (C) 2014 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* get_job_status_from_agents.c -- Wait and manage agent connections */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_SYS_SELECT_H
#include <sys/select.h>
#else
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_TIME_H
#include <time.h>
#endif

#if HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#include <lwc_log.h>
#include <sql_hosts.h>
#include <net_utils.h>
#include <net_utils_sock.h>
#include <net_utils_ssl.h>
#include <net_parse.h>
#include <xmem.h>
#include <get_job_status_from_agents.h>


/**
 * Poll agents that don't report their job status directly (because a firewall
 * denies them the access to the Schedwi server for instance) These agents are
 * marqued as such in the database.
 */
static void
poll_agents ()
{
	return;
}


/**
 * Error callback function for the sql_host_certificate() function
 */
static void
sql_host_certificate_error_logger (	void *data, const char *msg,
					int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
	_("Database error while trying to retrieve the agent parameters"));
	}
}


/**
 * Manage the connection with the agent (probably retrieve the status of a
 * finished job)
 *
 * @param[in] sock The socket connected with the agent.
 * @param[in] sockname Address of the peer socket (agent).
 * @param[in] len Size of the sockname object.
 * @return 0 on success and -1 on error (an error message is displayed using
 *         lwc_writeLog()) In any case the socket (in sock) is closed by this
 *         function.
 */
static int
manage_agent_connection (	int sock, struct sockaddr_storage *sockname,
				socklen_t *len)
{
	char **names;
	char use_ssl, *client_certificate;
	unsigned long int len_client_certificate;
	schedwi_BIO in;


	/* Retrieve the names associated with the in_addr structure */
	if (build_client_names_from_addr (sockname, *len, &names) != 0)
	{
		net_close_sock (sock);
		return -1;
	}

	/*
	 * Retrieve whether SSL must be used for the connexion and retrieve
	 * the associated certificate
	 */
	client_certificate = NULL;
	if (sql_host_certificate (	names, &use_ssl, &client_certificate,
					&len_client_certificate,
					sql_host_certificate_error_logger,
					NULL) != 0)
	{
		destroy_client_names (names);
		net_close_sock (sock);
		return -1;
	}

	/*
	 * No SSL
	 */
	if (use_ssl == 0) {
		in.use_ssl = 0;
		in.trusted_cert_ptr = NULL;
		in.fd = sock;
		if (client_certificate != NULL) {
			free (client_certificate);
		}
		destroy_client_names (names);
		net_parse (&in);
		net_close_sock (sock);
		return 0;
	}


	/*
	 * SSL
	 */
	in.trusted_cert_ptr = (gnutls_certificate_credentials_t *) xmalloc (
				sizeof (gnutls_certificate_credentials_t));
	in.use_ssl = 1;
	in.fd = sock;

	if (net_server_ssl (&(in.session), in.fd, in.trusted_cert_ptr,
			(const char **)names,
			client_certificate, len_client_certificate) != 0)
	{
		if (client_certificate != NULL) {
			free (client_certificate);
		}
		destroy_client_names (names);
		free (in.trusted_cert_ptr);
		net_close_sock (sock);
		return -1;
	}

	if (client_certificate != NULL) {
		free (client_certificate);
	}
	destroy_client_names (names);
	net_parse (&in);
	net_close_ssl (in.session, in.fd, in.trusted_cert_ptr);
	return 0;
}


/**
 * Wait for agent connections (no more than one minute)
 *
 * @param[in] sock_set The sockets to listen to.
 * @param[in] sock_max Highest-numbered file descriptor in the socket set.
 * @param[in] allow_from The list of allowed IP adresses (see the ALLOW_FROM
 *                       directive in the configuration file)
 * @return 0 on success and -1 on error (a message is displayed using
 *         lwc_writeLog())
 */
static int
wait_agents (fd_set *sock_set, int sock_max, const char *allow_from)
{
	time_t start;
	struct timeval max_wait;
	fd_set rfds;
	int ret, spent_sec, sock;
	struct sockaddr_storage sockname;
	socklen_t len;


	if (sock_set == NULL) {
		return 0;
	}

	start = time (NULL);
	while ((spent_sec = difftime (time (NULL), start)) < 60) {
		/* Wait for connections */
		rfds = *sock_set;
		max_wait.tv_sec = 60 - spent_sec;
		max_wait.tv_usec = 0;
		ret = select (sock_max, &rfds, NULL, NULL, &max_wait);
		if (ret == 0) {
			/* Timeout */
			break;
		}
		if (ret < 0) {
			if (errno == EINTR || errno == EAGAIN) {
				continue;
			}
			lwc_writeLog (	LOG_CRIT,
					_("Network error: select: %s"),
					strerror (errno));
			return -1;
		}

		/* Deal with the pending connections */
		do {
			/* Get one connection from the set */
			sock = net_accept (	&rfds, sock_max,
						&ret, allow_from,
						&sockname, &len);
			if (sock == -1) {
				return -1;
			}
			if (sock >= 0) {
				/* Manage the agent connection */
				if (manage_agent_connection (sock, &sockname,
								&len) < 0)
				{
					return -1;
				}
			}
		} while (sock != -2);  /* -2 means no more socket in the set */
	}
	return 0;
}


/**
 * Wait for agents to report the status of some jobs.
 *
 * @param[in] sock_set The sockets to listen to.
 * @param[in] sock_max Highest-numbered file descriptor in the socket set.
 * @param[in] allow_from The list of allowed IP adresses (see the ALLOW_FROM
 *                       directive in the configuration file)
 * @return 0 on success and an other code on error (a message is displayed
 *         using lwc_writeLog())
 */
int
get_job_status_from_agents (	fd_set *sock_set,
				int sock_max,
	       			const char *allow_from)
{
	/* TODO
	 * Poll agents that don't report their job status directly (because
	 * a firewall denies them the access to the Schedwi server for
	 * instance)
	 * These agents are marqued as such in the database.
	 */
	poll_agents ();

	/* Wait for 1 minute to agents */
	return wait_agents (sock_set, sock_max, allow_from);
}

/*------------------------======= End Of File =======------------------------*/
