/* Schedwi
   Copyright (C) 2007-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * env_utils.h -- Environment array management functions
 *
 * The environment array is an array of strings.  By convention these strings
 * have the form `name=value'.  This array is embedded in the environment_t
 * structure but can be used directly by system functions like exec.
 */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#endif

#include <xmem.h>
#include <env_utils.h>


/*
 * Init the environment structure
 */
void
init_environment (environment_t *e)
{
	if (e != NULL) {
		e->env = NULL;
		e->length = 0;
		e->idx = 0;
	}
}


/*
 * Free the environment structure
 */
void
destroy_environment (environment_t *e)
{
	int i;

	if (e != NULL) {
		if (e->env != NULL && e->length > 0) {
			for (i = 0; i < e->idx; i++) {
				if ((e->env)[i] != NULL) {
					free ((e->env)[i]);
				}
			}
			free (e->env);
		}
		e->env = NULL;
		e->length = 0;
		e->idx = 0;
	}
}


/*
 * Add a pair key-value at the end of the environment array.  The key and
 * value parameters are provided with their length.
 */
void
add_env_s (	environment_t *e,
		const char *key, size_t key_len,
		const char *val, size_t val_len)
{
	char **tmp;


	if (e == NULL || key == NULL || key_len == 0) {
		return;
	}

	if (e->idx + 1 >= e->length) {
		tmp = (char **) xrealloc (e->env,
					(e->length + 10) * sizeof (char **));
		e->env = tmp;
		e->length += 10;
	}

	(e->env)[e->idx] = (char *) xmalloc (key_len + val_len + 2);
	strncpy ((e->env)[e->idx], key, key_len);
	(e->env)[e->idx][key_len] = '=';
	if (val == NULL || val_len == 0) {
		(e->env)[e->idx][key_len + 1] = '\0';
	}
	else {
		strncpy ((e->env)[e->idx] + key_len + 1, val, val_len);
		(e->env)[e->idx][key_len + val_len + 1] = '\0';
	}
	(e->env)[++e->idx] = NULL;
}


/*
 * Add a pair key-value at the end of the environment array.
 */
void
add_env (environment_t *e, const char *key, const char *val)
{
	add_env_s (e, key, strlen (key), val, strlen (val));
}


/*
 * Search the provided string (var) in the env array (of size var_len)
 *
 * Return:
 *    The index of the var in the env array or
 *    -1 if not found
 */
int
search_environment (environment_t *e, const char *var, size_t var_len)
{
	int i;

	if (e == NULL || var == NULL || var_len == 0) {
		return -1;
	}

	for (i = 0; i < e->idx; i++) {
		if (	   strncmp (var, (e->env)[i], var_len) == 0
			&& (e->env)[i][var_len] == '=')
		{
			return i;
		}
	}
	return -1;
}


/*
 * Return the variable at the idx position in the array structure (or NULL
 * if idx is out of range or e is NULL)
 */
const char *
getvar_environment (environment_t *e, int idx)
{
	if (e == NULL || idx < 0 || idx >= e->idx) {
		return NULL;
	}
	return (e->env)[idx];
}

/*-----------------============== End Of File ==============-----------------*/
