/* Schedwi
   Copyright (C) 2007-2014 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* commands.c -- commands management functions */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#include <utils.h>
#include <lwc_log.h>
#include <sql_commands.h>
#include <xmem.h>
#include <commands.h>


/*
 * Convert a command to an int
 */
static int
command_to_int (command_t cmd)
{
	switch (cmd) {
		case COMMAND_CHANGE_STATUS:	return 1;
		case COMMAND_STOP:		return 2;
		default:			return 0;
	}
}


/*
 * Convert an int to a command
 */
static command_t
command_from_int (int cmd)
{
	switch (cmd) {
		case 1:		return COMMAND_CHANGE_STATUS;
		case 2:		return COMMAND_STOP;
		default:	return COMMAND_UNKNOWN;
	}
}


/*
 * Create a new command_action object
 *
 * Return:
 *   The new object (to be freed by the caller by command_action_destroy())
 */
static command_action_ptr
command_action_new (	int workload,
			unsigned long long int job_id,
			const char *user_details,
			int cmd,
			int parameter)
{
	command_action_ptr ptr;


	ptr = (command_action_ptr) xmalloc (sizeof (command_action));
	if (user_details != NULL) {
		ptr->user_details = xstrdup (user_details);
	}
	else {
		ptr->user_details = NULL;
	}

	ptr->workload = workload;
	ptr->job_id = job_id;
	ptr->cmd = command_from_int (cmd);
	ptr->parameter = parameter;
	return ptr;
}


/*
 * Destroy the provided command_action object
 */
static void
command_action_destroy (command_action_ptr ptr)
{
	if (ptr != NULL) {
		if (ptr->user_details != NULL) {
			free (ptr->user_details);
		}
		free (ptr);
	}
}


/*
 * Compare two command_action objects.  This function is used by
 * lwc_replaceUnsortedLL()
 */
static int
command_action_cmp (const void *key, const void *obj)
{
	const command_action_ptr k = (command_action_ptr)key;
	const command_action_ptr o = (command_action_ptr)obj;
	int ret;


	ret = k->workload - o->workload;
	return (ret != 0)? ret: (k->job_id - o->job_id);
}


/*
 * Error callback function for the sql_command_done() and sql_command_failed()
 * functions
 */
static void
sql_command_update_error_logger (	void *data, const char *msg,
					int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
	    _("Database error while trying to set the status of a command"));
	}
}


/*
 * Update the status of the command in the database
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error.  An error message has been logged by lwc_writeLog()
 */
int
command_done (command_action_ptr ptr)
{
	if (	   ptr != NULL
		&& sql_command_done (	ptr->workload,
					ptr->job_id,
					command_to_int (ptr->cmd),
					ptr->parameter,
					sql_command_update_error_logger,
					NULL) != 0)
	{
		return -1;
	}
	return 0;
}


/*
 * Update the status of the command in the database
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error.  An error message has been logged by lwc_writeLog()
 */
int
command_failed (command_action_ptr ptr)
{
	if (	   ptr != NULL
		&& sql_command_failed (	ptr->workload,
					ptr->job_id,
					command_to_int (ptr->cmd),
					ptr->parameter,
					sql_command_update_error_logger,
					NULL) != 0)
	{
		return -1;
	}
	return 0;
}


/*
 * Error callback function for the sql_command_get_commands() function
 */
static void
sql_command_get_error_logger (	void *data, const char *msg,
				int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
		    _("Database error while trying to get the command list"));
	}
}


/*
 * Callback function for the sql_command_get_commands() function.  Add the
 * new command to the list
 *
 * @return Always 0
 */
static int
command_get_list_cb (	void *lst,
			int workload, unsigned long long int job_id,
			const char *user_name, int cmd, int parameter)
{
	lwc_LL *list = (lwc_LL *) lst;
	command_action_ptr ptr;
	void *old;


	ptr = command_action_new (workload, job_id, user_name, cmd, parameter);
	old = NULL;
	lwc_replaceUnsortedLL (list, ptr, command_action_cmp, &old);
	/* Ignore the previous command for the same job/jobset */
	if (old != NULL) {
		/* Say that it's done */
		command_done ((command_action_ptr)old);
		command_action_destroy ((command_action_ptr)old);
	}
	return 0;
}


/*
 * Retrieve and return the commands to be run
 *
 * Return:
 *    The command list.  Each node is a command_action object.  This list must
 *    be freed by the caller by command_destroy_list()
 *   OR
 *    NULL in case of error (a message has been logged by lwc_writeLog())
 */
lwc_LL *
command_get_list ()
{
	lwc_LL *list;
	int ret;


	list = lwc_newLL ();
	ret = sql_command_get_commands (command_get_list_cb, list,
					sql_command_get_error_logger, NULL);
	if (ret != 0) {
		command_destroy_list (list);
		return NULL;
	}
	return list;
}


/*
 * Destroy the provided command list
 */
void
command_destroy_list (lwc_LL *list)
{
	lwc_delLL (list, (void (*)(const void *))command_action_destroy);
}

/*-----------------============== End Of File ==============-----------------*/
