/* Schedwi
   Copyright (C) 2013-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* check_waiting_can_start.c -- Check if a waiting job/jobset can start */

#include <schedwi.h>
#include <module.h>
#include <manual_trigger.h>
#include <testfile.h>
#include <lwc_log.h>
#include <sql_constraint_file.h>
#include <sql_hierarchy.h>
#include <sql_status.h>
#include <sql_common.h>
#include <sql_link.h>
#include <sql_job.h>
#include <check_waiting_can_start.h>


/*
 * Error callback function
 */
static void
sql_get_error_logger (void *data, const char *msg, int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
_("Database error while trying to check if a job/jobset is ready to start"));
	}
}


/**
 * Check if the provided waiting job/jobset can start now or not.  If the
 * job/jobset is not ready to start, then the reason is stored in the
 * job_status database table and ptr->wait_reason is set accordingly.
 *
 * @param[in] ptr Job/jobset details.
 * @param[in] hierarchy_list Hierarchy list (see sql_hierarchy.c) for this
 *                           job/jobset.
 * @param[in] now Current time (as in time(2))
 * @param[in] parent_status Parent jobset status.
 * @return 	0 if the job/jobset can start, 1 if not, 2 if it's too late
 *              to start the job/jobset (start limit) or -1 on error (a message
 *              has been logged).  Also, ptr->wait_reason is updated.
 */
int
check_waiting_can_start (	job_status_node_ptr ptr,
				lwc_LL *hierarchy_list,
				time_t now,
				job_status_state parent_status)
{
	time_t start_time;
	int reason, ret;
	short int start_limit;
	char exist;
	lwc_LL *rows;
	row_item_t *row;


	if (ptr == NULL || hierarchy_list == NULL) {
		return 0;
	}

	reason = 0;

	/* Check if the parent is running */
	if (parent_status != JOB_STATUS_STATE_RUNNING) {
		/* The parent is not running */
		reason |= WAIT_REASON_MASK_PARENT;
	}

	/* Check the start time */
	if (now < hierarchy_list_get_start_time (hierarchy_list)) {
		/* It's not time yet */
		reason |= WAIT_REASON_MASK_TIME;
	}

	/*
	 * Retrieve the start limit parameter for the job/jobset from the
	 * hierarchy.  start_limit is in minutes from the database. 0 means
	 * no limit.
	 */
	start_limit = hierarchy_list_get_start_limit (hierarchy_list);

	/* Get the time at which the job should have started */
	start_time = hierarchy_list_get_real_start_time (hierarchy_list);

	/* Check if it is not too late to start the job */
	if (start_limit > 0) {
		if (	   start_time > 0
			&& difftime (now, start_time + start_limit * 60) > 0)
		{
			/* Too late */
			return 2;
		}
	}

	/* Check if all the links are okay */
	ret = sql_link_are_resolved (	ptr->workload_date, ptr->job_id,
					sql_get_error_logger, NULL);
	if (ret < 0) {
		return -1;
	}
	if (ret > 0) {
		/* Some links are not satisfied */
		reason |= WAIT_REASON_MASK_LINKS;
	}

	/*
	 * Only do the following tests if so far everything is ready for the
	 * job/jobset to start. These tests may take some time so no need
	 * to run them if we know anyway that the job is not going to start.
	 */
	if (reason != 0) {
		/* Update the waiting reason in the database */
		job_status_node_set_wait_reason (ptr, reason);
		return 1;
	}

	/* Check the required files */
	if (sql_constraint_file_get (	ptr->workload_date, &rows,
					ptr->job_id,
					sql_get_error_logger,
					NULL) != 0)
	{
		return -1;
	}
	while ((row = (row_item_t *) lwc_delStartLL (rows)) != NULL) {
		/*
		   row[0] --> job ID
		   row[1] --> host ID
		   row[2] --> filename
		   row[3] --> present or missing
		*/

		/* ret = 1 if file is missing, ret = 0 if file is present */
		ret = testfile (hierarchy_list,
				ptr->workload_date,
				ptr->job_id,
				(unsigned long long int)
					sql_row_item2ll(&(row[1])),
				row[2].value_string,
				NULL);
		if (ret < 0) {
			sql_free_row (row);
			lwc_delLL (rows, (void (*)(const void *))sql_free_row);
			return -1;
		}
		exist = (char) sql_row_item2ll(&(row[3]));
		sql_free_row (row);
		if ((ret == 0 &&  exist == 0) || (ret != 0 &&  exist != 0)) {
			job_status_node_set_wait_reason (ptr, 
					WAIT_REASON_MASK_CONSTRAINT_FILE);
			lwc_delLL (rows, (void (*)(const void *))sql_free_row);
			return 1;
		}
	}
	lwc_delLL (rows, (void (*)(const void *))sql_free_row);

	/* Call the `schedwi_check' function of the modules */
	if (module_check (ptr, start_time, start_limit) != 0) {
		job_status_node_set_wait_reason (ptr, WAIT_REASON_MASK_MODULE);
		return 1;
	}

	/*
	 * Check if it's a manual job and it's waiting for acknowledgement.
	 * If yes, this function will alert the operators (only the first
	 * time it is run for this specific job/jobset)
	 */
	if (check_manual_trigger (	ptr, hierarchy_list,
					start_time, start_limit) != 0)
	{
		job_status_node_set_wait_reason (ptr, WAIT_REASON_MASK_MANUAL);
		return 1;
	}

	job_status_node_set_wait_reason (ptr, 0);
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
