/* Schedwi
   Copyright (C) 2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/**
  * @file check_new_workload.c
  * Check if it's time to load today's workload and load it.
  */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_TIME_H
#include <time.h>
#endif

#include <lwc_log.h>
#include <lwc_linkedlist.h>
#include <sql_hierarchy.h>
#include <sql_common.h>
#include <sql_status.h>
#include <sql_job.h>
#include <schedwi_time.h>
#include <calendar_list.h>
#include <job_status_set.h>
#include <result_mgnt.h>
#include <xmem.h>
#include <check_new_workload.h>


/*
 * Error callback function for the sql_status_check_workload() function
 */
static void
sql_status_check_workload_error_logger (void *data, const char *msg,
					int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
		_("Database error while checking the presence of a workload"));
	}
}


/*
 * Error callback function for the sql_status_delete() function
 */
static void
sql_status_delete_error_logger (void *data, const char *msg, int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
		    _("Database error while trying to delete a workload"));
	}
}


/*
 * Error callback function for the sql_status_copy_to_workload() function
 */
static void
sql_status_copy_error_logger (void *data, const char *msg, int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
	    _("Database error while trying to build the workload"));
	}
}


/*
 * Error callback function for the sql_job_today() and sql_job_tomorrow()
 * functions
 */
static void
sql_list_error_logger (void *data, const char *msg, int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
		_("Database error while trying to retrieve the job list"));
	}
}


/*
 * Error callback function for the sql_job_set_start_time() function
 */
static void
sql_start_time_error_logger (void *data, const char *msg, int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
	_("Database error while updating the start time of a job"));
	}
}


/**
 * For the given jobset, recursively add the jobs and jobsets to run today to
 * the job_status table.
 *
 * @param[in] workload_date Workload date of the job/jobset (YYYYMMDD).
 * @param[in] jobset_id Jobset ID.
 * @param[in] jobset_name_with_path Full jobset name.
 * @param[in] jobset_date Date at which the jobset has to run.
 * @param[in] jobset_start_time Time at which the jobset has to run (HHMM)
 * @param[in] hierarchy_list Hierarchy list (see sql_hierarchy.c) for this
 *                           jobset.
 * @param[in] all_calendars List of all the calendars.
 * @param[in] now Current time (time(2)).
 * @param[in] update_buffer The buffer used to store the database inserts into
 *                          the job_status table so they can be run in bulk.
 *
 * @return 0 on success and -1 on error (a message has already been logged with
 *         lwc_writeLog()).
 */
static int
build_job_status_recur (int workload_date,
			unsigned long long int jobset_id,
			const char *jobset_name_with_path,
			const schedwi_date *jobset_date,
			int jobset_start_time,
			lwc_LL *hierarchy_list,
			calendar_list_t_ptr all_calendars,
			time_t now,
			lwc_LL *update_buffer)
{
	char *calendar_list, *path, *err_msg;
	lwc_LL *job_list;
	row_item_t *child;
	unsigned long long int id;
	int start_time, reason;
	char job_or_jobset;
	size_t parent_name_length;
	result_t *result;
	schedwi_date date_tomorrow;


	/* Add the current jobset to the hierarchy list */
	if (jobset_id != 0) {
		err_msg = NULL;
		if (hierarchy_list_push_job (	hierarchy_list,
						workload_date, jobset_id,
						&err_msg) != 0)
		{
			if (err_msg != NULL) {
				lwc_writeLog (LOG_ERR, err_msg);
				free (err_msg);
			}
			else {
				lwc_writeLog (LOG_ERR,
			_("Database error while retrieving a job details"));
			}
			return -1;
		}
	}


	/*
	 * Today's jobs
	 */

	/* Get the list of calendars matching the given workload */
	calendar_list = get_calendar_list_for_the_day (	all_calendars,
							jobset_date);
	if (sql_job_today (	workload_date,
                         	&job_list,
                         	jobset_id,
                         	calendar_list,
                         	jobset_start_time,
                         	sql_list_error_logger,
                         	NULL) != 0)
	{
		free (calendar_list);
		if (jobset_id != 0) {
			hierarchy_list_pop_job (hierarchy_list);
		}
		return -1;
	}
	free (calendar_list);

	/* Build the tree from the list */
	parent_name_length = strlen (jobset_name_with_path);
	while ((child = (row_item_t *) lwc_delStartLL (job_list)) != NULL) {
		/*
		 * child[0] --> ID
		 * child[1] --> Type (0: jobset and 1: job)
		 * child[2] --> Start time
		 * child[3] --> Job/Jobset name
		 */

		/* Get the job/jobset details */
		id = (unsigned long long int) sql_row_item2ll (&(child[0]));
		job_or_jobset = (char) sql_row_item2ll (&(child[1]));
		start_time = (int) sql_row_item2ll (&(child[2]));

		/* Build the full path name of the job/jobset */
		path = (char *)xmalloc (parent_name_length + child[3].len + 2);
		if (id == ROOT_JOBSET) {
			path[0] = '/';
			path[1] = '\0';
		}
		else {
			if (jobset_id == ROOT_JOBSET) {
				path[0] = '/';
				strcpy (path + 1, child[3].value_string);
			}
			else {
				strcpy (path, jobset_name_with_path);
				path[parent_name_length] = '/';
				strcpy (path + parent_name_length + 1,
					child[3].value_string);
			}
		}
		sql_free_row (child);

		/* It's a jobset */
		if (job_or_jobset == JOBSET) {
			if (build_job_status_recur (	workload_date,
							id,
							path,
							jobset_date,
							start_time,
							hierarchy_list,
							all_calendars,
							now,
							update_buffer) != 0)

			{
				free (path);
				lwc_delLL (job_list,
					(void (*)(const void *)) sql_free_row);
				if (jobset_id != 0) {
					hierarchy_list_pop_job (
							hierarchy_list);
				}
				return -1;
			}
		}

		/* Add the job/jobset to the job_status table */
		if (id == ROOT_JOBSET) {
			reason = WAIT_REASON_MASK_TIME;
		}
		else if (start_time < 0) {
			reason = WAIT_REASON_MASK_PARENT;
		}
		else {
			/*
			reason = WAIT_REASON_MASK_PARENT|WAIT_REASON_MASK_TIME;
			*/
			reason = WAIT_REASON_MASK_TIME;
		}
		result = result_new ();
		result_set_id (result, workload_date, id);
		if (job_status_set_mem_add (	update_buffer,
						result, path,
						JOB_STATUS_STATE_WAITING,
						JOB_STATUS_STATE_UNDEFINED,
						now, reason) != 0)
		{
			result_destroy (result);
			free (path);
			lwc_delLL (	job_list,
					(void (*)(const void *)) sql_free_row);
			if (jobset_id != 0) {
				hierarchy_list_pop_job (hierarchy_list);
			}
			return -1;
		}
		result_destroy (result);
		free (path);

		/*
		 * Update the start time in the job_main_s table to be a
		 * real start time (time(2)) rather than the HHMM value
		 * previously copied from the job_main table.
		 */
		if (	   start_time >=0
			&& sql_job_set_start_time (
				workload_date, id,
				schedwi_time_convert (jobset_date, start_time),
				sql_start_time_error_logger, NULL) != 0)
		{
			lwc_delLL (	job_list,
					(void (*)(const void *)) sql_free_row);
			if (jobset_id != 0) {
				hierarchy_list_pop_job (hierarchy_list);
			}
			return -1;
		}
	}
	lwc_delLL (job_list, (void (*)(const void *)) sql_free_row);


	/*
	 * Tomorrow's jobs
	 * These jobs/jobsets belongs the current jobset but start before
	 * the jobset start time.  That means that they should run the next
	 * day.  For instance:
	 *   - The jobset_start_time (start time of the current jobset) is
	 *     23h00.
	 *   - In this jobset, there is a job with a start time of 03h00.
	 *   - Then the job will be plan for the next day at 03h00.
	 */

	schedwi_date_add_days (jobset_date, &date_tomorrow, 1);
	calendar_list = get_calendar_list_for_the_day (	all_calendars,
							&date_tomorrow);
	if (sql_job_tomorrow (	workload_date,
                         	&job_list,
                         	jobset_id,
                         	calendar_list,
                         	jobset_start_time,
                         	sql_list_error_logger,
                         	NULL) != 0)
	{
		free (calendar_list);
		if (jobset_id != 0) {
			hierarchy_list_pop_job (hierarchy_list);
		}
		return -1;
	}
	free (calendar_list);

	/* Build the tree from the list */
	while ((child = (row_item_t *) lwc_delStartLL (job_list)) != NULL) {
		/*
		 * child[0] --> ID
		 * child[1] --> Type (0: jobset and 1: job)
		 * child[2] --> Start time
		 * child[3] --> Job/Jobset name
		 */

		/* Get the job/jobset details */
		id = (unsigned long long int) sql_row_item2ll (&(child[0]));
		job_or_jobset = (char) sql_row_item2ll (&(child[1]));
		start_time = (int) sql_row_item2ll (&(child[2]));

		/* Build the full path name of the job/jobset */
		path = (char *)xmalloc (parent_name_length + child[3].len + 2);
		if (id == ROOT_JOBSET) {
			path[0] = '/';
			path[1] = '\0';
		}
		else {
			if (jobset_id == ROOT_JOBSET) {
				path[0] = '/';
				strcpy (path + 1, child[3].value_string);
			}
			else {
				strcpy (path, jobset_name_with_path);
				path[parent_name_length] = '/';
				strcpy (path + parent_name_length + 1,
					child[3].value_string);
			}
		}
		sql_free_row (child);

		/* It's a jobset */
		if (job_or_jobset == JOBSET) {
			if (build_job_status_recur (	workload_date,
							id,
							path,
							&date_tomorrow,
							start_time,
							hierarchy_list,
							all_calendars,
							now,
							update_buffer) != 0)
			{
				free (path);
				lwc_delLL (job_list,
					(void (*)(const void *)) sql_free_row);
				if (jobset_id != 0) {
					hierarchy_list_pop_job (
							hierarchy_list);
				}
				return -1;
			}
		}

		/* Add the job/jobset to the job_status table */
		if (id == ROOT_JOBSET) {
			reason = WAIT_REASON_MASK_TIME;
		}
		else if (start_time < 0) {
			reason = WAIT_REASON_MASK_PARENT;
		}
		else {
			reason = WAIT_REASON_MASK_PARENT|WAIT_REASON_MASK_TIME;
		}
		result = result_new ();
		result_set_id (result, workload_date, id);
		if (job_status_set_mem_add (	update_buffer,
						result, path,
						JOB_STATUS_STATE_WAITING,
						JOB_STATUS_STATE_UNDEFINED,
						now, reason) != 0)
		{
			result_destroy (result);
			free (path);
			lwc_delLL (	job_list,
					(void (*)(const void *)) sql_free_row);
			if (jobset_id != 0) {
				hierarchy_list_pop_job (hierarchy_list);
			}
			return -1;
		}
		result_destroy (result);
		free (path);

		/*
		 * Update the start time in the job_main_s table to be a
		 * real start time (time(2)) rather than the HHMM value
		 * previously copied from the job_main table.
		 */
		if (	   start_time >= 0
			&& sql_job_set_start_time (
				workload_date, id,
				schedwi_time_convert (	&date_tomorrow,
							start_time),
				sql_start_time_error_logger, NULL) != 0)
		{
			lwc_delLL (	job_list,
					(void (*)(const void *)) sql_free_row);
			if (jobset_id != 0) {
				hierarchy_list_pop_job (hierarchy_list);
			}
			return -1;
		}
	}
	lwc_delLL (job_list, (void (*)(const void *)) sql_free_row);

	if (jobset_id != 0) {
		hierarchy_list_pop_job (hierarchy_list);
	}
	return 0;
}


/**
 * Add the jobs and jobsets to run today to the job_status table.
 *
 * @param[in] workload_date Workload date of the job/jobset (YYYYMMDD).
 * @param[in] now Time to use (time(2)) as the workload day.
 * @return 0 on success and -1 on error (a message has already been logged with
 *         lwc_writeLog()).
 */
static int
build_job_status (int workload_date, time_t now)
{
	calendar_list_t_ptr all_calendars;
	schedwi_date workload_date2;
	lwc_LL *hierarchy_list, *update_buffer;


	/* Retrieve all the calendars */
	schedwi_date_from_int (workload_date, &workload_date2);
	all_calendars = new_calendar_list (&workload_date2);
	if (all_calendars == NULL) {
		return -1;
	}

	/* Recursively load the jobs and jobsets in the job_status table */
	update_buffer = lwc_newLL ();
	hierarchy_list = lwc_newLL ();
	if (build_job_status_recur (	workload_date, 0, "",
					&workload_date2, 0,
					hierarchy_list,
					all_calendars,
					now,
					update_buffer) != 0)
	{
		hierarchy_list_destroy (hierarchy_list);
		job_status_set_mem_destroy (update_buffer);
		destroy_calendar_list (all_calendars);
		return -1;
	}
	hierarchy_list_destroy (hierarchy_list);
	destroy_calendar_list (all_calendars);
	job_status_set_mem_flush (update_buffer);
	job_status_set_mem_destroy (update_buffer);
	return 0;
}


/**
 * Check if the workload for today is already loaded and load it otherwise.
 *
 * @param[in] today_workload_skip_minutes How long after midnight the workload
 *                                        for today can still be loaded.
 * @return 0 on success or -1 in case of error (a message has already been
 *         logged by lwc_writeLog())
 */
int
check_new_workload (long int today_workload_skip_minutes)
{
	time_t now, m;
	int workload_date, ret;
	static time_t midnight = 0;


	now = time (NULL);
	workload_date = schedwi_time_to_date_int (now);

	/*
	 * Check if the workload for today has already been loaded in the
	 * database.  If yes, nothing else has to be done.
	 */
	ret = sql_status_check_workload (workload_date,
					sql_status_check_workload_error_logger,
					NULL);
	if (ret < 0) {
		return -1;
	}

	if (ret > 0) {
		/* The workload for the day is already loaded */
		return 0;
	}

	/*
	 * Remove the workload from the database (*_s, job_status and a few
	 * other tables - see sql_status_delete()).
	 * Some entries may already exist in those tables if the program
	 * previously crashed during the loading of the tables or if the
	 * database was not available at the time.
	 */
	if (sql_status_delete (	workload_date, sql_status_delete_error_logger,
				NULL) != 0)
	{
		return -1;
	}

	/*
	 * Check how far we are from midnight.
	 * If the workload for today has not been loaded before
	 * `TODAY_WORKLOAD_SKIP' minutes past midnight, then skip the workload
	 * for today.
	 * For example, in case schedwisrv is started for the first time today
	 * in the afternoon, this mechanism prevents all the jobs that
	 * should have been run in the morning to be started at schedwisrv
	 * start time.
	 */
	m = schedwi_last_midnight (now);
	if (m + today_workload_skip_minutes * 60 < now) {
		if (midnight != m) {
			lwc_writeLog (	LOG_INFO,
_("Skipping workload for today: it is too late (more than %d minutes past midnight - see the TODAY_WORKLOAD_SKIP configuration directive)"),
					today_workload_skip_minutes);
			/* To make sure to only log once by day */
			midnight = m;
		}
		return 0;
	}

	/*
	 * Create the workload for today by copying all the tables into their
	 * corresponding table (same name but ending in _s)
	 */
	if (sql_status_copy_to_workload (	workload_date,
						sql_status_copy_error_logger,
						NULL) != 0)
	{
		return -1;
	}

	/*
	 * Load the jobs and jobsets to run in this workload into the
	 * job_status table (with a status of `waiting')
	 */
	if (build_job_status (workload_date, now) != 0) {
		sql_status_delete (workload_date, NULL, NULL);
		return -1;
	}

	return 0;
}

/*------------------------======= End Of File =======------------------------*/
