/* Schedwi
   Copyright (C) 2011-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* cert_utils.c -- Functions around GnuTLS */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#if HAVE_PWD_H
#include <pwd.h>
#endif

#if HAVE_GRP_H
#include <grp.h>
#endif

#if HAVE_CTYPE_H
#include <ctype.h>
#endif

#if HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif

#if HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <utils.h>
#include <lwc_log.h>
#include <conf.h>
#include <xmem.h>
#include <cert_utils.h>


#if GNUTLS_VERSION_NUMBER > 0x020b00
#include <gnutls/abstract.h>
#endif


/*
 * Substitute all the occurences of a character (`oldc') by an other
 * one (`newc') in the provided string.
 *
 * Return:
 *   A newly allocated string with all the occurences of `oldc' replaced
 */
static char *
tr (const char *instr, const char *oldc, const char *newc)
{
	char *outstr;
	size_t i, j, len_newc;


#if HAVE_ASSERT_H
	assert (instr != NULL && oldc != NULL && newc != NULL);
#endif

	outstr = (char *) xmalloc (strlen (instr) + 1);

	len_newc = strlen (newc);
	for (i = 0; instr[i] != '\0'; i++) {
		outstr[i] = instr[i];
		for (j = 0; oldc[j] != '\0'; j++) {
			if (instr[i] == oldc[j]) {
				outstr[i] = (j < len_newc) ? newc[j]
							   : newc[len_newc-1];
				break;
			}
		}
	}
	outstr[i] = '\0';
	return outstr;
}


/*
 * Serialize a PEM encoded string (replace \n by @)
 *
 * Return:
 *   The serialized string (to be freed by free())
 */
char *
PEM2string (const char *pemstr)
{
	return tr (pemstr, "\n\t\a\b\v\f\r", "@$ ");
}


/*
 * Unserialize a PEM encoded string (replace @ by \n)
 *
 * Return:
 *   The unserialized PEM string (to be freed by free())
 */
char *
string2PEM (const char *pemstr)
{
	return tr (pemstr, "@$", "\n\t");
}


/*
 * Change the owner/group of the provided file.  `user' and `group' can be
 * specified numerically or by name.  If `user' is NULL, only the group is set.
 * If `group' is NULL, only the `user' is set.
 *
 * Return:
 *      0 --> OK
 *  other --> Error (errno is set)
 */
static int
my_chown (const char *filename, const char *user, const char *group)
{
	uid_t uid;
	gid_t gid;


	if (filename == NULL) {
		return 0;
	}

	uid = -1;
	get_userid (user, &uid);
	gid = -1;
	get_groupid (group, &gid);

	if (uid == -1 && gid == -1) {
		return 0;
	}
#if HAVE_CHOWN
	return chown (filename, uid, gid);
#else
	return 0;
#endif
}


/*
 * Generata a RSA 2048 bits private key.
 *
 * Return:
 *   The generated private key (to be freed by gnutls_x509_privkey_deinit()) OR
 *   NULL is case of error (a message has been logged)
 */
static gnutls_x509_privkey_t
generate_private_key_int ()
{
	gnutls_x509_privkey_t key;
	int ret;


	ret = gnutls_x509_privkey_init (&key);
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot initialize the private key: %s"),
			gnutls_strerror (ret));
		return NULL;
	}

	ret = gnutls_x509_privkey_generate (key, GNUTLS_PK_RSA, 2048, 0);
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot generate the private key: %s"),
			gnutls_strerror (ret));
		gnutls_x509_privkey_deinit (key);
		return NULL;
	}
	return key;
}


/*
 * Write the private key to a file in PEM format.  Set the owner of the
 * private key file to `user' and `group' (if `user' and `group' are not NULL)
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (a message has been logged)
 */
static int
write_private_key (	gnutls_x509_privkey_t key, const char *outfile,
			const char *user, const char *group)
{
	int ret;
	unsigned char buffer[4 * 1024];
	size_t size;
	FILE *file;


	if (key == NULL || outfile == NULL) {
		return 0;
	}

	size = sizeof (buffer);
	ret = gnutls_x509_privkey_export (key, GNUTLS_X509_FMT_PEM,
					  buffer, &size);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
				_("SSL: Failed to export the private key: %s"),
				gnutls_strerror (ret));
		return -1;
	}

	file = fopen (outfile, "wb");
	if (file == NULL) {
		lwc_writeLog (LOG_ERR, _("%s: %s"), outfile, strerror (errno));
		return -1;
	}
	if (fwrite (buffer, 1, size, file) != size) {
		lwc_writeLog (	LOG_ERR, _("%s: write error: %s"),
				outfile, strerror (errno));
		fclose (file);
		my_unlink (outfile);
		return -1;
	}
	if (fclose(file) != 0) {
		my_unlink (outfile);
		lwc_writeLog (LOG_ERR, _("%s: %s"), outfile, strerror (errno));
		return -1;
	}

#if HAVE_CHMOD && defined S_IRUSR && defined S_IWUSR
	chmod (outfile, S_IRUSR|S_IWUSR);
#endif
	my_chown (outfile, user, group);

	return 0;
}


/*
 * Generate a private key (RSA, 2048 bits) and write it to a file (PEM format).
 * Set the owner of the private key file to `user' and `group' (if `user' and
 * `group' are not NULL)
 *
 * Return:
 *   The generated private key (to be freed by gnutls_x509_privkey_deinit()) OR
 *   NULL is case of error (a message has been logged)
 */
gnutls_x509_privkey_t
generate_private_key (const char *outfile, const char *user, const char *group)
{
	gnutls_x509_privkey_t key;


#if HAVE_ASSERT_H
	assert (outfile != NULL);
#endif

	key = generate_private_key_int ();
	if (key == NULL) {
		return NULL;
	}

	if (write_private_key (key, outfile, user, group) != 0) {
		gnutls_x509_privkey_deinit (key);
		return NULL;
	}

	return key;
}


/*
 * Convert a printable IP to binary.
 *
 * Return:
 *    The address len set in ip OR
 *    -1 in case of invalid address
 */
static int
string_to_ip (unsigned char *ip, const char *str)
{
	int len = strlen (str);


#if defined AF_INET6
	if (strchr (str, ':') != NULL || len > 16)
	{	/* IPv6 */
		if (inet_pton (AF_INET6, str, ip) <= 0) {
			return -1;
		}

		return 16;
	}
	else
#endif
	{	/* IPv4 */
		if (inet_pton (AF_INET, str, ip) <= 0) {
			return -1;
		}

		return 4;
	}
}


/*
 * Generage a certificate signing request in PEM format.  The NULL terminated
 * `dnsnames' array must contain all the names of the server.  The first
 * name will be used as the common name (CN).
 *
 * Return:
 *   The certificate signing request (to be freed by gnutls_x509_crq_deinit) OR
 *   NULL in case of error (a message has been logged)
 */
gnutls_x509_crq_t
generate_request (gnutls_x509_privkey_t key,
		  char **dnsnames, const char *ip)
{
	gnutls_x509_crq_t crq;
	int ret, len, i;
	unsigned char net_ip[20];
	const char *conf_c, *conf_o, *conf_ou, *conf_l, *conf_st;


#if HAVE_ASSERT_H
	assert (dnsnames != NULL && *dnsnames != NULL && **dnsnames != '\0');
#endif

	ret  = conf_get_param_string ("SSLCACountry", &conf_c);
	ret += conf_get_param_string ("SSLCAOrganization", &conf_o);
	ret += conf_get_param_string ("SSLCAUnit", &conf_ou);
	ret += conf_get_param_string ("SSLCALocality", &conf_l);
	ret += conf_get_param_string ("SSLCAState", &conf_st);
#if HAVE_ASSERT_H
	assert (ret == 0);
#endif

	ret = gnutls_x509_crq_init (&crq);
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot initialize the certificate request: %s"),
			gnutls_strerror (ret));
		return NULL;
	}

	/* Set the DN */
	ret = gnutls_x509_crq_set_dn_by_oid (crq,
				GNUTLS_OID_X520_COUNTRY_NAME, 0,
				conf_c, strlen (conf_c));
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot set the COUNTRY (%s): %s"),
			conf_c, gnutls_strerror (ret));
		gnutls_x509_crq_deinit (crq);
		return NULL;
	}

	ret = gnutls_x509_crq_set_dn_by_oid (crq,
			GNUTLS_OID_X520_ORGANIZATION_NAME, 0,
			conf_o, strlen (conf_o));
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot set the ORGANIZATION (%s): %s"),
			conf_o, gnutls_strerror (ret));
		gnutls_x509_crq_deinit (crq);
		return NULL;
	}

	ret = gnutls_x509_crq_set_dn_by_oid (crq,
				GNUTLS_OID_X520_ORGANIZATIONAL_UNIT_NAME, 0,
				conf_ou, strlen (conf_ou));
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot set the ORGANIZATIONAL UNIT (%s): %s"),
			conf_ou, gnutls_strerror (ret));
		gnutls_x509_crq_deinit (crq);
		return NULL;
	}

	ret = gnutls_x509_crq_set_dn_by_oid (crq,
				GNUTLS_OID_X520_LOCALITY_NAME, 0,
				conf_l, strlen (conf_l));
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot set the LOCALITY (%s): %s"),
			conf_l, gnutls_strerror (ret));
		gnutls_x509_crq_deinit (crq);
		return NULL;
	}

	ret = gnutls_x509_crq_set_dn_by_oid (crq,
				GNUTLS_OID_X520_STATE_OR_PROVINCE_NAME, 0,
				conf_st, strlen (conf_st));
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot set the STATE (%s): %s"),
			conf_st, gnutls_strerror (ret));
		gnutls_x509_crq_deinit (crq);
		return NULL;
	}

	/* The first name in the array is used as CN */
	ret = gnutls_x509_crq_set_dn_by_oid (crq,
					GNUTLS_OID_X520_COMMON_NAME, 0,
					*dnsnames, strlen (*dnsnames));
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot set the COMMON NAME (%s): %s"),
			*dnsnames , gnutls_strerror (ret));
		gnutls_x509_crq_deinit (crq);
		return NULL;
	}

	/*
	 * Add the names as alternative names.
	 * The crq buffer size is limited - tests have shown that only around
	 * 115 bytes for the names are available.
	 * So, we compute first the size of the names in dnsnames.  If too many
	 * data have to be stored, we'll then try to choose the best we can
	 * the ones that are going to make it in the certificate.
	 */
	len = 	  strlen (dnsnames[0]) * 2 /* In the CN and alt name */
		+ ((ip != NULL) ? strlen (ip) : 0);
	for (i = 1; dnsnames[i] != NULL; i++) {
		len += strlen (dnsnames[i]);
	}

	for (i = 0; dnsnames[i] != NULL; i++) {
		if (dnsnames[i][0] != '\0') {
			/* Too much data to store so let's select */
			if (len > 115) {
				if (   isdigit (dnsnames[i][0]) != 0
				    || strchr (dnsnames[i], ':') /* IPv6 */
				    || strcmp (dnsnames[i], "localhost") == 0
				    || strcmp (dnsnames[i], "localhost6") == 0)
				{
					len -= strlen (dnsnames[i]);
					if (   (ip == NULL || *ip == '\0')
					    && isdigit (dnsnames[i][0]) != 0)
					{
						/*
						 * Maybe there will still be
						 * place to store an IP address
						 */
						ip = dnsnames[i];
					}
					continue;
				}
			}
	 		ret = gnutls_x509_crq_set_subject_alt_name (crq,
						GNUTLS_SAN_DNSNAME,
						dnsnames[i],
						strlen (dnsnames[i]),
						GNUTLS_FSAN_APPEND);
			if (ret < 0) {
				lwc_writeLog (LOG_ERR,
				_("SSL: Cannot set a dns name (%s): %s"),
					dnsnames[i], gnutls_strerror (ret));
				gnutls_x509_crq_deinit (crq);
				return NULL;
			}
		}
	}

	/* IP */
	if (ip != NULL && *ip != '\0' && len <= 115) {
		len = string_to_ip (net_ip, ip);
		if (len < 0) {
			lwc_writeLog (LOG_ERR,
				_("SSL: Error parsing IP address %s"),
				ip);
			gnutls_x509_crq_deinit (crq);
			return NULL;
		}
		ret = gnutls_x509_crq_set_subject_alt_name (crq,
						GNUTLS_SAN_IPADDRESS,
						net_ip, len,
						GNUTLS_FSAN_APPEND);
		if (ret < 0) {
			lwc_writeLog (LOG_ERR,
				_("SSL: Cannot set the IP address (%s): %s"),
				ip, gnutls_strerror (ret));
			gnutls_x509_crq_deinit (crq);
			return NULL;
		}
	}

	/* Email */
/* As the crq buffer has a limited size, just skip the useless Email

	s = (char *) xmalloc(strlen(CSR_EMAIL_PREFIX) + strlen(*dnsnames) + 1);
	strcpy (s, CSR_EMAIL_PREFIX);
	strcat (s, *dnsnames);
	ret = gnutls_x509_crq_set_subject_alt_name (crq, GNUTLS_SAN_RFC822NAME,
						s, strlen (s),
						GNUTLS_FSAN_APPEND);
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot set the email address (%s): %s"),
			s, gnutls_strerror (ret));
		free (s);
		gnutls_x509_crq_deinit (crq);
		return NULL;
	}
	free (s);
*/

	/* This is NOT a CA's certificate request */
	ret = gnutls_x509_crq_set_basic_constraints (crq, 0, -1);
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot set basic constraints: %s"),
			gnutls_strerror (ret));
		gnutls_x509_crq_deinit (crq);
		return NULL;
	}

	/* Certificate usage */
	ret = gnutls_x509_crq_set_key_usage (crq,
		GNUTLS_KEY_KEY_ENCIPHERMENT | GNUTLS_KEY_DIGITAL_SIGNATURE);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
				_("SSL: Cannot specify the key usage: %s"),
				gnutls_strerror (ret));
		gnutls_x509_crq_deinit (crq);
		return NULL;
	}

	ret = gnutls_x509_crq_set_key_purpose_oid (crq,
						 GNUTLS_KP_TLS_WWW_CLIENT, 0);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
				_("SSL: Cannot specify the key purpose: %s"),
				gnutls_strerror (ret));
		gnutls_x509_crq_deinit (crq);
		return NULL;
	}

	ret = gnutls_x509_crq_set_key_purpose_oid (crq,
						 GNUTLS_KP_TLS_WWW_SERVER, 0);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
				_("SSL: Cannot specify the key purpose: %s"),
				gnutls_strerror (ret));
		gnutls_x509_crq_deinit (crq);
		return NULL;
	}

	/* Set the key */
	ret = gnutls_x509_crq_set_key (crq, key);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
				_("SSL: Cannot set the key: %s"),
				gnutls_strerror (ret));
		gnutls_x509_crq_deinit (crq);
		return NULL;
	}

	/* Signing */
#if GNUTLS_VERSION_NUMBER <= 0x020b00
	ret = gnutls_x509_crq_sign (crq, key);
#else
	{
		gnutls_privkey_t abs_key;

		ret = gnutls_privkey_init (&abs_key);
		if (ret < 0) {
			lwc_writeLog (	LOG_ERR,
			_("SSL: Cannot initialize the abstract key: %s"),
					gnutls_strerror (ret));
			gnutls_x509_crq_deinit (crq);
			return NULL;
		}
		ret = gnutls_privkey_import_x509 (abs_key, key, 0);
		if (ret < 0) {
			lwc_writeLog (	LOG_ERR,
				_("SSL: Cannot import the x509 key: %s"),
					gnutls_strerror (ret));
			gnutls_privkey_deinit (abs_key);
			gnutls_x509_crq_deinit (crq);
			return NULL;
		}
		ret = gnutls_x509_crq_privkey_sign (	crq, abs_key,
							GNUTLS_DIG_SHA256, 0);
		gnutls_privkey_deinit (abs_key);
	}
#endif
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
			_("SSL: Failed to sign the certificat request: %s"),
				gnutls_strerror (ret));
		gnutls_x509_crq_deinit (crq);
		return NULL;
	}

	return crq;
}


	/*
	 * Check the hostname.  One could use the provided
	 * gnutls_x509_crt_check_hostname() function.  However, we will check
	 * all the provided names.
	 * The following lines are then inspired from this function
	 * in GNUTLS (see rfc2818_hostname.c)
	 */
/*
 * Compares the provided hostnames with certificate's hostname
 * cert: should contain an gnutls_x509_crt_t structure
 * hostnames: A null terminated array of DNS name
 *
 * This function will check if the given certificate/request's subject matches
 * the given hostnames.  This is a basic implementation of the matching
 * described in RFC2818 (HTTPS), which uses the DNSName/IPAddress subject
 * alternative name PKIX extension.
 *
 * Returns: the (index + 1) in hostnames of the matching name OR
 *          zero on failure.
 */
int
my_gnutls_x509_crt_check_hostname (	void *cert_or_crq, type_data t,
					const char *const *hostnames)
{
	char dnsname[512];
	size_t dnsnamesize;
	char found_dnsname = 0;
	int ret = 0;
	int i, j;
	gnutls_x509_crt_t cert;
	gnutls_x509_crq_t crq;


	if (t == CERT) {
		cert = (gnutls_x509_crt_t) cert_or_crq;
		crq = NULL;
	}
	else {
		crq = (gnutls_x509_crq_t) cert_or_crq;
		cert = NULL;
	}

	/*
	 * Try matching against:
	 *  1) a DNS name as an alternative name (subjectAltName) extension in
	 *     the certificate
	 *  2) the common name (CN) in the certificate
	 */

	/*
	 * Check through all included subjectAltName extensions, comparing
	 * against all those of type DNSName and IPAddress.
	 */
	for (i = 0; ret >= 0; i++) {
		dnsnamesize = sizeof (dnsname);
		if (t == CERT) {
			ret = gnutls_x509_crt_get_subject_alt_name (cert, i,
							dnsname, &dnsnamesize,
							NULL);
		}
		else {
			ret = gnutls_x509_crq_get_subject_alt_name (crq, i,
							dnsname, &dnsnamesize,
							NULL, NULL);
		}
		if (ret == GNUTLS_SAN_DNSNAME) {
			found_dnsname = 1;
			for (j = 0; hostnames[j] != NULL; j++) {
				if (strcasecmp (hostnames[j], dnsname) == 0) {
					return (j + 1);
				}
			}
		}
#if HAVE_INET_NTOP
		else if (ret == GNUTLS_SAN_IPADDRESS) {
			char name[50];

			found_dnsname = 1;
			if (inet_ntop ((dnsnamesize == 4) ? AF_INET : AF_INET6,
					dnsname, name, 50) != NULL)
			{
				for (j = 0; hostnames[j] != NULL; j++) {
					if (strcasecmp (hostnames[j],
							name) == 0)
					{
						return (j + 1);
					}
				}
			}
		}
#endif
	}

	/* Check against the CN */
	if (!found_dnsname) {
		dnsnamesize = sizeof (dnsname);
		if (t == CERT) {
			ret = gnutls_x509_crt_get_dn_by_oid (cert,
						GNUTLS_OID_X520_COMMON_NAME, 0,
					 	0, dnsname, &dnsnamesize);
		}
		else {
			ret = gnutls_x509_crq_get_dn_by_oid (crq,
						GNUTLS_OID_X520_COMMON_NAME, 0,
					 	0, dnsname, &dnsnamesize);
		}
		if (ret != 0) {
			/* Got an error, can't find a name */
			return 0;
		}
		for (j = 0; hostnames[j] != NULL; j++) {
			if (strcasecmp (hostnames[j], dnsname) == 0) {
				return (j + 1);
			}
		}
	}

	/* Cannot find a matching name */
	return 0;
}


/*
 * Load a private key from a file.
 *
 * Return:
 *   The private key (to be freed by gnutls_x509_privkey_deinit()) OR
 *   NULL in case of error (a message has been logged)
 */
gnutls_x509_privkey_t
load_private_key (const char *infile)
{
	gnutls_x509_privkey_t key;
	int ret;
	gnutls_datum_t dat;
	size_t size;


	ret = gnutls_x509_privkey_init (&key);
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot initialize the private key: %s"),
			gnutls_strerror (ret));
		return NULL;
	}

	dat.data = (unsigned char *)read_file (infile, &size);
	dat.size = size;
	if (dat.data == NULL) {
		lwc_writeLog (	LOG_ERR, _("%s: read error: %s"),
				infile, strerror (errno));
		gnutls_x509_privkey_deinit (key);
		return NULL;
	}

	ret = gnutls_x509_privkey_import (key, &dat, GNUTLS_X509_FMT_PEM);
	/* Try in DER format */
	if (   ret < 0
	    && gnutls_x509_privkey_import (key, &dat, GNUTLS_X509_FMT_DER) < 0)
	{
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot import the private key from %s: %s"),
			infile, gnutls_strerror (ret));
		free (dat.data);
		gnutls_x509_privkey_deinit (key);
		return NULL;
	}
	free (dat.data);

	return key;
}

/*-----------------============== End Of File ==============-----------------*/
